/* File: CALJOY.C
** Description:
**   Demonstrates two methods for calibrating a joystick.
**   The first method simply assumes that the user is holding
**   they joystick in the center.  The second method asks the
**   user to move the joystick to the upper-left and lower-right
**   corners and hit a button.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <assert.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include "gamedefs.h"
#include "jstick.h"

/*
	Function: WaitJoystickButton
    Description:
    	Waits for a joystick button to be pressed and then
        returns the current X and Y position of the joystick.
        Assumes that a joystick is installed and joystick A
        is to be read.
*/
void WaitJoystickButton
	(
    JOYSTICK_STATE * JsState,
	UINT16 * x,
    UINT16 * y
    )
{

	assert(JsState != NULL);
	assert(x != NULL);
    assert(y != NULL);

    /* wait until both buttons aren't pressed */
    /* which prevents us from detecting another button */
    /* press when the user is just holding a button down */
    /* from the last time */
    do {
    	ReadJoysticks(JsState);
    } while (JsState->JsButtonA1 || JsState->JsButtonA2);

    /* read joystick until button is pressed */
	do {
    	ReadJoysticks(JsState);
    } while (! (JsState->JsButtonA1 || JsState->JsButtonA2));

    /* return x and y values when button was pressed */
    *x = JsState->JsAxisAX;
    *y = JsState->JsAxisAY;

    printf("%d, %d\n", JsState->JsAxisAX, JsState->JsAxisAY);
}

/*
	Function: GetJsCorners
    Description:
    	Asks the user to move the joystick to the upper left and
        lower right corners of its range and press a button.
        The function returns the minimum and maximum numbers
        received.
*/
void GetJsCorners
	(
    JOYSTICK_STATE * JsState,
    UINT16 * Xmin,
    UINT16 * Ymin,
    UINT16 * Xmax,
    UINT16 * Ymax
    )
{
	UINT16 XminTemp;	/* Temp variables are used to store the */
    UINT16 YminTemp;	/* results until error checking can be  */
    UINT16 XmaxTemp;	/* done on the values. */
    UINT16 YmaxTemp;
    BOOL Miscalibration;

    assert(JsState != NULL);
    assert(Xmin != NULL);
    assert(Ymin != NULL);
    assert(Xmax != NULL);
    assert(Ymax != NULL);

    Miscalibration = FALSE;

	do {
    	if (Miscalibration) {
        	printf("MISCALIBRATION!! Try again.\n");
        }
    	printf("Move joystick to upper left and hit a button.\n");
        WaitJoystickButton(JsState, &XminTemp, &YminTemp);

        printf("Move joystick to lower right and hit a button.\n");
        WaitJoystickButton(JsState, &XmaxTemp, &YmaxTemp);

        /* if the values don't make sense, indicate miscalibration */
        Miscalibration = FALSE;
        if (XminTemp > XmaxTemp) {
        	Miscalibration = TRUE;
        }
        if (YminTemp > YmaxTemp) {
        	Miscalibration = TRUE;
        }
    } while (Miscalibration);

    /* return values to caller */
    *Xmin = XminTemp;
    *Ymin = YminTemp;
    *Xmax = XmaxTemp;
    *Ymax = YmaxTemp;
}

/*
	Function: CalibrateJsCorners
    Description:
    	Gets the value of the joystick at the upper left and
        lower right corners and computes a midpoint point.
        Returns minimum, midpoint, and maximum value for
        the joystick.
*/
void CalibrateJsCorners
	(
    JOYSTICK_STATE * JsState,
    UINT16 * Xmin,
    UINT16 * Ymin,
    UINT16 * Xmid,
    UINT16 * Ymid,
    UINT16 * Xmax,
    UINT16 * Ymax
    )
{
	assert(JsState != NULL);
    assert(Xmin != NULL);
    assert(Ymin != NULL);
    assert(Xmid != NULL);
    assert(Ymid != NULL);
    assert(Xmax != NULL);
    assert(Ymax != NULL);

	/* get the corner points.  Note that we pass the pointers */
    /* straight through.  Don't pass addresses by mistake */
    /* (e.g., &Xmin). */
    GetJsCorners(JsState, Xmin, Ymin, Xmax, Ymax);

    /* calculate midpoint as average of min and max points */
    *Xmid = (*Xmin + *Xmax) / 2;
    *Ymid = (*Ymin + *Ymax) / 2;
}

/*
	Function: CalibrateJsMidpoint
    Description:
    	Calibrates a joystick's range using the midpoint method.
        Read's the value of the joystick at the time it is
        called and computes a min and max range from this.
        Note that this assumes that the joystick is centered
        when the routine is called.
*/
void CalibrateJsMidpoint
	(
    JOYSTICK_STATE * JsState,
    UINT16 * Xmin,
    UINT16 * Ymin,
    UINT16 * Xmid,
    UINT16 * Ymid,
    UINT16 * Xmax,
    UINT16 * Ymax
    )
{
	assert(JsState != NULL);
    assert(Xmin != NULL);
    assert(Ymin != NULL);
    assert(Xmid != NULL);
    assert(Ymid != NULL);
    assert(Xmax != NULL);
    assert(Ymax != NULL);

	/* read midpoint value */
    ReadJoysticks(JsState);
    *Xmid = JsState->JsAxisAX;
    *Ymid = JsState->JsAxisAY;

    /* assume min coordinate is (0,0) and max is twice midpoint */
    *Xmin = *Ymin = 0;

    *Xmax = 2 * *Xmid;
    *Ymax = 2 * *Ymid;
}

int main(void)
{
	UINT16 Xmin;
    UINT16 Ymin;
    UINT16 Xmax;
    UINT16 Ymax;
    UINT16 Xmid;
    UINT16 Ymid;
    JOYSTICK_STATE JsState;

    /* must call this before reading joysticks! */
    SenseJoysticks(&JsState);

	printf("\n\n*** Joystick calibration ***\n\n");

    if ((JsState.JsMask & 0x3) == 0) {
    	printf("Joystick A not installed.  Exiting...\n");
        exit(1);
    }

    /* do it by assuming joystick at midpoint and performing */
    /* a sample read */
    printf("Midpoint method:\n");
    CalibrateJsMidpoint(&JsState, &Xmin, &Ymin, &Xmid, &Ymid, &Xmax, &Ymax);
    printf("Upper left: (%d, %d)  Lower right: (%d, %d)  "
		"Midpoint: (%d, %d)\n",
    	Xmin, Ymin, Xmax, Ymax, Xmid, Ymid);

    /* do it by corners method */
    printf("Corners method:\n");
    CalibrateJsCorners(&JsState, &Xmin, &Ymin, &Xmid, &Ymid, &Xmax, &Ymax);
    printf("Upper left: (%d, %d)  Lower right: (%d, %d)  "
		"Midpoint: (%d, %d)\n",
    	Xmin, Ymin, Xmax, Ymax, Xmid, Ymid);

	return 0;
}
