/* File: ALIEN3.C
** Description:
**   Alien Alley, version 3.  Demonstration game.  Sound and scrolling
**   have been added to the basic game as done in ALIEN1.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>	/* for randomize() */
#include "gamedefs.h"
#include "animate.h"
#include "bitblt.h"
#include "collide.h"
#include "digmidif.h"
#include "digplay.h"
#include "jstick.h"
#include "keyboard.h"
#include "midpak.h"
#include "mouse.h"
#include "palette.h"
#include "pcx.h"
#include "retrace.h"
#include "setmodex.h"
#include "timer.h"
#include "vga.h"

/* CONSTANTS */
#define MOUSE_RANGE_Y		(MODE13H_HEIGHT)
#define MOUSE_RANGE_X		(2 * MODE13H_WIDTH)
#define MOUSE_THRESHOLD_UP	(MOUSE_RANGE_Y / 3)
#define MOUSE_THRESHOLD_DOWN (2 * (MOUSE_RANGE_Y / 3))
#define MOUSE_THRESHOLD_LEFT (MOUSE_RANGE_X / 3)
#define MOUSE_THRESHOLD_RIGHT (2 * (MOUSE_RANGE_X / 3))
#define INTRO_TEXT_COLOR	0x4
#define MAX_ALIENS			4
#define MAX_ALIEN_MISSILES	20
#define MAX_HERO_MISSILES	10
#define MAX_EXPLOSIONS		(MAX_ALIENS + 1) /* +1 for hero */
#define MAX_EXPLOSION_BITMAPS 5
#define ERASE_COLOR			0	/* color to use to erase sprites */
#define GUN_COLOR			8
#define GUN_BLINK_RATE		10
#define MAX_OBJECT_WIDTH	32
#define MAX_OBJECT_HEIGHT	32
#define HERO_X_VELOCITY		4 /* was 3 */
#define HERO_Y_VELOCITY		4 /* was 3 */
#define ALIEN_X_VELOCITY	3 /* was 2 */
#define ALIEN_Y_VELOCITY	3 /* was 2 */
#define HERO_MISSILE_VELOCITY	8 /* was 5 */
#define ALIEN_MISSILE_VELOCITY	6 /* was 4 */
#define MAX_MOVE_STEP		8	/* max sprite movement (hero missile) */
#define ALIEN_MOVE_TIME_VAR		50
#define ALIEN_MOVE_TIME_BASE	20
#define ALIEN_GEN_RATE_BASE		40
#define ALIEN_GEN_RATE_VAR		40
#define ALIEN_FIRE_LOCKOUT		60
#define ALIEN_FIRE_PROB_HERO	20
#define ALIEN_FIRE_PROB_RANDOM	10
#define ALIEN_PROX_THRESHOLD	20
#define HERO_GUN_OFFSET_LEFT	12
#define HERO_GUN_OFFSET_RIGHT	11
#define HERO_GUN_OFFSET_UP		4
#define ALIEN_GUN_OFFSET_LEFT	11
#define ALIEN_GUN_OFFSET_RIGHT	10
#define ALIEN_GUN_OFFSET_DOWN	3
#define DEATH_DELAY				60 /* 1 sec delay after player death */
#define POINTS_PER_ALIEN		10
#define MAX_HERO_SHIELDS		5
#define STATUS_HEIGHT		30	/* height of the bottom status window */
#define SHIELD_STATUS_COLOR		47
#define SHIELD_STATUS_INVERT_COLOR 173
#define STATUS_VERT_BORDER		5
#define STATUS_HORIZ_BORDER		8
#define SHIELDS_TEXT_LEFT		(STATUS_HORIZ_BORDER + 16)
#define SHIELDS_TEXT_TOP		(STATUS_VERT_BORDER + 5)
#define SCORE_TEXT_LEFT			190
#define SCORE_TEXT_TOP			(STATUS_VERT_BORDER + 7)
#define SCORE_NUMBERS_LEFT		237
#define SCORE_NUMBERS_TOP		(STATUS_VERT_BORDER + 7)
#define SHIELD_STATUS_WIDTH_MULT 2
#define SHIELD_STATUS_HEIGHT	10
#define SHIELD_STATUS_LEFT		96
#define SHIELD_STATUS_TOP       (STATUS_VERT_BORDER + 5)
#define SHIELD_STATUS_RIGHT		(SHIELD_STATUS_LEFT + (MAX_HERO_SHIELDS * \
	4 * SHIELD_STATUS_WIDTH_MULT) - 1)
#define SHIELD_STATUS_BOTTOM	(SHIELD_STATUS_TOP + \
	SHIELD_STATUS_HEIGHT - 1)
#define SHIELD_STATUS_OUTLINE_COLOR 7
#define STATUS_BACKGROUND_COLOR	27
#define EXPLOSION_FRAME_REPEAT_COUNT 3
#define HIGH_SCORE_TEXT_LEN		20
#define HIGH_SCORE_FILENAME		"AAHSCORE.DAT"
#define HIGH_SCORE_LINE			7
#define HIGH_SCORE_COLOR        2
#define SOUND_FX_FREQUENCY		11000
#define MAP_WIDTH				10 /* in tiles */
#define MAP_HEIGHT				32
#define TILE_WIDTH				32 /* in pixels */
#define TILE_HEIGHT				32
#define NUM_TILES				3
#define COUNTER_FRAME_TIME		17042 /* # of 1.193 MHz ticks in 1/70 sec. */

/* screen parameters */
#define SCREEN_WIDTH			MODEY_WIDTH
#define SCREEN_HEIGHT			MODEY_HEIGHT
#define SCREEN_WIDTH_BYTES		(SCREEN_WIDTH / 4)
#define VERTICAL_BORDER_LINES	(MAX_OBJECT_HEIGHT + MAX_MOVE_STEP)
#define VERTICAL_BORDER_BYTES	(VERTICAL_BORDER_LINES * SCREEN_WIDTH_BYTES)
#define REDUCED_SCREEN_HEIGHT	(SCREEN_HEIGHT - STATUS_HEIGHT)
#define REDUCED_SCREEN_BYTES	(REDUCED_SCREEN_HEIGHT * SCREEN_WIDTH_BYTES)
#define STATUS_BYTES			(STATUS_HEIGHT * SCREEN_WIDTH_BYTES)

/* scrolling parameters */
#define SCROLL_DISTANCE			2 /* = twice actual scroll per frame */
#define SCREEN_TOP_MIN			(STATUS_HEIGHT + VERTICAL_BORDER_LINES)
#define SCREEN_TOP_REINIT_LINES	(STATUS_HEIGHT + \
	(5 * VERTICAL_BORDER_LINES) + \
	(2 * REDUCED_SCREEN_HEIGHT))
#define SCREEN_TOP_REINIT_BYTES	(SCREEN_TOP_REINIT_LINES * \
	SCREEN_WIDTH_BYTES)
#define DUPLICATE_REGION_LINE	(STATUS_HEIGHT + VERTICAL_BORDER_LINES + \
	REDUCED_SCREEN_HEIGHT)
#define SCREEN_0_INIT_LINES		(STATUS_HEIGHT + \
	(3 * VERTICAL_BORDER_LINES) + REDUCED_SCREEN_HEIGHT - \
    (SCROLL_DISTANCE / 2))
#define SCREEN_0_INIT_BYTES		(SCREEN_0_INIT_LINES * SCREEN_WIDTH_BYTES)
#define SCREEN_1_INIT_LINES		SCREEN_TOP_REINIT_LINES
#define SCREEN_1_INIT_BYTES		(SCREEN_1_INIT_LINES * SCREEN_WIDTH_BYTES)
#define FREE_MEM_START_LINES	(SCREEN_TOP_REINIT_LINES + \
	REDUCED_SCREEN_HEIGHT + VERTICAL_BORDER_LINES)
#define FREE_MEM_START_BYTES    (FREE_MEM_START_LINES * SCREEN_WIDTH_BYTES)

/* STRUCTURES AND TYPES */
typedef struct {
	BOOL	Active;
    int		x;
    int		y;
    int		vx;
    int		vy;
    int		OldX[2];
    int		OldY[2];
    BOOL	Erase[2];
    BOOL	Draw;
    unsigned ObjectSpec;		/* object specific use */
    unsigned ObjectSpec2;
    RECT	Bounds;
} SPRITE;

typedef struct {
	char Text[HIGH_SCORE_TEXT_LEN + 1];
	int Score;
} HIGH_SCORE;

/* GLOBAL VARIABLES */
BOOL				MousePresent;
BOOL				JoystickPresent;
JOYSTICK_STATE		JsState;
UINT16				JsThresholdUp;
UINT16				JsThresholdDown;
UINT16				JsThresholdLeft;
UINT16				JsThresholdRight;
UINT8				BlackPalette[256][3];
UINT8				GamePalette[256][3];
int					Score;
SPRITE				Hero;
SPRITE				Alien[MAX_ALIENS];
SPRITE				HeroMissile[MAX_HERO_MISSILES];
SPRITE				AlienMissile[MAX_ALIEN_MISSILES];
SPRITE				Explosion[MAX_EXPLOSIONS];
PLANAR_BITMAP far *	HeroBitmap;
COLLISION_MAP far *	HeroCollisionMap;
PLANAR_BITMAP far *	AlienBitmap;
COLLISION_MAP far *	AlienCollisionMap;
PLANAR_BITMAP far *	MissileBitmap;
COLLISION_MAP far *	MissileCollisionMap;
PLANAR_BITMAP far * MissileTrailUpBitmap;
PLANAR_BITMAP far * MissileTrailDnBitmap;
PLANAR_BITMAP far * ExplosionBitmap[MAX_EXPLOSION_BITMAPS];
PLANAR_BITMAP far * Numbers[10];
unsigned			HeroWidth, HeroHeight;
unsigned			HeroXMin, HeroXMax;
unsigned			HeroYMin, HeroYMax;
unsigned			AlienWidth, AlienHeight;
unsigned			MissileWidth, MissileHeight;
unsigned			MissileTrailWidth, MissileTrailHeight;
int					HiddenPage;
UINT16				PageOffset[2];
int					AlienGenCounter;
int					HeroShields;
HIGH_SCORE			HighScore[10];
BOOL				SoundPresent;
UINT8 far *			MidiAddr;
long				MidiLength;
SNDSTRUC			ExplosionSound;
SNDSTRUC			LaserSound;
int					MapLine[2];
int					MemLine[2];
unsigned			ExplosionWidth, ExplosionHeight;
int 				Map[MAP_WIDTH][MAP_HEIGHT];
VIDEO_MEM_BITMAP far * Tile[NUM_TILES];
long				TickCount;

void DeInitSound(void);

/*
	Function: FatalError
    Description:
    	Handle a fatal error.
*/
void FatalError(char * Error, char * Routine, int Line)
{
	if (SoundPresent) {
    	DeInitSound();
    }
	SetVGAMode(0x3);
    fprintf(stderr, "ERROR: %s in %s, line %d!\n", Error, Routine, Line);
    exit(1);
}

/*
	Function: CalibrateJsMidpoint
    Description:
    	Calibrates a joystick's range using the midpoint method.
        Read's the value of the joystick at the time it is
        called and computes a min and max range from this.
        Note that this assumes that the joystick is centered
        when the routine is called.
*/
void CalibrateJsMidpoint
	(
    JOYSTICK_STATE * JsState,
    UINT16 * Xmin,
    UINT16 * Ymin,
    UINT16 * Xmid,
    UINT16 * Ymid,
    UINT16 * Xmax,
    UINT16 * Ymax
    )
{
	assert(JsState != NULL);
    assert(Xmin != NULL);
    assert(Ymin != NULL);
    assert(Xmid != NULL);
    assert(Ymid != NULL);
    assert(Xmax != NULL);
    assert(Ymax != NULL);

	/* read midpoint value */
    ReadJoysticks(JsState);
    *Xmid = JsState->JsAxisAX;
    *Ymid = JsState->JsAxisAY;

    /* assume min coordinate is (0,0) and max is twice midpoint */
    *Xmin = *Ymin = 0;

    *Xmax = 2 * *Xmid;
    *Ymax = 2 * *Ymid;
}

/*
	Function: DefaultHighScores
    Description:
    	Fills in the HighScore array with some defaults.
        Have fun with this.
*/
void DefaultHighScores(void)
{
	HighScore[0].Text[0] = '\0';
    strcat(HighScore[0].Text, "George Washington");
    HighScore[0].Score = 100;

	HighScore[1].Text[0] = '\0';
    strcat(HighScore[1].Text, "John Adams");
    HighScore[1].Score = 90;

	HighScore[2].Text[0] = '\0';
    strcat(HighScore[2].Text, "Thomas Jefferson");
    HighScore[2].Score = 80;

	HighScore[3].Text[0] = '\0';
    strcat(HighScore[3].Text, "James Madison");
    HighScore[3].Score = 70;

	HighScore[4].Text[0] = '\0';
    strcat(HighScore[4].Text, "James Monroe");
    HighScore[4].Score = 60;

	HighScore[5].Text[0] = '\0';
    strcat(HighScore[5].Text, "John Quincy Adams");
    HighScore[5].Score = 50;

	HighScore[6].Text[0] = '\0';
    strcat(HighScore[6].Text, "Andrew Jackson");
    HighScore[6].Score = 40;

	HighScore[7].Text[0] = '\0';
    strcat(HighScore[7].Text, "Martin Van Buren");
    HighScore[7].Score = 30;

	HighScore[8].Text[0] = '\0';
    strcat(HighScore[8].Text, "William H. Harrison");
    HighScore[8].Score = 20;

	HighScore[9].Text[0] = '\0';
    strcat(HighScore[9].Text, "John Tyler");
    HighScore[9].Score = 10;
}

/*
	Function: LoadHighScores
    Description:
    	Loads the high-score file from disk.  If a high-score file
        cannot be found or cannot be read, a default list of
        high-score entries is created.
*/
void LoadHighScores(void)
{
	FILE * HighScoreFile;
    char TextLine[81]; /* make sure high score lines are <= 80 chars */
    int i;
    int Result;

    HighScoreFile = fopen(HIGH_SCORE_FILENAME, "rt");
    if (HighScoreFile == NULL) { /* error on open, so default */
    	DefaultHighScores();
    	return;
    }

	for (i = 0; i < 10; i++) {
		if (fgets(TextLine, 81, HighScoreFile) == NULL) {
        	/* there aren't enough entries, fill in default */
            DefaultHighScores();
            return;
        }
        Result = sscanf(TextLine, "%20c %d", HighScore[i].Text,
        	&(HighScore[i].Score));
        if (Result != 2) {
        	/* corrupt file -- not a string and an integer on */
            /*   the same line */
            DefaultHighScores();
            return;
        }
	}

    fclose(HighScoreFile);
}

/*
	Function: SaveHighScores
    Description:
    	Writes the HighScore array out to the high-score file.
*/
void SaveHighScores(void)
{
	FILE * HighScoreFile;
    int i;

    HighScoreFile = fopen(HIGH_SCORE_FILENAME, "wt");
    if (HighScoreFile == NULL) {
    	/* error writing file, just ignore it */
    	return;
    }

    for (i = 0; i < 10; i++) {
    	fprintf(HighScoreFile, "%-20s %d\n", HighScore[i].Text,
        	HighScore[i].Score);
    }

    fclose(HighScoreFile);
}

/*
	Function: DeInitSound
    Description:
    	Unloads both MidPak and DigPak.
*/
void DeInitSound(void)
{
	DeInitMidPak();
    UnloadMidPak();
    DeInitDigPak();
    UnloadDigPak();
    if (MidiAddr != NULL) farfree(MidiAddr);
    if (ExplosionSound.sound != NULL) farfree(ExplosionSound.sound);
    if (LaserSound.sound != NULL) farfree(LaserSound.sound);
}

/*
	Function: InitSound
    Description:
    	Initialize sound stuff.
*/
BOOL InitSound(void)
{
    UINT8 far * Addr;
    long Length;

    /* load the drivers */
	if (!LoadDigPak("SOUNDRV.COM")) {
		return FALSE;
    }
    if (!InitDigPak()) {
    	UnloadDigPak();
		return FALSE;
    }
    if (!LoadMidPak("MIDPAK.COM", "MIDPAK.ADV", "MIDPAK.AD")) {
    	DeInitDigPak();
    	UnloadDigPak();
		return FALSE;
    }
    if (!InitMidPak()) {
    	UnloadMidPak();
        DeInitDigPak();
        UnloadDigPak();
		return FALSE;
    }

    /* make these NULL just in case we exit with a fatal error */
    MidiAddr = NULL;
	ExplosionSound.sound = NULL;
	LaserSound.sound = NULL;

 	/* load the MIDI file */
    if (LoadFile("alienste.xmi", &MidiAddr, NULL, &MidiLength)) {
    	FatalError("Can't load 'alienste.xmi'", "InitSound", __LINE__);
    	return FALSE;
    }
    RegisterXmidi((char far *)MidiAddr, MidiLength);

    /* load the sound effects */
	SetPlayMode(PCM_8_MONO);

	if (LoadFile("explode.8", &Addr, NULL, &Length)) {
    	FatalError("Can't load 'explode.8'", "InitSound", __LINE__);
    }
    ExplosionSound.frequency = SOUND_FX_FREQUENCY;
    ExplosionSound.sound = (char far *) Addr;
    ExplosionSound.sndlen = Length;
    MassageAudio(&ExplosionSound);

	if (LoadFile("laser.8", &Addr, NULL, &Length)) {
    	FatalError("Can't load 'laser.8'", "InitSound", __LINE__);
    }
    LaserSound.frequency = SOUND_FX_FREQUENCY;
    LaserSound.sound = (char far *) Addr;
    LaserSound.sndlen = Length;
    MassageAudio(&LaserSound);

    return TRUE;
}

/*
	Function: ProgramInit
    Description:
    	Performs all the program-wide initialization at start-up
        time.  This includes sensing the presence of alternate input
        devices and ensuring they are calibrated.
*/
void ProgramInit(void)
{
	UINT16 Xmin, Xmid, Xmax;
    UINT16 Ymin, Ymid, Ymax;
    RGB_TUPLE Black = {0, 0, 0};

    /* get into graphics */
    SetMode13h();

	/* detect mouse presence */
    if (ResetMouse() != 0) {
    	MousePresent = TRUE;
    }
    else {
    	MousePresent = FALSE;
    }

    /* detect and calibrate joystick */
    SenseJoysticks(&JsState);
    JsState.JsMask &= 0x3;  /* only enable joystick A */
    if (JsState.JsMask != 0) {
    	JoystickPresent = TRUE;
        CalibrateJsMidpoint(&JsState, &Xmin, &Ymin, &Xmid, &Ymid,
			&Xmax, &Ymax);
        JsThresholdUp		= Ymax / 3;
        JsThresholdDown		= (Ymax * 2) / 3;
        JsThresholdLeft		= Xmax / 3;
        JsThresholdRight	= (Xmax * 2) / 3;
    }
    else {
    	JoystickPresent = FALSE;
    }

    /* initialize palette */
    FillPaletteBlock(BlackPalette, 0, 256, &Black);

    /* load high-score file */
    LoadHighScores();

    /* init DIGPAK and MIDPAK stuff */
    SoundPresent = InitSound();
}

/*
	Function: ClearMode13hScreen
    Description:
    	Clears a mode 13h screen to color 0 (typically black).
*/
void ClearMode13hScreen(void)
{
	unsigned i;
	UINT8 far * Screen;

    Screen = MK_FP(VIDEO_MEM_SEGMENT, 0);

    for (i = 0; i < (320U * 200U); i++) {
    	*Screen++ = 0;
    }
}

/*
	Function: FadeIn
    Description:
    	Fades from black to the game palette in the
        specified amount of time.
*/
void FadeIn(int MilliSec)
{
	FadePaletteBlock(BlackPalette, GamePalette, 0, 256, MilliSec);
}

/*
	Function: FadeOut
    Description:
    	Fades from the current VGA palette to black.
*/
void FadeOut(int MilliSec)
{
	FadePaletteBlock(GamePalette, BlackPalette, 0, 256, MilliSec);
}

/*
	Function: SetCursorPosition
    Description:
    	Moves the cursor position to the indicated row and column
        using the VGA BIOS, function #2.  This BIOS routine
        works even in graphics mode and is used to position the
        cursor for drawing text to the screen in graphics
        modes.
*/
void SetCursorPosition(int Row, int Column)
{
	union REGS Regs;

	Regs.h.bh = 0; /* page */
	Regs.h.dh = Row; /* row */
	Regs.h.dl = Column; /* column */
	Regs.h.ah = 2; /* set cursor position */
	int86(VGA_BIOS_INT, &Regs, &Regs);
}

/*
	Function: DrawChar
    Description:
    	Draws a character onto the screen using the VGA BIOS calls.
        The character is drawn in the specified color.
*/
void DrawChar(char Char, int Color)
{
	union REGS Regs;

    Regs.h.bh = 0; /* page */
    Regs.h.al = Char; /* character */
    Regs.h.bl = Color; /* attribute byte (= color) */
    Regs.x.cx = 1; /* repeat count */
    Regs.h.ah = 9; /* write character/attribute pair */
    int86(VGA_BIOS_INT, &Regs, &Regs);
}

/*
	Function: DrawString
    Description:
    	Draws a string to the graphics screen using the VGA BIOS
        calls.  The string it drawn at the row and column indicated
        using the specified color.
*/
void DrawString(char *String, int Row, int Column, int Color)
{

    while (*String != '\0') {
	   	/* set cursor location */
        SetCursorPosition(Row, Column);
	    Column++; /* advance cursor */

        /* draw the character */
        DrawChar(*String++, Color);
    }
}

/*
	Function: CenterString
    Description:
    	Centers a string on the screen.  The function calculates
        the correct starting column position to center the string
        on the screen and then calls DrawString to do the actual
        drawing of the text.
*/
void CenterString(char *String, int Row, int Color)
{
	int Len;
    int Col;

    Len = strlen(String);
    Col = (40 - Len) / 2;

    DrawString(String, Row, Col, Color);
}

/*
	Function: IntroCredits
    Description:
        Displays the introduction credits.
*/
void IntroCredits(void)
{
    /* get into mode 13h */
	SetMode13h();

    /* load palette with VGA defaults */
    GetVGAPaletteBlock(GamePalette, 0, 256);

    /* set everything to black so we can draw without being seen */
    /* use two SetVGAPaletteBlock calls to avoid snow */
    SetVGAPaletteBlock(BlackPalette, 0, 128);
    SetVGAPaletteBlock(BlackPalette, 128, 128);

    /* first page of stuff */
    CenterString("Coriolis Group Books", 7, INTRO_TEXT_COLOR);
    CenterString("Presents", 10, INTRO_TEXT_COLOR);
    if (kbhit()) {
    	getch();
    	return;
    }
    FadeIn(1500);
    if (kbhit()) {
    	getch();
    	return;
    }
    delay(1500);
    if (kbhit()) {
    	getch();
    	return;
    }
    FadeOut(1500);
    if (kbhit()) {
    	getch();
    	return;
    }
    delay(500);

    /* second page of stuff */
    ClearMode13hScreen();
	CenterString("A", 7, INTRO_TEXT_COLOR);
    CenterString("Dave Roberts", 10, INTRO_TEXT_COLOR);
    CenterString("Production", 13, INTRO_TEXT_COLOR);
    if (kbhit()) {
    	getch();
    	return;
    }
    FadeIn(1500);
    if (kbhit()) {
    	getch();
    	return;
    }
    if (kbhit()) {
    	getch();
    	return;
    }
    delay(1500);
    if (kbhit()) {
    	getch();
    	return;
    }
    FadeOut(1500);
    if (kbhit()) {
    	getch();
    	return;
    }
    delay(500);
}

/*
	Function: TitlePage
    Description:
    	Displays the Alien Alley title page.
*/
void TitlePage(void)
{
	LINEAR_BITMAP far *	Image;

    /* start title music */
    if (SoundPresent) {
		PlaySequence(0);
    }

    /* clear screen */
	ClearMode13hScreen();

    /* set everything to black so we can draw without being seen */
    /* use two function calls to avoid snow */
    SetVGAPaletteBlock(BlackPalette, 0, 128);
    SetVGAPaletteBlock(BlackPalette, 128, 128);

    /* first page of stuff */
    Image = LoadPCX("title.pcx", GamePalette);
    if (Image == NULL) {
    	FatalError("Can't load 'title.pcx'", "TitlePage", __LINE__);
    }
    BltLinear(Image, 0, 0, MK_FP(VIDEO_MEM_SEGMENT, 0));
    farfree(Image);
    if (kbhit()) {
    	getch();
        SetVGAPaletteBlock(GamePalette, 0, 128);
        SetVGAPaletteBlock(GamePalette, 128, 128);
    	return;
    }
    FadeIn(1500);
}

/*
	Function: DisplayHighScores
    Description:
    	Displays the HighScore array on the screen.
*/
void DisplayHighScores(void)
{
	char DisplayLine[41]; /* max 40 characters wide + 1 for '\0' */
	int i;

    CenterString("*** High Scores ***", 2, 4);
    for (i = 0; i < 10; i++) {
    	sprintf(DisplayLine, "%2d. %-20s  %5d", i + 1, HighScore[i].Text,
        	HighScore[i].Score);
        CenterString(DisplayLine, HIGH_SCORE_LINE + i, HIGH_SCORE_COLOR);
    }
}

/*
	Function: DisplayHighScoreScreen
    Description:
    	Displays the high score screen from the title page.
*/
void DisplayHighScoreScreen(void)
{
    SetMode13h();
    GetVGAPaletteBlock(GamePalette, 0, 256);
    SetVGAPaletteBlock(BlackPalette, 0, 128);
    SetVGAPaletteBlock(BlackPalette, 128, 128);
    DisplayHighScores();
    FadeIn(500);
    if (getch() == 0)
    	getch();
    FadeOut(500);
}

/*
	Function: NewHighScore
    Description:
    	Manipulates the HighScore array to make room for the
        users score and gets the new text.
*/
void NewHighScore(int NewScore)
{
	int i;
    int Row;
    int Column;
    int StrLen;
    int Key;

    /* check to see if it's really a high score */
    if (NewScore <= HighScore[9].Score) {
    	return;
    }

    /* start high score music */
    if (SoundPresent) {
		PlaySequence(2);
    }

    /* move other scores down to make room */
    for (i = 8; i >= 0; i--) {
    	if (NewScore > HighScore[i].Score) {
        	strcpy(HighScore[i + 1].Text, HighScore[i].Text);
            HighScore[i + 1].Score = HighScore[i].Score;
        }
        else {
        	break;
        }
    }
    i++;

    /* blank out text of correct slot */
    HighScore[i].Text[0] = '\0';
    HighScore[i].Score = NewScore;

    /* display the text and fade in */
    SetMode13h();
    GetVGAPaletteBlock(GamePalette, 0, 256);
    SetVGAPaletteBlock(BlackPalette, 0, 128);
    SetVGAPaletteBlock(BlackPalette, 128, 128);
    DisplayHighScores();
    FadeIn(500);

    /* get user text string */
    Row		= HIGH_SCORE_LINE + i;
    Column	= 8;
    StrLen	= 0;
    do {
    	SetCursorPosition(Row, Column);
        DrawChar(127, 9);
        Key = getch();
        if (Key == 0) {
        	getch();
        }
        if (' ' <= Key && Key <= 126 && StrLen < HIGH_SCORE_TEXT_LEN) {
        	DrawChar(Key, 9);
            HighScore[i].Text[StrLen] = Key;
            StrLen++;
            HighScore[i].Text[StrLen] = '\0';
            Column++;
        }
        else if (Key == '\b' && StrLen > 0) {
        	DrawChar(' ', 9);
        	StrLen--;
            Column--;
            HighScore[i].Text[StrLen] = '\0';
        }
    } while (Key != '\r');

    /* erase cursor */
    DrawChar(' ', 9);

    /* fade to black... */
    FadeOut(500);
}

/*
	Function: DrawTile
    Description:
    	Draws the specified tile to the specified page.  X and Y are
        the tile location within the map.  MapTop specifies where
        the screen is within the map.  MemTop specifies where the
        screen is in memory.  Offset gives the current offset
        of the page in memory.
*/
void DrawTile(int x, int y, int MapTop, int MemTop, UINT16 Offset, BOOL Dup)
{
	int TileTop;	/* screen coordinates of tile */
	int TileLeft;
	int TileBottom;
    int MapBottom;	/* map line corresponding to bottom of screen */
    int DuplicateTop;

    /* figure out what map line corresponds to the bottom of the screen */
    MapBottom = (MapTop + REDUCED_SCREEN_HEIGHT - 1) %
		(MAP_HEIGHT * TILE_HEIGHT);

    /* see if the screen is wrapping around on the map */
    if (MapTop < MapBottom) {
    	/* if not, then compute location of tile relative to screen */
        /* in straight forward manner */
	    TileTop		= y * TILE_HEIGHT - MapTop;
    }
    else {
    	/* the screen is wrapping around on the map so do this */
        /* in a little more complicated way */

        /* first determine the map line of the top of the tile */
        TileTop		= y * TILE_HEIGHT;

        /* now compute the location relative to the screen top left */
        /* based on where the screen is in the map and the location */
        /* of the tile */
        if (TileTop <= MapBottom) {
        	/* it's in the lower portion of the screen so add in */
            /* the number of lines taken up by the top portion */
            TileTop += (MAP_HEIGHT * TILE_HEIGHT) - MapTop;
        }
        else {
        	/* it's in the upper half of the screen so do this */
            /* like the simple case */
            TileTop -= MapTop;
        }
    }

    /* figure out the other coordinates relative to screen */
   	TileBottom	= TileTop + TILE_HEIGHT - 1;
    TileLeft	= x * TILE_WIDTH;

    /* if the tile is completely off screen, don't draw it */
    if (TileBottom < 0 || TileTop >= REDUCED_SCREEN_HEIGHT)
   		return;

    /* draw the tile */
    BltVideoMemNoTransparent(Tile[Map[x][y]], TileLeft, TileTop,
		Offset);

    /* only duplicate tiles when we draw a new row on the screen */
    /* as it scrolls, not for dirty tiles */
    if (!Dup)
    	return;

    /* if the tile was drawn in the duplicate region, draw it toward */
    /* the end of video memory as well so the screen will be correct */
    /* when the screen offset is reset */
    DuplicateTop	= MemTop + TileTop;
    if (DuplicateTop < DUPLICATE_REGION_LINE) {
    	/* convert DuplicateTop to screen location relative to */
        /* SCREEN_TOP_REINIT_LINES */
    	DuplicateTop -= SCREEN_TOP_MIN;
        BltVideoMemNoTransparent(Tile[Map[x][y]], TileLeft, DuplicateTop,
        	SCREEN_TOP_REINIT_BYTES);
    }
}

/*
	Function: RedrawDirtyTiles
    Description:
    	Redraws the tiles that have been marked dirty in the erase
        routine.  This has the effect of erasing all the sprites
        on the hidden page.
*/
void RedrawDirtyTiles
	(
	BOOL DirtyMap[MAP_WIDTH][MAP_HEIGHT],
    int MapTop,
    int MemTop,
    UINT16 Offset
    )
{
	int i, j;

    for (i = 0; i < MAP_WIDTH; i++) {
    	for (j = 0; j < MAP_HEIGHT; j++) {
        	if (DirtyMap[i][j]) {
            	DrawTile(i, j, MapTop, MemTop, Offset, FALSE);
            }
        }
    }
}

/*
	Function: SetUpModeY
    Description:
    	Initializes the mode Y page start variables and sets up the
		status area.
*/
void SetUpModeY(void)
{
    unsigned	LineCompare;
    UINT8		VGARegTemp;
    BOOL		DirtyTiles[MAP_WIDTH][MAP_HEIGHT];
    int i, j;

    /* calculate offsets */
	PageOffset[0]	= SCREEN_0_INIT_BYTES;
    MemLine[0]		= SCREEN_0_INIT_LINES;
    MapLine[0]		= 0;

    PageOffset[1]	= SCREEN_1_INIT_BYTES;
    MemLine[1]		= SCREEN_1_INIT_LINES;
    MapLine[1]		= SCROLL_DISTANCE / 2;

    /* redraw the tiles around each page */
    for (i = 0; i < MAP_WIDTH; i++) {
    	for (j = 0; j < MAP_HEIGHT; j++) {
        	DirtyTiles[i][j] = TRUE;
        }
    }
    RedrawDirtyTiles(DirtyTiles, MapLine[0], MemLine[0], PageOffset[0]);
    RedrawDirtyTiles(DirtyTiles, MapLine[1], MemLine[1], PageOffset[1]);

    /* set start address to page 0 */
	PageFlip(PageOffset[0]);
    HiddenPage = 1;

    /* set line compare register */

    /* Mode X (and Mode Y and Mode 13h) are double scan modes, so */
    /*   the number of lines in terms of the VGA is actually */
    /*   double the pixel height */
	LineCompare = 2 * REDUCED_SCREEN_HEIGHT;
    WaitVerticalRetraceStart();
    asm cli;
    /* write lower eight bits of line compare */
    outportb(CRTC_INDEX_REG, LINE_COMPARE_INDEX);
    outportb(CRTC_DATA_REG, LineCompare & 0xFF);
    /* ninth bit is in bit 4 of overflow register */
    outportb(CRTC_INDEX_REG, OVERFLOW_INDEX);
    VGARegTemp = inportb(CRTC_DATA_REG);
    VGARegTemp = (VGARegTemp & 0xEF) | ((LineCompare >> 4) & 0x10);
    outportb(CRTC_DATA_REG, VGARegTemp);
    /* tenth bit is in bit 6 of max scan line register */
    outportb(CRTC_INDEX_REG, MAX_SCAN_LINE_INDEX);
    VGARegTemp = inportb(CRTC_DATA_REG);
    VGARegTemp = (VGARegTemp & 0xBF) | ((LineCompare >> 3) & 0x40);
    outportb(CRTC_DATA_REG, VGARegTemp);
    asm sti;
}

/*
	Function: LoadSprites
    Description:
    	Loads the hero, alien, and missile sprites and initializes the
		sprite structures.
*/
void LoadSprites(void)
{
	LINEAR_BITMAP far * LoadBitmap;
    int i;
    UINT16 VideoAddr;
    UINT16 Length;

    /* load hero spaceship */
    LoadBitmap = LoadPCX("hero.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'hero.pcx'", "LoadSprites", __LINE__);
    }
    HeroWidth			= LoadBitmap->Width;
    HeroHeight			= LoadBitmap->Height;
    HeroBitmap			= LinearToPlanar(LoadBitmap);
    HeroBitmap->OriginX	= HeroWidth / 2;
    HeroBitmap->OriginY = HeroHeight / 2;
    HeroCollisionMap	= CreateCollisionMap(LoadBitmap);
    farfree(LoadBitmap);
    HeroXMin	=	0 + (HeroWidth / 2) + 1;
    HeroXMax	=	SCREEN_WIDTH - ((HeroWidth / 2) + 1);
    HeroYMin	=	0 + (HeroHeight / 2) + 1;
    HeroYMax	=   REDUCED_SCREEN_HEIGHT - ((HeroHeight / 2) + 1);

    /* load alien spaceship */
    LoadBitmap = LoadPCX("alien.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'alien.pcx'", "LoadSprites", __LINE__);
    }
    AlienWidth			= LoadBitmap->Width;
    AlienHeight			= LoadBitmap->Height;
    AlienBitmap			= LinearToPlanar(LoadBitmap);
    AlienBitmap->OriginX	= AlienWidth / 2;
    AlienBitmap->OriginY	= AlienHeight / 2;
    AlienCollisionMap	= CreateCollisionMap(LoadBitmap);
    farfree(LoadBitmap);

    /* load missile */
    LoadBitmap = LoadPCX("missile.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'missile.pcx'", "LoadSprites", __LINE__);
    }
    MissileWidth			= LoadBitmap->Width;
    MissileHeight		= LoadBitmap->Height;
    MissileBitmap		= LinearToPlanar(LoadBitmap);
    MissileBitmap->OriginX	= MissileWidth / 2;
    MissileBitmap->OriginY	= MissileHeight / 2;
    MissileCollisionMap	= CreateCollisionMap(LoadBitmap);
    farfree(LoadBitmap);

    /* load missile trails */
    LoadBitmap = LoadPCX("mtrailu.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'mtrailu.pcx'", "LoadSprites", __LINE__);
    }
    MissileTrailWidth	= LoadBitmap->Width;
    MissileTrailHeight	= LoadBitmap->Height;
    MissileTrailUpBitmap	= LinearToPlanar(LoadBitmap);
    MissileTrailUpBitmap->OriginX = MissileTrailWidth / 2;
    MissileTrailUpBitmap->OriginY = -MissileHeight + MissileBitmap->OriginY;
    farfree(LoadBitmap);

    LoadBitmap = LoadPCX("mtraild.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'mtraild.pcx'", "LoadSprites", __LINE__);
    }
    MissileTrailDnBitmap	= LinearToPlanar(LoadBitmap);
    MissileTrailDnBitmap->OriginX = MissileTrailWidth / 2;
    MissileTrailDnBitmap->OriginY = MissileTrailHeight +
		MissileBitmap->OriginY;
    farfree(LoadBitmap);

    /* load explosion bitmaps */
    LoadBitmap = LoadPCX("exp1.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'exp1.pcx'", "LoadSprites", __LINE__);
    }
    ExplosionWidth	= LoadBitmap->Width;
    ExplosionHeight	= LoadBitmap->Height;
    ExplosionBitmap[0] = LinearToPlanar(LoadBitmap);
    ExplosionBitmap[0]->OriginX	= LoadBitmap->Width / 2;
    ExplosionBitmap[0]->OriginY	= LoadBitmap->Height / 2;
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("exp2.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'exp2.pcx'", "LoadSprites", __LINE__);
    }
    ExplosionBitmap[1] = LinearToPlanar(LoadBitmap);
    ExplosionBitmap[1]->OriginX	= LoadBitmap->Width / 2;
    ExplosionBitmap[1]->OriginY	= LoadBitmap->Height / 2;
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("exp3.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'exp3.pcx'", "LoadSprites", __LINE__);
    }
    ExplosionBitmap[2] = LinearToPlanar(LoadBitmap);
    ExplosionBitmap[2]->OriginX	= LoadBitmap->Width / 2;
    ExplosionBitmap[2]->OriginY	= LoadBitmap->Height / 2;
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("exp4.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'exp4.pcx'", "LoadSprites", __LINE__);
    }
    ExplosionBitmap[3] = LinearToPlanar(LoadBitmap);
    ExplosionBitmap[3]->OriginX	= LoadBitmap->Width / 2;
    ExplosionBitmap[3]->OriginY	= LoadBitmap->Height / 2;
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("exp5.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'exp5.pcx'", "LoadSprites", __LINE__);
    }
    ExplosionBitmap[4] = LinearToPlanar(LoadBitmap);
    ExplosionBitmap[4]->OriginX	= LoadBitmap->Width / 2;
    ExplosionBitmap[4]->OriginY	= LoadBitmap->Height / 2;
    farfree(LoadBitmap);

    /* load numbers */
    LoadBitmap = LoadPCX("0.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '0.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[0] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("1.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '1.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[1] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("2.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '2.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[2] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("3.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '3.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[3] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("4.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '4.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[4] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("5.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '5.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[5] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("6.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '6.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[6] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("7.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '7.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[7] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("8.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '8.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[8] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);
    LoadBitmap = LoadPCX("9.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load '9.pcx'", "LoadSprites", __LINE__);
    }
	Numbers[9] = LinearToPlanar(LoadBitmap);
    farfree(LoadBitmap);

    /* load the background tiles to video memory */
    LoadBitmap = LoadPCX("stars1.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'stars1.pcx'", "LoadSprites", __LINE__);
    }
    VideoAddr = FREE_MEM_START_BYTES;
    Tile[0] = LinearToVideoMemory(LoadBitmap, VideoAddr, &Length);
    farfree(LoadBitmap);

    LoadBitmap = LoadPCX("stars2.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'stars2.pcx'", "LoadSprites", __LINE__);
    }
    VideoAddr += Length;
    Tile[1] = LinearToVideoMemory(LoadBitmap, VideoAddr, &Length);
    farfree(LoadBitmap);

    LoadBitmap = LoadPCX("earth.pcx", NULL);
    if (LoadBitmap == NULL) {
    	FatalError("Can't load 'earth.pcx'", "LoadSprites", __LINE__);
    }
    VideoAddr += Length;
    Tile[2] = LinearToVideoMemory(LoadBitmap, VideoAddr, &Length);
    farfree(LoadBitmap);

    /* initialize Hero SPRITE */
    Hero.Active			= TRUE;
    Hero.x				= (HeroXMin + HeroXMax) / 2;
    Hero.y				= (HeroYMin + HeroYMax) / 2;
    Hero.Erase[0]		= FALSE;
    Hero.Erase[1]		= FALSE;
    Hero.Draw			= TRUE;

    /* initialize alien sprites */
    for (i = 0; i < MAX_ALIENS; i++) {
	    Alien[i].Active	= FALSE;
	    Alien[i].Draw	= FALSE;
    }

    /* initialize alien missiles */
    for (i = 0; i < MAX_ALIEN_MISSILES; i++) {
	    AlienMissile[i].Active	= FALSE;
	    AlienMissile[i].Draw		= FALSE;
    }

    /* initialize hero missiles */
    for (i = 0; i < MAX_HERO_MISSILES; i++) {
	    HeroMissile[i].Active	= FALSE;
        HeroMissile[i].Draw		= FALSE;
    }

    /* initialize explosions */
    for (i = 0; i < MAX_EXPLOSIONS; i++) {
    	Explosion[i].Active	= FALSE;
        Explosion[i].Draw	= FALSE;
    }
}

/*
	Function: FreeSprites
    Description:
    	Frees the memory occupied by the sprites.
*/
void FreeSprites(void)
{
	int i;

	farfree(HeroBitmap);
	farfree(HeroCollisionMap);
	farfree(AlienBitmap);
	farfree(AlienCollisionMap);
	farfree(MissileBitmap);
	farfree(MissileCollisionMap);
	farfree(MissileTrailUpBitmap);
	farfree(MissileTrailDnBitmap);
    for (i = 0; i < MAX_EXPLOSION_BITMAPS; i++) {
    	farfree(ExplosionBitmap[i]);
    }
    for (i = 0; i < 10; i++) {
    	farfree(Numbers[i]);
    }
    for (i = 0; i < NUM_TILES; i++) {
    	farfree(Tile[i]);
    }
}

/*
	Function: GetJoystickInput
    Description:
    	Reads the current joystick position and button state and
        sets UserInput... accordingly.
*/
BOOL GetJoystickInput
	(
	BOOL * UserInputUp,
	BOOL * UserInputDown,
	BOOL * UserInputLeft,
	BOOL * UserInputRight,
	BOOL * UserInputFire
	)
{
    ReadJoysticks(&JsState);

    *UserInputUp	= JsState.JsAxisAY < JsThresholdUp;
    *UserInputDown	= JsState.JsAxisAY > JsThresholdDown;
    *UserInputLeft	= JsState.JsAxisAX < JsThresholdLeft;
    *UserInputRight	= JsState.JsAxisAX > JsThresholdRight;
    *UserInputFire	= JsState.JsButtonA1;

	if (kbhit()) {
    	if (getch() == 0) {
        	getch();
        }
        return TRUE;
    }
    else {
    	return FALSE;
    }
}

/*
	Function: GetMouseInput
    Description:
    	Reads the position of the mouse and the current button state
        and sets UserInput... accordingly.
*/
BOOL GetMouseInput
	(
	BOOL * UserInputUp,
	BOOL * UserInputDown,
	BOOL * UserInputLeft,
	BOOL * UserInputRight,
	BOOL * UserInputFire
	)
{
	UINT16 Buttons;
    UINT16 x;
    UINT16 y;

    Buttons = PollMouseStatus(&x, &y);

	*UserInputUp	= y < MOUSE_THRESHOLD_UP;
    *UserInputDown	= y > MOUSE_THRESHOLD_DOWN;
    *UserInputLeft	= x < MOUSE_THRESHOLD_LEFT;
    *UserInputRight	= x > MOUSE_THRESHOLD_RIGHT;
    *UserInputFire	= (Buttons & LEFT_BUTTON_MASK) != 0;

	if (kbhit()) {
    	if (getch() == 0) {
			getch();
        }
        return TRUE;
    }
    else {
    	return FALSE;
    }
}

/*
	Function: GetKeyboardInput
    Description:
    	Updates the "UserInput..." variables used by the MoveSprites
        routine from the keyboard input device.
*/
BOOL GetKeyboardInput
	(
	BOOL * UserInputUp,
	BOOL * UserInputDown,
	BOOL * UserInputLeft,
	BOOL * UserInputRight,
	BOOL * UserInputFire
	)
{
	*UserInputUp	= GetKeyState(KEY_UP);
	*UserInputDown	= GetKeyState(KEY_DOWN);
    *UserInputRight	= GetKeyState(KEY_RIGHT);
    *UserInputLeft	= GetKeyState(KEY_LEFT);
    *UserInputFire	= GetKeyState(KEY_SPACE) ||
		GetKeyState(KEY_CONTROL) || GetKeyState(KEY_ALT);

	return GetKeyState(KEY_ESC);
}

/*
	Function: GetInput
    Description:
    	Get's player input and updates the hero's x & y variables
        as well as the variable to start a new missile firing.
        Control is a character specifying the method of input
        gathering.  Returns whether the user wants to quit the
        program or not.
*/
BOOL GetInput
	(
	int Control,
	BOOL * UserInputUp,
	BOOL * UserInputDown,
	BOOL * UserInputLeft,
	BOOL * UserInputRight,
	BOOL * UserInputFire
	)
{
	BOOL Quit;

	switch (Control) {
    	case 'j':
        	Quit = GetJoystickInput(UserInputUp, UserInputDown,
				UserInputLeft, UserInputRight, UserInputFire);
            break;
        case 'm':
        	Quit = GetMouseInput(UserInputUp, UserInputDown,
				UserInputLeft, UserInputRight, UserInputFire);
            break;
        case 'k':
        	Quit = GetKeyboardInput(UserInputUp, UserInputDown,
				UserInputLeft, UserInputRight, UserInputFire);
            break;
   }
   return Quit;
}

/*
	Function: CreateHeroMissile
    Description:
    	Finds a non-active hero missile in the HeroMissile
        array and initializes it.
*/
void CreateHeroMissile(int x, int y)
{
	int i;

    for (i = 0; i < MAX_HERO_MISSILES; i++) {
    	if (!HeroMissile[i].Active) {
        	HeroMissile[i].Active	= TRUE;
            HeroMissile[i].x			= x;
            HeroMissile[i].y			= y;
            HeroMissile[i].vx		= 0;
            HeroMissile[i].vy		= -HERO_MISSILE_VELOCITY;
            HeroMissile[i].Erase[0]	= FALSE;
            HeroMissile[i].Erase[1]	= FALSE;
            HeroMissile[i].Draw		= TRUE;
           	/* initialize bounding rect */
            ComputeBoundingRect(HeroMissile[i].x, HeroMissile[i].y,
			MissileBitmap->OriginX, MissileBitmap->OriginY,
				MissileWidth, MissileHeight, &(HeroMissile[i].Bounds));
            return;
        }
    }
}

/*
	Function: CreateAlien
    Description:
    	Finds a free alien in the Alien array and initializes it.
*/
void CreateAlien(void)
{
	int i;

    for (i = 0; i < MAX_ALIENS; i++) {
       	if (!Alien[i].Active) {
        	Alien[i].Active		= TRUE;
            Alien[i].x			= random(SCREEN_WIDTH);
            Alien[i].x = MAX(MAX_OBJECT_WIDTH / 2, MIN(Alien[i].x,
				SCREEN_WIDTH - (MAX_OBJECT_WIDTH / 2)));
            Alien[i].y			= -(MAX_OBJECT_HEIGHT / 2);
            Alien[i].vx			= random((2 * ALIEN_X_VELOCITY) + 1) -
				ALIEN_X_VELOCITY;
            Alien[i].vy			= ALIEN_Y_VELOCITY;
            Alien[i].Erase[0]	= FALSE;
            Alien[i].Erase[1]	= FALSE;
            Alien[i].Draw		= TRUE;
            Alien[i].ObjectSpec	= ALIEN_MOVE_TIME_BASE +
               	random(ALIEN_MOVE_TIME_VAR);
            Alien[i].ObjectSpec2 = 0; /* ability to fire immediately */
            /* initialize alien bounding rect */
            ComputeBoundingRect(Alien[i].x, Alien[i].y,
				AlienBitmap->OriginX, AlienBitmap->OriginY,
                AlienWidth, AlienHeight, &(Alien[i].Bounds));
           	return;
        }
    }
}

/*
	Function: CreateAlienMissile
    Description:
    	Finds a free alien missile in the AlienMissile array and
        initializes it.  The x and y positions of the missile
        are set from the x and y parameters which will place
        them somewhere near an alien gun.
*/
void CreateAlienMissile(int x, int y)
{
	int i;

    for (i = 0; i < MAX_ALIEN_MISSILES; i++) {
    	if (!AlienMissile[i].Active) {
        	AlienMissile[i].Active		= TRUE;
            AlienMissile[i].x			= x;
            AlienMissile[i].y			= y;
            AlienMissile[i].vx			= 0;
            AlienMissile[i].vy			= ALIEN_MISSILE_VELOCITY;
            AlienMissile[i].Erase[0]		= FALSE;
            AlienMissile[i].Erase[1]		= FALSE;
            AlienMissile[i].Draw			= TRUE;
           	/* initialize bounding rect */
            ComputeBoundingRect(AlienMissile[i].x, AlienMissile[i].y,
			MissileBitmap->OriginX, MissileBitmap->OriginY,
				MissileWidth, MissileHeight, &(AlienMissile[i].Bounds));
        	return;
        }
    }
}

/*
	Function: StartLaserSound
    Description:
    	Starts a laser sound playing if the sound driver is
        installed.
*/
void StartLaserSound(void)
{
	int Status;

	if (!SoundPresent)
    	return;

   	Status = AudioPendingStatus();
    if (Status == NOTPLAYING) {
    	DigPlay2(&LaserSound);
    }
	/* else too many sounds, so defer to other stuff */
}

/*
	Function: StartExplosionSound
    Description:
    	Starts an explosion sound playing if the sound driver is
        installed.
*/
void StartExplosionSound(void)
{
	if (!SoundPresent)
    	return;


    StopSound();
    DigPlay2(&ExplosionSound);
}

/*
	Function: MoveSprites
    Description:
    	Takes care of moving hero ship and alien sprites, based on
        user input and their behavioral algorithms.  MoveSprites
        is also where missiles are generated and off-screen images
        are removed from play.
*/
void MoveSprites
	(
    BOOL UserInputUp,
    BOOL UserInputDown,
    BOOL UserInputLeft,
    BOOL UserInputRight,
    BOOL UserInputFire
	)
{
	int i;
    static LastFireInput;
    int AlienFireResult;
    int AlienProximity;

	/* first, take care of the hero */
    if (UserInputUp) {
    	Hero.y -= HERO_Y_VELOCITY;
    }
    if (UserInputDown) {
    	Hero.y += HERO_Y_VELOCITY;
    }
    if (UserInputLeft) {
    	Hero.x -= HERO_X_VELOCITY;
    }
    if (UserInputRight) {
    	Hero.x += HERO_X_VELOCITY;
    }
    /* limit player movement */
    Hero.y = MAX(HeroYMin, MIN(HeroYMax, Hero.y));
    Hero.x = MAX(HeroXMin, MIN(HeroXMax, Hero.x));
    /* update hero bounding rect */
    ComputeBoundingRect(Hero.x, Hero.y, HeroBitmap->OriginX,
    	HeroBitmap->OriginY, HeroWidth, HeroHeight,
		&(Hero.Bounds));

    /* update hero missiles */
    for (i = 0; i < MAX_HERO_MISSILES; i++) {
    	if (HeroMissile[i].Draw) {
        	/* update position */
        	HeroMissile[i].y += HeroMissile[i].vy;
            /* stop drawing when it's off screen */
            if (HeroMissile[i].y < - (MAX_OBJECT_HEIGHT / 2)) {
            	HeroMissile[i].Draw = FALSE;
            }
            else {
            	/* if still onscreen, update bounding rect */
                ComputeBoundingRect(HeroMissile[i].x, HeroMissile[i].y,
                	MissileBitmap->OriginX, MissileBitmap->OriginY,
                    MissileWidth, MissileHeight, &(HeroMissile[i].Bounds));
            }
        }
    }

    /* generate hero missiles */
    if (UserInputFire && !LastFireInput && Hero.Draw) {
		CreateHeroMissile(Hero.x - HERO_GUN_OFFSET_LEFT,
			Hero.y - HERO_GUN_OFFSET_UP);
        CreateHeroMissile(Hero.x + HERO_GUN_OFFSET_RIGHT,
			Hero.y - HERO_GUN_OFFSET_UP);
        StartLaserSound();
    }
    LastFireInput = UserInputFire;

    /* update alien missiles */
    for (i = 0; i < MAX_ALIEN_MISSILES; i++) {
    	if (AlienMissile[i].Draw) {
        	/* update position */
        	AlienMissile[i].y += AlienMissile[i].vy;
            /* stop drawing when it's off screen */
            if (AlienMissile[i].y > (REDUCED_SCREEN_HEIGHT +
				(MAX_OBJECT_HEIGHT / 2))) {
            	AlienMissile[i].Draw = FALSE;
            }
            else {
            	/* if still onscreen, update bounding rect */
                ComputeBoundingRect(AlienMissile[i].x, AlienMissile[i].y,
                	MissileBitmap->OriginX, MissileBitmap->OriginY,
                    MissileWidth, MissileHeight, &(AlienMissile[i].Bounds));
            }
        }
    }

    /* move aliens */
    for (i = 0; i < MAX_ALIENS; i++) {
    	if (Alien[i].Draw) {
            if (Alien[i].ObjectSpec == 0) {
            	/* pick a new direction */
                Alien[i].vx = random((2 * ALIEN_X_VELOCITY) + 1) -
					ALIEN_X_VELOCITY;
                Alien[i].ObjectSpec = ALIEN_MOVE_TIME_BASE +
                	random(ALIEN_MOVE_TIME_VAR);
            }
            else {
            	Alien[i].ObjectSpec--;
            }
            /* update alien position */
            Alien[i].x += Alien[i].vx;
            Alien[i].y += Alien[i].vy;

            /* clip alien movement horizontally */
            Alien[i].x = MAX(MAX_OBJECT_WIDTH / 2, MIN(Alien[i].x,
				SCREEN_WIDTH - (MAX_OBJECT_WIDTH / 2)));

            /* move alien to top when it gets to bottom */
        	if (Alien[i].y > (REDUCED_SCREEN_HEIGHT +
				(MAX_OBJECT_HEIGHT / 2))) {
                Alien[i].y = - (MAX_OBJECT_HEIGHT / 2);
            }

            /* update alien bouding rect */
            ComputeBoundingRect(Alien[i].x, Alien[i].y,
				AlienBitmap->OriginX, AlienBitmap->OriginY,
                AlienWidth, AlienHeight, &(Alien[i].Bounds));

            /* generate alien missiles */
            if (Alien[i].ObjectSpec2 == 0) {
            	AlienFireResult	= random(100); /* in percent */
                AlienProximity	= Alien[i].x - Hero.x;
                if (AlienProximity < 0) {
                	AlienProximity = -AlienProximity;
                }
                if (((AlienProximity < ALIEN_PROX_THRESHOLD) &&
                	(AlienFireResult < ALIEN_FIRE_PROB_HERO)) ||
					(AlienFireResult < ALIEN_FIRE_PROB_RANDOM)) {
                    CreateAlienMissile(Alien[i].x -
						ALIEN_GUN_OFFSET_LEFT, Alien[i].y +
						ALIEN_GUN_OFFSET_DOWN);
                    CreateAlienMissile(Alien[i].x +
						ALIEN_GUN_OFFSET_RIGHT, Alien[i].y +
						ALIEN_GUN_OFFSET_DOWN);
	                Alien[i].ObjectSpec2 = ALIEN_FIRE_LOCKOUT;
                    StartLaserSound();
                }
            }
            else {
            	Alien[i].ObjectSpec2--;
            }
        }
    }

    /* generate aliens */
    if (AlienGenCounter == 0) {
    	/* generate an alien */
        CreateAlien();
        /* reinit generate counter */
        AlienGenCounter = ALIEN_GEN_RATE_BASE +
			random(ALIEN_GEN_RATE_VAR);
    }
    else {
    	AlienGenCounter--;
    }

    /* update explosions -- note, we don't really "move" them, just */
    /* make the animation go */
    for (i = 0; i < MAX_EXPLOSIONS; i++) {
    	if (Explosion[i].Draw) {
        	if (Explosion[i].ObjectSpec2 == 0) {
                Explosion[i].ObjectSpec++;
                Explosion[i].ObjectSpec2 = EXPLOSION_FRAME_REPEAT_COUNT;
                if (Explosion[i].ObjectSpec >= MAX_EXPLOSION_BITMAPS) {
                	Explosion[i].Draw = FALSE;
                }
            }
           else {
            	Explosion[i].ObjectSpec2--;
            }
        }
    }
}

/*
	Function: CreateExplosion
    Description:
    	Starts an explosion occuring at the appropriate x and y
        coordinates.
*/
void CreateExplosion(int x, int y)
{
	int i;

    for (i = 0; i < MAX_EXPLOSIONS; i++) {
    	if (!Explosion[i].Active) {
        	Explosion[i].Active		= TRUE;
            Explosion[i].x			= x;
            Explosion[i].y			= y;
            Explosion[i].Erase[0]	= FALSE;
            Explosion[i].Erase[1]	= FALSE;
            Explosion[i].Draw		= TRUE;
            Explosion[i].ObjectSpec	= 0; /* current explosion bitmap */
			Explosion[i].ObjectSpec2 = EXPLOSION_FRAME_REPEAT_COUNT;
            return;
        }
    }
}

/*
	Function: CheckCollisions
    Description:
    	Check for collisions between various objects and start
        explosions if they collide.  Collision detection is
        performed between:
        	* aliens and hero
            * aliens and hero missiles
            * hero and alien missiles
        Note that all tests are performed between objects that are
        currently being drawn, not just active objects.
*/
void CheckCollisions(void)
{
	int i, j;

	/* check between hero and aliens */
    for (i = 0; i < MAX_ALIENS; i++) {
    	/* Use C short circuit boolean evaluation in a big way. */
        /* Make sure both hero and alien are still being drawn */
        /* (they may still be active but have been removed */
		/* from the screen and are just being erased). */
        /* If they are still onscreen, then perform a rectangle test. */
        /* If the rectangle collision indicates a possible hit, then */
        /* perform a bitmap test. */
    	if (Hero.Draw && Alien[i].Draw &&
			CollisionTestRect(&(Hero.Bounds), &(Alien[i].Bounds)) &&
        	CollisionTestBitmap(HeroCollisionMap, AlienCollisionMap,
            	Hero.Bounds.Left, Hero.Bounds.Top,
				Alien[i].Bounds.Left, Alien[i].Bounds.Top)) {

        	Hero.Draw = FALSE;
            CreateExplosion(Hero.x, Hero.y);
            Alien[i].Draw = FALSE;
            CreateExplosion(Alien[i].x, Alien[i].y);
            StartExplosionSound();
        }
    }

    /* check between aliens and hero missiles */
    for (i = 0; i < MAX_ALIENS; i++) {
    	if (!Alien[i].Draw) {
        	continue;
        }
    	for (j = 0; j < MAX_HERO_MISSILES; j++) {
        	/* do similiar short circuit, mondo huge test as above */
        	if (HeroMissile[j].Draw &&
                CollisionTestRect(&(Alien[i].Bounds),
					&(HeroMissile[j].Bounds)) &&
                CollisionTestBitmap(AlienCollisionMap,
					MissileCollisionMap, Alien[i].Bounds.Left,
                    Alien[i].Bounds.Top, HeroMissile[j].Bounds.Left,
                    HeroMissile[j].Bounds.Top)) {

    			Alien[i].Draw		= FALSE;
                HeroMissile[j].Draw	= FALSE;
                CreateExplosion(Alien[i].x, Alien[i].y);
                Score += POINTS_PER_ALIEN;
                StartExplosionSound();
            	break; /* alien is destroyed */
            }
        }
    }

    /* check between hero and alien missiles */
    for (i = 0; i < MAX_ALIEN_MISSILES; i++) {
    	/* again, rely on short circuiting */
    	if (AlienMissile[i].Draw && Hero.Draw &&
        	CollisionTestRect(&(AlienMissile[i].Bounds),
				&(Hero.Bounds)) &&
            CollisionTestBitmap(MissileCollisionMap, HeroCollisionMap,
            	AlienMissile[i].Bounds.Left, AlienMissile[i].Bounds.Top,
                Hero.Bounds.Left, Hero.Bounds.Top)) {

            AlienMissile[i].Draw	= FALSE; /* destroy missile in any case */
            if (HeroShields == 0) {
	            Hero.Draw			= FALSE;
	            CreateExplosion(Hero.x, Hero.y);
                StartExplosionSound();
	            break; /* hero is destroyed */
            }
            else {
            	/* take away a bit of shields */
            	HeroShields--;
            }
        }
    }
}

/*
	Function: MarkDirtyTiles
    Description:
    	Takes the dirty map and sets to TRUE those tiles that
        are under the bitmap specified by the coordinates, width,
        height, and origin.
*/
void MarkDirtyTiles
	(
    BOOL DirtyMap[MAP_WIDTH][MAP_HEIGHT],
    int TopLine,
    int x,
    int y,
    unsigned Width,
    unsigned Height,
    int OriginX,
    int OriginY
    )
{
	int TopRow;
    int BottomRow;
    int LeftCol;
    int RightCol;

    /* compute tile row and column of each side of the sprite */
    LeftCol = (x - OriginX) / TILE_WIDTH;
    RightCol = (x - OriginX + (int) Width) / TILE_WIDTH;
    TopRow	= (TopLine + y - OriginY) / TILE_HEIGHT;
    BottomRow = (TopLine + y - OriginY + (int) Height) / TILE_HEIGHT;

    /* make sure these are in range on the map */
    /* if out of range, fold them back around */
    LeftCol		= (MAP_WIDTH + LeftCol) % MAP_WIDTH;
    RightCol	= (MAP_WIDTH + RightCol) % MAP_WIDTH;
    TopRow		= (MAP_HEIGHT + TopRow) % MAP_HEIGHT;
    BottomRow	= (MAP_HEIGHT + BottomRow) % MAP_HEIGHT;

    /* mark the tiles corresponding to the four corners as dirty. */
    /* NOTE: this technique assumes that sprites are the same size */
    /* or smaller than the tiles.  If the sprites are larger, */
    /* then a sprite could straddle a tile and it would not be */
    /* marked as dirty */
    DirtyMap[LeftCol][TopRow]		= TRUE;
    DirtyMap[RightCol][TopRow]		= TRUE;
    DirtyMap[LeftCol][BottomRow]	= TRUE;
    DirtyMap[RightCol][BottomRow]	= TRUE;
}

/*
	Function: ClearDirtyTiles
    Description:
    	Clears the dirty tile array to FALSE.
*/
void ClearDirtyTiles(BOOL DirtyMap[MAP_WIDTH][MAP_HEIGHT])
{
	int i, j;

    for (i = 0; i < MAP_WIDTH; i++) {
    	for (j = 0; j < MAP_HEIGHT; j++) {
        	DirtyMap[i][j] = FALSE;
        }
    }
}

/*
	Function: EraseSprites
    Description:
    	Erase all current bitmaps from the hidden screen.  If the
        erasure marks the last time that the object will be erased
        because it is no longer being drawn, deactivate the object.
*/
BOOL EraseSprites(void)
{
	int i;
    static unsigned DeathCounter;
    static BOOL DirtyTileMap[MAP_WIDTH][MAP_HEIGHT];


    /* clear out the dirty tile array */
    ClearDirtyTiles(DirtyTileMap);

	/* do player and possibly deactivate */
    if (Hero.Active && Hero.Erase[HiddenPage]) {
		MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
        	Hero.OldX[HiddenPage], Hero.OldY[HiddenPage],
            HeroWidth, HeroHeight,
            HeroBitmap->OriginX, HeroBitmap->OriginY);
        Hero.Erase[HiddenPage] = FALSE;
	    if (!(Hero.Draw || Hero.Erase[0] || Hero.Erase[1])) {
			Hero.Active = FALSE;
        	DeathCounter = DEATH_DELAY;
	    }
    }


    /* erase and deactivate hero missiles */
    for (i = 0; i < MAX_HERO_MISSILES; i++) {
    	if (HeroMissile[i].Active && HeroMissile[i].Erase[HiddenPage]) {
        	/* erase missile itself */
			MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
            	HeroMissile[i].OldX[HiddenPage],
				HeroMissile[i].OldY[HiddenPage],
                MissileWidth, MissileHeight,
                MissileBitmap->OriginX, MissileBitmap->OriginY);
            /* erase missile trail */
			MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
            	HeroMissile[i].OldX[HiddenPage],
                HeroMissile[i].OldY[HiddenPage],
                MissileTrailWidth, MissileTrailHeight,
                MissileTrailUpBitmap->OriginX,
                MissileTrailUpBitmap->OriginY);
            HeroMissile[i].Erase[HiddenPage] = FALSE;
        }
        /* deactivate missile if we aren't going to draw or */
        /*   erase it anymore */
        if (!(HeroMissile[i].Draw || HeroMissile[i].Erase[0] ||
			HeroMissile[i].Erase[1])) {
			HeroMissile[i].Active = FALSE;
        }
    }

    /* erase and deactivate aliens */
    for (i = 0; i < MAX_ALIENS; i++) {
    	if (Alien[i].Active && Alien[i].Erase[HiddenPage]) {
			MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
            	Alien[i].OldX[HiddenPage],
                Alien[i].OldY[HiddenPage],
                AlienWidth, AlienHeight,
                AlienBitmap->OriginX,
                AlienBitmap->OriginY);
            Alien[i].Erase[HiddenPage] = FALSE;
        }
        /* deactive alien if it's been destroyed */
        if (!(Alien[i].Draw || Alien[i].Erase[0] ||
			Alien[i].Erase[1])) {
            Alien[i].Active = FALSE;
        }
    }

    /* erase and deactivate alien missiles */
    for (i = 0; i < MAX_ALIEN_MISSILES; i++) {
    	if (AlienMissile[i].Active && AlienMissile[i].Erase[HiddenPage]) {
        	/* erase missile itself */
			MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
            	AlienMissile[i].OldX[HiddenPage],
				AlienMissile[i].OldY[HiddenPage],
                MissileWidth, MissileHeight,
                MissileBitmap->OriginX, MissileBitmap->OriginY);
            /* erase missile trail */
			MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
            	AlienMissile[i].OldX[HiddenPage],
                AlienMissile[i].OldY[HiddenPage],
                MissileTrailWidth, MissileTrailHeight,
                MissileTrailDnBitmap->OriginX,
                MissileTrailDnBitmap->OriginY);
            AlienMissile[i].Erase[HiddenPage] = FALSE;
        }
        /* deactivate missile if we aren't going to draw or */
        /*   erase it anymore */
        if (!(AlienMissile[i].Draw || AlienMissile[i].Erase[0] ||
			AlienMissile[i].Erase[1])) {
			AlienMissile[i].Active = FALSE;
        }
    }

    /* erase and deactivate explosions */
    for (i = 0; i < MAX_EXPLOSIONS; i++) {
    	if (Explosion[i].Active && Explosion[i].Erase[HiddenPage]) {
			MarkDirtyTiles(DirtyTileMap, MapLine[HiddenPage],
            	Explosion[i].OldX[HiddenPage],
                Explosion[i].OldY[HiddenPage],
                ExplosionWidth, ExplosionHeight,
                ExplosionBitmap[0]->OriginX,
                ExplosionBitmap[0]->OriginY);
            Explosion[i].Erase[HiddenPage] = FALSE;
        }
        /* deactivate if explosion has run its course */
        if (!(Explosion[i].Draw || Explosion[i].Erase[0] ||
			Explosion[i].Erase[1])) {
            Explosion[i].Active = FALSE;
        }
    }

    /* do the actual erase by redrawing dirty tiles */
    RedrawDirtyTiles(DirtyTileMap, MapLine[HiddenPage], MemLine[HiddenPage],
		PageOffset[HiddenPage]);

    /* hero has died -- signal game over after brief delay */
    if (!Hero.Active) {
    	if (DeathCounter == 0) {
        	return TRUE;
        }
        else {
        	DeathCounter--;
        }
    }
    return FALSE;
}

/*
	Function: DrawSprites
    Description:
    	Draw all active objects that should be drawn on the
        screen.
*/
void DrawSprites(void)
{
	int i;

    /* do explosions */
    for (i = 0; i < MAX_EXPLOSIONS; i++) {
    	if (Explosion[i].Draw) {
            /* draw explosion */
            BltPlanar(ExplosionBitmap[Explosion[i].ObjectSpec],
            	Explosion[i].x, Explosion[i].y, PageOffset[HiddenPage]);
            Explosion[i].Erase[HiddenPage]	= TRUE;
            Explosion[i].OldX[HiddenPage]	= Explosion[i].x;
            Explosion[i].OldY[HiddenPage]	= Explosion[i].y;
        }
    }

    /* draw hero missiles */
    for (i = 0; i < MAX_HERO_MISSILES; i++) {
    	if (HeroMissile[i].Draw) {
        	/* draw missile itself */
            BltPlanar(MissileBitmap, HeroMissile[i].x, HeroMissile[i].y,
				PageOffset[HiddenPage]);
            /* draw missile trail */
            BltPlanar(MissileTrailUpBitmap, HeroMissile[i].x, HeroMissile[i].y,
				PageOffset[HiddenPage]);
            HeroMissile[i].Erase[HiddenPage]	= TRUE;
            HeroMissile[i].OldX[HiddenPage]	= HeroMissile[i].x;
            HeroMissile[i].OldY[HiddenPage]	= HeroMissile[i].y;
        }
    }

    /* draw alien missiles */
    for (i = 0; i < MAX_ALIEN_MISSILES; i++) {
    	if (AlienMissile[i].Draw) {
        	/* draw missile itself */
            BltPlanar(MissileBitmap, AlienMissile[i].x, AlienMissile[i].y,
				PageOffset[HiddenPage]);
            /* draw missile trail */
            BltPlanar(MissileTrailDnBitmap, AlienMissile[i].x,
				AlienMissile[i].y, PageOffset[HiddenPage]);
            AlienMissile[i].Erase[HiddenPage]	= TRUE;
            AlienMissile[i].OldX[HiddenPage]	= AlienMissile[i].x;
            AlienMissile[i].OldY[HiddenPage]	= AlienMissile[i].y;
        }
    }

    /* do aliens */
    for (i = 0; i < MAX_ALIENS; i++) {
    	if (Alien[i].Active && Alien[i].Draw) {
        	BltPlanar(AlienBitmap, Alien[i].x, Alien[i].y,
            	PageOffset[HiddenPage]);
			Alien[i].Erase[HiddenPage]	= TRUE;
            Alien[i].OldX[HiddenPage]	= Alien[i].x;
            Alien[i].OldY[HiddenPage]	= Alien[i].y;
        }
    }

	/* do player */
    if (Hero.Active && Hero.Draw) {
    	BltPlanar(HeroBitmap, Hero.x, Hero.y, PageOffset[HiddenPage]);
        Hero.Erase[HiddenPage]	= TRUE;
        Hero.OldX[HiddenPage]	= Hero.x;
        Hero.OldY[HiddenPage]	= Hero.y;
    }
}

/*
	Function: DrawShieldStatus
    Description:
    	Updates the shield status at the bottom of the screen.
*/
void DrawShieldStatus(void)
{
	int i, j;
    UINT8 far * Screen;
    UINT16 LineAdder;

    SetMMR(0xF);
    Screen = MK_FP(VIDEO_MEM_SEGMENT, 0);
    Screen += SHIELD_STATUS_TOP * (GScreenVirtualWidth / 4) +
		(SHIELD_STATUS_LEFT / 4);
    LineAdder = (GScreenVirtualWidth / 4) - (MAX_HERO_SHIELDS *
		SHIELD_STATUS_WIDTH_MULT);

    for (j = 0; j < SHIELD_STATUS_HEIGHT; j++) {
		for (i = 0; i < (MAX_HERO_SHIELDS * SHIELD_STATUS_WIDTH_MULT);
			i++) {
    		if (i < (HeroShields * SHIELD_STATUS_WIDTH_MULT)) {
        		*Screen++ = SHIELD_STATUS_COLOR;
	        }
	        else {
	        	*Screen++ = SHIELD_STATUS_INVERT_COLOR;
	        }
	    }
        Screen += LineAdder;
    }
}

/*
	Function: DrawScore
    Description:
    	Draw the player score at the bottom of the screen.
*/
void DrawScore(void)
{
	char ScoreText[6]; /* five digits plus '\0' */
    int i, j;

    itoa(Score, ScoreText, 10);

    for (i = 0; i < 6; i++) { /* quickly find length */
    	if (ScoreText[i] == '\0') {
        	break;
        }
    }

	j = SCORE_NUMBERS_LEFT;
    while (i < 6) { /* draw leading zeros */
    	BltPlanarNoTransparent(Numbers[0], j, SCORE_NUMBERS_TOP, 0);
        i++;
        j += 8;
    }

    for (i = 0; ScoreText[i] != '\0'; i++) {
    	BltPlanarNoTransparent(Numbers[ScoreText[i] - '0'], j,
			SCORE_NUMBERS_TOP, 0);
        j += 8;
    }
}

/*
	Function: DrawStatus
    Description:
    	Draws the status area at the bottom of the screen
        showing the player's current score and shield strength.
*/
void DrawStatus(void)
{
	DrawShieldStatus();
    DrawScore();
}

/*
	Function: InitStatus
    Description:
    	Draw the background and "Shield" and "Score" bitmaps.
*/
void InitStatus(void)
{
	LINEAR_BITMAP far * LinearBM;
    PLANAR_BITMAP far * PlanarBM;

    LinearBM = LoadPCX("cpanel.pcx", NULL);
    if (LinearBM == NULL) {
    	FatalError("Can't load 'cpanel.pcx'", "InitStatus", __LINE__);
    }
    PlanarBM = LinearToPlanar(LinearBM);
    WaitVerticalRetraceStart();
    BltPlanarNoTransparent(PlanarBM, 0, 0, 0);
    farfree(LinearBM);
    farfree(PlanarBM);
}

/*
	Function: DrawNewMapRow
    Description:
    	Draws a new set of tiles at the top of the map as the
        screen moves up.
*/
void DrawNewMapRow(int Row, int MapTop, int MemTop, UINT16 Offset)
{
	int i;

    for (i = 0; i < MAP_WIDTH; i++) {
    	DrawTile(i, Row, MapTop, MemTop, Offset, TRUE);
    }
}

/*
	Function: ScrollScreen
    Description:
    	Scroll the hidden page up a few lines and draw the next row
        of tiles above, if necessary.  Moves the screen
*/
void ScrollScreen(void)
{
	int OldMapRow;
    int NewMapRow;

    /* move the page toward the start of video memory */
	MemLine[HiddenPage] -= SCROLL_DISTANCE;
    PageOffset[HiddenPage] -= SCREEN_WIDTH_BYTES * SCROLL_DISTANCE;
    if (MemLine[HiddenPage] < SCREEN_TOP_MIN) {
    	/* hidden page is too close to top, so move it */
        /* toward the end of video memory */
        MemLine[HiddenPage] = SCREEN_TOP_REINIT_LINES -
			(SCREEN_TOP_MIN - MemLine[HiddenPage]);
        PageOffset[HiddenPage] = MemLine[HiddenPage] * SCREEN_WIDTH_BYTES;
    }

    /* move the screen up in the map */
    OldMapRow			= MapLine[HiddenPage] / TILE_HEIGHT;
	MapLine[HiddenPage]	-= SCROLL_DISTANCE;
    if (MapLine[HiddenPage] < 0) {
    	/* we've wrapped on the map */
    	MapLine[HiddenPage] = (MAP_HEIGHT * TILE_HEIGHT) +
			MapLine[HiddenPage];
    }
    NewMapRow			= MapLine[HiddenPage] / TILE_HEIGHT;
    if (OldMapRow != NewMapRow) {
    	/* the top of the screen is in a new row of tiles, so */
        /* draw it */
    	DrawNewMapRow(NewMapRow, MapLine[HiddenPage], MemLine[HiddenPage],
			PageOffset[HiddenPage]);
    }
}

/*
	Function: InitMap
    Description:
    	Initialize the map with random tiles.
*/
void InitMap(void)
{
	int i, j;
    int RandomNum;

    for (i = 0; i < MAP_WIDTH; i++) {
    	for (j = 0; j < MAP_HEIGHT; j++) {
        	/* seed with mostly stars and a few planets */
        	RandomNum = random(100);
            if (RandomNum < 2) {
            	Map[i][j] = 2; /* earth */
            }
            else if (RandomNum < 30) {
            	Map[i][j] = 1; /* stars2 */
			}
			else {
            	Map[i][j] = 0; /* stars1 */
			}
        }
    }
}

/*
	Function: ClockInterrupt
    Description:
    	Simply increments the TickCount variable every 1/70th of
        a second when MidPak is not loaded.
*/
void interrupt ClockInterrupt(void)
{
	TickCount++;
    outportb(PIC, NONSPECIFIC_EOI);
}

/*
	Function: Play
    Description:
    	Play the game!
*/
void Play(int Control)
{
	BOOL	UserInputUp;
	BOOL	UserInputDown;
	BOOL	UserInputLeft;
	BOOL	UserInputRight;
	BOOL	UserInputFire;
	int		GunBlinkCounter;
	int		GunBlinkState;
	RGB_TUPLE Black = {0, 0, 0};
	RGB_TUPLE GunColor;
    BOOL	GameOver;
    BOOL	GameOverInput;
    BOOL	GameOverDeath;
    LINEAR_BITMAP far * Image;
    long far * Clock;
    long	OldClock;
    void interrupt (*SavedVector)();

	/* fade screen to remove title page */
    FadeOut(500);

    /* initialize all counters, etc. */
    Score = 0;
    AlienGenCounter = ALIEN_GEN_RATE_BASE;
    HeroShields = MAX_HERO_SHIELDS;
    InitMap();

    /* set mode X */
    SetModeY();

    /* load sprites */
    /* do this after we get into mode X since we're using */
    /* video mem bitmaps and we don't want them clobbered */
    /* when SetModeX clears video memory */
    LoadSprites();

    /* now up Mode X since we draw the initial screens here */
    SetUpModeY();

    /* start game music and get sound effects off disk */
    if (SoundPresent) {
		PlaySequence(1);
    }

    /* setup palette stuff */
    /* palette.pcx is a single pixel bitmap with the palette */
    /*   this is done because it's easy to load the palette */
    /*   using LoadPCX */
    Image = LoadPCX("palette.pcx", GamePalette);
    if (Image == NULL) {
    	FatalError("Can't load 'palette.pcx'", "ProgramInit", __LINE__);
    }
    farfree(Image); /* ignore the single pixel bitmap */
    SetVGAPaletteBlock(GamePalette, 0, 128);
    SetVGAPaletteBlock(GamePalette, 128, 128);

    /* install keyboard or mouse handlers */
    /* do nothing for joystick */
    if (Control == 'm') {
    }
    else if (Control == 'k') {
	    SetButtonKeysMode();
    }

    /* set up gun blink stuff */
    GetPaletteEntry(GamePalette, GUN_COLOR, &GunColor);
	GunBlinkCounter		= GUN_BLINK_RATE;
    GunBlinkState		= 1; /* gun blink on */

    /* initialize the status screen */
    InitStatus();

    if (SoundPresent) {
    	Clock = MidPakClockAddress();
    }
    else {
    	Clock = &TickCount;
        *Clock = 0;
        SavedVector = HookAndProgramSysTimer(ClockInterrupt,
			SELECT_0 | RW_LSB_MSB | MODE_SQUARE_WAVE | TYPE_BINARY,
        	COUNTER_FRAME_TIME);
    }
    OldClock = *Clock;

    /* enter main animation loop */
    GameOver = FALSE;
    while (!GameOver) {
    	/* get user input */
        GameOverInput = GetInput(Control, &UserInputUp, &UserInputDown,
			&UserInputLeft, &UserInputRight, &UserInputFire);

        /* move sprites */
        MoveSprites(UserInputUp, UserInputDown, UserInputLeft,
			UserInputRight, UserInputFire);

        /* check for collisions */
        CheckCollisions();

        /* erase */
        GameOverDeath = EraseSprites();

        /* scroll screen */
        ScrollScreen();

        /* draw */
        DrawSprites();

        /* update status */
        DrawStatus();

        /* wait for page flip */
        PageFlip(PageOffset[HiddenPage]);
   	    HiddenPage ^= 1; /* flip HiddenPage to other state */

       	/* blink the guns */
        if (GunBlinkCounter == 0) {
   	    	if (GunBlinkState == 1) {
       	    	SetVGAPaletteEntry(GUN_COLOR, &Black);
           	}
            else {
   	        	SetVGAPaletteEntry(GUN_COLOR, &GunColor);
       	    }
           	GunBlinkState ^= 1; /* flip it to other state */
            GunBlinkCounter = GUN_BLINK_RATE;
		}
       	else {
       		GunBlinkCounter--;
        }

        /* delay a bit */
        if (SoundPresent) {
        	/* make sure each frame takes at least 4/120 = 2/60 of */
        	while (*Clock < OldClock + 4);
        }
        else {
        	/* No MidPak, so wait 2/70 of a second, minimum */
        	while (*Clock < OldClock + 2);
        }
        OldClock = *Clock;

        /* player either aborts or dies */
        GameOver = GameOverInput || GameOverDeath;
    }

    /* restore timer vector and DOS time */
    if (!SoundPresent) {
    	UnhookAndRestoreSysTimer(SavedVector);
        RestoreDOSTime();
    }

    /* uninstall mouse and/or keyboard handlers */
    if (Control == 'm') {
    }
    else if (Control == 'k') {
    	SetNormalKeysMode();
    }

    /* free all memory used to play */
    FreeSprites();

    /* fade to black... */
    FadeOut(250);

    /* return to Mode 13h, as we were */
    SetMode13h();
}

/*
	Function: main
    Description:
    	Main program loop.  Init's the program, draws intro screens
        and title pages, and waits for user to hit keystroke
        to indicated what they want to do.
*/
int main()
{
	BOOL	Quit;
    BOOL	DrawTitle;
    char	Key;


	ProgramInit();

    IntroCredits();

    Quit = FALSE;
    DrawTitle = TRUE;

    while (!Quit) {
    	if (DrawTitle) {
	    	TitlePage();
			DrawTitle = FALSE;
		}

        Key = getch();
        Key = tolower(Key);

        randomize(); /* make sure we get a different game */

        switch (Key) {
        	case 0: /* extended key (F1, for instance) */
            	/* burn next key input which should be */
                /* a scan code corresponding to the key */
            	getch();
                break;
        	case 0x1B: /* escape */
            case 'q':
            	Quit = TRUE;
                break;
            case 'j':
            	if (JoystickPresent) {
                	Play(Key);
                    NewHighScore(Score);
                    DrawTitle = TRUE;
                }
                break;
            case 'm':
            	if (MousePresent) {
                	Play(Key);
                    NewHighScore(Score);
                    DrawTitle = TRUE;
                }
               break;
            case ' ': /* space */
            	Key = 'k';
                /* fall through */
            case 'k':
            	Play(Key);
                NewHighScore(Score);
                DrawTitle = TRUE;
                break;
            case 's':
            	DisplayHighScoreScreen();
                DrawTitle = TRUE;
                break;
            default:
            	DrawTitle = FALSE;
            	break;
        }
    }

    SaveHighScores();
    if (SoundPresent) {
	    DeInitSound();
    }

    SetVGAMode(0x3);

    return 0;
}