/* File: PALETTE.C
** Description:
**   Routines to manipulating the VGA palette, including doing palette
**   fading.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <assert.h>
#include <stdio.h>
#include "gamedefs.h"
#include "palette.h"
#include "retrace.h"
#include "vga.h"

#define STEPS_PER_SECOND	(60)	/* number of fade steps per second */
									/*   = 1 per VGA scan frame */

/*
	Function: SetVGAPaletteEntry
    Description:
    	Modifies the VGA palette entry specified by the Index
        parameter to the RGB color value specified by
        the Rgb parameter.  The individual R, G, and B values
        should be ready to be sent directly to the VGA
        (i.e., should range from 0-63).

		This function may need to be called during a vertical
		or horizontal retrace period on some VGAs to avoid
		causing snow.
*/
void SetVGAPaletteEntry(int Index, RGB_TUPLE * Rgb)
{
    assert(0 <= Index && Index <= 255);
	assert(Rgb != NULL);
    assert(Rgb->r < 64);
    assert(Rgb->g < 64);
    assert(Rgb->b < 64);

    asm cli;

	outportb(COLOR_ADDRESS_WRITE, Index);
    outportb(COLOR_DATA, Rgb->r);
    outportb(COLOR_DATA, Rgb->g);
    outportb(COLOR_DATA, Rgb->b);

    asm sti;
}

/*
	Function: GetVGAPaletteEntry
    Description:
    	Gets the R, G, and B components of the color value
        currently in the VGA palette at Index.  The
        function returns the color components in the
        RGB_TUPLE pointed to by Rgb.
*/
void GetVGAPaletteEntry(int Index, RGB_TUPLE * Rgb)
{
    assert(0 <= Index && Index <= 255);
	assert(Rgb != NULL);

    asm cli;

    outportb(COLOR_ADDRESS_READ, Index);

    Rgb->r = inportb(COLOR_DATA);
    Rgb->g = inportb(COLOR_DATA);
    Rgb->b = inportb(COLOR_DATA);

    asm sti;
}

/*
	Function: SetVGAPaletteBlock
    Description:
    	Sets a subset of the VGA palette registers to new values.
        A RAM copy of the palette to load is pointed to by
        the Palette parameter.  The block of palette registers
        to transfer to the VGA is specified by the Start and
        Length parameters.  Note that Length specifies the
        number of RGB entries, not the actual length of
        the block in bytes.

		This function may need to be called during a vertical
		or horizontal retrace period on some VGAs to avoid
		causing snow.
*/
void SetVGAPaletteBlock(UINT8 Palette[256][3], int Start, int Length)
{
    UINT8 * BlockStartPtr;

	assert(Palette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);

    asm cli;

    outportb(COLOR_ADDRESS_WRITE, Start);

    BlockStartPtr = &Palette[Start][0];

#if defined(__TINY__) || defined(__SMALL__) || defined(__MEDIUM__)
	asm {
		/* near data pointers so no need to load DS with correct value */
        push ds
        mov si,BlockStartPtr
    }
#else
	asm {
		/* far data pointers, so load DS, too */
        push ds
		lds si,BlockStartPtr
    }
#endif
	asm {
    	mov cx,Length
        mov dx,COLOR_DATA
    }
    Loop:
    asm {
    	outsb		/* write R component */
        outsb		/* write G component */
        outsb		/* write B component */
        dec cx
        jnz Loop
        pop ds
    }

    asm sti;
}

/*
	Function: GetVGAPaletteBlock
    Description:
    	Gets a subset of the VGA palette registers and stores
        them in a UINT8 array.  The subset block to retrieve
        is identified by the Start and Length parameters.
        Note that Length specifies the number of RGB entries,
        not the length of the block in bytes.
*/
void GetVGAPaletteBlock(UINT8 Palette[256][3], int Start, int Length)
{
    UINT8 * BlockStartPtr;

	assert(Palette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);

    asm cli;

    outportb(COLOR_ADDRESS_READ, Start);

    BlockStartPtr = &Palette[Start][0];

#if defined(__TINY__) || defined(__SMALL__) || defined (__MEDIUM__)
    asm {
		/* near data pointers, so move ds into es */
    	mov ax,ds
        mov es,ax
        mov di,BlockStartPtr
    }
#else
	asm {
		/* far data pointers so load es,di from BlockStartPtr */
        les di,BlockStartPtr
    }
#endif
	asm {
    	mov cx,Length
        mov dx,COLOR_DATA
    }
    Loop:
    asm {
    	insb		/* get R component */
        insb		/* get G component */
        insb		/* get B component */
        dec cx
        jnz Loop
    }

    asm sti;
}

/*
	Function: SetPaletteEntry
    Description:
    	Sets the RGB components of a UINT8 entry to the values
        specified by the Rgb parameter.
*/
void SetPaletteEntry(UINT8 Palette[256][3], int Index, RGB_TUPLE * Rgb)
{
	assert(Palette != NULL);
    assert(0 <= Index && Index <= 255);
    assert(Rgb != NULL);
    assert(Rgb->r < 64);
    assert(Rgb->g < 64);
    assert(Rgb->b < 64);

	Palette[Index][0] = Rgb->r;
	Palette[Index][1] = Rgb->g;
	Palette[Index][2] = Rgb->b;
}

/*
	Function: GetPaletteEntry
    Description:
    	Sets Rgb to the color components from the Palette at
        Index.
*/
void GetPaletteEntry(UINT8 Palette[256][3], int Index, RGB_TUPLE * Rgb)
{
	assert(Palette != NULL);
    assert(0 <= Index && Index <= 255);
    assert(Rgb != NULL);

	Rgb-> r = Palette[Index][0];
	Rgb-> g = Palette[Index][1];
	Rgb-> b = Palette[Index][2];
}

/*
	Function: FillPaletteBlock
    Description:
    	Fills a block of palette entries with a single color.
        The block begins at index Start and is Length entries
        long.  The fill color is specified by Rgb.  This
        function is useful for clearing a palette to all
        black or all white to create an ending point for a
        fade.
*/
void FillPaletteBlock
	(
	UINT8 Palette[256][3],
	int Start,
	int Length,
	RGB_TUPLE * Rgb
	)
{
	int Index;

	assert(Palette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);
    assert(Rgb != NULL);
    assert(Rgb->r < 64);
    assert(Rgb->g < 64);
    assert(Rgb->b < 64);

    for (Index = Start; Index < Start + Length; Index++) {
    	Palette[Index][0] = Rgb->r;
    	Palette[Index][1] = Rgb->g;
    	Palette[Index][2] = Rgb->b;
    }
}

/*
	Function: CopyPaletteBlock
    Description:
    	Copies a block of palette entries from one palette to
        another.
*/
void CopyPaletteBlock
	(
	UINT8 DestPalette[256][3],
	UINT8 SourcePalette[256][3],
	int Start,
	int Length
	)
{
	int Index;

    assert(DestPalette != NULL);
    assert(SourcePalette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);

    for (Index = 0; Index < Start + Length; Index++) {
    	DestPalette[Index][0] = SourcePalette[Index][0];
    	DestPalette[Index][1] = SourcePalette[Index][1];
    	DestPalette[Index][2] = SourcePalette[Index][2];
    }
}

/*
	Function: ComputeFadeStep
    Description:
    	Computes a palette for a single step of a palette fade.  The
        starting palette and ending palette for the fade are given.
        The amount to fade each entry in the block is computed
        as the ratio of the current step to the number of steps
        in the fade.  Note that when step = 0, the resulting
        palette equals the starting palette and when step =
        NumSteps, the resulting palette equals the ending palette.
        Only the palette entries specified in the block will be
        altered in the FadeStepPalette.  A calling routine
        should use CopyPaletteBlock from either the starting or
        ending palettes to ensure that FadeStepPalette contains
        correct entries outside the specified block.
*/
void ComputeFadeStep
	(
	UINT8 StartPalette[256][3],
	UINT8 EndPalette[256][3],
	int Start,
	int Length,
	long NumSteps,
	long CurrentStep,
	UINT8 FadeStepPalette[256][3]
	)
{
	int Index;
    long RDelta;
    long GDelta;
    long BDelta;

	assert(StartPalette != NULL);
	assert(EndPalette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);
    assert(NumSteps > 0);
    assert(0 <= CurrentStep && CurrentStep <= NumSteps);
    assert(FadeStepPalette != NULL);

    for (Index = Start; Index < Start + Length; Index++) {
		RDelta = EndPalette[Index][0] - StartPalette[Index][0];
        RDelta = (RDelta * CurrentStep) / NumSteps;

		GDelta = EndPalette[Index][1] - StartPalette[Index][1];
        GDelta = (GDelta * CurrentStep) / NumSteps;

		BDelta = EndPalette[Index][2] - StartPalette[Index][2];
        BDelta = (BDelta * CurrentStep) / NumSteps;

        FadeStepPalette[Index][0] = StartPalette[Index][0] + RDelta;
        FadeStepPalette[Index][1] = StartPalette[Index][1] + GDelta;
        FadeStepPalette[Index][2] = StartPalette[Index][2] + BDelta;
    }
}

/*
	Function: FadePaletteBlock
    Description:
    	Fades a block of palette entries on the screen.  The starting
        and ending palettes are given as well as the block to fade
        and the length of time to fade.  The routine computes the
        number of fade steps from the length of time given.  It
        then computes each fade step and sets the block of VGA
        palette entries to the values computed for the step.
        Note that the VGA palette corresponding to the block
        specified should equal what is contained in StartPalette,
        otherwise a color jump will result when the palette is
        set on the first fade step.  The routine will leave the
        VGA palette entries corresponding to the block specified
        equal to the values in EndPalette.  If the routine is
        passed a length greater than 128, it will fade half the
        color entries on every other retrace and alternate between
        the two halfs.  This avoids snow.
*/
void FadePaletteBlock
	(
	UINT8 StartPalette[256][3],
	UINT8 EndPalette[256][3],
    int Start,
    int Length,
	int Milliseconds
	)
{
	UINT8 IntermedPalette[256][3];
	long TotalSteps;
    long Step;

    assert(StartPalette != NULL);
    assert(EndPalette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);
    assert(Milliseconds > 0);

    if (Length <= 128) {
	    TotalSteps = ((long) Milliseconds * STEPS_PER_SECOND) / 1000;
    }
    else {
	    TotalSteps = ((long) (Milliseconds / 2) * STEPS_PER_SECOND) / 1000;
    }

    for (Step = 0; Step <= TotalSteps; Step++) {
    	ComputeFadeStep(StartPalette, EndPalette, Start, Length,
        	TotalSteps, Step, IntermedPalette);
        if (Length <= 128) {
	        WaitVerticalRetraceStart();
    	    SetVGAPaletteBlock(IntermedPalette, Start, Length);
        }
        else {
	        WaitVerticalRetraceStart();
    	    SetVGAPaletteBlock(IntermedPalette, Start, Length / 2);
	        WaitVerticalRetraceStart();
    	    SetVGAPaletteBlock(IntermedPalette, Start + (Length / 2),
				Length - (Length / 2));
        }
    }
}

/*
	Function: RotatePaletteBlock
    Description:
    	Rotates a block of palette entries around one another.  The
        block is specified by Start and Length.  The rotation amount
        is specified by Rotation.  A positive rotation value will
        make Palette[Start] = Palette[Start + Rotation].  Rotations
        are clipped to the block specified (i.e., the rotation wraps
        around within the block).
*/
void RotatePaletteBlock
	(
	UINT8 Palette[256][3],
	int Start,
	int Length,
	int Rotation
	)
{
	UINT8 TempPalette[256][3];
	int Index;
    int SourceIndex;

    assert(Palette != NULL);
    assert(0 <= Start && Start <= 255);
    assert(Length > 0);
    assert(Start + Length <= 256);

    if (Rotation < 0) {
    	Rotation = Length + Rotation;
    }

    for (Index = 0; Index < Length; Index++) {
   		SourceIndex = (Index + Rotation) % Length;
   		TempPalette[Start + Index][0] = Palette[Start + SourceIndex][0];
    	TempPalette[Start + Index][1] = Palette[Start + SourceIndex][1];
   		TempPalette[Start + Index][2] = Palette[Start + SourceIndex][2];
    }

    CopyPaletteBlock(Palette, TempPalette, Start, Length);
}
