/* File: MOUSEEVT.C
** Description:
**   Module for dealing with the mouse using event-driven programming.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <dos.h>
#include <stdio.h>
#include "gamedefs.h"
#include "mouse.h"
#include "mouseevt.h"

/* constants */
#define ALL_EVENTS (0x7F)

/* types */
typedef struct {
	UINT16 Length;
    UINT16 Read;
    UINT16 Write;
    MOUSE_EVENT far * Queue;
} MOUSE_QUEUE;

/* module variables */
static MOUSE_QUEUE MouseQueue;
static BOOL HandlerInstalled = FALSE;

/*
	Function: MouseEventQueueHandler
    Description:
    	This is the mouse event handler that is installed when
        InstallMouseEventQueue() is called.  The hander
        take the generated mouse event and adds it to the
        mouse queue specified by the MouseQueue variable.
        Note that this routine must be defined as
        "far _loadds _saveregs" to ensure that it returns
        with a RETF, that DS is loaded with the program's
        data segment (rather than leaving DS set to the
        mouse driver's data segment), and that none of
        the registers are modified.
*/
static void far _loadds _saveregs MouseEventQueueHandler(void)
{
	UINT16 Event;
    UINT16 ButtonState;
    UINT16 X;
    UINT16 Y;
    UINT16 XMickeys;
    UINT16 YMickeys;
    MOUSE_EVENT far * EventQueueEntry;
    UINT16 EventBit;

    /* save registers into variables */
    /* DO THIS IMMEDIATELY TO PREVENT OVERWRITING ANYTHING! */
    Event		= _AX;
    ButtonState	= _BX;
    X			= _CX;
    Y			= _DX;
    XMickeys	= _SI;
    YMickeys	= _DI;

    EventBit = 1;

    /* separate all signalled events into individual queue entries */
    while (EventBit & ALL_EVENTS) {
    	if (Event & EventBit) {
    		EventQueueEntry = &(MouseQueue.Queue[MouseQueue.Write]);

    		/* if the current write position in the queue is full, */
    		/* discard all following events */
    		if (EventQueueEntry->InUse == TRUE) {
    			return;
    		}

    		/* copy info to event queue entry */
    		EventQueueEntry->Event		= Event & EventBit;
    		EventQueueEntry->ButtonState= ButtonState;
    		EventQueueEntry->X			= X;
    		EventQueueEntry->Y			= Y;
    		EventQueueEntry->XMickeys	= XMickeys;
    		EventQueueEntry->YMickeys	= YMickeys;

    		/* mark entry as now being in use */
    		EventQueueEntry->InUse = TRUE;

    		/* increment write index and handle wrap-around */
    		MouseQueue.Write++;
    		if (MouseQueue.Write >= MouseQueue.Length) {
    			MouseQueue.Write = 0;
    		}
		}
        EventBit <<= 1;
    }
}

/*
	Function: InstallMouseEventQueue
    Description:
    	This function takes an event mask and a queue size as
        parameters.  The function creates an event queue with
        the specified number of entries and initializes it.
		The function installs the handler using the
		SetMouseEventHandler function defined in MOUSE.C.
		The event mask parameter is passed to
		SetMouseEventHandler.  The handler is defined above,
        MouseEventQueueHandler.  The function returns TRUE if
        all went well, FALSE if memory for the queue could not
        be allocated.
*/
int InstallMouseEventQueue
	(
    UINT16 Length,
    UINT16 EventMask
    )
{
	UINT16 i;

    assert(HandlerInstalled == FALSE);
    assert(Length != 0);

	/* create queue and return error if space couldn't be allocated */
	MouseQueue.Queue = farmalloc(Length * sizeof(MOUSE_EVENT));
    if (MouseQueue.Queue == NULL) {
    	return FALSE;
    }

    /* initialize queue pointers */
    MouseQueue.Length	= Length;
    MouseQueue.Write	= 0;
    MouseQueue.Read		= 0;

    /* initialize the InUse flags of each entry to not in use */
    for (i = 0; i < Length; i++) {
    	MouseQueue.Queue[i].InUse = FALSE;
    }

    /* set up the handler and event mask */
    SetMouseEventHandler(EventMask, MouseEventQueueHandler);

    HandlerInstalled = TRUE;

    return TRUE;
}

/*
	Function: RemoveMouseEventQueue
    Description:
    	Sets the handler function in the mouse driver to null and
        deallocates the mouse event queue memory.
*/
void RemoveMouseEventQueue(void)
{
	assert(HandlerInstalled == TRUE);

	/* remove handler (specify all events masked, no handler address). */
    /* note that we must do this before freeing the memory. */
    /* if we do after, we could get a mouse event after */
    /* freeing the memory. */
    SetMouseEventHandler(0, NULL);

    /* now that there are no mouse events to be generated, */
    /* free the memory */
	farfree(MouseQueue.Queue);

    HandlerInstalled = FALSE;
}

/*
	Function: QueryMouseEventWaiting
    Description:
    	Returns TRUE if there is a mouse event in the queue,
        FALSE if not.
*/
int QueryMouseEventWaiting(void)
{
	assert(HandlerInstalled == TRUE);

	if (MouseQueue.Queue[MouseQueue.Read].InUse == TRUE) {
    	return TRUE;
    }
    else {
    	return FALSE;
    }
}

/*
	Function: PeekMouseEvent
    Description:
    	Returns the mouse event at the head of the queue.  If
        a mouse event is present, the function returns TRUE and
        sets the InUse flag of the supplied MOUSE_EVENT structure
        to TRUE.  If no event is present, these two indicators
        are false.
*/
int PeekMouseEvent
	(
    MOUSE_EVENT * MouseEvent
    )
{
	MOUSE_EVENT far * CurrentEvent;

    assert(MouseEvent != NULL);
    assert(HandlerInstalled == TRUE);

    CurrentEvent = &(MouseQueue.Queue[MouseQueue.Read]);

    /* unconditionally copy what's sitting in the read spot */
    MouseEvent->InUse		= CurrentEvent->InUse;
    MouseEvent->Event		= CurrentEvent->Event;
    MouseEvent->ButtonState	= CurrentEvent->ButtonState;
    MouseEvent->X			= CurrentEvent->X;
    MouseEvent->Y			= CurrentEvent->Y;
    MouseEvent->XMickeys	= CurrentEvent->XMickeys;
    MouseEvent->YMickeys	= CurrentEvent->YMickeys;

    /* indicate whether there was really anything there */
    return CurrentEvent->InUse;
}

/*
	Function: GetMouseEvent
    Description:
    	Copies the mouse event at the head of the event queue to
        the MOUSE_EVENT structure supplied as a parameter and
        removes the event from the queue.  It returns TRUE if all
        goes well, FALSE if there is no event on the queue.
        The InUse flag of the MOUSE_EVENT structure will also
        indicate whether an event was waiting.  If InUse it
        TRUE, and event was pending and the other info in
        the MOUSE_EVENT structure is valid.
*/
int GetMouseEvent
	(
    MOUSE_EVENT * MouseEvent
    )
{
	assert(MouseEvent != NULL);
    assert(HandlerInstalled == TRUE);

    /* copy the mouse event using PeekMouseEvent */
    PeekMouseEvent(MouseEvent);

    /* if an event was present, free the queue position and */
    /* bump the read pointer (account for wrap) */
    if (MouseEvent->InUse == TRUE) {
    	MouseQueue.Queue[MouseQueue.Read].InUse = FALSE;
    	MouseQueue.Read++;
        if (MouseQueue.Read >= MouseQueue.Length) {
        	MouseQueue.Read = 0;
        }
    }

    return MouseEvent->InUse;
}
