/* File: FILENAME.C
** Description:
**   Routines for manipulating DOS file names.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include "filename.h"

/*
	Function: FindBackward
    Description:
    	A module specific function that searches for a character
        within the string argument.  The search starts from
        the last character of the string and progresses toward
        the first character.  The routine returns the position
        of the found character (0 to strlen-1).  If the
        character is not found, the routine returns -1.
*/
static int FindBackward(char * String, char FindChar)
{
	int Index;

    Index = strlen(String) - 1;

    while (Index >= 0) {
    	if (String[Index] == FindChar)
        	break;
        Index--;
    }

    return Index;
}

/*
	Function: GetPath
    Description:
    	Returns the path component of a full or partial filename.
*/
char * GetPath(char * FullPath, int LeaveTrailingSlash)
{
	int Len;
    char * Path;

    assert(FullPath != NULL);

	Len = FindBackward(FullPath, PATH_SEPARATOR) + 1;
    if (Len > 0 && !LeaveTrailingSlash) {
    	Len--;
    }

    Path = (char *) malloc(Len + 1);
    if (Path == NULL) {
    	return NULL;
    }

    if (Len > 0) {
	    strncpy(Path, FullPath, Len);
    }
    Path[Len] = '\0';

    return Path;
}

/*
	Function: GetFilename
    Description:
    	Returns the filename portion of a path.
*/
char * GetFilename(char * FullPath)
{
	int Len;
    int FilenameStart;
    char * Filename;

    assert(FullPath != NULL);

	FilenameStart = FindBackward(FullPath, PATH_SEPARATOR) + 1;
    Len = strlen(FullPath) - FilenameStart;

    Filename = (char *) malloc(Len + 1);
    if (Filename == NULL) {
    	return NULL;
    }

    if (Len > 0) {
	    strncpy(Filename, FullPath + FilenameStart, Len);
    }
    Filename[Len] = '\0';

    return Filename;
}

/*
	Function: GetRoot
    Description:
    	Returns the root of the filename of the path passed to
        the function.
*/
char * GetRoot(char * FullPath)
{
	char * Filename;
    int Len;
    char * Root;

    assert(FullPath != NULL);

    Filename = GetFilename(FullPath);
    if (Filename == NULL) {
    	return NULL;
    }

	Len = FindBackward(Filename, EXTENSION_SEPARATOR);

    if (Len == -1) {
    	/* there is no "." in the filename, so the whole thing */
        /* is the root */
    	return Filename;
    }

	Root = (char *) malloc(Len + 1);
    if (Root == NULL) {
    	return NULL;
    }

    if (Len > 0) {
    	strncpy(Root, Filename, Len);
    }
    Root[Len] = '\0';

    free(Filename);

    return Root;
}

/*
	Function: GetExtension
    Description:
    	Returns the extension of the filename at the end of the
        given path.  The extension is returned with the "." preceeding
        it.  If there is no extension (e.g., filename is "foobar"), then
        a null string ("") is returned.  If a period (".") is the last
        character in the filename, just the period is returned.
        This distinguishes between filenames of the form "foobar" versus
        "foobar.".
*/
char * GetExtension(char * FullPath)
{
	char * Filename;
    int ExtensionStart;
    int Len;
    char * Extension;

    assert(FullPath != NULL);

    Filename = GetFilename(FullPath);
    if (Filename == NULL) {
    	return NULL;
    }

	ExtensionStart = FindBackward(Filename, EXTENSION_SEPARATOR);

    if (ExtensionStart == -1) {
    	/* there is no "." in the filename, so the extension */
        /* is the null string ("") */
        Len = 0;
    }
	else {
	    Len = strlen(Filename) - ExtensionStart;
    }

	Extension = (char *) malloc(Len + 1);
    if (Extension == NULL) {
    	return NULL;
    }

    if (Len > 0) {
    	strncpy(Extension, Filename + ExtensionStart, Len);
    }
    Extension[Len] = '\0';

    free(Filename);

    return Extension;
}
