/* File: DIGMIDIF.C
** Description:
**   Interface routines for loading, initializing and unloading
**   Digpak and Midpak drivers.  These routines are based on
**   similiar routines provided as example code in the DigPak/MidPak
**   distribution archive.
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <alloc.h>
#include <assert.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <stdio.h>
#include "gamedefs.h"
#include "digplay.h" /* supplied with DigPak */
#include "midpak.h"  /* supplied with MidPak */

/* private module variables */
static BOOL		DigPakLoaded 	= FALSE;
static BOOL		DigPakInit		= FALSE;
static UINT8 far *DigPakAddr;
static UINT16	DigPakParaSeg;

static BOOL		MidPakLoaded	= FALSE;
static BOOL		MidPakInit		= FALSE;
static BOOL		AdLoaded		= FALSE;
static UINT8 far *MidPakAddr;
static UINT16	MidPakParaSeg;
static UINT8 far *AdvAddr;
static UINT16	AdvParaSeg;
static UINT8 far *AdAddr;

/*
	Function: LoadFile
    Description:
    	Loads a file into far memory.  The filename is specified by
        the Filename parameter.  The routine returns the address
        of the file in the Addr parameter.  If the Para parameter
        is non-NULL, the file is loaded aligned to a paragraph and
        the segment portion of the aligned address is returned
        in Para.  Note that the Addr parameter is still set to
        the true start of the address block to which the file
        is loaded and should be used to farfree the block when
        done.  The length is returned in Len.
*/
int LoadFile(char * Filename, UINT8 far *(*Addr), UINT16 *Para, long *Len)
{
	int Handle;
    int Status;
    long Length;
    UINT8 far * Buffer;
    UINT8 far * BufferPara;
    UINT16 BufferSeg;
    UINT16 BufferOff;
    unsigned NumRead;

    assert(Filename != NULL);

	/* figure out file length */
    Handle = open(Filename, O_RDONLY | O_BINARY);
    if (Handle == -1) {
    	return 1; /* error */
    }
    Length = filelength(Handle);
    if (Length == -1L) {
    	close(Handle);
        return 1; /* error */
    }

    /* allocate buffer */
    if (Para) {
    	/* we want this paragraph aligned */
    	Buffer = farmalloc(Length + 16);
    	if (Buffer == NULL) {
    		close(Handle);
    		return 1;
    	}

    	/* if not paragraph aligned, make it so */
    	BufferSeg = FP_SEG(Buffer);
    	BufferOff = FP_OFF(Buffer);
    	BufferSeg = BufferSeg + (BufferOff >> 4) + 1;
    	BufferOff = 0;
    	BufferPara = MK_FP(BufferSeg, BufferOff);
    }
    else {
    	Buffer = farmalloc(Length);
        if (Buffer == NULL) {
        	close(Handle);
            return 1;
        }
        BufferPara = Buffer;
    }

    /* load file into buffer */
    Status = _dos_read(Handle, BufferPara, (unsigned) Length, &NumRead);
    if (Status != 0 || (NumRead != (unsigned) Length)) {
    	farfree(Buffer);
    	close(Handle);
        return 1;
    }
    close(Handle);

    /* update return values */
    if (Addr != NULL) {
		*Addr = Buffer;
    }
    if (Para != NULL) {
	    *Para = BufferSeg;
    }
    if (Len != NULL) {
    	*Len = Length;
    }

    return 0;
}

/*
	Function: LoadDigPak
    Description:
    	Loads a DigPak driver into memory.  Returns FALSE if
        something goes wrong.
*/
BOOL LoadDigPak(char *Filename)
{
	int Error;

	/* check to see that we haven't already loaded it */
    assert(!DigPakLoaded && !DigPakInit);

    /* load the driver */
	Error = LoadFile(Filename, &DigPakAddr, &DigPakParaSeg, NULL);
    if (!Error) { /* everything OK */
        DigPakLoaded = TRUE;
        return TRUE;
    }
	return FALSE; /* something went wrong */
}

/*
	Function: InitDigPak
    Description:
    	Initializes the DigPak driver already loaded.  Returns
        FALSE if the driver can't be initialized.
*/
BOOL InitDigPak(void)
{
	/* make sure we've loaded it */
	assert(DigPakLoaded);

    /* initialize and if everything is okay, return TRUE */
    if (InitDP(DigPakParaSeg)) {
    	DigPakInit = TRUE;
        return TRUE;
    }

    /* something went wrong */
    DigPakInit = FALSE;
    return FALSE;
}

/*
	Function: DeInitDigPak
    Description:
    	Deinitializes the DigPak driver.
*/
void DeInitDigPak(void)
{
	/* make sure it's already loaded and initialized */
	assert(DigPakLoaded && DigPakInit);
    DeInitDP(DigPakParaSeg);
    DigPakInit = FALSE;
}

/*
	Function: UnloadDigPak
    Description:
    	Unloads the DigPakDriver and frees its memory.
*/
void UnloadDigPak(void)
{
	assert(DigPakLoaded);

    if (DigPakInit) {
    	DeInitDigPak();
    }
    farfree(DigPakAddr);
    DigPakLoaded = FALSE;
}

/*
	Function: LoadMidPak
    Description:
    	Loads the specified MidPak driver files into memory.
        Returns TRUE if all is well, FALSE otherwise.
*/
BOOL LoadMidPak(char * Filename, char *AdvFilename, char *AdFilename)
{
	int Error;

	assert(!MidPakLoaded && !MidPakInit);

    Error = LoadFile(Filename, &MidPakAddr, &MidPakParaSeg, NULL);
    if (!Error) {
    	Error = LoadFile(AdvFilename, &AdvAddr, &AdvParaSeg, NULL);
        if (!Error) {
        	if (AdFilename) {
            	Error = LoadFile(AdFilename, &AdAddr, NULL, NULL);
                if (Error) {
                	/* error loading .AD file */
                    farfree(AdvAddr);
                    farfree(MidPakAddr);
                    return FALSE;
                }
                AdLoaded = TRUE;
            }
            else {
            	AdLoaded = FALSE;
            }
            /* everything OK */
            MidPakLoaded = TRUE;
            return TRUE;
        }
        else {
        	/* error loading .ADV file */
        	farfree(MidPakAddr);
            return FALSE;
        }
    }
    return FALSE;
}

/*
	Function: InitMidPak
    Description:
    	Initializes a MidPak driver.  Returns FALSE if the driver
        can't be initialized, TRUE otherwise.
*/
BOOL InitMidPak(void)
{
	/* make sure it's been loaded first */
	assert(MidPakLoaded);

    /* initialize it */
    /* note that the midpak.h file declares the AdAddr parameter as */
    /*   char far * rather than unsigned char far *, hence the cast */
    if (!InitMP(MidPakParaSeg, AdvParaSeg, (char far *) AdAddr)) {
    	MidPakInit = TRUE;
        return TRUE;
    }

    /* something went wrong */
    MidPakInit = FALSE;
	return FALSE;
}

/*
	Function: DeInitMidPak
    Description:
    	Deinitializes a MidPak driver.
*/
void DeInitMidPak(void)
{
	/* make sure we're loaded and initialized */
	assert(MidPakLoaded && MidPakInit);

    MidiStop(); /* make sure we don't keep playing */
    DeInitMP(MidPakParaSeg);
    MidPakInit = FALSE;
}

/*
	Function: UnloadMidPak
    Description:
    	Unloads a MidPak driver.  Frees the memory associated
        with the main driver, the .ADV file, and the .AD file.
*/
void UnloadMidPak(void)
{
	/* make sure it's loaded */
	assert(MidPakLoaded);

    if (MidPakInit) {
    	DeInitMidPak();
    }
    if (AdLoaded) {
    	farfree(AdAddr);
    }
    farfree(AdvAddr);
    farfree(MidPakAddr);
}
