/* File: ANIMATE.C
** Description:
**   Routines for performing animation.  Includes routines for page
**   flipping, off-screen display buffers, and dirty rectangles.
** Author:
**   David G. Roberts
** Copyright:
**   Copyright 1994, David G. Roberts
*/

#include <assert.h>
#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include "gamedefs.h"
#include "animate.h"
#include "retrace.h"
#include "vga.h"

/*
	Function: SetScreenStart
    Description:
    	Sets the VGA screen start address to the offset specified.
        This has the effect of flipping display pages or of scrolling
        the screen a few lines (a flip is really just a massive,
        page-long scroll, right?).
*/
void SetScreenStart(UINT16 Offset)
{
	UINT8 OffsetLow;
    UINT8 OffsetHigh;

    OffsetLow = LOWBYTE(Offset);
    OffsetHigh = HIGHBYTE(Offset);

    /* disable interrupts to ensure this is virtually atomic */
    asm cli;

    outportb(CRTC_INDEX_REG, START_ADDRESS_HIGH_INDEX);
    outportb(CRTC_DATA_REG, OffsetHigh);

    outportb(CRTC_INDEX_REG, START_ADDRESS_LOW_INDEX);
    outportb(CRTC_DATA_REG, OffsetLow);

    asm sti;

}

/*
	Function: PageFlip
    Description:
    	Sets the VGA screen start address to the offset specified.
        The routine sets the new address while the VGA is in display
        mode and waits for retrace to become active before returning.
        This ensures that the address is set correctly (e.g., the
        VGA doesn't reload when only half of the address has been
        set) and that the old page is really invisible before we
        start manipulating things on it.
*/
void PageFlip(UINT16 Offset)
{
	WaitDisplayMode();
    SetScreenStart(Offset);
    WaitRetraceMode();
}

/*
	Function: CopyRects
    Description:
    	Copies an array of dirty rectangles from a virtual screen
        buffer to video memory.
*/
void CopyRects(RECT RectArray[], int NumRects, UINT8 far * VirtualBase)
{
	int RectCounter;
    /* int LineCounter; used in C version */
    /* int ColumnCounter; used in C version */
	UINT8 far * Screen;
    UINT8 far * VirtualScreen;
    UINT16 Offset;
    int LineIncrement;
    int Top, Left, Bottom, Right;

    for (RectCounter = 0; RectCounter < NumRects; RectCounter++) {
    	/* quickly copy these to simply the source code and */
        /* eliminate array index and structure member address */
        /* calculations */
    	Top		= RectArray[RectCounter].Top;
        Left	= RectArray[RectCounter].Left;
        Bottom	= RectArray[RectCounter].Bottom;
        Right	= RectArray[RectCounter].Right;

        assert(Top <= Bottom);
        assert(Left <= Right);

        /* initialize pointers to virtual and physical screens */
        Offset = MODE13H_WIDTH * Top + Left;
		Screen = MK_FP(VIDEO_MEM_SEGMENT, Offset);
        VirtualScreen = VirtualBase + Offset;

        LineIncrement = MODE13H_WIDTH - (Right - Left);

		/*
        This bit of C code is equivalent to the inline assembly
        language, below, and is shown here for reference.

        for (LineCounter = Top; LineCounter < Bottom; LineCounter++) {
            for (ColumnCounter = Left; ColumnCounter < Right;
				ColumnCounter++) {
                *Screen++ = *VirtualScreen++;
            }
            Screen += LineIncrement;
            VirtualScreen += LineIncrement;
        }
		*/

        /* do that copy in assembly because it's time critical */
        asm {
        	push ds
        	mov dx,Right
            sub dx,Left
            mov bx,Bottom
            sub bx,Top
            lds si,VirtualScreen
            les di,Screen
        }
        rowloop:
        asm {
        	mov cx,dx
            shr cx,1
            rep movsw
            adc cx,0
            rep movsb
            add si,LineIncrement
            add di,LineIncrement
            dec bx
            jnz rowloop
            pop ds
        }
    }
}