/* This program is an aid to solving various ciphers.
**
** main.C written by Mike Thomas
** wart@ugcs.caltech.edu
**
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include "term.h"
#include "types.h"
#include "ctypes.h"

char get_type(char *type);
void show_cipher_types(void);
int wordcmp(const char **s1, const char **s2);

int main(int argc, char **argv){
  FILE *ifptr;
  char option, type;
  char temp_str[STRINGLENGTH];
  char ifile[STRINGLENGTH];
  char **filelist;
  cipherdata *cptr;
  DIR *dptr;
  struct dirent *direntp;
  int i, num_files;

  *ifile = (char) NULL;
  --argc, ++argv;
  while(argc){
    if(**argv != '-'){
      strcpy(ifile, *argv);
      ++argv, --argc;
    }
    else{
      switch(*++*argv){
	case 'f':
	  ++argv, --argc;
	  if(argc){
	    strcpy(ifile, *argv);
	    ++argv, --argc;
	  }
	  break;
	case 't':
	  ++argv, --argc;
	  sscanf(*argv, "%s", temp_str);
	  --argc, ++argv;
	  type = get_type(temp_str);
	  break;
      }
    }
  }

  ready_term();
  set_input_mode(TRUE, TRUE, FALSE);

  /* Get the name of the file that holds the cipher.
  */

  if( ! *ifile){
    strcpy(ifile, "?");
    while(strcmp(ifile, "?") == 0){
      prompt("In what file does the cipher reside ('?' to list files)? ");
      read_line(ifile);

      /* If a '?' was entered, then print a list of all files in the
      ** current directory.
      */
      if(strcmp("?", ifile) == 0){
	dptr = opendir(".");
	i = 0;

	/* We need to save the list of filenames and sort them
	** ourself because readdir does not return the list
	** in sorted order.
	*/

	  /* First count the total number of files and allocate memory
	  ** for an array of filenames.  We won't actually use every
	  ** file because files beginning with "." are ignored.
	  */

	num_files = 0;
	rewinddir(dptr);
	while( (direntp = readdir(dptr)) != NULL) num_files++;
	rewinddir(dptr);
	filelist = new char*[num_files];
	for(i = 0; i < num_files; i++){
	  filelist[i] = NULL;
	}

	/* Now read in each filename and put it in the array
	*/

	i = 0;
	while( (direntp = readdir(dptr)) != NULL){
	  if(direntp->d_name[0] != '.'){
	    filelist[i++] = strdup(direntp->d_name);
	  }
	}
	filelist[i] = NULL;
	num_files = i;

	/* Sort the list of filenames and print them out
	*/

	qsort(filelist, i, sizeof(char *), wordcmp);

	for(i = 0; i < num_files; i++){
	  msgprint((i/DRAWABLE_HEIGHT)*DRAWABLE_HEIGHT, (i%DRAWABLE_HEIGHT), 
	       filelist[i]);
	}
	delete[] filelist;
	(void)closedir(dptr);
      }
    }
  }

  ifptr = fopen(ifile, "r");
  if(!ifptr){
    msgerror("Bad input file:  %s\n", ifile);
    unready_term();
    putchar('\n');
    exit(0);
  }

  /* Find out what type of cipher we're dealing with.
  */

  cptr = new cipherdata;
  type = cptr->read_header(ifptr);
  rewind(ifptr);

  do{
    if(!type){
      clear_to_prompt();
      show_cipher_types();
      prompt("What type is the cipher? ");
      read_line(temp_str);
      type = get_type(temp_str);
    }
    delete cptr;
    switch(type){
      case HOMOPHONIC:
	(homophonic *)cptr = (homophonic *)new homophonic;
	break;
      case PHILLIPS:
	(phillips *)cptr = (phillips *)new phillips;
	break;
      case COLUMNAR:
	(columnar *)cptr = (columnar *)new columnar;
	break;
      case SWAGMAN:
	(swagman *)cptr = (swagman *)new swagman;
	break;
      case NIHILIST:
	(nihilist *)cptr = (nihilist *)new nihilist;
	break;
      case CADENUS:
	(cadenus *)cptr = (cadenus *)new cadenus;
	break;
      case QUAG:
	(quag *)cptr = (quag *)new quag;
	break;
      case ARISTOCRAT:
	(aristocrat *)cptr = (aristocrat *)new aristocrat;
	break;
      case PORTA:
	(porta *)cptr = (porta *)new porta;
	break;
      case VARIANT:
	(variant *)cptr = (variant *)new variant;
	break;
      case BEAUFORT:
	(beaufort *)cptr = (beaufort *)new beaufort;
	break;
      case VIGENERE:
	(vigenere *)cptr = (vigenere *)new vigenere;
	break;
      case VARSLIDE:
	(varslide *)cptr = (varslide *)new varslide;
	break;
      case BEASLIDE:
	(beaslide *)cptr = (beaslide *)new beaslide;
	break;
      case VIGSLIDE:
	(vigslide *)cptr = (vigslide *)new vigslide;
	break;
      case POLLUX:
	(pollux *)cptr = (pollux *)new pollux;
	break;
      case MORBIT:
	(morbit *)cptr = (morbit *)new morbit;
	break;
      case FMORSE:
	(fmorse *)cptr = (fmorse *)new fmorse;
	break;
      case REDEFENCE:
	(redefence *)cptr = (redefence *)new redefence;
	break;
      case RAILFENCE:
	(railfence *)cptr = (railfence *)new railfence;
	break;
      case BACONIAN:
	(baconian *)cptr = (baconian *)new baconian;
	break;
      case CHECKER:
	(checker *)cptr = (checker *)new checker;
	break;
      case TRIDIGITAL:
	(tridigit *)cptr = (tridigit *)new tridigit;
	break;
      case KEYPHRASE:
	(keyphrase *)cptr = (keyphrase *)new keyphrase;
	break;
      case GRANDPRE:
	(grandpre *)cptr = (grandpre *)new grandpre;
	break;
      case VIGAUTOKEY:
	(vig_autokey *)cptr = (vig_autokey *)new vig_autokey;
	break;
      case VARAUTOKEY:
	(var_autokey *)cptr = (var_autokey *)new var_autokey;
	break;
      case BEAAUTOKEY:
	(bea_autokey *)cptr = (bea_autokey *)new bea_autokey;
	break;
      default:
	msgerror("Type %s not recognized.", temp_str);
	type = (char) NULL;
	break;
    }
    cptr->set_type(type);
  } while(!type);

  clear_to_prompt();

  /* Read in the cipher
  */
  
  if(cptr->read_cipher(ifptr)){
    msgerror("Corrupt save file.");
    unready_term();
    putchar('\n');
    exit(0);
  }
  (void) fclose(ifptr);


  /* Now display the cipher and print the menu.  This is where the 
  ** good stuff starts.  
  **
  ** Set the option to anything but QUIT so that the program doesn't
  ** terminate immediately.  The initial option can be anything because
  ** the program isn't going to act on it.
  */

  option = SAVE;
  clear_after_prompt();
  while(option != QUIT){

    /* Show the menu and get the user's option.
    */

    cptr->show_key();
    cptr->show_cipher();
    cptr->show_menu();
    prompt("Your choice? ");
    refresh();
    option = get_char();
    option |= ' ';
    clear_after_prompt();

    if(cptr->execute_option(option) == FALSE){
      msgerror("That is not a valid option.");
    }
    refresh();
  }

  unready_term();
  (void) putchar('\n');
  return(0);
}

char
get_type(char *string){
  char type;

  if(strcmp(string, "1") == 0)
    type = ARISTOCRAT;
  else if(strcmp(string, "2") == 0)
    type = QUAG;
  else if(strcmp(string, "3") == 0)
    type = BACONIAN;
  else if(strcmp(string, "4") == 0)
    type = VIGENERE;
  else if(strcmp(string, "5") == 0)
    type = VARIANT;
  else if(strcmp(string, "6") == 0)
    type = BEAUFORT;
  else if(strcmp(string, "7") == 0)
    type = COLUMNAR;
  else if(strcmp(string, "8") == 0)
    type = NIHILIST;
  else if(strcmp(string, "9") == 0)
    type = CADENUS;
  else if(strcmp(string, "10") == 0)
    type = SWAGMAN;
  else if(strcmp(string, "11") == 0)
    type = PHILLIPS;
  else if(strcmp(string, "12") == 0)
    type = HOMOPHONIC;
  else if(strcmp(string, "13") == 0)
    type = RAILFENCE;
  else if(strcmp(string, "14") == 0)
    type = REDEFENCE;
  else if(strcmp(string, "15") == 0)
    type = POLLUX;
  else if(strcmp(string, "16") == 0)
    type = MORBIT;
  else if(strcmp(string, "17") == 0)
    type = FMORSE;
  else if(strcmp(string, "18") == 0)
    type = PORTA;
  else if(strcmp(string, "19") == 0)
    type = CHECKER;
  else if(strcmp(string, "20") == 0)
    type = GRANDPRE;
  else if(strcmp(string, "21") == 0)
    type = TRIDIGITAL;
  else if(strcmp(string, "22") == 0)
    type = VIGSLIDE;
  else if(strcmp(string, "23") == 0)
    type = VARSLIDE;
  else if(strcmp(string, "24") == 0)
    type = BEASLIDE;
  else if(strcmp(string, "25") == 0)
    type = KEYPHRASE;
  else if(strcmp(string, "26") == 0)
    type = VIGAUTOKEY;
  else if(strcmp(string, "27") == 0)
    type = BEAAUTOKEY;
  else if(strcmp(string, "28") == 0)
    type = VARAUTOKEY;
  else
    type = (char) NULL;

  return type;
}

void
show_cipher_types(){
  msgprint(5, 0, " 1)  Aristocrat\t\t 2)  Quagmire");
  msgprint(5, 1, " 3)  Baconian\t\t 4)  Vigenere");
  msgprint(5, 2, " 5)  Variant\t\t 6)  Beaufort");
  msgprint(5, 3, " 7)  Col. Transposition\t 8)  Nihilist Transposition");
  msgprint(5, 4, " 9)  Cadenus\t\t10)  Swagman");
  msgprint(5, 5, "11)  Phillips\t\t12)  Homophonic");
  msgprint(5, 6, "13)  Railfence\t\t14)  Redefence");
  msgprint(5, 7, "15)  Pollux\t\t16)  Morbit");
  msgprint(5, 8, "17)  Fractionated Morse\t18)  Porta");
  msgprint(5, 9, "19)  Checkerboard\t\t20)  Grandpre");
  msgprint(5, 10,"21)  Tridigital\t\t22)  Vig Slidefair");
  msgprint(5, 11,"23)  Var Slidefair\t\t24)  Bea Slidefair");
  msgprint(5, 12,"25)  Key phrase\t\t26)  Vigenere Autokey");
  msgprint(5, 13,"27)  Beaufort Autokey\t28)  Variant Autokey");
}

int 
wordcmp(const char **word1, const char **word2){
  return strcmp(*word1, *word2);
}
