/* OpenDoors BBS Framework 1.00
 * Written 1995 - 1996 by Brian Pirie. Released to the public domain.
 *
 *
 *        File: ex_bbs.c
 *
 * Description: If you want to create your own custom BBS system from the
 *              ground up using OpenDoors, this program provides a simple
 *              framework that you can build upon.
 *
 *              This program has the logic to accept already established
 *              incoming calls or to answer the modem when the line rings. Once
 *              a connection has been established, this program tests the
 *              remote system's display capabilities and then prompts the user
 *              to log in. The program maintains a simple user file that
 *              you may extend to add whatever information you require, and
 *              provides a new user login proceedure. The program also handles
 *              user daily time limits, which are reset on the user's first
 *              call of the day.
 *
 *              The rest is left up to you to create and structure however you
 *              want. See the source code comments for instructions on where
 *              you should add your own code.
 *
 *              To start up the BBS in local mode, simply run it with the
 *              -local command line option. To start up the BBS in remote
 *              mode, the serial port number and locked/initial BPS rate
 *              MUST BOTH be specified on the command line. The BBS system
 *              always exits after each call, and so it will generally be
 *              run from a batch file.
 *
 *              This program requires OpenDoors 6.00 or later, and can be
 *              compiled as either a plain-DOS or native-Windows 95/NT program.
 *
 *
 *   Revisions: Date          Ver   Who  Change
 *              ---------------------------------------------------------------
 *              Mar 12, 1996  1.00  BP   New file header format.
 */

/* MULTINODE_AWARE should be defined for multi-node aware file access. */
#define MULTINODE_AWARE

/* Include required header files. */

/* Standard C run-time library. */
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#ifdef MULTINODE_AWARE
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <share.h>
#endif

/* OpenDoors. */
#include "opendoor.h"

/* BBS Framework. */
#include "ex_bbs.h"


/* Default configuration settings. */
tConfigurationRecord Config =
{
   "ATM0&C1E0",                   /* Modem initialization command string. */
   "RING",                        /* Phone is ringing result string. */
   "ATA",                         /* Modem answer command string. */
   "CONNECT",                     /* Modem connect result string. */
   "bbs.log",                     /* Log file name. */
   200,                           /* Maximum inactivity time, in seconds */
   10,                            /* New user security. */
   60,                            /* Default daily time limit. */
   5,                             /* Login time limit. */
   5,                             /* Maximum number of password retries. */
};


/* Current user record. */
tUserRecord CurrentUserRecord;


/* Hard-coded filenames. */
#define FILENAME_TITLE "logo"     /* Displayed before prompting for name. */
#define FILENAME_NEWS  "news"     /* Displayed after user logs in. */
#define FILENAME_USER  "user.dat" /* Name of user file. */


/* Other configurable constants. */
#define RECEIVE_STR_SIZE         80
#define FILE_ACCESS_MAX_WAIT     15


/* main() or WinMain() function - Program execution begins here. */
#ifdef ODPLAT_WIN32
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
   LPSTR lpszCmdLine, int nCmdShow)
#else
int main(int argc, char *argv[])
#endif
{
   /* If you are going to read the configuration record information from an */
   /* external configuration file (rather than using the hard-coded         */
   /* defaults), then you should ADD CONFIGURATION FILE I/O HERE.           */


#ifdef ODPLAT_WIN32
   /* In Windows, pass in nCmdShow value to OpenDoors. */
   od_control.od_cmd_show = nCmdShow;

   /* Ignore unused parameters. */
   (void)hInstance;
   (void)hPrevInstance;
#endif

   /* Setup program and version information. */
   strcpy(od_control.od_prog_name, "OpenDoors BBS");
   strcpy(od_control.od_prog_version, "Version 1.00");
   strcpy(od_control.od_prog_copyright, "Written 1995-1996 by Brian Pirie");

   /* Handle command-line arguments. */
#ifdef ODPLAT_WIN32
   od_parse_cmd_line(lpszCmdLine);
#else
   od_parse_cmd_line(argc, argv);
#endif

   /* Turn off certain default OpenDoors features. */
   od_control.od_disable
     = DIS_INFOFILE                  /* Don't attempt to read a drop file. */
     | DIS_CARRIERDETECT             /* Don't monitor carrier detection yet. */
     | DIS_TIMEOUT                   /* Don't monitor user timeouts yet. */
     | DIS_NAME_PROMPT;              /* Don't prompt for name in local mode. */
   od_control.od_nocopyright = TRUE; /* Don't display OpenDoors copyright. */
   od_set_statusline(STATUS_NONE);   /* Don't display status line yet. */

   /* Place your OpenDoors registration information here. */
   od_registration_key = 0000000000L;
   strcpy(od_registered_to, "Your Name");

   /* Now, initialize OpenDoors. */
   od_init();

   /* Don't monitor user inactivity yet. */
   od_control.od_inactivity = 0;

   /* Attempt to establish a connection. */
   EstablishConnection();

   /* Now that we have a connection, start the BBS session. */
   BBSSession();

   /* Shutdown OpenDoors and exit program. */
   od_exit(0, TRUE);
   return(0);
}


/* EstablishConnection() - Function that will return only after a connection */
/*                         has been establish.                               */
void EstablishConnection(void)
{
   char szReceivedStr[RECEIVE_STR_SIZE + 1];
   int nIndex;

   /* First, check whether we are operating in local mode. If so, return */
   /* immediately, because we always have a local "connection".          */
   if(od_control.od_force_local)
   {
      return;
   }

   /* Next, check whether we have an already established connection on this */
   /* line. If so, then we don't need to wait for an incoming call.         */
   if(od_carrier())
   {
      return;
   }

   /* Check that we have been started with required command line parameters. */
   if(od_control.baud == 0)
   {
      od_printf("Error: -port x and -bps x must both be specified on command line\n\r");
      od_printf("to have the BBS handle inbound calls.\n\r");
      od_printf("Use -local to log in in local mode.\n\r\n\r");
      od_printf("Press [Enter] to exit.\n\r");
      od_get_answer("\n\r");
      od_exit(1, FALSE);
   }

   /* Otherwise, we must wait for an incoming call. */
   od_disp_emu("The OpenDoors BBS is waiting for a call...\n\r", FALSE);
   od_disp_emu("Press [ALT]-[D] to exit.\n\r\n\r", FALSE);

   /* Initialize the modem. */
   SendCommandToModem(Config.szModemInit);

   /* Loop until we have a connection. */
   for(;;)
   {
      /* Get the next line of data from the mode. */
      for(nIndex = 0; nIndex < RECEIVE_STR_SIZE; ++nIndex)
      {
         szReceivedStr[nIndex] = od_get_key(TRUE);
         szReceivedStr[nIndex + 1] = '\0';
         if(szReceivedStr[nIndex] == '\r' || szReceivedStr[nIndex] == '\n')
         {
            break;
         }
      }

      if(strstr(szReceivedStr, Config.szRingString))
      {
         /* If line rings, then tell the modem to answer. */
         SendCommandToModem(Config.szAnswerString);
      }
      else if(strstr(szReceivedStr, Config.szConnectString))
      {
         /* If we have a connection, then exit wait for connect loop. */
         break;
      }
   }

   /* Wait for a few seconds before doing anything after connection. */
   od_sleep(2000);
}


/* SendCommandToModem() - Sends a simple command to the modem. */
void SendCommandToModem(char *pszModemCommand)
{
   od_disp(pszModemCommand, strlen(pszModemCommand), FALSE);
   od_disp("\r\n", 2, FALSE);
}


/* BBSSession() - Controls the flow of events during a call. */
void BBSSession(void)
{
   tUserRecord *pCurrentUserRecord = &CurrentUserRecord;

   /* Allow the user to log in. */
   if(!BBSLogin(pCurrentUserRecord))
   {
      /* If the user did not log in, then do not proceed with this session. */
      return;
   }

   /* Display news file to the user. */
   od_send_file(FILENAME_NEWS);

   /* Prompt for enter before proceeding. */
   PromptForEnter();


   /* ADD YOUR MAIN BBS CODE HERE. */
   /* The user is now online and ready for the main menu, bulletin menu, */
   /* or whatever you want to do once the user is online.                */
}


/* BBSLogin() - Handles the user login proceedure. */
BOOL BBSLogin(tUserRecord *pCurrentUserRecord)
{
   DWORD dwPasswordAttempts;
   struct tm CurrentTime;
   struct tm LastCallTime;
   time_t CurrentTimeInSeconds;

   /* Now that we have established a connection, re-enable timeouts */
   /* and carrier detection.                                        */
   od_control.user_timelimit = Config.dwLoginTimeLimit;
   od_control.od_inactivity = (int)Config.dwMaximumInactivity;
   od_control.od_disable = 0;

   /* Clear any garbage out of the inbound keyboard buffer. */
   od_clear_keybuffer();

   /* Clear the screen and perform test of remote terminal's display */
   /* capabilities.                                                  */
   od_printf("\n\r");
   od_clr_scr();
   od_printf("Testing your software's display capabilities...\n\r");
   od_autodetect(DETECT_NORMAL);

   /* Display title file before prompting for a user's name, if one has */
   /* been provided.                                                    */
   od_send_file(FILENAME_TITLE);

   /* Prompt the user for their name. */
   od_printf("\n\r`bright yellow`OpenDoors BBS 1.00");

   /* Loop until the user has logged in. */
   for(;;)
   {
      od_printf("`bright yellow`\n\rPlease enter your full name: `bright white`");

      /* Obtain the user's name. */
      od_input_str(od_control.user_name, 35, 32, 126);

      /* If the user did not enter a valid name, then prompt again. */
      if(strlen(od_control.user_name) == 0
         || strchr(od_control.user_name, ' ') == NULL)
      {
         continue;
      }

      /* Check for the user's name in the user file. */

      if(ReadUser(pCurrentUserRecord, od_control.user_name))
      {
         /* If the user's record has been found, then confirm that this */
         /* isn't a second person with the same name.                   */
         od_printf("\n\r%s `bright yellow`from `bright white`%s`bright yellow`? (`bright white`Y`bright yellow`/`bright white`N`bright yellow`)",
            pCurrentUserRecord->szName,
            pCurrentUserRecord->szLocation);
         if(od_get_answer("YN") == 'N')
         {
            od_printf("\n\r");
            continue;
         }
         od_printf("\n\r");

         /* This is the correct user, so prompt for the password. */
         for(dwPasswordAttempts = 0;
            dwPasswordAttempts < Config.dwPasswordRetries;
            ++dwPasswordAttempts)
         {
            od_printf("`bright yellow`\n\rPassword: `bright white`");
            PasswordInput(od_control.user_password, 15);
            if(stricmp(od_control.user_password,
               pCurrentUserRecord->szPassword) == 0)
            {
               /* If the correct password was entered, then proceed. */
               break;
            }
         }
         od_printf("`bright yellow`");

         /* If maximum number of password retries have been exceeded, then */
         /* end this connection.                                           */
         if(dwPasswordAttempts >= Config.dwPasswordRetries)
         {
            od_printf("Maximum number of password retries exceeded.\n\r");
            od_exit(1, TRUE);
         }

         /* If the user's last call was before today, then reset the amount */
         /* of time that the user has used today.                           */
         memcpy(&LastCallTime,
            localtime(&pCurrentUserRecord->LastCallTime),
            sizeof(struct tm));
         CurrentTimeInSeconds = time(NULL);
         memcpy(&CurrentTime, localtime(&CurrentTimeInSeconds),
            sizeof(struct tm));
         if(LastCallTime.tm_yday != CurrentTime.tm_yday
            || LastCallTime.tm_year != CurrentTime.tm_year)
         {
            pCurrentUserRecord->nTimeUsed = 0;
         }

         /* Update user's total number of calls. */
         pCurrentUserRecord->dwNumCalls++;

         /* Update the time of the user's last call. */
         pCurrentUserRecord->LastCallTime = time(NULL);

         /* Update the user's record on disk. */
         WriteUser(pCurrentUserRecord);

         /* Exit loop. */
         break;
      }

      /* If user's name was not found, then check whether it is correct. */
      else
      {
         /* Confirm entered name with the user. */
         od_printf("`bright yellow`\n\rYour name was not found in our records. Either you have entered your name\n\r");
         od_printf("incorrectly, or this is the first time you have called. If this is your first\n\r");
         od_printf("call, then you may proceed to log in as a new user.\n\r\n\r");

         od_printf("Name entered: `bright white`%s`bright yellow`\n\r\n\r",
            od_control.user_name);

         od_printf("Is this name correct? (`bright white`Y`bright yellow`/`bright white`N`bright yellow`)");

         if(od_get_answer("YN") == 'N')
         {
            /* If the user's name was incorrectly entered, prompt again. */
            continue;
         }

         /* If the name is correct, proceed with the new user proceedure. */
         if(!NewUser(pCurrentUserRecord))
         {
            od_printf("Sorry, unable to create your new user record.");
            od_exit(1, TRUE);
         }

         /* Exit loop. */
         break;
      }
   }

   /* Fill od_control with information from user record. */
   od_control.user_num = pCurrentUserRecord->dwUserNumber;
   strcpy(od_control.user_name, pCurrentUserRecord->szName);
   strcpy(od_control.user_location, pCurrentUserRecord->szLocation);
   strcpy(od_control.user_homephone, pCurrentUserRecord->szPhone);
   strcpy(od_control.user_password, pCurrentUserRecord->szPassword);
   od_control.user_security = pCurrentUserRecord->dwSecurity;
   od_control.user_numcalls = pCurrentUserRecord->dwNumCalls;

   /* Determine the maximum time for this connection. */
   od_control.user_timelimit = (int)Config.dwDailyTimeLimit
      - pCurrentUserRecord->nTimeUsed;

   /* Turn on status line. */
   od_set_statusline(STATUS_NORMAL);

   /* Clear the screen. */
   od_clr_scr();

   /* Set OpenDoors to call EndBBSSession() at exit time. */
   od_control.od_before_exit = EndBBSSession;

   /* Open log file. */
   strcpy(od_control.od_logfile_name, Config.szLogfileName);
   od_log_open();

   /* Return with success, indicating that we have a user logged in. */
   return(TRUE);
}


/* NewUser() - Handles the new user account creation proceedure. */
BOOL NewUser(tUserRecord *pCurrentUserRecord)
{
   od_printf("\n\r");

   /* Fill the user record with the default new user information that */
   /* we already know.                                                */
   strcpy(pCurrentUserRecord->szName, od_control.user_name);
   pCurrentUserRecord->FirstCallTime = time(NULL);
   pCurrentUserRecord->LastCallTime = time(NULL);
   pCurrentUserRecord->nTimeUsed = 0;
   pCurrentUserRecord->dwSecurity = Config.dwNewUserSecurity;
   pCurrentUserRecord->dwNumCalls = 1;

   /* Prompt for the user's location. */
   od_printf("`bright yellow`\n\rWhere are you calling from?\n\r");
   od_printf("[-------------------------]`bright white`\n\r ");
   od_input_str(pCurrentUserRecord->szLocation, 25, 32, 126);

   /* Prompt for the user's phone number. */
   od_printf("`bright yellow`\n\rWhat is your phone number?\n\r");
   od_printf("[---------------]`bright white`\n\r ");
   od_input_str(pCurrentUserRecord->szPhone, 15, 32, 126);

   /* Prompt for the user's password. */
   for(;;)
   {
      char szVerifiedPassword[16];

      /* Obtain password from user. */
      od_printf("`bright yellow`\n\rWhat password do you wish to use to access this system?\n\r");
      od_printf("[---------------]`bright white`\n\r ");
      PasswordInput(pCurrentUserRecord->szPassword, 15);

      /* Check that this is a valid password. */
      if(strlen(pCurrentUserRecord->szPassword) < 4)
      {
         /* In the case of an invalid password, prompt again. */
         od_printf("`bright yellow`\n\rThis is not a valid password.\n\r");
         od_printf(
            "Your password must be at least 4 characters in length.\n\r");
         continue;
      }

      /* Verify password. */
      od_printf("`bright yellow`\n\rPlease re-enter your password to verify:\n\r");
      od_printf("[---------------]`bright white`\n\r ");
      PasswordInput(szVerifiedPassword, 15);
      od_printf("`bright yellow`");

      /* If passwords match, then accept password. */
      if(strcmp(szVerifiedPassword, pCurrentUserRecord->szPassword) == 0)
      {
         break;
      }

      /* If passwords don't match, then tell the user. */
      od_printf("\n\rPasswords don't match!\n\r");
   }

   /* Add this user to the user file. */
   return(CreateUser(pCurrentUserRecord));
}


/* EndBBSSession() - Called by od_exit() to end the current BBS session. */
void EndBBSSession(void)
{
   tUserRecord *pCurrentUserRecord = &CurrentUserRecord;

   /* Log out the current user. */
   LogoutUser(pCurrentUserRecord);
}


/* LogoutUser() - Called to log out the current user and updates the user's */
/*                record in the user file.                                  */
BOOL LogoutUser(tUserRecord *pCurrentUserRecord)
{
   /* Determine how much time (if any) the user has used. */
   pCurrentUserRecord->nTimeUsed = Config.dwDailyTimeLimit
      - od_control.user_timelimit;

   /* Update the user's record. */
   return(WriteUser(pCurrentUserRecord));
}


/* ReadUser() - Searches the user file for a user with the specified name. */
/*              If found, the provided user record is filled with          */
/*              information about the user.                                */
BOOL ReadUser(tUserRecord *pCurrentUserRecord, char *pszUserName)
{
   FILE *fpUserFile;
   int hUserFile;
   tUserRecord UserRecord;

   /* Attempt to open the user file for exclusive access. */
   fpUserFile = ExclusiveFileOpen(FILENAME_USER, "rb", &hUserFile);
   if(fpUserFile == NULL) return(FALSE);

   /* Loop through the records in the user file. */
   while(!feof(fpUserFile))
   {
      /* Attempt to read the next record from the file (if any). */
      if(fread(&UserRecord, sizeof(tUserRecord), 1, fpUserFile) != 1)
      {
         break;
      }

      /* Check whether the user names match. */
      if(stricmp(pszUserName, UserRecord.szName) == 0)
      {
         /* We have found this user, so copy the user information into the */
         /* current user record, close the file and return with success.   */
         memcpy(pCurrentUserRecord, &UserRecord, sizeof(tUserRecord));
         ExclusiveFileClose(fpUserFile, hUserFile);
         return(TRUE);
      }
   }

   /* Close the user file. */
   ExclusiveFileClose(fpUserFile, hUserFile);

   /* We haven't found a matching user, so return with failure. */
   return(FALSE);
}


/* CreateUser() - Adds a new user to the user file, using the information in */
/*                the provided user record, and updating the user record     */
/*                with the user number that was assigned to this user.       */
BOOL CreateUser(tUserRecord *pCurrentUserRecord)
{
   FILE *fpUserFile;
   int hUserFile;

   /* Attempt to open the user file for exclusive access. */
   fpUserFile = ExclusiveFileOpen(FILENAME_USER, "w+b", &hUserFile);
   if(fpUserFile == NULL) return(FALSE);

   /* Move to the end of the user file. */
   fseek(fpUserFile, 0, SEEK_END);

   /* Determine what user number should be assigned to this new user. */
   pCurrentUserRecord->dwUserNumber = ftell(fpUserFile) / sizeof(tUserRecord);
   
   /* Write the user's record to the file. */
   fwrite(pCurrentUserRecord, sizeof(tUserRecord), 1, fpUserFile);

   /* Close the user file and return with success. */
   ExclusiveFileClose(fpUserFile, hUserFile);
   return(TRUE);
}


/* WriteUser() - Updates the user file with the information about the */
/*               current user.                                        */
BOOL WriteUser(tUserRecord *pCurrentUserRecord)
{
   FILE *fpUserFile;
   int hUserFile;

   /* Attempt to open the user file for exclusive access. */
   fpUserFile = ExclusiveFileOpen(FILENAME_USER, "wb", &hUserFile);
   if(fpUserFile == NULL) return(FALSE);

   /* Move to the position of this user's record in the user file. */
   fseek(fpUserFile, sizeof(tUserRecord) * pCurrentUserRecord->dwUserNumber,
      SEEK_SET);

   /* Write the user's record to the file. */
   fwrite(pCurrentUserRecord, sizeof(tUserRecord), 1, fpUserFile);

   /* Close the user file and return with success. */
   ExclusiveFileClose(fpUserFile, hUserFile);
   return(TRUE);
}


/* PromptForEnter() - Prompts the user to press the enter key. */
void PromptForEnter(void)
{
   od_printf("`bright yellow`Press [Enter] to continue.");
   od_get_answer("\n\r");
   od_printf("\n\r");
}


/* ExclusiveFileOpen() - Opens a file for exclusive access by this node. */
FILE *ExclusiveFileOpen(char *pszFileName, char *pszMode, int *phHandle)
{
#ifdef MULTINODE_AWARE
   /* If Vote is being compiled for multinode-aware file access, then   */
   /* attempt to use compiler-specific share-aware file open functions. */
   FILE *fpFile = NULL;
   time_t StartTime = time(NULL);
   int hFile;

   /* Attempt to open the file while there is still time remaining. */    
   while((hFile = sopen(pszFileName, O_BINARY | O_RDWR, SH_DENYRW,
      S_IREAD | S_IWRITE)) == -1)
   {
      /* If we have been unable to open the file for more than the */
      /* maximum wait time, or if open failed for a reason other   */
      /* than file access, then attempt to create a new file and   */
      /* exit the loop.                                            */
      if(errno != EACCES ||
         difftime(time(NULL), StartTime) >= FILE_ACCESS_MAX_WAIT)
      {
         hFile = sopen(pszFileName, O_BINARY | O_CREAT, SH_DENYRW,
            S_IREAD | S_IWRITE);
         break;
      }

      /* If we were unable to open the file, call od_kernel, so that    */
      /* OpenDoors can continue to respond to sysop function keys, loss */
      /* of connection, etc.                                            */
      od_kernel();

      /* Now, give other processes a chance to run. */
      od_sleep(250);
   }

   /* Attempt to obtain a FILE * corresponding to the handle. */
   if(hFile != -1)
   {
      fpFile = fdopen(hFile, pszMode);
      if(fpFile == NULL)
      {
         close(hFile);
      }
   }

   /* Pass file handle back to the caller. */
   *phHandle = hFile;

   /* Return FILE pointer for opened file, if any. */   
   return(fpFile);
#else
   /* Ignore unused parameters. */
   (void)phHandle;

   /* If Vote is not being compiled for multinode-aware mode, then just */
   /* use fopen to access the file.                                     */
   return(fopen(pszFileName, pszMode));
#endif
}


/* ExclusiveFileClose() - Closes a file opened by ExclusiveFileOpen(). */
void ExclusiveFileClose(FILE *pfFile, int hHandle)
{
   fclose(pfFile);
#ifdef MULTINODE_AWARE
   close(hHandle);
#else
   /* Ignore unused parameters. */
   (void)hHandle;
#endif
}


/* PasswordInput() - Inputs a string in password mode. */
void PasswordInput(char *pszInput, INT nMaxLength)
{
   char chKeyPressed;
   INT nPosition = 0;

   for(;;)
   {
      chKeyPressed = od_get_key(TRUE);

      if(chKeyPressed == '\r' || chKeyPressed == '\n')
      {
         pszInput[nPosition] = '\0';
         od_disp_str("\n\r");
         return;
      }
      else if(chKeyPressed == 8)
      {
         if(nPosition > 0)
         {
            od_disp_str("\b \b");
            --nPosition;
         }
      }
      else if(chKeyPressed >= ' ' && chKeyPressed <= 126
         && nPosition < nMaxLength)
      {
         od_putch('*');
         pszInput[nPosition++] = chKeyPressed;
      }
   }
}
