/* YAREdit 1.00
 * (C) Copyright 1996 by Brian Pirie (pirie@msn.com). All Rights Reserved.
 *
 *
 *        File: yaredit.c
 *
 * Description: Implementation of YAREdit full screen external editor for
 *              RemoteAccess
 *
 *              Requires OpenDoors 6.10 or later to compile.
 *
 *   Revisions: Date          Ver   Who  Change
 *              ---------------------------------------------------------------
 *              Mar 13, 1996  1.00  BP   Created.
 */

/* Header files. */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "opendoor.h"

/* Global variables. */
BOOL bSave;
char *pszMessageFilename = "MSGTMP.";
char *pszInfoFilename = "MSGINF.";

/* Message information. */
char szMessageFrom[36];
char szMessageTo[36];
char szMessageSubject[73];
UINT unMessageNum;
char szAreaName[41];
BOOL bPrivate;

/* Main editor colors. */
char btTextColor = 0x03;
char btTopBarLeftColor = 0x74;
char btTopBarCenterColor = 0x7f;
char btTopBarRightColor = 0x71;
char btFieldNameColor = 0x0e;
char btFieldContentsColor = 0x0a;
char btFlagColor = 0x0f;
char btSeparatorColor = 0x10;

/* Full-screen chat options. */
char abtWindowColor[2] = {0x0b, 0x0c};   /* Text colour used for each person */
char btBarColor = 0x70;                   /* Colour of window seperation bar */
char abtTopLine[2] = {13, 1}; /* Specifies location of each window on screen */
char abtBottomLine[2] = {23, 11};    /* Line number of bottom of each window */
char btBarLine = 12;                 /* Line number of window seperation bar */
char btScrollDistance = 2;        /* Distance to scroll window when required */
char btShellWindowTitle = 0x7f;/* Colour of title of DOS shell notice window */
char btShellWindowBoarder = 0x70;      /* Colour of DOS shell window boarder */
char btShellWindowText = 0x74;         /* Colour of text in DOS shell window */

char btCursorWindow;
char aszCurrentWord[2][81];
int anWordLength[2];
int anCursorCol[2];
int anCursorLine[2];
unsigned char chKey;
char abtScreenBuffer[4004];

/* DOS shell variables. */
void *pShellWindow;
INT nPreShellRow;
INT nPreShellColumn;

/* Function prototypes. */
void EditFile(char *pszFilename);
tODEditMenuResult MenuFunction(void *pUnused);
void OnlineHelp(void);
void LoadFile(char *pszFilename, char **ppszFileContents, size_t *pBufferSize);
void SaveFile(char *pszFilename, char *pszFileContents);
void LoadMessageInfo(char *pszFilename);
void SaveMessageInfo(char *pszFilename);
void ReadLine(FILE *pFile, char *pszDest, size_t SizeOfDest);
void FullScreenChat(void);
void ChatNewLine(void);
void DisplayShellWindow(void);
void RemoveShellWindow(void);
void DisplayPrivateFlag(void);
void EditFields(void);


/* main() or WinMain() function - Program execution begins here. */
#ifdef ODPLAT_WIN32
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
   LPSTR lpszCmdLine, int nCmdShow)
#else
int main(int argc, char *argv[])
#endif
{
#ifdef ODPLAT_WIN32
   /* In Windows, pass in nCmdShow value to OpenDoors. */
   od_control.od_cmd_show = nCmdShow;

   /* Ignore unused parameters. */
   (void)hInstance;
   (void)hPrevInstance;
#endif
   
   /* Set program's name for use by OpenDoors and registration information. */
   strcpy(od_control.od_prog_name, "YAREdit");
   strcpy(od_control.od_prog_version, "Version 1.00");
   strcpy(od_control.od_prog_copyright, "Copyright 1996 by Brian Pirie");

   /* Call the standard command-line parsing function. */
#ifdef ODPLAT_WIN32
   od_parse_cmd_line(lpszCmdLine);
#else
   od_parse_cmd_line(argc, argv);
#endif

   /* Default to using the RemoteAccess personality. */
   od_control.od_default_personality = PER_RA;

   /* Initialize OpenDoors. */
   od_init();

   /* Setup full screen chat function. */
   od_control.od_cbefore_chat = FullScreenChat;
   od_control.od_after_chat = NULL;

   /* Provide a popup window when the sysop shells to DOS. */
   od_control.od_cbefore_shell = DisplayShellWindow;
   od_control.od_cafter_shell = RemoveShellWindow;
   od_control.od_before_shell = NULL;
   od_control.od_after_shell = NULL;

   /* Edit the file. */
   EditFile(pszMessageFilename);

   /* Exit program. */
   od_exit(0, FALSE);
   return(0);
}


/* The EditFile() function loads a text file, allows the user to edit it,  */
/* and then saves the contents of the file to the disk if the user did not */
/* cancel.                                                                 */
void EditFile(char *pszFilename)
{
   tODEditOptions EditOptions;
   char *pszBuffer = NULL;
   size_t BufferSize = 0;
   struct tm *pTimeStructure;
   time_t CurrentTime;

   /* Attempt to load the contents of the file. */
   LoadFile(pszFilename, &pszBuffer, &BufferSize);

   /* Load message information. */
   LoadMessageInfo(pszInfoFilename);

   /* If we were unable to open the file, then return with failure. */
   if(pszBuffer == NULL)
   {
      pszBuffer = malloc(BufferSize = 4000);
      *pszBuffer = '\0';
   }

   /* Begin with a clear screen. */
   od_clr_scr();

   /* Display an information bar along the top of the window. */
   od_set_cursor(1, 1);
   od_set_attrib(btTopBarLeftColor);
   od_disp_str("  ");
   od_disp_str(szAreaName);
   od_repeat(' ', 41 - strlen(szAreaName));
   od_set_attrib(btTopBarCenterColor);
   od_disp_str("[ESC] for YAREdit menu       ");
   od_set_attrib(btTopBarRightColor);
   od_printf("#%-5u  ", unMessageNum);

   /* Display date and time line. */
   od_set_cursor(2, 1);
   od_set_attrib(btFieldNameColor);
   od_disp_str("Dt: ");
   od_set_attrib(btFieldContentsColor);
   CurrentTime = time(NULL);
   pTimeStructure = localtime(&CurrentTime);
   od_printf("%02.2d-%02.2d-%02.2d %02.2d:%02.2d",
      pTimeStructure->tm_mday,
      pTimeStructure->tm_mon,
      pTimeStructure->tm_year % 100,
      pTimeStructure->tm_hour,
      pTimeStructure->tm_min);

   /* Display "Private" flag. */
   DisplayPrivateFlag();

   /* Display sender's name line. */
   od_set_cursor(3, 1);
   od_set_attrib(btFieldNameColor);
   od_disp_str("By: ");
   od_set_attrib(btFieldContentsColor);
   od_disp_str(szMessageFrom);

   /* Display receiver's name line. */
   od_set_cursor(4, 1);
   od_set_attrib(btFieldNameColor);
   od_disp_str("To: ");
   od_set_attrib(btFieldContentsColor);
   od_disp_str(szMessageTo);

   /* Display message subject line. */
   od_set_cursor(5, 1);
   od_set_attrib(btFieldNameColor);
   od_disp_str("Re: ");
   od_set_attrib(btFieldContentsColor);
   od_disp_str(szMessageSubject);

   /* Display separator line. */
   od_set_cursor(6, 1);
   od_set_attrib(btSeparatorColor);
   od_clr_line();

   /* Setup edit options structure with information to adjust how */
   /* od_multiline_edit() behaves.                                */

   /* First reset the contents of the edit options structure. */
   memset(&EditOptions, 0, sizeof(tODEditOptions));

   /* Setup area of screen that will be used. */
   EditOptions.nAreaLeft = 1;
   EditOptions.nAreaRight = 80;
   EditOptions.nAreaTop = 7;
   EditOptions.nAreaBottom = 23;

   /* We want to save the message with paragraph breaks. */
   EditOptions.TextFormat = FORMAT_PARAGRAPH_BREAKS;

   /* Setup menu callback function. */
   EditOptions.pfMenuCallback = MenuFunction;

   /* Setup memory reallocation function to allow od_multiline_edit() to */
   /* grow the size of the buffer.                                       */
   EditOptions.pfBufferRealloc = realloc;

   /* Set the appropriate text color before starting the editor. */
   od_set_attrib(btTextColor);

   /* Start up the editor. */
   od_multiline_edit(pszBuffer, BufferSize, &EditOptions);

   /* Save the file if the user did not cancel. */
   if(bSave)
   {
      SaveFile(pszFilename, EditOptions.pszFinalBuffer);
      SaveMessageInfo(pszInfoFilename);
   }

   /* Deallocate buffer memory. */
   free(EditOptions.pszFinalBuffer);
}


/* MenuFunction() allows you to present your own menu and control */
/* what happens when  the user presses the escape key.            */
tODEditMenuResult MenuFunction(void *pUnused)
{
   (void)pUnused;

   /* Use od_popup_menu() to provide a simple menu for the user. */
   switch(od_popup_menu("YAREdit Menu",
      "^Send Message|^Cancel Message|^Help",
      44, 2, 0, MENU_ALLOW_CANCEL))
   {
      /* User chose to save changes. */
      case 1:
         bSave = TRUE;
         return(EDIT_MENU_EXIT_EDITOR);

      /* User chose to cancel changes. */
      case 2:
         bSave = FALSE;
         return(EDIT_MENU_EXIT_EDITOR);

      /* User chose to edit header. */
      case 4:
         EditFields();
         break;

      /* User chose to toggle private flag. */
      case 5:
         bPrivate = !bPrivate;
         DisplayPrivateFlag();
         od_set_attrib(btTextColor);
         break;

      /* User asked for online help. */
      case 3:
         OnlineHelp();
         break;
   }

   /* Restore original text color. */
   od_set_attrib(btTextColor);

   /* Return, telling the editor to continue normally. */
   return(EDIT_MENU_DO_NOTHING);
}


/* OnlineHelp() displays a window with online help to the user. */
void OnlineHelp(void)
{
   void *pWindow;

   /* Create a window. */
   pWindow = od_window_create(5, 3, 75, 22, " YAREdit 1.00 Help ", 0x70, 0x7f, 0x70,
      0);

   /* Display help in window. */
   od_set_cursor(4, 7);
   od_printf("`red on white`Use the following keys to compose your message:");
   od_set_cursor(6, 7);
   od_printf("   [CTRL]-[Y]: Deletes the current line.");
   od_set_cursor(7, 7);
   od_printf("        [INS]: Toggles insert or overwrite mode.");
   od_set_cursor(8, 7);
   od_printf("[CURSOR KEYS]: You can move around in the file using arrow keys,");
   od_set_cursor(9, 7);
   od_printf("               [PAGE UP], [PAGE DOWN], [HOME] and [END]. If your");
   od_set_cursor(10, 7);
   od_printf("               terminal doesn't support these keys, you can also");
   od_set_cursor(11, 7);
   od_printf("               ^E for up, ^S for left, ^D for right, ^X for down.");
   od_set_cursor(12, 7);
   od_printf("     [DELETE]: Deletes text to the right of the cursor. If your");
   od_set_cursor(13, 7);
   od_printf("               terminal doesn't suppor this key, you can also use");
   od_set_cursor(14, 7);
   od_printf("               ^G.");
   od_set_cursor(15, 7);
   od_printf("  [BACKSPACE]: Moves cursor right, deleting characters.");
   od_set_cursor(16, 7);
   od_printf("        [TAB]: Advances cursor to next tab stop.");
   od_set_cursor(17, 7);
   od_printf("     [ESCAPE]: Activates menu. ^Z can be used as an alternative.");
   od_set_cursor(18, 7);

   od_set_cursor(20, 7);
   od_printf("YAREdit is a FREE external full screen editor for use with the");
   od_set_cursor(21, 7);
   od_printf("RA BBS system. YAREdit is written by Brian Pirie (pirie@msn.com).");

   /* Wait for user to press enter or escape. */
   od_get_answer("\n\r\033");

   /* Remove the window. */
   od_window_remove(pWindow);
}


/* LoadFile() - Loads the contents of a file into the buffer. */
void LoadFile(char *pszFilename, char **ppszFileContents, size_t *pBufferSize)
{
   FILE *pFile;
   char *pchCtrlZ;

   /* Open file. */
   pFile = fopen(pszFilename, "rt");
   if(pFile == NULL) return;

   /* Determine the length of the file. */
   for(*pBufferSize = 0; !feof(pFile); ++*pBufferSize)
   {
      if(fgetc(pFile) == EOF) break;
   }

   fseek(pFile, 0, SEEK_SET);

   /* Attempt to allocate enough memory to hold the entire file contents. */
   *ppszFileContents = malloc(*pBufferSize + 1);
   if(*ppszFileContents == NULL) return;

   /* Now, fill buffer from file. */
   fread(*ppszFileContents, sizeof(char), *pBufferSize, pFile);
   (*ppszFileContents)[*pBufferSize] = '\0';

   /* Close file. */
   fclose(pFile);

   /* Trim trailing Ctrl-Z character, if any. */
   pchCtrlZ = strchr(*ppszFileContents, 26);
   if(pchCtrlZ != NULL) pchCtrlZ = '\0';
}


/* SaveFile() - Saves the contents of the buffer into a file. */
void SaveFile(char *pszFilename, char *pszFileContents)
{
   FILE *pFile;

   /* Open file. */
   pFile = fopen(pszFilename, "wt");
   if(pFile == NULL) return;

   /* Write text to file. */
   fwrite(pszFileContents, sizeof(char), strlen(pszFileContents), pFile);

   /* Close file. */
   fclose(pFile);
}

                                                /* FULL-SCREEN CHAT FUNCTION */
void FullScreenChat(void)
{
   btCursorWindow=0;                               /* Reset working variables */
   anWordLength[0]=anWordLength[1]=0;
   anCursorCol[0]=anCursorCol[1]=1;
   anCursorLine[0]=abtTopLine[0];
   anCursorLine[1]=abtTopLine[1];


                         /* If ANSI or AVATAR graphics mode is not available */
   if(!od_control.user_ansi && !od_control.user_avatar)
   {
      /* Then use OpenDoor's line chat mode instead. */
      return;
   }

   od_control.od_chat_active=TRUE;

   od_save_screen(abtScreenBuffer);           /* Save current screen contents. */

                                                     /* DRAW THE CHAT SCREEN */
   od_set_attrib(abtWindowColor[0]);
   od_clr_scr();                                         /* Clear the screen */

   od_set_cursor(btBarLine,1);                  /* Draw window separation bar */
   od_set_attrib(btBarColor);
   od_clr_line();
   od_set_cursor(btBarLine,67);
   od_printf("Ctrl-A: Clear");
   od_set_cursor(btBarLine,1);
   od_printf(" Top : %-.28s    Bottom : %-.28s    ",
      od_control.sysop_name, od_control.user_name);

   od_set_cursor(abtTopLine[0],1);    /* Locate cursor where typing will begin */
   od_set_attrib(abtWindowColor[0]);           /* Set appropriate text colour */

                                                           /* MAIN CHAT LOOP */
   for(;;)                             /* (Repeats for each character typed) */
   {
      do
      {
         chKey=(char)od_get_key(FALSE);    /* Get next keystroke from keyboard */

                                                    /* CHECK FOR SYSOP ABORT */
         if((chKey==27 && od_control.od_last_input==1) /* If sysop pressed ESC */
            || !od_control.od_chat_active)
         {
            od_set_attrib(0x07);                     /* Reset display colour */
            od_clr_scr();                                /* Clear the screen */

            od_control.od_chat_active=FALSE;           /* Turn off chat mode */
            od_restore_screen(abtScreenBuffer);      /* Restore orignal screen */
            return;                                 /* Exit full-screen chat */

         }
      } while(chKey==0);

                                                     /* CHECK FOR NEW TYPIST */
      if(od_control.od_last_input!=btCursorWindow)/* If new person typing now */
      {                               /* Switch cursor to appropriate window */
         btCursorWindow=od_control.od_last_input;       /* Set current typist */

                                                /* Move cursor to new window */
         od_set_cursor(anCursorLine[btCursorWindow],anCursorCol[btCursorWindow]);

         od_set_attrib(abtWindowColor[btCursorWindow]);  /* Change text colour */
      }


      if(chKey==13 || chKey==10)           /* IF USER PRESSED [ENTER] / [RETURN] */
      {
         anWordLength[btCursorWindow]=0;      /* Enter constitutes end of word */

         ChatNewLine();                               /* Move to next line */
      }


      else if(chKey==8)                           /* IF USER PRESS [BACKSPACE] */
      {
         if(anCursorCol[btCursorWindow] > 1)       /* If not at left of screen */
         {
            --anCursorCol[btCursorWindow];    /* Move cursor back on character */
            if(anWordLength[btCursorWindow] > 0) --anWordLength[btCursorWindow];
            od_printf("\b \b");          /* Erase last character from screen */
         }
      }


      else if(chKey==32)                            /* IF USER PRESSED [SPACE] */
      {
         anWordLength[btCursorWindow]=0;    /* [Space] constitutes end of word */

         if(anCursorCol[btCursorWindow]==79)              /* If at end of line */
            ChatNewLine();                     /* Move cursor to next line */
         else                                       /* If not at end of line */
         {
            ++anCursorCol[btCursorWindow];        /* Increment cursor position */
            od_putch(32);                                 /* Display a space */
         }
      }


      else if(chKey==1)                    /* IF USER PRESSED CLEAR WINDOW KEY */
      {                                               /* Clear user's window */
         od_scroll(1,abtTopLine[btCursorWindow],79,abtBottomLine[btCursorWindow],
            abtBottomLine[btCursorWindow]-abtTopLine[btCursorWindow]+1,0);

         anWordLength[btCursorWindow]=0;  /* We are no longer composing a word */

         anCursorCol[btCursorWindow]=1;               /* Reset cursor position */
         anCursorLine[btCursorWindow]=abtTopLine[btCursorWindow];
         od_set_cursor(anCursorLine[btCursorWindow],anCursorCol[btCursorWindow]);
      }


      else if(chKey>32)                 /* IF USER TYPED A PRINTABLE CHARACTER */
      {                                    /* PERFORM WORD WRAP IF NECESSARY */
         if(anCursorCol[btCursorWindow]==79)    /* If cursor is at end of line */
         {
                                               /* If there is a word to wrap */
            if(anWordLength[btCursorWindow]>0 && anWordLength[btCursorWindow]<78)
            {
                                         /* Move cursor to beginning of word */
               od_set_cursor(anCursorLine[btCursorWindow],
                          anCursorCol[btCursorWindow]-anWordLength[btCursorWindow]);

               od_clr_line();                /* Erase word from current line */

               ChatNewLine();                  /* Move cursor to next line */

                                                           /* Redisplay word */
               od_disp(aszCurrentWord[btCursorWindow],anWordLength[btCursorWindow],
                                                                          TRUE);
               anCursorCol[btCursorWindow]+=anWordLength[btCursorWindow];
            }

            else                            /* If there is no word to "wrap" */
            {
               ChatNewLine();                  /* Move cursor to next line */
               anWordLength[btCursorWindow]=0;             /* Start a new word */
            }
         }

                                            /* ADD CHARACTER TO CURRENT WORD */
                              /* If there is room for more character in word */
         if(strlen(aszCurrentWord[btCursorWindow])<79)     /* Add new character */
            aszCurrentWord[btCursorWindow][anWordLength[btCursorWindow]++]=chKey;

                                            /* DISPLAY NEWLY TYPED CHARACTER */
         ++anCursorCol[btCursorWindow];
         od_putch(chKey);
      }
   }
}



              /* FUNCTION USED BY FULL-SCREEN CHAT TO START A NEW INPUT LINE */
void ChatNewLine(void)
{                                        /* If cursor is at bottom of window */
   if(anCursorLine[btCursorWindow]==abtBottomLine[btCursorWindow])
   {                                  /* Scroll window up one line on screen */
      od_scroll(1,abtTopLine[btCursorWindow],79, abtBottomLine[btCursorWindow],
                btScrollDistance, 0);
      anCursorLine[btCursorWindow]-=(btScrollDistance - 1);
   }

   else                              /* If cursor is not at bottom of window */
   {
      ++anCursorLine[btCursorWindow];             /* Move cursor down one line */
   }

                                         /* Move cursor's position on screen */
   od_set_cursor(anCursorLine[btCursorWindow],anCursorCol[btCursorWindow]=1);

   od_set_attrib(abtWindowColor[btCursorWindow]);        /* Change text colour */
}


void DisplayShellWindow(void)
{
   od_get_cursor(&nPreShellRow, &nPreShellColumn);
   if((pShellWindow = od_window_create(17, 9, 63, 15, "DOS Shell",
                                     btShellWindowBoarder, btShellWindowTitle, 
                                     btShellWindowText, 0)) == NULL) return;

   od_set_attrib(btShellWindowText);
   od_set_cursor(11, 26);
   od_printf("The Sysop has shelled to DOS");
   od_set_cursor(13, 21);
   od_printf("He/She will return in a few moments...");
}


void RemoveShellWindow(void)
{
   od_window_remove(pShellWindow);
   od_set_cursor(nPreShellRow, nPreShellColumn);
   if(od_control.od_chat_active)
   {
      od_set_attrib(abtWindowColor[btCursorWindow]);
   }
   else
   {
      od_set_attrib(btTextColor);
   }
}


void LoadMessageInfo(char *pszFilename)
{
   FILE *pFile;
   char szTemp[30];

   pFile = fopen(pszFilename, "rt");
   if(pFile == NULL)
   {
      od_printf("Unable to load MSGINF. Press [Enter] to continue.\n\r");
      od_get_answer("\n\r");
   }

   ReadLine(pFile, szMessageFrom, sizeof(szMessageFrom));
   ReadLine(pFile, szMessageTo, sizeof(szMessageTo));
   ReadLine(pFile, szMessageSubject, sizeof(szMessageSubject));
   ReadLine(pFile, szTemp, sizeof(szTemp));
   unMessageNum = atoi(szTemp);
   ReadLine(pFile, szAreaName, sizeof(szAreaName));
   ReadLine(pFile, szTemp, sizeof(szTemp));
   bPrivate = (stricmp(szTemp, "YES") == 0);

   fclose(pFile);
}


void ReadLine(FILE *pFile, char *pszDest, size_t SizeOfDest)
{
   static char szLine[300];
   fgets(szLine, sizeof(szLine), pFile);
   if(szLine[strlen(szLine) - 1] == '\n') szLine[strlen(szLine) - 1] = '\0';
   memcpy(pszDest, szLine, SizeOfDest - 1);
   pszDest[SizeOfDest - 1] = '\0';
}


void SaveMessageInfo(char *pszFilename)
{
   FILE *pFile;

   pFile = fopen(pszFilename, "wt");
   if(pFile == NULL)
   {
      return;
   }

   fprintf(pFile, "%s\n", szMessageFrom);
   fprintf(pFile, "%s\n", szMessageTo);
   fprintf(pFile, "%s\n", szMessageSubject);
   fprintf(pFile, "%u\n", unMessageNum);
   fprintf(pFile, "%s\n", szAreaName);
   fprintf(pFile, bPrivate ? "YES\n" : "NO\n");

   fclose(pFile);
}


void DisplayPrivateFlag(void)
{
   od_set_cursor(2, 23);
   od_set_attrib(btFlagColor);
   od_disp_str(bPrivate ? "(Private)" : "         ");
}


void EditFields(void)
{
   INT nField = 0;
   WORD wResult;

   for(;;)
   {
      switch(nField)
      {
         case 0:
            wResult = od_edit_str(szMessageTo,
               "***********************************", 4, 5,
               btFieldContentsColor, btFieldContentsColor, 176,
               EDIT_FLAG_FIELD_MODE | EDIT_FLAG_EDIT_STRING |
               EDIT_FLAG_SHOW_SIZE |
               EDIT_FLAG_AUTO_DELETE);
            break;
         case 1:
            wResult = od_edit_str(szMessageSubject,
               "************************************************************************",
               5, 5, btFieldContentsColor, btFieldContentsColor, 176,
               EDIT_FLAG_FIELD_MODE | EDIT_FLAG_EDIT_STRING |
               EDIT_FLAG_SHOW_SIZE |
               EDIT_FLAG_AUTO_DELETE);
            break;
      }

      switch(wResult)
      {
         case EDIT_RETURN_CANCEL:
         case EDIT_RETURN_ACCEPT:
            od_set_attrib(btTextColor);
            return;
         case EDIT_RETURN_NEXT:
         case EDIT_RETURN_PREVIOUS:
            if(nField == 0) nField = 1; else nField = 0;
            break;
      }
   }
}
