#ifndef     __COLLECT_H
#define     __COLLECT_H

#include    <assert.h>
#include    <mem.h>


const unsigned MAXPTRLIST =
             ( unsigned )((65536uL - 16 )/sizeof( void* ));


/* ------------------------------------------------------------ *\
|   Collection Template:  Stores pointers to Ts                  |
\* ------------------------------------------------------------ */
template <class T>
class tPTRLIST
{

    public:
        tPTRLIST( unsigned aLimit, unsigned aDelta );
       ~tPTRLIST();

        T* at( unsigned index )
        {
            assert( index < count );
            return( items[index] );
        }

        T* operator []( unsigned index )
        {
            assert( index < count );
            return( items[index] );
        }

        void atFree( unsigned index );
        void atRemove( unsigned index );
        void remove( T* item )
        {
            atRemove( indexOf( item ) );
        }

        void removeAll()
        {
            count = 0;
        }

        void free( T* item )
        {
            remove( item );
            freeItem( item );
        }

        void atInsert( unsigned index, T* item );

        void atPut( unsigned index, T* item )
        {
            assert( index < count );
            items[index] = item;
        }

        unsigned insert( T* item );
        void setLimit( unsigned aLimit );

        unsigned getCount()
        {
            return count;
        }

        void    freeAll( void );
        unsigned indexOf( T* item );

        void forEach  ( void( *actn )( T*, void* ), void *arg );
        T*   firstThat( int ( *Test )( T*, void* ), void *arg );
        T*   lastThat ( int ( *Test )( T*, void* ), void *arg );
        int  numThat  ( int ( *Test )( T*, void* ), void *arg );

        void forEach  ( void( *actn )( T*, T* ), T *arg );
        T*   firstThat( int ( *Test )( T*, T* ), T *arg );
        T*   lastThat ( int ( *Test )( T*, T* ), T *arg );
        int  numThat  ( int ( *Test )( T*, T* ), T *arg );


    protected:
        tPTRLIST() :
                count( 0 ),
                items( 0 ),
                limit( 0 ),
                delta( 0 ),
                shouldDelete( 1 )
        {
            items = 0;
        }

        public:
        T   **items;
        unsigned count;
        unsigned limit;
        unsigned delta;
        int      shouldDelete;

    private:
        void freeItem( T* item );
};



/* ------------------------------------------------------------ *\
|   Constructor of Collection Template...                        |
\* ------------------------------------------------------------ */
template<class T>
tPTRLIST<T>::tPTRLIST( unsigned aLimit, unsigned aDelta ) :
    count( 0 ),
    items( 0 ),
    limit( 0 ),
    delta( aDelta ),
    shouldDelete( 1 )
{
    setLimit( aLimit );
}



/* ------------------------------------------------------------ *\
|   Destructor of Collection Template...                         |
\* ------------------------------------------------------------ */
template<class T> 
tPTRLIST<T>::~tPTRLIST()
{
    if ( shouldDelete )
        freeAll();
    setLimit( 0 );
}



/* ------------------------------------------------------------ *\
|   atFree: removes and frees item at index from collection...   |
\* ------------------------------------------------------------ */
template<class T> 
void tPTRLIST<T>::atFree( unsigned index )
{
    T *item = at( index );
    atRemove( index );
    freeItem( item );
}



/* ------------------------------------------------------------ *\
|   atRemove: removes and item at index from collection..        |
\* ------------------------------------------------------------ */
template<class T>
void tPTRLIST<T>::atRemove( unsigned index )
{
    assert( index < count );
    count--;
    memmove( ( void* )&items[index], ( void* )&items[index+1],
             (count-index)*sizeof( T* ) );
}



/* ------------------------------------------------------------ *\
|   atInsert: inserts item at index in the collection...         |
\* ------------------------------------------------------------ */
template<class T> 
void tPTRLIST<T>::atInsert( unsigned index, T* item )
{
    if ( count == limit )
        setLimit( count + delta );

    if ( count-index != 0 )
        memmove(  ( void* )&items[index+1], ( void* )&items[index],
                  (count-index)*sizeof( T* )  );
    count++;
    items[index] = item;
}



/* ------------------------------------------------------------ *\
|   insert: inserts item at end of collection...                 |
\* ------------------------------------------------------------ */
template<class T> 
unsigned tPTRLIST<T>::insert( T* item )
{
    unsigned loc = count;
    atInsert( count, item );
    return( loc );
}



/* ------------------------------------------------------------ *\
|   setLimit: resizes the array of pointers to items...          |
\* ------------------------------------------------------------ */
template<class T> 
void tPTRLIST<T>::setLimit( unsigned aLimit )
{
    if ( aLimit < count )
         aLimit = count;
    assert( aLimit <= MAXPTRLIST );

    if ( aLimit != limit )
    {
        T  **aItems;

        if ( aLimit == 0 )
            aItems = 0;
        else
        {
            aItems = new T *[aLimit];
            if ( count != 0 )
                memcpy( aItems, items, count*sizeof( T* ) );
        }

        if ( items != ( T** )0 )
            delete items;

        items = aItems;
        limit = aLimit;
    }
}

        

/* ------------------------------------------------------------ *\
|   indexOf: returns the index of an item in the collection...   |
\* ------------------------------------------------------------ */
template<class T> 
unsigned tPTRLIST<T>::indexOf( T* item )
{
    for( unsigned i=0; i<count; i++ )
        if ( items[i] == item )
            return i;
    assert( 0 );
    return( -1 );
}




/* ------------------------------------------------------------ *\
|   freeAll:  frees all items in the collection...               |
\* ------------------------------------------------------------ */
template<class T> 
void tPTRLIST<T>::freeAll( void )
{
    for( unsigned i=0; i<count; i++ )
        freeItem( at(i) );
    count = 0;
}



/* ------------------------------------------------------------ *\
|   freeItem:  Deletes individual item in collection...          |
\* ------------------------------------------------------------ */
template<class T> 
void tPTRLIST<T>::freeItem( T* item )
{
    delete item;
}



/* ------------------------------------------------------------ *\
|   firstThat: iterates through collection for condition...      |
|              This version expects an extra void* pointer.      |
\* ------------------------------------------------------------ */
template<class T> 
T* tPTRLIST<T>::firstThat( int( *Test )( T*, void* ), void *arg )
{
    for( unsigned i=0; i<count; i++ )
    {
        if ( Test( items[i], arg ) != 0 )
            return items[i];
    }
    return ( T* )0;
}



/* ------------------------------------------------------------ *\
|   lastThat: iterates backwards through collection for condition|
|             This version expects an extra void* pointer.       |
\* ------------------------------------------------------------ */
template<class T> 
T* tPTRLIST<T>::lastThat( int( *Test )( T*, void* ), void *arg )
{
    for( unsigned i=count; i>0; i-- )
    {
        if ( Test( items[i-1], arg ) != 0 )
            return( items[i-1] );
    }
    return( T* )0;
}



/* ------------------------------------------------------------ *\
|   forEach: iterates through whole collection...                |
|            this version expects and extra void* pointer.       |
\* ------------------------------------------------------------ */
template<class T> 
void tPTRLIST<T>::forEach( void ( *action )( T*, void* ), void *arg )
{
    for ( unsigned i=0; i<count; i++ )
        action( items[i], arg );
}



/* ------------------------------------------------------------ *\
|   numThat: returns number of items meeting condition...        |
\* ------------------------------------------------------------ */
template<class T> 
int tPTRLIST<T>::numThat( int( *Test )( T*, void* ), void *arg )
{
    int cnt = 0;
    for( unsigned i=0; i<count; i++ )
    {
        if ( Test( items[i], arg ) != 0 )
            cnt++;
    }
    return( cnt );
}



/* ------------------------------------------------------------ *\
|   firstThat: iterates through collection for condition...      |
|              This version expect an extra T* pointer.          |
\* ------------------------------------------------------------ */
template<class T> 
T* tPTRLIST<T>::firstThat( int( *Test )( T*, T* ), T *arg )
{
    for( unsigned i=0; i<count; i++ )
    {
        if ( Test( items[i], arg ) != 0 )
            return items[i];
    }
    return ( T* )0;
}



/* ------------------------------------------------------------ *\
|   lastThat: iterates backwards through collection for condition|
|             This version expect an extra T* pointer.           |
\* ------------------------------------------------------------ */
template<class T>
T* tPTRLIST<T>::lastThat( int( *Test )( T*, T* ), T *arg )
{
    for( unsigned i=count; i>0; i-- )
    {
        if ( Test( items[i-1], arg ) != 0 )
            return( items[i-1] );
    }
    return( T* )0;
}



/* ------------------------------------------------------------ *\
|   forEach: iterates through whole collection...                |
|            This version expect an extra T* pointer.            |
\* ------------------------------------------------------------ */
template<class T>
void tPTRLIST<T>::forEach( void ( *action )( T*, T* ), T* arg )
{
    for ( unsigned i=0; i<count; i++ )
        action( items[i], arg );
}



/* ------------------------------------------------------------ *\
|   numThat: returns number of items meeting condition...        |
\* ------------------------------------------------------------ */
template<class T> 
int tPTRLIST<T>::numThat( int( *Test )( T*, T* ), T *arg )
{
    int cnt = 0;
    for( unsigned i=0; i<count; i++ )
    {
        if ( Test( items[i], arg ) != 0 )
            cnt++;
    }
    return( cnt );
}

// typedef tPRTLIST<int*> TIstar;

#endif  // __COLLECT_H
