/* RPLSETD.CMD - REXX program to convert an existing OS/2 Remote IPL client  */
/*               to use the 32-bit VGA and XGA device drivers or the 16-bit  */
/*               8514 device driver.                                         */
/*                                                                           */
/*    Copyright: (C) Copyright IBM Corp. 1993                                */
/*                                                                           */
/*    Note 1: All the variables are global to all subroutines.               */
/*                                                                           */
/*    Note 2: All NLS message strings are centrally located in the           */
/*            subroutine Initialize_NLS_Messages at the end of this          */
/*            rexx procedure.                                                */

driver = ''
client = ''
logfile = ''
Rspfile = ''
os2csdlevel = ''
currentOS2 = 'OS2.20'
newOS2 = 'OS2.21'
CRLF = '0A0D'X
ERROR_INFO_type = 1
SYNTAX_type = 2
changeOS2 = 0
basedevflag = 1
clientlist.0 = 0
msgdata = ''

/* initialize NLS error and informational message strings */
Call Initialize_NLS_Messages ERROR_INFO_type

Parse Arg  parm.1 parm.2 parm.3 parm.4 parm.5 parm.6 .
/* Check for help request (parm.1 = ? or '') */
If parm.1 = '?' | parm.1 = '' Then Call Syntax_Help

Call Process_Input_Parameters
If result <> 0 Then Call Error_Exit

If Rspfile <> '' Then Do
   Call Read_Rspfile
   If result <> 0 Then Do
      Say errprefix || message.1 Rspfile || CRLF || message.2
      exit result
   end
end

/* if logfile specified, delete old log file (if it exists) & log header */
If Logfile <> '' Then Do
   Call Delete_File Logfile
   msgdata = 'RPLSETD ' || message.3 || Date('U')  Time()
   Call Lineout Logfile, msgdata
   If result <> 0 Then Do
      Say message.4 || Logfile
      exit 4
   end
   Call Lineout Logfile, ' '
end

/* were required parameters  entered ?*/
If driver = '' | clientlist.0 = 0 Then Call Syntax_Help

Call Validate_Display_DriverID
If result <> 0 Then Call Error_Exit

ddbootdrv = ''
dd_data.0 = 0

/* build list of directories in RPL\IBMCOM to process */
lanlist.0 = 6
lanlist.1 = 'TOKENRNG'
lanlist.2 = 'ETHERNET'
lanlist.3 = 'PCNETA'
lanlist.4 = 'PCNET'
lanlist.5 = 'ELNKII'
lanlist.6 = 'ELNKMC'

Call Determine_RIPL_Directory
If result <> 0 Then Call Error_Exit

/* get LAN Server version */
Call Get_LAN_Server_Version
if result <> 0 Then Exit

/* get version of OS/2 in target OS/2 RIPL directory */
Call Get_RIPL_OS2_Version
if result <> 0 Then Call Error_Exit

/* determine if desired device driver is installed */
If changeOS2 = 1 Then ddpath = rpldir'\'newOS2'\OS2\DLL\'
else  ddpath = rpldir'\'currentOS2'\OS2\DLL\'
filename = ddpath||driver||'.DLL'
if File_Exist(filename) = 0 Then Do
   ddpath = substr(ddpath, 1, length(ddpath) - 1)
   msgdata = errprefix || message.5
   Call Display_Log_Msg msgdata, driver, dpath
   Call Error_Exit
end

/* If OS/2 2.1 is in RIPL tree, check to see if 2.1 migration has been run. */
If version21 = 1 & File_Exist(rpldir'\RPLMIG21.$$$') = 0 Then Do
   msgdata = errprefix || message.23
   Call Display_Log_Msg msgdata, ''
   Call Error_Exit
end

do clnum = 1 to clientlist.0
   /* update each client in list */
   client = clientlist.clnum
   Call Update_Client_Files
end

If Logfile <> '' Then Call Lineout Logfile
exit
/* end of main routine */


/*************** Start of subroutines ***************************/


/* Subroutine Error_Exit */
/* This subroutine cleans up on an error exit. */
Error_Exit:
   If Logfile <> '' Then Do
      Call Lineout Logfile
      Say message.7 || Logfile
   end
   exit 4


/* Subroutine Get_LAN_Server_Version */
/* This subroutine determines the version of LAN Server installed on  */
/* the Remote IPL server. */
Get_LAN_Server_Version:
   /* Read the first 60 bytes of SYSLEVEL.SRV and check the current server level*/
   filename = lanpath'\SYSLEVEL.SRV'
   syslevel = ''
   Do i = 1 to 60
      syslevel = syslevel || charin(filename)
   end
   Call Charout filename
   verp = pos('IP', syslevel)
   If verp <> 0 Then LS_version = substr(syslevel, verp+3, 4)
   If LS_version = '' Then Do
      msgdata = errprefix || message.8
      Call Display_Log_Msg msgdata, ''
      return 4
   end
   return 0



Get_RIPL_OS2_Version:
   /* Read the first 60 bytes of RIPL SYSLEVEL.OS2 and get the current */
   /* OS/2 CSD level and version id */
   If changeOS2 = 1 Then spath = rpldir'\'newOS2
   else  spath = rpldir'\'currentOS2
   filename = spath'\OS2\INSTALL\SYSLEVEL.OS2'
   syslevel = ''
   If File_Exist(filename) = 1 Then Do
      Do i = 1 to 60
         syslevel = syslevel || charin(filename)
      end
      Call Charout filename
      verp = pos('XR', syslevel)
   end
   else verp = 0

   If verp <> 0 Then os2csdlevel = substr(syslevel, verp, 8)
   else os2csdlevel = unknown
   If os2csdlevel = unknown | substr(os2csdlevel, 4, 4) = '2000' Then Do
      msgdata = errprefix || message.9
      Call Display_Log_Msg msgdata, spath, os2csdlevel
      return 4
   end

   /* the following is to differentiate between 2.0+SP, 2.00.1, and 2.1 */
   version21 = 0
   If substr(os2csdlevel, 4, 4) = '6055' Then basedevflag = 1
   else If substr(os2csdlevel, 4, 4) = '2010' & substr(syslevel, 41, 1) = '20'x  Then basedevflag = 0
        else Do
           /* 2.1 */
           basedevflag = 1
           version21 = 1
        end
   return 0


/* Subroutine Update_Client_Files */
/* This subroutine calls the subroutines to update each client file. */
Update_Client_Files:
   /* update the client copy of OS2INI.20 */
   Call Update_Client_Ini
   If result = 0 Then Do
      msgdata = client || message.10
      Call Display_Log_Msg msgdata, client

      /* update the client CONFIG.20 file.  If the client is DEFALT20, then */
      /* there is special processing to update each of the master CONFIG.20 */
      /* files in the RPL\IBMCOM tree. */
      if client = 'DEFALT20' Then Do
         /* for DEFALT20 update each of the lantype CONFIG.20 and CONFIGRI.20 files */
         Do lli = 1 to lanlist.0
            lantype = lanlist.lli
            configfilename = rpldir'\IBMCOM\'lantype'\CONFIG.20'
            If File_Exist(configfilename) <> 0 Then Do
               /* a CONFIG.20 file is present, update it */
               Call Update_Client_Config_20
               If result = 0 Then Do
                  msgdata = lantype || message.11
                  Call Display_Log_Msg msgdata, ''
               end
               else Do
                  msgdata = errprefix || client || lantype || message.12
                  Call Display_Log_Msg msgdata, ''
               end
            end

            configfilename = rpldir'\IBMCOM\'lantype'\CONFIGRI.20'
            If File_Exist(configfilename) <> 0 Then Do
               /* a CONFIGRI.20 file is present, update it */
               Call Update_Client_Configri_20
               If result = 0 Then Do
                  msgdata = lantype || message.25
                  Call Display_Log_Msg msgdata, ''
               end
               else Do
                  msgdata = errprefix || client || lantype || message.26
                  Call Display_Log_Msg msgdata, ''
               end
            end
         end
      end
      else Do
         /* not the DEFALT20 client */
         Call Update_Client_Config_20
         If result = 0 Then Do
            msgdata = client || message.11
            Call Display_Log_Msg msgdata, ''
         end
         else Do
            msgdata = errprefix || client || message.12
            Call Display_Log_Msg msgdata, ''
         end

         Call Update_Client_Configri_20
         If result = 0 Then Do
            msgdata = client || message.25
            Call Display_Log_Msg msgdata, ''
         end
         else Do
            msgdata = errprefix || client || message.26
            Call Display_Log_Msg msgdata, ''
         end
      end

      /* Update the client.FIT file */
      Call Update_Client_Fit
      If result = 0 Then Do
         msgdata = client||message.13
         Call Display_Log_Msg msgdata, ''
      end
      else Do
         /* error updating client fit */
         msgdata = errprefix || client||message.14
         Call Display_Log_Msg msgdata, ''
      end

      If client <> 'DEFALT20' Then Do
         /* not DEFALT20, check to see if RPL.MAP needs to be updated */
         If changeOS2 = 1 Then Do
            /* update workstation record in RPL.MAP */
            Call Update_Workstation_Record
            If result = 0 Then Do
               msgdata = client||message.27
               Call Display_Log_Msg msgdata, ''
            end
            else Do
               /* error updating workstation record */
               msgdata = errprefix||client||message.28
               Call Display_Log_Msg msgdata, ''
            end
         end
      end
   end
   else Do
      /* INI update failed, skip CONFIG.20, FIT, & RPL.MAP update */
      msgdata = errprefix || client || message.15 || client '.'
      Call Display_Log_Msg msgdata, ''
   end

   msgdata = '  '
   Call Display_Log_Msg msgdata, ''
   return 0



/* Subroutine Update_Client_Ini */
/* This subroutine updates the OS2INI.20 file for a client workstation */
/* to add the required profile data to support the specified device driver. */
Update_Client_Ini:

   rc = 0
   If newOS2 = 'OS2.21' Then Do
      /* check to see if OS2INI.21 exists */
      ininame = rpldir'USER\'client'\OS2\'
      If File_Exist(ininame'OS2INI.21') = 1 Then Do
         /* copy 21 filenames into 20 filenames */
         '@ATTRIB -r 'ininame'OS2*.20'
         '@ATTRIB -r 'ininame'OS2*.21'
         '@COPY 'ininame'OS2INI.21 ' ininame'OS2INI.20 1>nul 2>nul'
         '@COPY 'ininame'OS2SYINI.21 ' ininame'OS2SYINI.20 1>nul 2>nul'
         Call Delete_File ininame'OS2INI.21B'
         Call Delete_File ininame'OS2SYINI.21B'
      end
   end

   /* build fully qualified path name for OS2INI.20 */
   ininame = rpldir'USER\'client'\OS2\OS2INI.20'
   /* Execute RPLRXUTL to update client copy of OS2.INI (OS2INI.20) */
   cmdline = '@RPLRXUTL.EXE /D:'driver ' /C:'ininame '1>nul'
   address CMD cmdline
   return rc




/* Subroutine Update_Client_Fit */
/* This subroutine creates a new client.FIT by making the following */
/* changes to the original FIT: */
/*   - Comment out the DISPLAY.DLL entries. */
/*   - Enables the correct .FON entries for the specified display driver. */
/*   - If requested, changes the target OS2 root directory. */
Update_Client_Fit:

   /* build fully qualified path names for client.??? */
   fitname = rpldir'\FITS\'client'.FIT'
   tmpfitname = rpldir'\FITS\'client'.TMP'
   bakfitname = rpldir'\FITS\'client'.BAK'
   /* Call SysFileDelete tmpfitname */
   Call Delete_File tmpfitname
   Call Delete_File bakfitname

   If File_Exist(fitname) Then Do
      /* client.FIT does exists, build a new FIT */
      os2verflag = 0
      do until lines(fitname) = 0
         data = translate(linein(fitname))
         If data <> '' & substr(data, 1, 2) <> '\\' Then Do
            /* not a null line or a UNC line, check line */
            datap = pos(':\OS2\DLL\DISPLAY.DLL', data)
            if datap <> 0 Then Do
               if substr(data,1,1) <> ';' Then Do
                  /* comment out DISPLAY.DLL entry */
                  newdata = ';'data
                  data = newdata
               end
            end
            else Do
               /* check for .FON entry and enable/disable the entry */
               /* depending on the driver type */
               helvp = 0
               courp = 0
               timesp = 0
               helvp = pos(':\OS2\DLL\HELV.FON', data)
               If helvp = 0 Then Do
                  courp = pos(':\OS2\DLL\COURIER.FON',data)
                  If courp = 0 Then Do
                     timesp = pos(':\OS2\DLL\TIMES.FON',data)
                  end
               end
               If helvp <> 0 | courp <> 0 | timesp <> 0 Then Do
                 datal = length(data)
                 If driver = 'IBMVGA32' | driver = 'IBMVGA' Then Do
                    If substr(data, datal-2, 3) <> 'VGA' Then Do
                       If substr(data, 1,1) <> ';' Then Do
                          /* disable non-VGA .FON entry */
                          newdata = ';'data
                          data = newdata
                       end
                    end
                    else If substr(data, 1, 1) = ';' Then Do
                       /* enable VGA .FON entry */
                       data = substr(data, 2)
                    end
                 end
                 else If driver = 'IBMXGA32' | driver = 'IBM8514' Then Do
                    If substr(data, datal-2, 3) <> 'BGA' Then Do
                       If substr(data, 1, 1) <> ';' Then Do
                          /* disable non-XGA/8514 .FON entry */
                          newdata = ';'data
                          data = newdata
                       end
                    end
                    else If substr(data, 1, 1) = ';' Then Do
                       /* enable XGA/8514 .FON entry */
                       data = substr(data, 2)
                    end
                 end
               end
            end

            /* check for possible operating system change */
            if changeOS2 = 1 Then Do
               If substr(data, 1, 2) = '; ' Then srvfname = word(data, 3)
               else srvfname = word(data, 2)
               os2p = pos(currentOS2'\', srvfname)
               if os2p <> 0 Then Do
                  /* old OS2 entry found, update it */
                  os2p = pos(currentOS2, data, length(word(data,1))+1)
                  newdata = substr(data, 1, os2p-1)
                  newdata = newdata || newOS2
                  newdata = newdata || substr(data, os2p+length(currentOS2))
                  data = newdata
               end
            end
         end
         /* pick up boot drive id from OS2KRNL entry */
         os2p = pos(':\OS2KRNL ', translate(data))
         If os2p <> 0 Then bootdrv = substr(data, os2p-1, 1)

         /* check for OS2VER reference, set flag if found */
         os2p = pos(':\OS2VER ', translate(data))
         If os2p <> 0 Then os2verflag = 1

         /* write data record to new FIT */
         Call Lineout tmpfitname, data
      end

      /* does os2ver statment need to be added? */
      If os2verflag = 0 Then Do
         /* add os2ver statement to end of file */
         Call lineout tmpfitname, '  '
         data = bootdrv':\OS2VER    'newOS2'\OS2VER'
         Call Lineout tmpfitname, data
      end

      Call Lineout tmpfitname   /* close file */
      Call Lineout fitname      /* close file */

      /* rename original client.FIT */
      cmdstr = '@rename 'fitname' 'client'.BAK'
      address CMD cmdstr
      if rc = 0 Then Do
         /* rename client.TMP to be client.FIT */
         cmdstr = '@rename 'tmpfitname' 'client'.FIT'
         address CMD cmdstr
         if rc = 0 Then Do
            /* delete orginal client.FIT */
            /* Call SysFileDelete bakfitname */
            cmdstr = '@DEL 'bakfitname
            address CMD cmdstr
         end
         else Do
            /* rename error, try to restore original */
            cmdstr = '@rename 'bakfitname'  'client'.FIT'
            address CMD cmdstr
      end
   end
   else return 4
   return 0



/* Subroutine Update_Client_Config_20 */
/* This subroutine creates a new CONFIG.20 for the client that has the */
/* appropriate display device drivers enabled. */
Update_Client_Config_20:

   /* if LS 2.0 file, convert it to new format */
   If LS_version = '6000' Then Do
      Call Update_LS20_Config_20
      If result <> 0 Then Do
         msgdata = errprefix || message.16
         Call Display_Log_Msg msgdata, rpldir, lantype
         return 4
      end
   end

   /* build fully qualified path names for client CONFIG.20 */
   If client = 'DEFALT20' Then Do
      /* DEFALT20 is special case, it must be called for each lantype */
      configname = rpldir'\IBMCOM\'lantype'\CONFIG.20'
      tmpconfigname = rpldir'\IBMCOM\'lantype'\CONFIG.20T'
      bakconfigname = rpldir'\IBMCOM\'lantype'\CONFIG.20B'
   end
   else Do
      configname = rpldir'\MACHINES\'client'\CONFIG.20'
      tmpconfigname = rpldir'\MACHINES\'client'\CONFIG.20T'
      bakconfigname = rpldir'\MACHINES\'client'\CONFIG.20B'
   end
   /* Call SysFileDelete tmpconfigname */
   Call Delete_File tmpconfigname
   Call Delete_File bakconfigname

   If File_Exist(configname) Then Do
      /* copy records to first DEVINFO (XGA) */
      flag = 0
      do until lines(configname) = 0
         data = linein(configname)
         ucdata = translate(data)
         If pos('DEVINFO=', ucdata) <> 0 Then Leave
         xgacomp = pos('ing 6 state', data)
         If xgacomp <> 0 & driver = 'IBMXGA32' Then Do
            /* update XGA comment statement */
            newdata = substr(data, 1, xgacomp+3)||'7'||substr(data, xgacomp+5)
            data = newdata
         end
         else Do
            If version21 = 1 Then Do
               /* version 2.1 */
               /* check for AUTOSTART statement */
               If pos('AUTOSTART=', ucdata) <> 0 Then Do
                  If pos('CONNECTIONS', ucdata) = 0 Then data = data || ',CONNECTIONS'
               end
               else Do
                  /* check for SHELL statement */
                  pv = pos('SHELL=', ucdata)
                  If pv <> 0 Then Do
                     pv = pos('/P', ucdata)
                     If pv <> 0 Then data = substr(data, 1, pv-1)
                  end
               end
            end
            If version21 = 1 | basedevflag = 0 Then Do
               /* version 2.1 or 2.00.1 */
               /* check for EPATH statement */
               pv = pos('EPATH=', ucdata)
               If pv <> 0 Then Do
                  /* replace with EPMPATH */
                  newdata = substr(data, 1, pv-1) || 'EPMPATH' || substr(data, pv+5)
                  data = newdata
               end
            end
         end
         Call Lineout tmpconfigname, data
      end

      /* process XGA statements */
      xgasysflag = 0
      position = pos(':\OS2\', ucdata)
      bootdrv2 = substr(data, position-1, 1)
      vgap = pos(',BGA,', ucdata)
      If vgap <> 0 Then Do
         newdata = substr(data, 1, vgap) || 'VGA' || substr(data, vgap+4)
         data = newdata
      end
      flag = 0
      Do while flag = 0
         If driver = 'IBMXGA32' Then Do
            If translate(substr(data, 1, 4)) = 'REM ' Then Do
               data = substr(data, 5)
               ucdata = translate(data)
            end
            If xgasysflag = 0 Then Do
               If pos('XGA.SYS', ucdata) <> 0 Then Do
                  /* replace XGA.SYS statement with version appropriate with */
                  /* current value of basedevflag. */
                  If basedevflag = 0 Then newdata = 'DEVICE='bootdrv2':\OS2\XGA.SYS'
                  else newdata = 'BASEDEV=XGA.SYS'
                  data = newdata
                  xgasysflag = 1
               end
            end
            if pos('\OS2\MDOS\VVGA.SYS', ucdata) <> 0 Then Do
               If xgasysflag = 0 Then Do
                  /* insert the XGA.SYS record */
                  If basedevflag = 0 Then newdata = 'DEVICE='bootdrv2':\OS2\XGA.SYS'
                  else newdata = 'BASEDEV=XGA.SYS'
                  Call Lineout tmpconfigname, newdata
                  xgasysflag = 1
               end
            end
         end
         else Do
            /* make sure records are REMed out */
            if substr(ucdata, 1, 4) <> 'REM ' Then Do
               newdata = 'REM '||data
               data = newdata
            end
            If pos('XGA.SYS', ucdata) <> 0 Then Do
               /* replace XGA.SYS statement with version appropriate with */
               /* current value of basedevflag. */
               If basedevflag = 0 Then newdata = 'REM DEVICE='bootdrv2':\OS2\XGA.SYS'
               else newdata = 'REM BASEDEV=XGA.SYS'
               data = newdata
               xgasysflag = 1
            end
            else if pos('\OS2\MDOS\VVGA.SYS', ucdata) <> 0 Then Do
               If xgasysflag = 0 Then Do
                  /* insert the XGA.SYS record */
                  If basedevflag = 0 Then newdata = 'REM DEVICE='bootdrv2':\OS2\XGA.SYS'
                  else newdata = 'REM BASEDEV=XGA.SYS'
                  Call Lineout tmpconfigname, newdata
               end
            end
         end
         Call lineout tmpconfigname, data
         if pos('SET VIO_XGA=', ucdata) <> 0 Then flag = 1
         else Do
            data = linein(configname)
            ucdata = translate(data)
         end
      end

      /* copy records to next DEVINFO (8514) */
      flag = 0
      do until lines(configname) = 0
         data = linein(configname)
         ucdata = translate(data)
         If pos('DEVINFO=', ucdata) <> 0 Then Leave
         Call Lineout tmpconfigname, data
      end

      /* process 8514 statements */
      flag = 0
      Do while flag = 0
         If driver = 'IBM8514' Then Do
            If translate(substr(data, 1, 4)) = 'REM ' Then Do
               data = substr(data, 5)
            end
         end
         else Do
            /* make sure records are REMed out */
            if substr(ucdata, 1, 4) <> 'REM ' Then Do
               newdata = 'REM '||data
               data = newdata
            end
         end
         Call lineout tmpconfigname, data
         if pos('\V8514A.SYS', ucdata) <> 0 Then flag = 1
         else Do
            data = linein(configname)
            ucdata = translate(data)
         end
      end

      /* copy records to the VGA DEVINFO */
      do until lines(configname) = 0
         data = linein(configname)
         ucdata = translate(data)
         If pos('DEVINFO=SCR,VGA', ucdata) <> 0 Then Leave
         Call Lineout tmpconfigname, data
      end
      newdata = ''

      /* process VGA statements */
      flag = 0
      Do while flag = 0
         If driver = 'IBMVGA' | driver = 'IBMVGA32' Then Do
            If substr(ucdata, 1, 4) = 'REM ' Then Do
               data = substr(data, 5)
            end
         end
         else Do
            /* make sure records are REMed out */
            if substr(ucdata, 1, 4) <> 'REM ' Then Do
               newdata = 'REM '||data
               data = newdata
            end
         end
         Call lineout tmpconfigname, data
         if pos('\VVGA.SYS', ucdata) <> 0 Then flag = 1
         else Do
            data = linein(configname)
            ucdata = translate(data)
         end
      end

      /* copy remaining records */
      ipfkeys = 0
      faxpm = 0
      do until lines(configname) = 0
         data = linein(configname)
         ucdata = translate(data)
         If pos('IPF_KEYS=', ucdata) <> 0 Then ipfkeys = 1
         If pos('FAXPM=', ucdata) <> 0 Then faxpm = 1
         Call Lineout tmpconfigname, data
      end

      If ipfkeys = 0 Then Do
         If basedevflag = 0 | version21 = 1 Then Do
            /* Add 2.00.1 and 2.1 statement */
            Call Lineout tmpconfigname,'SET IPF_KEYS=SBCS'
         end
      end

      If faxpm = 0 Then Do
         If version21 = 1 Then Do
            /* Add 2.1 statement */
            Call Lineout tmpconfigname,'SET FAXPM='bootdrv2':\OS2\APPS'
         end
      end

      Call Lineout tmpconfigname   /* close file */
      Call Lineout configname      /* close file */

      /* rename original CONFIG.20 */
      cmdstr = '@rename 'configname' CONFIG.20B'
      address CMD cmdstr
      if rc = 0 Then Do
         /* rename CONFIG.20T to be CONFIG.20 */
         cmdstr = '@rename 'tmpconfigname' CONFIG.20'
         address CMD cmdstr
         if rc = 0 Then Do
            /* delete orginal CONFIG.20 */
            /* Call SysFileDelete bakconfigname */
            cmdstr = '@DEL 'bakconfigname
            address CMD cmdstr
         end
         else Do
            /* rename error, try to restore original */
            cmdstr = '@rename 'bakconfigname' CONFIG.20'
            address CMD cmdstr
         end
      end
   end
   else return 4   /* file not found */
   return 0


/* Subroutine Update_Client_Configri_20 */
/* This subroutine creates a new CONFIGRI.20 for the client that has the */
/* DRIVER=x:\OS2\TESTCFG.SYS statement added to it */
Update_Client_Configri_20:

   /* build fully qualified path names for client CONFIGRI.20 */
   If client = 'DEFALT20' Then Do
      /* DEFALT20 is special case, it must be called for each lantype */
      configname = rpldir'\IBMCOM\'lantype'\CONFIGRI.20'
      tmpconfigname = rpldir'\IBMCOM\'lantype'\CONFIGRI.20T'
      bakconfigname = rpldir'\IBMCOM\'lantype'\CONFIGRI.20B'
   end
   else Do
      configname = rpldir'\MACHINES\'client'\CONFIGRI.20'
      tmpconfigname = rpldir'\MACHINES\'client'\CONFIGRI.20T'
      bakconfigname = rpldir'\MACHINES\'client'\CONFIGRI.20B'
   end
   /* Call SysFileDelete tmpconfigname */
   Call Delete_File tmpconfigname
   Call Delete_File bakconfigname

   bootdrv = 'Z'
   If File_Exist(configname) Then Do
      /* copy records looking for TESTCFG.SYS statement */
      flag = 0
      do until lines(configname) = 0
         data = linein(configname)
         ucdata = translate(data)
         dpos = pos('SET PATH=', ucdata)
         if dpos <> 0 Then Do
            /* get bootdrive id from SET PATH statement */
            dpos = pos(':\OS2;', ucdata)
            if dpos <> 0 Then bootdrv = substr(ucdata, dpos-1, 1)
         end
         dpos = pos('DEVICE=', ucdata)
         If dpos <> 0 Then Do
            If pos('\OS2\TESTCFG.SYS', ucdata, dpos+7) <> 0 Then Do
               /* DEVICE statement already present */
               If word(ucdata, 1) = 'REM' Then Do
                  /* statement REMed out, un-REM it */
                  newdata = substr(data, dpos)
                  data = newdata
               end
               flag = 1
            end
         end
         Call lineout tmpconfigname, data
      end
      If flag = 0 Then Do
         /* add DEVICE statement to end of file */
         Call lineout tmpconfigname, 'DEVICE='bootdrv':\OS2\TESTCFG.SYS'
      end

      Call Lineout tmpconfigname   /* close file */
      Call Lineout configname      /* close file */

      /* rename original CONFIGRI.20 */
      cmdstr = '@rename 'configname' CONFIGRI.20B'
      address CMD cmdstr
      if rc = 0 Then Do
         /* rename CONFIGRI.20T to be CONFIGRI.20 */
         cmdstr = '@rename 'tmpconfigname' CONFIGRI.20'
         address CMD cmdstr
         if rc = 0 Then Do
            /* delete orginal CONFIGRI.20 */
            /* Call SysFileDelete bakconfigname */
            cmdstr = '@DEL 'bakconfigname
            address CMD cmdstr
         end
         else Do
            /* rename error, try to restore original */
            cmdstr = '@rename 'bakconfigname' CONFIGRI.20'
            address CMD cmdstr
         end
      end
   end
   else return 4   /* file not found */
   return 0



/* Subroutine Update_LS20_Config_20 */
/* This subroutine converts a LAN Server 2.0 CONFIG.20 file to the new format */
Update_LS20_Config_20:

   /* build fully qualified path names for client CONFIG.20 */
   If client = 'DEFALT20' Then Do
      /* DEFALT20 is special case, it must be called for each lantype */
      configname = rpldir'\IBMCOM\'lantype'\CONFIG.20'
      tmpconfigname = rpldir'\IBMCOM\'lantype'\CONFIG.20T'
      bakconfigname = rpldir'\IBMCOM\'lantype'\CONFIG20.BAK'
   end
   else Do
      configname = rpldir'\MACHINES\'client'\CONFIG.20'
      tmpconfigname = rpldir'\MACHINES\'client'\CONFIG.20T'
      bakconfigname = rpldir'\MACHINES\'client'\CONFIG20.BAK'
   end
   /* Call SysFileDelete tmpconfigname */
   Call Delete_File tmpconfigname
   Call Delete_File bakconfigname

   /* determine bootdrive id from PROTSHELL statement */
   do until lines(configname) = 0
      data = translate(linein(configname))
      If substr(data, 1, 9) = 'PROTSHELL' Then Do
         ddi = pos(':\', data)
         newbootdrv = substr(data,ddi - 1, 1)
         Leave
      end
   end
   Call Lineout configname     /* close file */
   if newbootdrv = '' Then return 4

   /* check to see if file has already been updated with new device driver */
   /* statements */
   /* Call SysFileSearch '=XGA.SYS', configname, 'xgasys.' */
   /* read client CONFIG.20 looking for the XGA.SYS entry */
   do until lines(configname) = 0
      data = translate(linein(configname))
      xgasysp = pos('XGA.SYS', data)
      if xgasysp <> 0 then Do
         If xgasysp > 1 Then Do
            prevchar = substr(data, xgasysp-1, 1)
            If prevchar = '=' | prevchar = '\' Then Leave
            else xgasysp = 0   /* ignore entry */
         end
         else xgasysp = 0   /* ignore entry */
      end
   end
   Call Lineout configname     /* close file */
   If xgasysp = 0 Then Do
      /* file has not been updated, update it */
      /* check to see if dd_data needs to be initialized */
      If ddbootdrv <> newbootdrv Then Do
         dd_data.0 = 0
         ddbootdrv = newbootdrv
      end
      If dd_data.0 = 0 Then Do
         /* initialize dd_data */
         dd_data.0 = 27
         dd_data.1 = 'REM Use the following 7 statements for workstations with XGA displays:'
         dd_data.2 = 'REM DEVINFO=SCR,VGA,'ddbootdrv':\OS2\VIOTBL.DCP'
         dd_data.3 = 'REM DEVICE='ddbootdrv':\OS2\XGARING0.SYS'

         If basedevflag = 0 Then dd_data.4 = 'REM DEVICE='ddbootdrv':\OS2\XGA.SYS'
         else dd_data.4 = 'REM BASEDEV=XGA.SYS'

         dd_data.5 = 'REM DEVICE='ddbootdrv':\OS2\MDOS\VVGA.SYS'
         dd_data.6 = 'REM DEVICE='ddbootdrv':\OS2\MDOS\VXGA.SYS '
         dd_data.7 = 'REM SET VIDEO_DEVICES=VIO_XGA'
         dd_data.8 = 'REM SET VIO_XGA=DEVICE(BVHVGA,BVHXGA)'
         dd_data.9 = ' '
         dd_data.10 = 'REM Use the following 5 statements for workstations with 8514/A adapters:'
         dd_data.11 = 'REM DEVINFO=SCR,BGA,'ddbootdrv':\OS2\VIOTBL.DCP'
         dd_data.12 = 'REM SET VIDEO_DEVICES=VIO_8514A'
         dd_data.13 = 'REM SET VIO_8514A=DEVICE(BVHVGA,BVH8514A)'
         dd_data.14 = 'REM DEVICE='ddbootdrv':\OS2\MDOS\VVGA.SYS'
         dd_data.15 = 'REM DEVICE='ddbootdrv':\OS2\MDOS\V8514A.SYS'
         dd_data.16 = ' '
         dd_data.17 = 'REM Use the following 4 statements for workstations with EGA displays:'
         dd_data.18 = 'REM DEVINFO=SCR,EGA,'ddbootdrv':\OS2\VIOTBL.DCP'
         dd_data.19 = 'REM SET VIDEO_DEVICES=VIO_EGA'
         dd_data.20 = 'REM SET VIO_EGA=DEVICE(BVHEGA)'
         dd_data.21 = 'REM DEVICE='ddbootdrv':\OS2\MDOS\VEGA.SYS'
         dd_data.22 = ' '
         dd_data.23 = 'REM Use the following 4 statements for workstations with VGA displays:'
         dd_data.24 = 'DEVINFO=SCR,VGA,'ddbootdrv':\OS2\VIOTBL.DCP'
         dd_data.25 = 'SET VIDEO_DEVICES=VIO_VGA'
         dd_data.26 = 'SET VIO_VGA=DEVICE(BVHVGA)'
         dd_data.27 = 'DEVICE='ddbootdrv':\OS2\MDOS\VVGA.SYS'
      end

      If File_Exist(configname) = 1 Then Do
         /* copy records until the DEVINFO=SCR,.. statement */
         do until lines(configname) = 0
            data = translate(linein(configname))
            If substr(data, 1, 12) = 'DEVINFO=SCR,' Then Leave
            Call Lineout tmpconfigname,data
         end

         Call Lineout tmpconfigname, ' '   /* blank line */

         /* write new device driver statements to file */
         do ddi= 1 to dd_data.0
           Call lineout tmpconfigname, dd_data.ddi
         end

         Call Lineout tmpconfigname, ' '   /* blank line */

         /* skip display records until POINTDD.SYS statement */
         do until lines(configname) = 0
            data = translate(linein(configname))
            skip = 0
            ddp1 = pos('XGARING0.SYS', data)
            If ddp1 <> 0 Then skip = 1
            else Do
               ddp1 = pos('VXGA.SYS', data)
               if ddp1 <> 0 Then skip = 1
               else Do
                  ddp1 = pos('VIO_XGA', data)
                  if ddp1 <> 0 Then skip = 1
                  else Do
                     ddp1 = pos('VIO_VGA', data)
                     if ddp1 <> 0 Then skip = 1
                     else Do
                        ddp1 = pos('VVGA.SYS', data)
                        if ddp1 <> 0 Then skip = 1
                        else Do
                           ddp1 = pos('VGA DISPLAYS:', data)
                           if ddp1 <> 0 Then skip = 1
                           else Do
                              ddp1 = pos('XGA DISPLAYS:', data)
                              if ddp1 <> 0 Then skip = 1
                           end
                        end
                     end
                  end
               end
            end
            If substr(data, 15, 11) = 'POINTDD.SYS' Then Leave
            If skip = 0 Then Do
              If data <> '' Then Call Lineout tmpconfigname, data
            end
         end

         Call Lineout tmpconfigname, data  /* write pointdd.sys */

         /* copy remaining records */
         do until lines(configname) = 0
            data = linein(configname)
            Call Lineout tmpconfigname, data
         end

         Call Lineout configname
         Call Lineout tmpconfigname

         /* rename original CONFIG.20 */
         cmdstr = '@rename 'configname' CONFIG20.BAK'
         address CMD cmdstr
         if rc = 0 Then Do
            /* rename CONFIG.20T to be CONFIG.20 */
            cmdstr = '@rename 'tmpconfigname' CONFIG.20'
            address CMD cmdstr
            if rc <> 0 Then Do
               /* rename error, try to restore original */
               cmdstr = '@rename 'bakconfigname' CONFIG.20'
               address CMD cmdstr
               return 4
            end
         end
         else return 4
      end
   end
   return 0


/* subroutine to delete a file */
Delete_File:
   arg delfilename
   If lines(delfilename) <> 0 Then Do
      Call stream delfilename,C,'close'
      cmdline = '@DEL 'delfilename
      address CMD cmdline
   end
   else Call stream delfilename,C,'close'  /* must do a pseudo-close to prevent */
                                       /* a zero length file from being created */
   return 0



/* subroutine to display/log messages */
Display_Log_Msg:
   Parse Arg msgstring, parm.1, parm.2, parm.3, parm.4, parm.5, parm.6
   if parm.1 <> '' Then Do
      /* substitution data supplied */
      startpoint = 1
      newmsg = ''
      sdpos = pos('%', msgstring, startpoint)
      Do while sdpos <> 0
         j = substr(msgstring, sdpos+1, 1)
         newmsg = newmsg || substr(msgstring, startpoint, sdpos-startpoint) || parm.j
         startpoint = sdpos+2
         sdpos = pos('%', msgstring, startpoint)
      end
      newmsg = newmsg || substr(msgstring, startpoint)
      msgstring = newmsg
   end

   If Logfile = '' Then Do
      /* display message locally */
      Say msgstring
   end
   else Do
      /* log message */
      Call lineout Logfile, msgstring
   end
   return 0





/* subroutine to test for existance of a file */
File_Exist:
   Arg testfilename
   If lines(testfilename) = 1 Then rc = 1
   else rc = 0
   /* lines() leaves file open, must force close */
   Call stream testfilename,C,'close'
   return rc



Determine_RIPL_Directory:
   /* find out where IBMLAN is installed */
   os2path = value('PATH',,'OS2ENVIRONMENT')

   ibmlanp = pos(':\IBMLAN', translate(os2path))
   if ibmlanp = 0 Then Do
     msgdata = errprefix || message.17
     Call Display_Log_Msg msgdata, ''
     return 4
   end

   landrv = substr(os2path, ibmlanp-1, 1)
   lanpath = landrv':\IBMLAN'

   fname = lanpath'\IBMLAN.INI'

   /* read IBMLAN.INI looking for the RPLDIR entry */
   do until lines(fname) = 0
      data = linein(fname)
      rpldirp = pos('RPLDIR', translate(data))
      if rpldirp <> 0 then do
         rpldirp = pos('=', data)
         rpldir = strip(substr(data, rpldirp+1))
         Leave
      end
   end
   Call stream fname,C,'close'

   if rpldir = '' then do
      msgdata = errprefix || message.18
      Call Display_Log_Msg msgdata, ''
      return 4
   end
   return 0



/* subroutine to verify that the display driver id is valid */
Validate_Display_DriverID:
   /* initialize display driver ids */
   displaydrv.0 = 3
   displaydrv.1 = 'IBMVGA32'
   displaydrv.2 = 'IBMXGA32'
   displaydrv.3 = 'IBM8514'

   /* validate driver parameter */
   flag = 0
   Do i=1 to displaydrv.0
      If driver=displaydrv.i Then Do
         flag = 1
         leave
      end
   end
   if flag = 0 Then Do
      Call Display_Log_Msg message.19, ''
      return 4
   end
   return 0



/* subroutine to update client workstation record in RPL.MAP */
Update_Workstation_Record:
   rplmap = rpldir'\RPL.MAP'
   tmprplmap = rpldir'\RPLMAP.TMP'
   updateerr = 0
   Call Delete_File(tmprplmap)
   Do until lines(rplmap) = 0
      data = linein(rplmap)
      field2 = translate(word(data, 2))
      If field2 = client Then Do
         /* may be a client record, check field 4 for FIT */
         field4 = word(data,4)
         If translate(substr(field4, 1, 5)) = 'FITS\' Then Do
            /* this is the client workstation record */
            field12 = word(data, 12)
            f12p = pos(field12, data)
            ucnewos2 = translate(newOS2)
            /* determine OS type for field 12 */
            Select
               when ucnewos2 = 'OS2.20' Then os2type = '20'
               when ucnewos2 = 'OS2.20a' Then os2type = '20A'
               when ucnewos2 = 'OS2.21' Then os2type = '21'
               otherwise Do
                  os2type = ''
                  updateerr = 1
               end
            end
            If os2type <> '' Then Do
               /* update field 12 for correct OS type */
               newdata = substr(data, 1, f12p+1) || os2type
               f12p = pos('_', data, f12p+2)
               if f12p <> 0 Then newdata = newdata || substr(data, f12p)
               else Do
                    /* error, use original record */
                    newdata = data
                    updateerr = 1
               end
               data = newdata
            end
         end
      end
      Call lineout tmprplmap, data
   end
   Call lineout rplmap
   Call lineout tmprplmap
   If updateerr = 0 Then Do
      /* no error, replace RPL.MAP with new RPL.MAP */
      '@copy 'rplmap  rpldir'\RPLMAP.BAK 1>nul 2>nul'
      Call Delete_File(rplmap)
      '@rename 'tmprplmap 'RPL.MAP 1>nul 2>nul'
   end
   else Do
      /* update error, delete tmp file, return error */
      Call Delete_File(tmprplmap)
      return 4
   end
   return 0





/* subroutine to read user provided response file and set keyword variables */
Read_Rspfile:
   /* read response file and get keyword values */
   clientlist.0 = 0
   Do until lines(Rspfile) = 0
      data = translate(linein(Rspfile))
      If substr(data, 1, 13) = 'DISPLAYDRIVER' Then Do
         epos = pos('=', data)
         driver = strip(substr(data, epos+1))
      end
      else If substr(data, 1, 10) = 'CLIENTLIST' | ,
              substr(data, 1, 6) = 'CLIENT' Then Do
         /* multiple CLIENT and/or CLIENTLIST entries are permitted */
         epos = pos('=', data)
         clientdata = strip(substr(data, epos+1))
         cpos = pos(',', clientdata)
         if cpos = 0 Then Do
            /* only one client specified */
            clnum = clientlist.0 + 1
            clientlist.clnum = clientdata
            clientlist.0 = clnum
         end
         else Do
            /* list of clients provided */
            startpoint = 1
            clnum = clientlist.0 + 1
            cpos = pos(',', clientdata, startpoint)
            /* scan list and put each name in clientlist array */
            Do while (cpos <> 0)
               clientlist.clnum = strip(substr(clientdata, startpoint, cpos-startpoint))
               clientlist.0 = clnum
               startpoint = cpos+1
               clnum = clnum + 1
               cpos = pos(',', clientdata, startpoint)
            end
            if startpoint < length(clientdata) Then Do
               /* last name in list */
               clientlist.clnum = strip(substr(clientdata, startpoint))
               clientlist.0 = clnum
            end
         end
      end
      else If substr(data, 1, 13) = 'CURRENTOS2DIR' Then Do
         /* optional OS/2 root directory parameter */
         epos = pos('=', data)
         currentOS2 = strip(substr(data, epos+1))
         changeOS2 = 1
      end
      else If substr(data, 1, 9) = 'NEWOS2DIR' Then Do
         /* optional OS/2 root directory parameter */
         epos = pos('=', data)
         newOS2 = strip(substr(data, epos+1))
         changeOS2 = 1
      end
   end
   Call lineout Rspfile  /* close file */

   /* check for required keywords */
   If driver = '' | clientlist.0 = 0 Then Do
      /* invalid/incomplete response file */
      return 4
   end
   return 0



/* subroutine to process command line parameters */
Process_Input_Parameters:

   Do i = 1 to 6
      If parm.i = '' Then Leave
      parmtype = translate(substr(parm.i,1,2))
      select
         when parmtype = '/H' Then Call Syntax_Help

         when parmtype = '/h' Then Call Syntax_Help

         when parmtype = '/D' Then Do
            if substr(parm.i, 3, 1) = ':' Then driver = substr(parm.i, 4)
            else driver = substr(parm.i, 3)
            driver = translate(driver)
         end

         when parmtype = '/C' Then Do
            if substr(parm.i, 3, 1) = ':' Then clientdata = substr(parm.i, 4)
            else clientdata = substr(parm.i, 3)
            clientdata = translate(clientdata)
            cpos = pos(',', clientdata)
            if cpos = 0 Then Do
               clientlist.0 = 1
               clientlist.1 = clientdata
            end
            else Do
               /* list of clients provided */
               startpoint = 1
               clnum = 1
               cpos = pos(',', clientdata, startpoint)
               /* scan list and put each name in clientlist array */
               Do while (cpos <> 0)
                  clientlist.clnum = substr(clientdata, startpoint, cpos-startpoint)
                  clientlist.0 = clnum
                  startpoint = cpos+1
                  clnum = clnum + 1
                  cpos = pos(',', clientdata, startpoint)
               end
               if startpoint < length(clientdata) Then Do
                  /* last name in list */
                  clientlist.clnum = substr(clientdata, startpoint)
                  clientlist.0 = clnum
               end
            end
         end

         when parmtype = '/O' Then Do
            if substr(parm.i, 3, 1) = ':' Then currentOS2 = substr(parm.i, 4)
            else currentOS2 = substr(parm.i, 3)
            changeOS2 = 1
         end

         when parmtype = '/N' Then Do
            if substr(parm.i, 3, 1) = ':' Then newOS2 = substr(parm.i, 4)
            else newOS2 = substr(parm.i, 3)
            changeOS2 = 1
         end

         when parmtype = '/L' Then Do
            if substr(parm.i, 3, 1) = ':' Then Logfile = substr(parm.i, 4)
            else Logfile = substr(parm.i, 3)
         end

         when parmtype = '/R' Then Do
            if substr(parm.i, 3, 1) = ':' Then Rspfile = substr(parm.i, 4)
            else Rspfile = substr(parm.i, 3)
         end

         otherwise Do
           msgdata = errprefix || message.20
           Call Display_Log_Msg msgdata, parm.i
           return 4
         end
      end
   end
   return 0



Determine_Installed_OS2_Levels:
   syslevel = '\OS2\INSTALL\SYSLEVEL.OS2'
   If File_Exist(rpldir'\OS2'syslevel) = 1 Then os213 = 1
   else os213 = 0

   If File_Exist(rpldir'\OS2.20'syslevel) = 1 Then os220 = 1
   else os220 = 0

   If File_Exist(rpldir'\OS2.20a'syslevel) = 1 Then os220a = 1
   else os220a = 0

   If File_Exist(rpldir'\OS2.21'syslevel) = 1 Then os221 = 1
   else os221 = 0
   return


/* Subroutine Syntax_Help */
/* This subroutine displays the syntax help panel. */
Syntax_Help:
  If Logfile = '' Then Do
     Do i = 30 to 34
        Call Display_Log_Msg message.i, ''
     end
     Call Display_Log_Msg message.22, ''
     Pull dummy

     Do i = 35 to 37
        Call Display_Log_Msg message.i, ''
     end
     Call Display_Log_Msg message.22, ''
     Pull dummy

     Call Display_Log_Msg message.38, ''
     exit
  end
  else Do
     Call Display_Log_Msg message.21, p1, p2, p3, p4, p5, p6
     Call Error_Exit
  end


Initialize_NLS_Messages:
/***********************************************************************/
/*                            NLS messages                             */
/*                                                                     */
/* The following message strings should be translated for NLS support. */
/*                                                                     */
/* Note: The %1, %2, etc. strings represent dynamic data fields that   */
/*       are filled in when the message is displayed.  Do not          */
/*       translate the %1, %2, etc. fields.                            */
/*       The 'CRLF,' sequence that ends some lines represents          */
/*       formatting control (new line) and should not be translated.   */
/*       However, after translation, each message may need to be       */
/*       manually reformatted to prevent excessive lenght message      */
/*       lines. The message display code does not automatically        */
/*       reformat long message lines.                                  */
/***********************************************************************/
arg message_type

errprefix = 'ERROR: '
unknown = 'UNKNOWN'

/* start of error and informational messages */
message.1 = 'Missing keyword in '
message.2 = '       Displaydriver, Client or Clientlist, and Logfile are required.'
message.3 = 'HISTORY LOG: '
message.4 = 'Unable to open the Logfile '
message.5 = 'The display driver %1.DLL is not installed in the Remote IPL' CRLF,
            '      directory %2.'
message.6 = ''
message.7 = 'Error detected.  Check log file: '
message.8 = 'The version of LAN Server is prior to Version 2.0.'
message.9 = 'The version of OS/2 in the %1 directory tree is' CRLF,
            '      %2.  This version of OS/2 does not support 32 bit display drivers.' CRLF,
            '      Processing is being terminated.'
message.10 = ' copy of OS2INI.20 was updated.'
message.11 = ' copy of CONFIG.20 was updated.'

message.12 = ' copy of CONFIG.20 was not updated.'
message.13 = '.FIT was updated.'
message.14 = '.FIT was not updated.'
message.15 = ' copy of OS2INI.20 was not updated.  Either the file did' CRLF,
             '      not exist or the Remote client is still powered up and has the' CRLF,
             '      file locked.  Processing terminated for '
message.16 = 'Not able to upgrade the format of file %1\IBMCOM\%2\CONFIG.20.'
message.17 = 'The PATH environment variable does not contain a reference to' CRLF,
             '      \IBMLAN.  Either LAN Server 2.0 is not installed or the PATH' CRLF,
             '      variable has been modified to remove \IBMLAN references.  If' CRLF,
             '      LAN Server 2.0 is installed, use the SET command to update the' CRLF,
             '      PATH variable to specify the path for the directory \IBMLAN\NETPROG' CRLF,
             '      and then rerun RPLSETD.CMD.'
message.18 = 'RPLDIR parameter was not found in IBMLAN.INI.'
message.19 = 'Invalid display driver specified.'
message.20 = 'The option %1 is not a valid option.'
message.21 = 'The following RPLSETD command syntax is not valid:' CRLF,
             '      RPLSETD  %1 %2 %3 %4 %5 %6'
message.22 = CRLF 'Press the Enter key for next screen.'
message.23 = 'The RPLMIG21.CMD procedure has not been run to upgrade Remote IPL' CRLF,
             'to support OS/2 2.1.  Run RPLMIG21.CMD prior to using RPLSETD.CMD.'
message.24 = ''
message.25 = ' copy of CONFIGRI.20 was updated.'
message.26 = ' copy of CONFIGRI.20 not was updated.'
message.27 = ' workstation record in RPL.MAP was updated.'
message.28 = ' workstation record in RPL.MAP could not be updated.'

/* start of syntax help messages */
message.30 = 'RPLSETD   [/H]  [ /D:driver  /C:client[,client2,client3,...] ] ' CRLF,
             '          [/O:current_OS2]  [/N:new_OS2] [/L:Logfile]   [/R:Response_file]'
message.31 = '    where /H is a request for this help panel.  If specified, it must'CRLF,
             '             be first parameter.'
message.32 = '          /D:driver (DISPLAYDRIVER) is the new display type for the client' CRLF,
             '                 definition.  The following display types are valid:' CRLF,
             '                   IBMVGA32  - 32 bit VGA display driver' CRLF,
             '                   IBMXGA32  - 32 bit XGA display driver' CRLF,
             '                   IBM8514   - 16 bit 8514 display driver'
message.33 = '          /C:client (CLIENT or CLIENTLIST) is the name of one or more OS/2' CRLF,
             '                 Remote IPL client workstations.  If more than one client is'CRLF,
             '                 specified, each client name must be separated with a comma.'CRLF,
             '                 Imbedded blanks are not allowed.  Multiple CLIENT and/or'CRLF,
             '                 CLIENTLIST parameters are allowed in the response file.'
message.34 = '          /O:current_OS2 (CURRENTOS2DIR) is the root directory (under' CRLF,
             '                 \IBMLAN\RPL) for the current OS/2 2.0 operating system that' CRLF,
             '                 the client is booting.  This parameter is case sensitive.' CRLF,
             '                 The default is OS2.20.  /O or /N should only be specified' CRLF,
             '                 when the version of the client operating system should be' CRLF,
             '                 changed.'
message.35 = '          /N:new_OS2 (NEWOS2DIR) is the root directory (under \IBMLAN\RPL) for' CRLF,
             '                 the OS/2 2.0 operating system that the client is to be' CRLF,
             '                 switched to.  This parameter is case sensitive.  The default' CRLF,
             '                 is OS2.21.  /O or /N should only be specified when the' CRLF,
             '                 version of the client operating system should be changed.'
message.36 = '          /L:Logfile is the fully qualified name of a file to ' CRLF,
             '                  which all messages and errors are to be logged.  If' CRLF,
             '                  logfile is specified, no messages or errors will be' CRLF,
             '                  displayed on the screen.'
message.37 = '          /R:Response_file is the fully qualified name of a file' CRLF,
             '                  that contains all the inputs in keyword form(Keyword=value).'CRLF,
             '                  The keywords are indicated above in () following the /X:' CRLF,
             '                  syntax.  If this parameter is specified, all other command' CRLF,
             '                  line parameters except Logfile are ignored.  It is' CRLF,
             '                  recommended that the Logfile parameter be specfied whenever' CRLF,
             '                  the Response_file parameter is used.'
message.38 = '    Note: The following conditions must be meet before running RPLSETD:' CRLF,
             '          1. The Remoteboot Service should be stopped.' CRLF,
             '          2. The Remote IPL workstations to be upgraded must do a' CRLF,
             '             System Shutdown.'CRLF,
             '          3. Remote IPL workstations to be upgraded can not have a session' CRLF,
             '             with the RIPL server.  If necessary, force the session to' CRLF,
             '             disconnect using the server full screen interface. '
return
