#include "winbuf.h"

// -------------------------------------------------------
//  Constructor
//
//  bBufferedRedraw_ -- initial buffer redraw flag
//
TWinBuffered::TWinBuffered(PTWindowsObject AParent, 
                           LPSTR           ATitle, 
                           PTModule        AModule,
                           BOOL            bBufferedRedraw_)
             :TWindow(AParent, ATitle, AModule),
              bBufferedRedraw(bBufferedRedraw_)
{
    //
    //  Buffer isn't created until WMSize().
    //
    hbmBuffer = 0;
}

// -------------------------------------------------------
//  Destructor
//
TWinBuffered::~TWinBuffered()
{
    if(hbmBuffer) DeleteBitmap(hbmBuffer);
}

// -------------------------------------------------------
//  If our buffered redraw flag is set, free our existing
//  bitmap buffer, if any, and try to allocate another 
//  one.  If we get it, redraw its contents -- and at the
//  same time, draw the window contents.  Validate the 
//  window after the drawing is complete.
//
void TWinBuffered::WMSize(RTMessage msg)
{
    TWindow::WMSize(msg);

    size.cx = msg.LP.Lo;
    size.cy = msg.LP.Hi;

    if(bBufferedRedraw == FALSE)
        return; 

    //
    // If no size, nothing to do.
    //
    if((size.cx == 0) && (size.cy == 0))
        return;

    //
    // If we have a buffer, delete it.
    //
    if(hbmBuffer)
        DeleteBitmap(hbmBuffer);
    
    //
    // Try to create a new one.
    //
    HDC hdcScreen = GetDC(0);
    hbmBuffer = CreateCompatibleBitmap(hdcScreen,
                                       size.cx, size.cy);
    ReleaseDC(0, hdcScreen);

    //
    // If no luck, we're done.
    //
    if(hbmBuffer == 0) 
        return;

    //
    // Clear the contents of the buffer with light gray.
    //
    HDC     hdcBuf  = CreateCompatibleDC(0);
    HBITMAP hbmSave = SelectBitmap(hdcBuf, hbmBuffer);

    RECT rc;
    rc.left = rc.top = 0;
    rc.right = size.cx;
    rc.bottom = size.cy;
    FillRect(hdcBuf, &rc, GetStockBrush(LTGRAY_BRUSH));

    //
    // Clear the window with light gray, then draw in the 
    // window and buffer at the same time.
    //
    HDC hdcWin = GetDC(HWindow); 
    FillRect(hdcWin, &rc, GetStockBrush(LTGRAY_BRUSH));
    prvDraw(hdcBuf, hdcWin);

    //
    // Clean up.
    //
    ReleaseDC(HWindow, hdcWin);
    SelectBitmap(hdcBuf, hbmSave);
    DeleteDC(hdcBuf);

    //
    // Validate entire window so no WM_PAINT.
    //
    ValidateRgn(HWindow, NULL);
}

// -------------------------------------------------------
//  If we're supposed to do buffered redrawing and we've got
//  a buffer, use the buffered drawing function, else use 
//  the regular drawing function.
//
void TWinBuffered::Paint(HDC hdc, PAINTSTRUCT &ps)
{
    if(bBufferedRedraw && hbmBuffer)
        prvDrawWithBuffer(hdc, ps);
    else
        prvDraw(hdc, 0);    // 0 indicates no second DC 
}

// -------------------------------------------------------
//  Use BitBlt() to copy all or part of our buffer to the 

//  DC provided.  Use the rcPaint member of the PAINTSTRUCT
//  to figure out how much to draw.
//
void TWinBuffered::prvDrawWithBuffer(HDC         hdc, 
                                     PAINTSTRUCT &ps)
{
    //
    //  Create a DC that's compatible with the one that's
    //  passed in, then select our buffer bitmap into it.
    //
    HDC     hdcLocal = CreateCompatibleDC(hdc);
    HBITMAP hbmSave  = SelectBitmap(hdcLocal, hbmBuffer);
    
    //
    //  Copy the required portion from the buffer to the DC.
    //
    RECT &rc = ps.rcPaint;
    int nWidth  = rc.right-rc.left;
    int nHeight = rc.bottom-rc.top;

    BitBlt(hdc,                 // dest DC
           rc.left, rc.top,     // dest upper left corner
           nWidth, nHeight,     // dest size
           hdcLocal,            // source DC
           rc.left, rc.top,     // source upper left corner
           SRCCOPY);

    //
    //  Clean up.
    //
    SelectBitmap(hdcLocal, hbmSave);
    DeleteDC(hdcLocal);
}

// ---------------------------------------------------------
//  Turn buffering on or off.  If it's turned on, force a 
//  redraw by sending a dummy WM_SIZE message to ourselves.
//  If turned off, force a redraw with InvalidateRect().
//
void TWinBuffered::SetbBufferedRedraw(BOOL b)
{
    bBufferedRedraw = b;

    if(b) {
        RECT rc;
        GetClientRect(HWindow, &rc);
        SendMessage(HWindow,
                    WM_SIZE,
                    (WPARAM)0,
                    MAKELPARAM(rc.right, rc.bottom));
    } else {
        if(hbmBuffer) {
            DeleteBitmap(hbmBuffer);
            hbmBuffer = 0;
            InvalidateRect(HWindow, NULL, TRUE);
        }
    }
}

