/*  Copyright (C) 1993   Marc Stern  (internet: stern@mble.philips.be)  */

#include "strings.h"
#include "tools.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>




/***
 *
 *  Function   :   getsetupparam
 *
 *  Topics     :   Read paramaters in setup file.
 *
 *  Decisions  :   Expected format is a list of lines of the form
 *
 *		                  <keyword>=<value>
 *
 *                 - All letters are treated (and returned) as uppercases.
 *                 - Blanks and tabs are skipped.
 *
 *                 - <value> can be enclosed between double quotes
 *		     to preserve lowercases and blanks.
 *
 * 		   - 2 consecutive double quotes allow to keep
 *                   a double quote in <value>.
 *
 *                 - A number sign (#) in first column forces the line
 *                   to be ignored.
 *
 *
 *  Parameters :    in    FILE *file        input file
 *                  in    char *keyword     keyword to match
 *                  out   char *value       value associated to keyword
 *
 *  Return code:     0     OK
 *                  -1     keyword not found
 *                  -2     file problem
 *
 *  Precond    :   Input file must be opened with read access.
 *
 */

static int near getsetupparam( FILE *file, char *keyword, char *value )

{ char buffer[256], *ptr;
  int length;

  if ( fseek(file, 0, SEEK_SET) ) return -2;

  while ( fgets(buffer, 255, file) )
	{
	  if ( *buffer == '#' ) continue;

	  strreduce( buffer );
	  strreduce( keyword );

          length = strlen( keyword );
	  if ( strncmp(buffer, keyword, length) ) continue;
	  ptr = buffer + length;
          if ( *ptr++ != '=' ) continue;

	  strcpy( value, ptr );
	  return 0;
	}

  return -1;

}



/***
 *
 *  Function   :   getsetup
 *
 *  Topics     :   Read paramaters in setup file.
 *
 *  Decisions  :   Expected format is a list of lines of the form
 *
 *		                  <keyword>=<value>
 *
 *                 - All letters are treated (and returned) as uppercases.
 *                 - Blanks and tabs are skipped.
 *
 *                 - <value> can be enclosed between double quotes
 *		     to preserve lowercases and blanks.
 *
 * 		   - 2 consecutive double quotes allow to keep
 *                   a double quote in <value>.
 *
 *                 - A number sign (#) in first column forces the line
 *                   to be ignored.
 *
 *                 - The input formats in parameter 'format' are C standard
 *                   ones ( %d, %s, %f,...).
 *
 *                 - Special format added: "%b" will translate
 *                   True/False, Yes/No, On/Off, 1/0 into integer 1/0
 *
 *                 - Special format added: "%S" will result into copying
 *                   all the input line (with spaces in it if any).
 *                   It will not be processed by scanf-like processing.
 *
 *  Parameters :    in    FILE *file        input file
 *                  in    char *format      "keyword1=format keyword2=format"
 *                  out   pointer list
 *
 *  Return code:    number of found parameters
 *                  -1     syntax error
 *                  -2     file error
 *
 *  Precond    :   Input file must be opened with read access.
 *
 */

int getsetup( FILE *file, const char *format, ... )

{ va_list list;
  int status, count = 0;
  char par_value[255], fmt[255], *keyword, *ptr;
  void *param;

  va_start(list, format);
  strcpy( fmt, format );
  for ( keyword = strtok(fmt, "="); keyword; keyword = strtok(NULL, "=") )
      {
        param = va_arg( list, void* );
        status = getsetupparam( file, keyword, par_value );
        if ( status == -2 ) return -2;
        if ( ! status ) count++;
        format = strtok( NULL, " \t\r\n" );
        /* handle boolean (%b) format */
        ptr = strstr( format, "%b" );
        if ( ptr++ )
           {
             char buffer[255] = "";
/*  When available use _stricmp instead of strcmp */
/*  in case of string has been between quotes     */
/*  and has not been translated into uppercases.  */
#if ! defined(_MSC_VER) && ! defined(__TURBOC__)
# define _stricmp strcmp
#endif
             *ptr = 's';

             if ( sscanf(par_value, format, buffer) != 1 ) return -1;
             if ( _stricmp(buffer, "TRUE") ||
                  _stricmp(buffer, "YES") ||
                  _stricmp(buffer, "ON") ||
                  _stricmp(buffer, "1")
                ) *(int*)param = 1;
             else *(int*)param = 0;

             continue;
           }

        /* Handle %S to get spaces inside strings */
        ptr = strstr( format, "%S" );
        if ( ptr )
           {
             strcpy( ((char*)param), par_value );
             continue;
           }

        if ( sscanf(par_value, format, param) != 1 ) return -1;
      }

  va_end( list );

  return count;
}

