
/* HLPLOT.C */
/* ======================================================================= */
/*    Plotting routine using floating horizon for hidden line removal.     */
/*    This is an implementation and modification of the algorithm given    */
/*    in "Procedural Elements for Computer Graphics", by D. F. Rogers      */
/*    Compiler: Turbo C, v 2.0                                             */
/* ======================================================================= */

#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <graphics.h>
#include <math.h>

/* for dimensioning horizons, assume VGA max  */
#define HSCREEN 640
#define VSCREEN 480
/* upper bound on x- and z-divisions */
#define MAXDIV  101

#define MAX(x,y)  (x)>(y) ? x : y
#define MIN(x,y)  (x)<(y) ? x : y

extern int sxmax,symax;         /* maximum x- and y-values on screen */
extern int left,top,right,pbot; /* plot window corners */

/* function data */
extern int numX,numZ;           /* x- and z-divisions used */
extern float X[MAXDIV], Z[MAXDIV];    /* x- and z-values */
extern float Y[MAXDIV][MAXDIV];       /* Y[j][i] = f(X[i],Z[j]) */

int Upper[HSCREEN], Lower[HSCREEN];    /* horizons */
int Xleft, Xright, Yleft, Yright;      /* for keeping track of edges */
double ax, bx, ay, by;          /* scaling constants */
double a00, a01, a11, a20, a21; /* entries in transformation matrix */
int xp, yp;                     /* coordinates of projected point */
int Xi, Yi;                     /* coordinates of intermediate point */
int pflag,cflag;                /* visibility flags */

double roundoff = 0.0000001;    /* for floating point calculations */

/* function prototypes */
int
Efill(int, int, int *, int *),  /* fill edge */
extend(int, int, int, int),     /* extend segment, keeping visibility */
horizon(int, int, int, int),    /* update horizon arrays */
plot(void),
sign(float),                    /* return -1,0,1 for sign */
setup_xform(float, float),      /* set xform constants */
transform(float, float, float), /* set (xp,yp) = xform(x,y,z) */
visibility(int, int);           /* check visibility */

int c;                          /* drawing color for Aline() function */

plot()
{
  int i, j, row;
  float Xprev, Yprev, x, y, z;
  int iXprev, iYprev, ix, iy;
  /* extra variables for demo program ... */
  int x1[2 * MAXDIV], y1[2 * MAXDIV], x2[2 * MAXDIV], y2[2 * MAXDIV];
  int u, umax;

  c = 15;                     /* color for Aline() function */
  setcolor(15);               /* color for line(), if used in place of Aline */
  setviewport(left + 1, top + 1, right - 1, pbot - 1, 1);
  clearviewport();
  for (j = numZ - 1; j >= 0; j--) {     /* start at nearest vertical plane */
    z = Z[j];
    u = 0;
    Xprev = X[0];               /* last point is (Xprev,Yprev) */
    Yprev = Y[j][0];
    transform(Xprev, Yprev, z); /* apply viewing transformation */
    iXprev = xp;
    iYprev = yp;                /* (iXprev,iYprev) is transformed point */
    Efill(iXprev, iYprev, &Xleft, &Yleft);      /* fill left side */
    pflag = visibility(iXprev, iYprev); /* save visibility of previous point */
    for (i = 1; i < numX; i++) {/* step through x-values */
      x = X[i];
      y = Y[j][i];
      transform(x, y, z);       /* transform current point.. */
      ix = xp;
      iy = yp;                  /* .. to (ix,iy) */
      cflag = visibility(ix, iy);       /* check visibility of current point
                                         * .. */
      if (cflag == pflag) {     /* .. and take approriate action */
        if (cflag == 1 || cflag == -1) {        /* cflag = pflag = 1 or -1   */
          x1[u] = iXprev, y1[u] = iYprev, x2[u] = ix, y2[u++] = iy;
          horizon(iXprev, iYprev, ix, iy);      /* fill horizon from P to C  */
        }
      } else {
        if (cflag == 0) {       /* cflag = 0 and pflag = 1 or -1 */
          extend(iXprev, iYprev, ix, iy);       /* I = last visible pt on PC  */
          x1[u] = iXprev, y1[u] = iYprev, x2[u] = Xi, y2[u++] = Yi;
          horizon(iXprev, iYprev, Xi, Yi);      /* fill horizon from P to I */
        } else if (cflag == 1)
          if (pflag == 0) {     /* cflag = 1 and pflag = 0  */
            extend(iXprev, iYprev, ix, iy);     /* I = isect(PC,Upper) */
            x1[u] = Xi, y1[u] = Yi, x2[u] = ix, y2[u++] = iy;
            horizon(Xi, Yi, ix, iy);    /* fill horizon from I to C */
          } else {              /* cflag = 1 and pflag = -1 */
            extend(iXprev, iYprev, ix, iy);     /* I = isect(PC,Lower) */
            x1[u] = iXprev, y1[u] = iYprev, x2[u] = Xi, y2[u++] = Yi;
            horizon(iXprev, iYprev, Xi, Yi);    /* fill horizon from P to I */
            extend(iXprev, iYprev, ix, iy);     /* I = isect(IC,Upper) */
            x1[u] = Xi, y1[u] = Yi, x2[u] = ix, y2[u++] = iy;
            horizon(Xi, Yi, ix, iy);    /* fill horizon from I to C */
          }
        else if (pflag == 0) {  /* cflag = -1 and pflag = 0 */
          extend(iXprev, iYprev, ix, iy);       /* I = isect(PC,Lower) */
          x1[u] = Xi, y1[u] = Yi, x2[u] = ix, y2[u++] = iy;
          horizon(Xi, Yi, ix, iy);      /* fill horizon from I to C */
        } else {                /* cflag = -1 and pflag = 1 */
          extend(iXprev, iYprev, ix, iy);       /* I = isect(PC,Upper) */
          x1[u] = iXprev, y1[u] = iYprev, x2[u] = Xi, y2[u++] = Yi;
          horizon(iXprev, iYprev, Xi, Yi);      /* fill horizon from P to I */
          extend(iXprev, iYprev, ix, iy);       /* I = isect(IC,Lower) */
          x1[u] = Xi, y1[u] = Yi, x2[u] = ix, y2[u++] = iy;
          horizon(Xi, Yi, ix, iy);      /* fill horizon I to C */
        }
      }
      pflag = cflag;            /* make current point = previous point */
      iXprev = ix;
      iYprev = iy;
    }
    Efill(ix, iy, &Xright, &Yright);    /* fill right edge */
    umax = u;
    draw_slice(umax, x1, y1, x2, y2);
  }
  setviewport(0, 0, sxmax, symax, 0);
  return 1;
}

/* ---------- draw slice of graph corresponding to Z[j] ------------------- */
draw_slice(int umax, int *x1, int *y1, int *x2, int *y2)
{
  int u;
    for (u = 0; u < umax; u++)
      Aline(x1[u], y1[u], x2[u], y2[u], c);
/* Aline() is the fast EGAVGA line drawing routine given by Michael Abrash,
   Programmer's Journal, v. 7.3, May/June, 1989. It may be replaced with
   the standard line() function, but with some loss in speed */
}

/* =========== alphabetical listing of 'plot()' subfunctions =============== */

/* ---------------------------- fill edge ---------------------------------- */
Efill(int x, int y, int *Xedge, int *Yedge)
{
  if (*Xedge != -1)
    horizon(*Xedge, *Yedge, x, y);
  *Xedge = x;
  *Yedge = y;
}

/* ----- extend given segment to maintain visibility of first endpoint ----- */
/*          (replacement for intersect() function in Rogers' book)           */
extend(int X1, int Y1, int X2, int Y2)
{
  double Yi1, slope;
  int v, v1, v2;

  v1 = visibility(X1, Y1);
  v2 = visibility(X2, Y2);

  if (X1 == X2) {
    Xi = X1;
    if (v1 == 1)
      Yi = Upper[Xi];
    else if (v1 == -1)
      Yi = Lower[Xi];
    else if (v1 == 0)
      if (v2 == 1)
        Yi = Upper[Xi];
      else
        Yi = Lower[Xi];
    return;
  }
  slope = (double) (Y2 - Y1) / (X2 - X1);
  Xi = X1;
  Yi = Y1;
  Yi1 = (double) Y1;
  if (v1 == 1)
    while (Yi1 >= Upper[Xi] && Upper[Xi] > 0 && Xi < X2) {
      Xi++;
      Yi1 += slope;
    }
  else if (v1 == -1)
    while (Yi1 <= Lower[Xi] && Lower[Xi] < VSCREEN && Xi < X2) {
      Xi++;
      Yi1 += slope;
    }
  else if (v1 == 0)
    if (v2 == 1)
      while (Yi1 < Upper[Xi] - roundoff) {
        Xi++;
        Yi1 += slope;
      }
    else
      while (Yi1 > Lower[Xi] + roundoff) {
        Xi++;
        Yi1 += slope;
      }
  Yi = (int) (Yi1 +.5);
}

/* ------------------------ update horizon arrays -------------------------- */
horizon(int X1, int Y1, int X2, int Y2)
{
  float slope;
  int x, y;
  int y0;

  if (X1 < 0)
    X1 = 0;                     /* the approximate scaling routine */
  if (X2 >= 639)
    X2 = 639;                   /* may give unacceptable values here */

  if (X2 < X1)
    return;

  if (X2 == X1) {
    Upper[X2] = MAX(Upper[X2], Y2);
    Lower[X2] = MIN(Lower[X2], Y2);
    return;
  }
  for (x = X1; x <= X2; x++) {
    y0 = (Y2 - Y1) * (x - X1);
    if (y0 > (Upper[x] - Y1) * (X2 - X1))
      Upper[x] = Y1 + y0 / (X2 - X1);
    if (y0 < (Lower[x] - Y1) * (X2 - X1))
      Lower[x] = Y1 + y0 / (X2 - X1);
  }
  return;
}

/* ------------------------------------------------------------------------- */
sign(float x)
{
  if (x > roundoff)
    return 1;
  if (x < -roundoff)
    return -1;
  return 0;
}

/* ------------- calculate transformation constants --------------------- */
setup_xform(float theta, float alpha)
{
  float cost, sint, cosa, sina;

  cost = cos(theta), sint = sin(theta);
  cosa = cos(alpha), sina = sin(alpha);
  a00 = cost;
  a01 = sint * sina;
  a11 = cosa;
  a20 = sint;
  a21 = -cost * sina;
}

/* ----------- transform point (x,y,z) to image point (xp,yp) -------------- */

transform(float x, float y, float z)
{
  float t;
  float x1, y1, x2, y2;

  x1 = a00 * x + a20 * z;
  y1 = a01 * x + a11 * y + a21 * z;
  x2 = ax * x1 + bx;
  y2 = ay * y1 + by;
  xp = x2 +.5;
  yp = y2 +.5;
}

/* ------------------- test visibility of current point -------------------- */
visibility(int x, int y)
{
  int flag;

  if (y < Upper[x] && y > Lower[x])
    flag = 0;
  if (y >= Upper[x])
    flag = 1;
  if (y <= Lower[x])
    flag = -1;
  return flag;
}
