//
// Mini Java Editor
//
// By Lim Thye Chean
//

import Table;
import Hash;
import java.io.*;
import java.awt.*;
import java.util.*;

class MJE extends Frame {
  static final String about = new String("Mini Java Editor v0.91 - " +
    "by Lim Thye Chean ");

  String path = null;
  Table project = new Table();
  Hash windows = new Hash();

  MJEFiles list = new MJEFiles(this, 10);
  String directory = null;
  String options = "";
  Font font;
  String fonts[];
  String styles[] = {"Plain", "Bold", "Italic", "Bold-Italic"};

  Label status = new Label("Ready.");
  Menu editMenu = new Menu("Edit");
  Menu projectMenu = new Menu("Project");
  MenuItem saveItem = new MenuItem("Save Project");
  MenuItem saveAsItem = new MenuItem("Save Project As...");
  MenuItem closeItem = new MenuItem("Close Project");

  CheckboxMenuItem fontItems[];
  CheckboxMenuItem styleItems[];
  CheckboxMenuItem sizeItems[];

  OptionsDialog optionsDialog = null;
  InfoDialog aboutDialog = null;

  public static void main(String args[]) {;
    MJE ed = new MJE("Mini Java Editor");

    if (args.length > 0) {
      ed.setTitle(new File(args[0]).getName());

      ed.path = args[0];
      ed.open(ed.path);
      ed.saveItem.enable();
    }
  }

  MJE(String title) {

  // Initialize

    font = new Font("Courier", Font.PLAIN, 12);
    fonts = getToolkit().getFontList();
    MenuBar menuBar = new MenuBar();

  // Create file menu

    Menu fileMenu = new Menu("File");
    fileMenu.add("New Editor");
    fileMenu.add("-");
    fileMenu.add("New File");
    fileMenu.add("Open File...");
    fileMenu.add("-");
    fileMenu.add("New Project...");
    fileMenu.add("Open Project...");
    fileMenu.add(saveItem);
    fileMenu.add(saveAsItem);
    fileMenu.add(closeItem);
    fileMenu.add("-");
    fileMenu.add("Exit");
    menuBar.add(fileMenu);

  // Create edit menu

    editMenu.add("Add File...");
    editMenu.add("Add All Files...");
    editMenu.add("Add All Java Files...");
    editMenu.add("Remove File");
    editMenu.add("Remove All Files");
    editMenu.add("-");
    editMenu.add("Move File Up");
    editMenu.add("Move File Down");
    editMenu.add("Sort Files");
    menuBar.add(editMenu);

  // Add project menu
 
    projectMenu.add("Compile");
    projectMenu.add("-");
    projectMenu.add("Build");
    projectMenu.add("Build All");
    projectMenu.add("-");
    projectMenu.add("Options...");
    menuBar.add(projectMenu);

  // Create help menu

    Menu helpMenu = new Menu("Help");
    helpMenu.add("About Mini Java Editor...");
    menuBar.add(helpMenu);
    menuBar.setHelpMenu(helpMenu);

  // Show editor

    setMenuBar(menuBar);
    editMenu.disable();
    projectMenu.disable();
    saveItem.disable();
    saveAsItem.disable();
    setTitle(title);

    add("Center", list);
    add("South", status);
    resize(250, 250);
    show();
  }

// Set font

  public void changeFont(Font fnt) {
    font = fnt;
    Table wins = windows.keyNames();

    for (int i = 0; i < wins.size(); i++)
      ((MJEWindow) windows.get(wins.getString(i))).changeFont(fnt);
  }

// Open project

  public boolean open(String path) {
    try {
      project.clear();
      FileInputStream fs = new FileInputStream(path);
      DataInputStream ds = new DataInputStream(fs);

    // Read font

      font = new Font(ds.readLine(), new Integer(ds.readLine()).intValue(),
	new Integer(ds.readLine()).intValue());

    // Read project

      options = ds.readLine();
      String str = ds.readLine();
 
      while (str != null) {
	String[] item = new String[2];
	item[0] = str;
	item[1] = ds.readLine();

	project.add(item);
	str = ds.readLine();
      }
    } catch (Exception err) {
      status.setText("Cannot open file.");
      return false;
    }

    return true;
  }

// Save project

  public void save(String path) {
    try {
      status.setText("Saving project...");

      FileOutputStream fs = new FileOutputStream(path);
      PrintStream ps = new PrintStream(fs);
      String ls = System.getProperty("line.separator");

      ps.println(font.getName());
      ps.println(font.getStyle());
      ps.println(font.getSize());
      ps.print(options);
      ps.print(ls);

      for (int i = 0; i < project.size(); i++) {
	String[] item = (String[]) project.get(i);
	ps.println(item[0]);
	ps.print(item[1]);
	ps.print(ls);
      }

      status.setText("Ready.");
    } catch (Exception err) {
      status.setText("Cannot save project.");
    }
  }

// Compile file

  public boolean compile(String file) {
    try {
      MJECompiler compiler = new MJECompiler();
      String pth;

      if (new File(file).exists())
        pth = file;
      else if (new File(directory + file).exists())
        pth = directory + file;
      else {
        status.setText("Cannot locate " + file + ".");
        return true;
      }

      status.setText("Compiling " + pth + "...");

      if (compiler.compile(pth, options)) {
        status.setText("Ready.");
	return true;
      }
      else {
        MJEWindow win;

        if (windows.containsKey(pth)) {
	  win = (MJEWindow) windows.get(pth);
  	  win.show();
        }
        else {
      	  win = new MJEWindow(file, font);
          win.open(pth);
	  win.editor = this;
          windows.set(pth, win);
        }

        Table errs = compiler.errors;
        win.output.delItems(0, win.output.countItems() - 1);

        for (int i = 0; i < errs.size(); i++)
          win.output.addItem((String) errs.get(i));

        status.setText("Error count: " + errs.size());
      }
    }
    catch (Exception err) {
      status.setText("Cannot locate " + file + ".");
    }

    return false;
  }

// Exit editor

  public void exit() {
    System.exit(0);
  }

// Handle system event

  public boolean handleEvent(Event evt) {
    if (evt.id == Event.WINDOW_DESTROY && evt.target == this)
      exit();
 
    return super.handleEvent(evt);
  }

// Handle component events

  public boolean action(Event evt, Object obj) {
    String label = (String) obj;
  
  // Handle file menu

    if (label.equals("New Editor")) {
      new MJE("Mini Java Editor");
      return true;
    }

    if (label.equals("New File")) {
      new MJEWindow("Untitled", font);
      return true;
    }

    if (label.equals("Open File...")) {
      FileDialog dialog = new FileDialog(this, "Open File...", 
	FileDialog.LOAD);

      dialog.show();
      String file = dialog.getFile();

      if (file != null) {
	String path = dialog.getDirectory() + file;
	MJEWindow win = new MJEWindow(file, font);
        win.editor = this;
        win.open(path);
      }

      return true;
    }

    if (label.equals("New Project...")) {
      FileDialog dialog = new FileDialog(this, "New Project...", 
	FileDialog.LOAD);

      dialog.show();
      String file = dialog.getFile();

      if (file != null) {
	if (new File(file).exists()) {
	  status.setText("Project already exists.");
	  return true;
	}

	if (file.endsWith(".project")) {
	  directory = dialog.getDirectory();
	  path = directory + file;

	  setTitle(file + " - Mini Java Editor");
	  saveItem.enable();
	  saveAsItem.enable();
	  closeItem.enable();
	  editMenu.enable();
	  projectMenu.enable();
	  list.clear();
          project.clear();
	  status.setText("Ready.");
	}
	else
	  status.setText("Please use the file extension '.project'");
      }
      
      return true;
    }

    if (label.equals("Open Project...")) {
      FileDialog dialog = new FileDialog(this, "Open Project...", 
	FileDialog.LOAD);

      dialog.show();
      String file = dialog.getFile();

      if (file != null) {
	if (file.endsWith(".project")) {
	  directory = dialog.getDirectory();
	  path = directory + file;

	  if (open(path)) {
	    setTitle(file + " - Mini Java Editor");
	    saveItem.enable();
	    saveAsItem.enable();
	    closeItem.enable();
	    editMenu.enable();
	    projectMenu.enable();
	    list.refresh();
	    status.setText("Ready.");
	  }
	}
      }

      return true;
    }

    if (label.equals("Save Project")) {
      if (path != null)
        save(path);
    }

    if (label.equals("Save Project As...")) {
      FileDialog dialog = new FileDialog(this, "Save Project As...", 
	FileDialog.SAVE);
      
      dialog.setFile("Untitled.project");
      dialog.show();
      String file = dialog.getFile();

      if (file != null) {
	directory = dialog.getDirectory();
	path = directory + file;
	setTitle(file + " - Mini Java Editor");
	save(path);
      }

      return true;
    }

    if (label.equals("Close Project")) {
      directory = null;
      list.clear();
      saveItem.disable();
      saveAsItem.disable();
      editMenu.disable();
      projectMenu.disable();
      project.clear();

      return true;
    }

    if (label.equals("Exit"))
      exit();

  // Handle edit menu

    if (label.equals("Add File...")) {
      FileDialog dialog = new FileDialog(this, "Add File...", 
	FileDialog.LOAD);

      dialog.show();
      String file = dialog.getFile();

      if (file != null) {
	String dir = dialog.getDirectory();
	String str;
	String[] item = new String[2];

	if (dir.equals(directory)) 
	  item[0] = file;
	else
	  item[0] = dir + file;

	item[1] = "0";
	project.add(item);
	list.refresh();
      }

      return true;
    }

    if (label.equals("Add All Files...")) {
      String str;
      String[] files = new File(directory).list();

      for (int i = 0; i < files.length; i++) {
	String[] item = new String[2];

	item[0] = files[i];
	item[1] = "0";
	project.add(item);
      }

      list.refresh();
      return true;
    }

   if (label.equals("Add All Java Files...")) {
      String str;
      String[] files = new File(directory).list();

      for (int i = 0; i < files.length; i++) {
	if (files[i].endsWith(".java")) {
	  String[] item = new String[2];

	  item[0] = files[i];
	  item[1] = "0";
	  project.add(item);
	}
      }

      list.refresh();
      return true;
    }

    if (label.equals("Remove File")) {
      for (int i = 0; i < project.size(); i++) {
	String str = ((String[]) project.get(i))[0];

	if (str.equals(list.getSelectedItem())) {
	  project.removeAt(i);

	  if (windows.containsKey(str)) {
	    MJEWindow win = (MJEWindow) windows.get(str);
	    win.dispose();
	  }
	}
      }

      list.refresh();
      return true;
    }

    if (label.equals("Remove All Files")) {
      for (int i = 0; i < project.size(); i++) {
        String str = ((String[]) project.get(i))[0];

	if (windows.containsKey(str)) {
	  MJEWindow win = (MJEWindow) windows.get(path);
	  win.dispose();
	}
      }

      project.clear();
      list.clear();
    }
     
    if (label.equals("Move File Up")) {
      int pos = list.getSelectedIndex();

      if (pos > 0) {
	String str = list.getItem(pos);
	list.delItem(pos);
	list.addItem(str, pos - 1);
	list.select(pos - 1);

	Object item = project.get(pos);
	project.removeAt(pos);
	project.addAt(item, pos - 1);
      }
    }

    if (label.equals("Move File Down")) {
      int pos = list.getSelectedIndex();

      if (pos > -1 || pos < (list.countItems() - 1)) {
	String str = list.getItem(pos);
	list.delItem(pos);
	list.addItem(str, pos + 1);
	list.select(pos + 1);

	Object item = project.get(pos);
	project.removeAt(pos);
	project.addAt(item, pos + 1);
      }
    }

    if (label.equals("Sort Files")) {
      for (int i = project.size() - 1; i > 0; i--) {
	for (int j = 0; j < i; j++) {
	  String[] file1 = (String[]) project.get(j);
	  String[] file2 = (String[]) project.get(j + 1);

	  if (file1[0].compareTo(file2[0]) > 0) {
	    project.removeAt(j);
	    project.addAt(file1, j + 1);
  	  }
        }
      }

      list.refresh();
    }

    if (label.equals("Compile")) {
      String str = (String) list.getSelectedItem();

      if (str.endsWith(".java"))
	compile(str);
    }
      
    if (label.equals("Build")) {
      int build = 0;

      for (int i = 0; i < project.size(); i++) {
	String[] item = (String[]) project.get(i);

	if (item[0].endsWith(".java")) {
	  File file = new File(item[0]);
	  long time = file.lastModified();

	  if (time != new Long(item[1]).longValue()) {
	    if (compile(item[0])) {
	      item[1] = new Long(time).toString();
	      build++;
	    }
	  }
	}
      }

      if (build > 0)
        save(path);

      return true;
    }

    if (label.equals("Build All")) {
      String str = (String) list.getSelectedItem();

      for (int i = 0; i < project.size(); i++) {
	String[] item = (String[]) project.get(i);

	if (item[0].endsWith(".java"))
	  if (compile(item[0]))
	    item[1] = new Long(new File(item[0]).lastModified()).toString();
      }

      save(path);
      return true;
    }

    if (label.equals("Options...")) {
      if (optionsDialog == null) {
      	optionsDialog = new OptionsDialog(this);
	optionsDialog.pack();
      }

      optionsDialog.show();
      return true;
    }
 
  // Handle help menu

    if (label.equals("About Mini Java Editor...")) {
      if (aboutDialog == null) {
        aboutDialog = new InfoDialog(this, label, MJE.about); 
        aboutDialog.pack();
      }

      aboutDialog.show();
      return true;
    }
    
    return false; 
  }
}

// 
// File list area
//

class MJEFiles extends List {
  MJE editor;

  public MJEFiles(MJE ed, int rows) {
    super(rows, false);
    editor = ed;
  }

  public void clear() {
    delItems(0, countItems() - 1);
  }

  public void refresh() {
    clear();

    for (int i = 0; i < editor.project.size(); i++)
      addItem(((String[]) editor.project.get(i))[0]);
  }
 
  public boolean action(Event evt, Object obj) {
    Hashtable wins = editor.windows;
    String dir = editor.directory;
    String str = (String) obj;
    String path;

    if (new File(str).exists())
      path = str;
    else if (new File(editor.directory + str).exists())
      path = editor.directory + str;
    else {
      editor.status.setText("Cannot locate " + str + ".");
      return true;
    }

    if (wins.containsKey(path))
      ((MJEWindow) wins.get(path)).show();
    else {
      MJEWindow win = new MJEWindow(str, editor.font);
      win.open(path);
      win.editor = editor;
      wins.put(path, win);
    }

    return true;
  }
}
