//
// Mini Java Editor window
//

import java.io.*;
import java.awt.*;

class MJEWindow extends Frame {
  TextArea area;
  MJE editor;
  MJEOutput output;
  String clipboard;
  String savePath = null;
  GotoDialog gotoDialog = null;
  FindDialog findDialog = null;

  MenuItem saveItem = new MenuItem("Save");
  MenuItem compileItem = new MenuItem("Compile");

  MJEWindow(String title, Font fnt) {

  // Initialize

    area = new TextArea(34, 80);
    output = new MJEOutput(this, 6);

    MenuBar menuBar = new MenuBar();

  // Create file menu

    Menu fileMenu = new Menu("File");
    fileMenu.add(new MenuItem("New"));
    fileMenu.add(new MenuItem("Open..."));
    fileMenu.add(saveItem);
    fileMenu.add(new MenuItem("Save As..."));
    fileMenu.add(new MenuItem("-"));
    fileMenu.add(compileItem);
    menuBar.add(fileMenu);

  // Create edit menu

    Menu editMenu = new Menu("Edit");
    editMenu.add(new MenuItem("Cut"));
    editMenu.add(new MenuItem("Copy"));
    editMenu.add(new MenuItem("Paste"));
    editMenu.add(new MenuItem("Select All"));
    editMenu.add(new MenuItem("-"));
    editMenu.add(new MenuItem("Go To Line..."));
    editMenu.add(new MenuItem("Find..."));
    menuBar.add(editMenu);
 
  // Show window

    setMenuBar(menuBar);
    saveItem.disable();
    compileItem.disable();

    changeFont(fnt);
    setTitle(title);
    output.clear();

    add("Center", area);
    add("South", output);
    pack();
    show();
  }

// Open file

  public boolean open(String path) {
    if (path != null) {
      setTitle(getFile(path));
      savePath = path;
      area.setText(readFile(path));
      saveItem.enable();
      output.clear();

      if (path.endsWith(".java")) 
	compileItem.enable();
      else
	compileItem.disable();

      return true;
    }

    return false;
  }

  public boolean save() {
    writeFile(savePath, area.getText());
    return true;
  }

// Read file

  public String readFile(String fl) {
    String text = new String();

    try {
      FileInputStream fs = new FileInputStream(fl);
      DataInputStream ds = new DataInputStream(fs);
      String str = ds.readLine();

      while (str != null) {
  	text = text + str + "\n";
	str = ds.readLine();
      }
    } catch (Exception err) {
      output.set("Cannot read file.");
    }

    return text;
  }

// Write file

  public void writeFile(String fl, String txt) {
    try {
      StringBufferInputStream ss = new StringBufferInputStream(txt);
      DataInputStream ds = new DataInputStream(ss);
      FileOutputStream fs = new FileOutputStream(fl);
      PrintStream ps = new PrintStream(fs);
      String ls = System.getProperty("line.separator");
      String str = ds.readLine();
   
      while (str != null) {
	ps.print(str);
	ps.print(ls);
  	str = ds.readLine();
      }
    } catch (Exception err) {
      output.set("Cannot write file.");
    }
  }

// Handle system event

  public boolean handleEvent(Event evt) {
    if (evt.id == Event.WINDOW_DESTROY && evt.target == this)
      hide();

    return super.handleEvent(evt);
  }

// Handle component events

  public boolean action(Event evt, Object obj) {
    String label = (String) obj;
    String file = null;
  
  // Handle file menu

    if (label.equals("New")) {
      area.setText("");
      setTitle("Untitled");
      saveItem.disable();
      output.clear();

      return true;
    }

    if (label.equals("Open...")) {
      FileDialog dialog = new FileDialog(this, "Open...", FileDialog.LOAD);

      dialog.show();
      file = dialog.getFile();

      if (file != null)
	open(dialog.getDirectory() + file);

      return true;
    }

    if (label.equals("Save")) {
      writeFile(savePath, area.getText());
    }

    if (label.equals("Save As...")) {
      FileDialog dialog = new FileDialog(this, "Save As...", FileDialog.SAVE);
      
      dialog.setFile(savePath);
      dialog.show();
      file = dialog.getFile();

      if (file != null) {
	setTitle(file);
	savePath = dialog.getDirectory() + file;
        saveItem.enable();
	writeFile(file, area.getText());
      }

      return true;
    }

    if (label.equals("Compile")) {
      MJECompiler compiler = new MJECompiler();

      output.delItems(0, output.countItems() - 1);
      output.addItem("Saving " + savePath + "...");   
      writeFile(savePath, area.getText());
      output.addItem("Compiling " + savePath + "...");     

      if (compiler.compile(savePath, editor.options))
	output.addItem("No error.");
      else {
	Table errs = compiler.errors;

	for (int i = 0; i < errs.size(); i++)
      	  output.addItem((String) errs.elementAt(i));

	output.addItem("Error count: " + errs.size());
      }
    }

  // Handle edit menu

    if (label.equals("Cut")) {
      clipboard = area.getSelectedText();
      
	area.replaceText("", area.getSelectionStart(), 
        area.getSelectionEnd());

	return true;
    }

    if (label.equals("Copy")) {
      clipboard = area.getSelectedText();
	return true;
    }

    if (label.equals("Paste")) {
      int start = area.getSelectionStart();
      int end = area.getSelectionEnd();

      if (start == end) 
        area.insertText(clipboard, start);
      else
        area.replaceText(clipboard, start, end);

      return true;
    }

    if (label.equals("Select All"))
      area.selectAll();

    if (label.equals("Go To Line...")) {
      if (gotoDialog == null) {
      	gotoDialog = new GotoDialog(this);
      	gotoDialog.pack();
      }

      gotoDialog.show();	
      return true;
    }

    if (label.equals("Find...")) {
      if (findDialog == null) {
      	findDialog = new FindDialog(this);
      	findDialog.pack();
      }

      findDialog.show();	
      return true;
    }
    
    return false; 
  }

// Change font

  public void changeFont(Font fnt) {
    if (fnt != null)
      area.setFont(fnt);
  }

// Get text area

  public TextArea getTextArea() {
    return area;
  }

// Get file name from path

  public String getFile(String path) {
    char sp = System.getProperty("file.separator").charAt(0);

    for (int i = path.length() - 1; i > 0; i--)
      if (path.charAt(i) == sp)
	return path.substring(i + 1, path.length());

    return path;  
  }

// Go to line

  public void gotoLine(int line) {
    gotoLine(line, 0);
  }

  public void gotoLine(int line, int col) {
    int i = 1, pos = 0;
    String str = area.getText();

    while (i < line) {
      if (str.charAt(pos) == '\n')
	i++;

      pos++;
    }

    area.requestFocus();
    area.select(pos + col, pos + col);
  }      
}

//
// Output area
//

class MJEOutput extends List {
  MJEWindow win;

  public MJEOutput(MJEWindow w, int rows) {
    super(rows, false);
    win = w;
  }

  public void clear() {
    set("Ready.");
  }

  public void set(String str) {
    delItems(0, countItems() - 1);
    addItem(str);
  }

  public boolean action(Event evt, Object obj) {
    String str = (String) obj;

    if (str.charAt(0) == '*') {
      str = getItem(getSelectedIndex() + 1);
    }

    if (str.charAt(0) == ' ') { 
      for (int i = 5; i < 9; i++)
        if (str.charAt(i) == ' ') {
	  int line = new Integer(str.substring(4, i)).intValue();
	  int col = 0;

	  for (int j = i + 2; j < i + 5; j++)
	    if (str.charAt(j) == ')') 
	      col = new Integer(str.substring(i + 2, j)).intValue();

	  win.gotoLine(line, col);
	  break;
        }
    }

    return true;
  }
}
