/*
 * object.c
 * Handle create and subsequent garbage collection of objects.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include <stdio.h>
#include <assert.h>
#include "gtypes.h"
#include "itypes.h"
#include "object.h"
#include "classMethod.h"
#include "baseClasses.h"
#include "gc.h"
#include "errors.h"

/*
 * Primitive object allocation.
 */
inline
object*
newObject(int sz)
{
	object* obj;

	obj = (object*)calloc(sz, sizeof(char));
	if (obj == 0) {
		throwException(OutOfMemoryError);
	}
	return (obj);
}

/*
 * Create a new object based on the given class type.
 */
object*
alloc_object(classes* c, bool perm)
{
	object* obj;

	obj = newObject(sizeof(object) + c->fsize * sizeof(u4));

	obj->size = 0;
	obj->mtable = c->mtable;

	add_object(obj, perm);

	return (obj);
}

/*
 * Create a new class object.
 */
classes*
alloc_class(void)
{
	classes* obj;

	obj = (classes*)newObject(sizeof(classes));

	if (ClassClass != 0) {
		obj->head.mtable = ClassClass->mtable;
		obj->head.size = ClassClass->fsize;
	}

	add_object(&obj->head, true);

	return (obj);
}

/*
 * Create a new array of data items (not objects).
 */
object*
alloc_array(int sz, int type)
{
	object* obj;
	int elemsz;
	classes* class;

	assert(type < MAXTYPES);
	assert(TYPE_SIZE(type) != 0);

	obj = newObject(sizeof(object) + sz*TYPE_SIZE(type));

	class = lookupArray(TYPE_ARRAYSIG(type));
	obj->size = sz;
	obj->mtable = class->mtable;

	add_object(obj, false);

	return (obj);
}

/*
 * Create a new array of objects.
 */
object*
alloc_objectarray(int sz, char* csig)
{
	char sig[CLASSMAXSIG];
	object* obj;
	classes* class;

	obj = (object*)newObject(sizeof(object) + sz*sizeof(u4));

	/* Build signature for array type */
	strcpy(sig, "[");
	strcat(sig, csig);

	class = lookupArray(addString(sig));
	obj->size = sz;
	obj->mtable = class->mtable;

	add_object(obj, false);

	return (obj);
}

/*
 * Build a multi-dimensional array.
 */
object*
alloc_multiarray(int* dims, char* cname)
{
	object* obj;
	object** array;
	int i;

	obj = alloc_objectarray(dims[0], cname+1);
	if (dims[1] > 0) {
		array = (object**)obj->data;
		for (i = 0; i < dims[0]; i++) {
			array[i] = alloc_multiarray(dims+1, cname+1);
		}
	}

	add_object(obj, false);

	return (obj);
}
