/*
 * exception.c
 * Handle exceptions.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include "config.h"
#include <stdio.h>
#include <assert.h>
#include <signal.h>
#if defined(HAVE_ASM_SIGNAL_H)
#include <asm/signal.h>
#endif
#if defined(HAVE_UCONTEXT_H)
#include <ucontext.h>
#endif
#include "object.h"
#include "classMethod.h"
#include "exception.h"
#include "baseClasses.h"
#include "lookup.h"
#include "thread.h"
#include "md.h"

object*	ClassFormatError;
object*	ClassFormatError;
object*	LinkageError;
object*	NoClassDefFoundError;
object*	NoSuchFieldError;
object*	NoSuchMethodError;
object*	OutOfMemoryError;
object*	UnsatisfiedLinkError;
object*	VirtualMachineError;
object*	ClassCircularityError;
object*	NegativeArraySizeException;
object*	ClassCastException;
object*	IllegalMonitorStateException;
object*	NullPointerException;
object*	ArithmeticException;
object*	ArrayIndexOutOfBoundsException;

static void dispatchException(object*, exceptionFrame*);

static void nullException();
static void arithmeticException();

#define	SIG_T	void(*)()

/*
 * Setup the internal exceptions.
 */
void
initExceptions(void)
{
#define	EXCEPTION(s) alloc_object(lookupClass(addString(#s)), true)

	ClassFormatError = EXCEPTION(java/lang/ClassFormatError);
	LinkageError = EXCEPTION(java/lang/LinkageError);
	NoClassDefFoundError = EXCEPTION(java/lang/NoClassDefFoundError);
	NoSuchFieldError = EXCEPTION(java/lang/NoSuchFieldError);
	NoSuchMethodError = EXCEPTION(java/lang/NoSuchMethodError);
	OutOfMemoryError = EXCEPTION(java/lang/OutOfMemoryError);
	UnsatisfiedLinkError = EXCEPTION(java/lang/UnsatisfiedLinkError);
	VirtualMachineError = EXCEPTION(java/lang/VirtualMachineError);
	ClassCircularityError = EXCEPTION(java/lang/ClassCircularityError);
	NegativeArraySizeException = EXCEPTION(java/lang/NegativeArraySizeException);
	ClassCastException = EXCEPTION(java/lang/ClassCastException);
	IllegalMonitorStateException = EXCEPTION(java/lang/IllegalMonitorStateException);
	NullPointerException = EXCEPTION(java/lang/NullPointerException);
	ArithmeticException = EXCEPTION(java/lang/ArithmeticException);
	ArrayIndexOutOfBoundsException = EXCEPTION(java/lang/ArrayIndexOutOfBoundsException);

	initClasses();

#if !defined(DEBUG)
	/* Catch signal we need to convert to exceptions */
#if defined(SIGSEGV)
	signal(SIGSEGV, nullException);
#endif
#if defined(SIGBUS)
	signal(SIGBUS, nullException);
#endif
#if defined(SIGFPE)
	signal(SIGFPE, arithmeticException);
#endif
#endif
}

/*
 * Throw an exception.
 */
void
throwException(object* eobj, void* fptr)
{
	if (eobj == 0) {
		fprintf(stderr, "Exception thrown on null object ... aborting\n");
		exit(1);
	}
	dispatchException(eobj, FIRSTFRAME(fptr));
}

/*
 * Search for a matching exception.
 */
static
void
dispatchException(object* eobj, exceptionFrame* firstFrame)
{
	exceptionInfo einfo;
	classes* class;
	exceptionFrame* frame;

	class = eobj->mtable->class;

	for (frame = firstFrame; FRAMEOKAY(frame); frame = NEXTFRAME(frame)) {
		findExceptionInMethod((nativecode*)frame->retpc, class, &einfo);
		if (einfo.handler == 0) {
			continue;
		}
		/* Goto the exception handler */
		CALL_KAFFE_EXCEPTION(frame, einfo, eobj);
	}

	fprintf(stderr, "Failed to catch exception ... aborting:\n\t%s\n", class->name);
	fflush(stderr);
	exit(1);
}

/*
 * Null exception - catches bad memory accesses.
 */
static
void
nullException(EXCEPTIONPROTO)
{
	exceptionFrame firstFrame;

#if !defined(DEBUG)
	/* Reset signal handler - necessary for SysV, does no harm for BSD */
	signal(sig, (SIG_T)nullException);
#endif
	EXCEPTIONFRAME(firstFrame, ctx);
	dispatchException(NullPointerException, &firstFrame);
}

/*
 * Division by zero.
 */
static
void
arithmeticException(EXCEPTIONPROTO)
{
	exceptionFrame firstFrame;

#if !defined(DEBUG)
        /* Reset signal handler - necessary for SysV, does no harm for BSD */
        signal(sig, (SIG_T)arithmeticException);
#endif
	EXCEPTIONFRAME(firstFrame, ctx);
	dispatchException(ArithmeticException, &firstFrame);
}
