/*
 * soft2.c
 * Soft instruction implementations.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#define ADBG(s)
#define MDBG(s)
#define SDBG(s)

#include <stdio.h>
#include <assert.h>
#include <stdarg.h>
#include "gtypes.h"
#include "itypes.h"
#include "object.h"
#include "needs.h"
#include "classMethod.h"
#include "lookup.h"
#include "errors.h"
#include "locks.h"

#define	MAXDIMS	8

#ifdef NEED_soft_unimplemented
void
soft_unimplemented()
{
	fprintf(stderr, "Unimplemented soft instruction.\n");
	fflush(stderr);
	abort();
}
#endif

#ifdef NEED_soft_multianewarray
void
soft_multianewarray(int dims, classes* class, int sizen, ...)
{
	/* dims, class, sizen, ... size1 */
	va_list ap;
	int arraydims[MAXDIMS];
	int i;
	int* arg;
	object* obj;

	assert(dims < MAXDIMS);

	/* Extract the dimensions into an array */
	va_start(ap, sizen);
	for (i = 0; i < dims; i++) {
		arg = &va_arg(ap, int);
		if (*arg < 0) {
			throwException(NegativeArraySizeException);
		}
		arraydims[dims-(i+1)] = *arg;
	}
	arraydims[i] = 0;
	va_end(ap);

	/* Mmm, okay now build the array using the wonders of recursion */
	obj = alloc_multiarray(arraydims, class->name);

	/* Return the base object */
	*arg = (int)obj;
}
#endif

#ifdef NEED_soft_new
void
soft_new(classes* c)
{
	object* obj;

	obj = alloc_object((classes*)c, false);

ADBG(	printf("New object of type %s (%d,%x)\n", c->name, c->fsize, obj);
		fflush(stdout);						)

	*(&c) = (classes*)obj;
}
#endif

#ifdef NEED_soft_newarray
void
soft_newarray(int type, int size)
{
	object* obj;

	if (size < 0) {
		throwException(NegativeArraySizeException);
	}

	obj = alloc_array(size, type);

ADBG(	printf("New object of %d type (%d,%x)\n", type, size, obj);
	fflush(stdout);							)

	*(&size) = (int)obj;
}
#endif

#ifdef NEED_soft_anewarray
void
soft_anewarray(classes* c, int size)
{
	object* obj;

	if (size < 0) {
		throwException(NegativeArraySizeException);
	}

	obj = alloc_objectarray(size, c->sig);

ADBG(	printf("New object array of type %s (%d,%x)\n", c->name, size, obj);
		fflush(stdout);						)

	*(&size) = (int)obj;
}
#endif

#ifdef NEED_soft_athrow
void
soft_athrow(object* o)
{
	throwException(o);
}
#endif

#if defined(NEED_soft_checkcast) || defined(NEED_soft_instanceof)
bool isInterface(classes*, classes*);
#endif

#ifdef NEED_soft_checkcast
void
soft_checkcast(classes* c, object* o)
{
	classes* oc;
	int i;

	/* Null can be cast to anything */
	if (o == 0) {
		return;
	}

	/* First look down the superclass list */
	for (oc = o->mtable->class; oc != 0; oc = oc->superclass) {
		if (oc == c) {
			return;
		}
	}

	/* If this fails, look at the interfaces */
	if (isInterface(o->mtable->class, c)) {
		return;
	}

	throwException(ClassCastException);
}

#endif

#ifdef NEED_soft_instanceof
void
soft_instanceof(classes* c, object* o)
{
	classes* oc;

	if (o != 0) {
		for (oc = o->mtable->class; oc != 0; oc = oc->superclass) {
			if (oc == c) {
				*(&o) = (object*)1;
				return;
			}
		}
		/* If this fails, look at the interfaces */
		if (isInterface(o->mtable->class, c)) {
			*(&o) = (object*)1;
			return;
		}
		*(&o) = (object*)0;
	}
}
#endif

#ifdef NEED_soft_lookupmethod
void
soft_lookupmethod(methodTable* tab, strpair* pair)
{
	void* func;

	func = findMethod(tab->class, pair);

#if !defined(DEBUG)
	/* We do not fill in the tag in debug mode, so forcing a lookup
	 * on every method call.  This can help debugging but makes things
	 * go real slow.
	 */
	tab->m[pair->hash % MAXMETHOD].tag = pair;
#endif
	tab->m[pair->hash % MAXMETHOD].method = func;

MDBG(	printf("Calling %s:%s%s @ 0x%x\n",
		tab->class->name, pair->s1, pair->s2, func);
		fflush(stdout);						)

	if (func == 0) {
		throwException(NoSuchMethodError);
	}
}
#endif

#ifdef NEED_soft_monitorenter
void
soft_monitorenter(object* obj)
{
SDBG(	printf("Monitor enter %x\n", obj);			)
	lockMutex(obj);
}
#endif

#ifdef NEED_soft_monitorexit
void
soft_monitorexit(object* obj)
{
SDBG(	printf("Monitor exit %x\n", obj);			)
	unlockMutex(obj);
}
#endif

#ifdef NEED_soft_badarrayindex
void
soft_badarrayindex(void)
{
	throwException(ArrayIndexOutOfBoundsException);
}
#endif

#if defined(NEED_soft_checkcast) || defined(NEED_soft_instanceof)
/*
 * Does this interface match the class.  Recuse down interface.
 *  list if not.
 */
bool
isInterface(classes* ic, classes* c)
{
	int i;

	for (i = 0; i < ic->interface_len; i++) {
		if (ic->interface[i] == c || isInterface(ic->interface[i], c)) {
			return (true);
		}
	}
	return (false);
}
#endif
