//****************************************************************************
//*                                                                          *
//*    Class: AwarePanel.java                                                *
//* Comments: A panel with cool 3D effects                                   *
//*                                                                          *
//****************************************************************************

//****************************************************************************
//* Package                                                                  *
//****************************************************************************

package							jif.awt;

//****************************************************************************
//* Imports                                                                  *
//****************************************************************************

import							java.awt.*;
import							java.util.Vector;

//****************************************************************************
//* AwarePanel                                                               *
//****************************************************************************

public class AwarePanel
extends Panel
{

//****************************************************************************
//* Constants                                                                *
//****************************************************************************

	//	The TAB key!
	final static public int		TAB_KEY = 9;

	//	Text Styles...
	public final static int		TEXT_NORMAL = 0;
	public final static int		TEXT_LOWERED = 1;
	public final static int		TEXT_RAISED = 2;
	public final static int		TEXT_SHADOW = 3;

	//	Text placement...
	public final static int		CENTER = 0;
	public final static int		LEFT = 1;
	public final static int		RIGHT = 2;

	//	3D Look
	public final static int		NONE = 0;
	public final static int 	FLAT = 0x001;
	public final static int 	GROOVED = 0x002;
	public final static int 	LOWERED = 0x004;
	public final static int 	ROUNDED = 0x08;
	public final static int 	RAISED = 0x010;
	public final static int 	RIDGED = 0x020;
	public final static int		CAPTION = 0x40;

    //	Miscellaneous...
    /**
    * This is the space between the start of the text and the
    * edge of the panel.
    */
	public final static int		TEXT_OFFSET = 2;

//****************************************************************************
//* Members                                                                  *
//****************************************************************************

	Color						light = Color.white;
	Color						shadow = Color.darkGray;

	int							textStyle = TEXT_NORMAL;
	int							textPlacement = CENTER;
	int							style = NONE;
	int							thickness = 2;
	int							myWidth, myHeight;
	String						text;

	/**
	* Our current position in the tab order.
	*/
	int							currentPos = 0;

//****************************************************************************
//* Constructors                                                             *
//****************************************************************************

	public
	AwarePanel()
	{
		this.style = NONE;
	}

	public
	AwarePanel( int style )
	{
		this.style = style;
	}

	public
	AwarePanel( int style, int width, int height )
	{
		super();
		this.style = style;
		this.myWidth = width;
		this.myHeight = height;
		resize( myWidth, myHeight );
	}

	public
	AwarePanel( int style, int width, int height, String text )
	{
		super();
		this.style = style;
		this.myWidth = width;
		this.myHeight = height;
		resize( myWidth, myHeight );
		setText( text, TEXT_NORMAL, CENTER );
	}

//****************************************************************************
//* setStyle                                                                 *
//****************************************************************************

	public void
	setStyle( int style )
	{
		this.style = style;
	}

//****************************************************************************
//* setText                                                                  *
//****************************************************************************

	public void
	setText( String newText, int textStyle, int placement )
	{
		this.text = newText;
		setTextStyle( textStyle );
		setTextPlacement( placement );
		repaint();
	}

	public void
	setText( String newText )
	{
		this.text = newText;
		setTextStyle( TEXT_NORMAL );
		setTextPlacement( LEFT );
		repaint();
	}

//****************************************************************************
//* setTextStyle                                                             *
//****************************************************************************

	public void
	setTextStyle( int textStyle )
	{
		this.textStyle = textStyle;
	}

//****************************************************************************
//* setTextPlacement                                                         *
//****************************************************************************

	public void
	setTextPlacement( int placement )
	{
		this.textPlacement = placement;
	}

//****************************************************************************
//* setThickness                                                             *
//****************************************************************************

	public void
	setThickness( int thick )
	{
		thickness = thick;
	}

//****************************************************************************
//* paint                                                                    *
//****************************************************************************

	public void
	paint( Graphics g )
	{
		if ( style > 0 )
		{
			drawFrame( g );

			if ( text != "" )
				drawText( g );
		}
	}

//****************************************************************************
//* drawFrame                                                                *
//****************************************************************************

	public void
	drawFrame( Graphics g )
	{
		boolean 	status = false;
		int 		i, j, offset = 0, rounded = 0;
		Dimension	bounds = size();

		if ( isStyleSet( CAPTION ) )
		{
			//	Fix up for font...
			FontMetrics fm = g.getFontMetrics();
			offset = fm.getHeight() / 2;
		}

		for ( i = 0, j = 1; i < thickness; i++, j += 2 )
		{
			if ( i != 0 )
				rounded = 0;
			else
				if ( isStyleSet( ROUNDED ) )
					rounded = 1;

			if ( isStyleSet( RAISED ) ||
				( isStyleSet( RIDGED ) && ( j < thickness ) ) ||
				( isStyleSet( GROOVED ) && ( j > thickness ) ) )
				g.setColor( light );
			else
				g.setColor( shadow );

			//	Draw top line...
			g.fillRect( i + rounded,
				i + offset,
				bounds.width - ( 2 * ( i + rounded ) ),
				1 );

			//	Draw the left side...
			g.fillRect( i,
				i + rounded + offset,
				1,
				bounds.height - ( 2 * i + 1 + rounded ) - offset );

			if ( isStyleSet( RAISED ) || isStyleSet( FLAT ) ||
				( isStyleSet( RIDGED ) && ( j < thickness ) ) ||
				( isStyleSet( GROOVED ) && ( j > thickness ) ) )
				g.setColor( shadow );
			else
				g.setColor( light );

			//	Draw the bottom line...
			g.fillRect( i + rounded,
				bounds.height - ( i + 1 ),
				bounds.width - ( 2 * ( i + rounded ) ),
				1 );

			//	Draw the right side...
			g.fillRect( bounds.width - ( i + 1 ),
				i + rounded + offset,
				1,
				bounds.height - ( 2 * i + 1 + rounded ) - offset );
		}
	}

//****************************************************************************
//* preferredSize                                                            *
//****************************************************************************

	public Dimension
	preferredSize()
	{
		return( minimumSize() );
	}

//****************************************************************************
//* minimumSize                                                              *
//****************************************************************************

	public Dimension
	minimumSize()
	{
		return( new Dimension( myWidth, myHeight ) );
	}

//****************************************************************************
//* resize                                                                   *
//****************************************************************************

	public void
	resize( int width, int height )
	{
		myWidth = width;
		myHeight = height;
		super.resize( width, height );
	}


//****************************************************************************
//* drawText                                                                 *
//****************************************************************************

	public void
	drawText( Graphics g )
	{
		Color				oldColor = g.getColor();
		int					xx = 0, yy = 0;
		FontMetrics			fm = g.getFontMetrics();
		Dimension			bounds = size();

		//	Decide where to place the text...
		if ( !isStyleSet( CAPTION ) )
		{
			switch ( textPlacement )
			{
				case CENTER:
					xx = ( bounds.width / 2 ) - ( fm.stringWidth( text ) / 2 );
					yy = ( bounds.height / 2 ) - ( fm.getHeight() / 2 );
					break;

				case LEFT:
					xx = TEXT_OFFSET;
					yy = ( bounds.height / 2 ) - ( fm.getHeight() / 2 );
					break;

				case RIGHT:
					xx = bounds.width - TEXT_OFFSET -
						fm.stringWidth( text );

					yy = ( bounds.height / 2 ) - ( fm.getHeight() / 2 );
					break;
			}
		}
		else
		{
			xx = thickness + fm.charWidth( 'W' );
			yy = 0;

			//	Fill a rectangle in the bounding space of the string...
			g.setColor( getBackground() );
			g.fillRect( xx, yy, fm.stringWidth( text ), fm.getHeight() );
		}

		//	Adjust for drawString weirdness...
		yy += fm.getHeight() - fm.getDescent() - 1;

		if ( textStyle == TEXT_LOWERED )
		{
			//	Draw highlight to right and below text
			g.setColor( light );
			g.drawString( text, xx + 1, yy + 1 );
		}

		if ( textStyle == TEXT_RAISED )
		{
			//	Draw highlight to left and above text
			g.setColor( light );
			g.drawString( text, xx - 1, yy - 1 );
		}

		if ( textStyle == TEXT_SHADOW )
		{
			//	Draw shadow to right and below text
			g.setColor( shadow );
			g.drawString( text, xx + 1, yy + 1 );
		}

		g.setColor( Color.black );
		g.drawString( text, xx, yy );

		//	Restore the old color...
		g.setColor( oldColor );
	}

//****************************************************************************
//* isStyleSet                                                               *
//****************************************************************************

	/**
	* Check the style bitmap for inclusion.
	* @param check The bits to check
	*/

	public boolean
	isStyleSet( int check )
	{
		return( ( style & check ) != 0 );
	}

//****************************************************************************
//* focusForward                                                             *
//****************************************************************************

	/**
	* Sends the focus to the next component
	*/

	public synchronized void
	focusForward( Component c )
	{
		//	Go from our current position and find next widget...
		currentPos = findNextComponent( currentPos );

		//	Now set the focus...
		getComponent( currentPos ).requestFocus();
	}

//****************************************************************************
//* focusBackward                                                            *
//****************************************************************************

	/**
	* Sends the focus to the previous component
	*/

	public synchronized void
	focusBackward( Component c )
	{
		//	Go from our current position and find last widget...
		currentPos = findPreviousComponent( currentPos );

		//	Set the focus...
		getComponent( currentPos ).requestFocus();
	}

//****************************************************************************
//* findNextComponent                                                        *
//****************************************************************************

	/**
	* Finds the next widget in the component list...
    *
    * @param startPos The starting position
	*/

	protected int
	findNextComponent( int startPos )
	{
		int			i = startPos + 1;

		do
		{
			//	Check for a wrap...
			if ( i >= countComponents() )
				i = 0;

			Component c = getComponent( i );

			//	Skip anything disabled or labels...
			if ( c instanceof Label || !c.isEnabled() )
			{
				i++;
				continue;
			}

			//	I am the one!
			return( i );
		}
		while ( i != startPos );

		//	Wow, no more!
		return( 0 );
	}

//****************************************************************************
//* findPreviousComponent                                                    *
//****************************************************************************

	/**
	* Finds the previous widget in the component list...
	*
	* @param startPos The starting position
	*/

	protected int
	findPreviousComponent( int startPos )
	{
		int			i = startPos - 1;

		do
		{
			//	Check for a wrap...
			if ( i < 0 )
				i = countComponents() - 1;

			Component c = getComponent( i );

			//	Skip anything disabled or labels...
			if ( c instanceof Label || !c.isEnabled() )
			{
				i--;
				continue;
			}

			//	I am the one!
			return( i );
		}
		while ( i != startPos );

		//	Wow, no more!
		return( 0 );
	}

//****************************************************************************
//* setFocus                                                                 *
//****************************************************************************

	/**
	* This routine is called when any of the child components get
	* the focus. It is a feeble attempt to figure out what the current
	* position in the tab order I am. However, as of the JDK v1.0.2,
	* the focus events are not working properly.
	*
	* @param c The component who got the focus
	*/

	public void
	setFocus( Component target )
	{
		//	Search for the component that just got the focus...
		for ( int i = 0; i < countComponents(); i++ )
		{
			Component c = getComponent( i );

			if ( target == c )
			{
				//	Skip disabled components...
				if ( !c.isEnabled() )
					return;

				getComponent( i ).requestFocus();
				currentPos = i;

				//	Select text by default...
				if ( getComponent( i ) instanceof TextField )
					( ( TextField )getComponent( i ) ).selectAll();

				//	We're done...
				return;
			}
		}

		return;
	}

//****************************************************************************
//* generateUpdateSQL                                                        *
//****************************************************************************

	/**
	* Generates an SQL UPDATE statement for the AwareTextFields on this
	* panel that have changed.
	* @param tableName The name of the table to update
	*/

	public String
	generateUpdateSQL( String tableName )
	{
		Component	cList[] = getComponents();
		Vector		valueList = new Vector( 5 );
		String		s;
		String		whereClause = " WHERE ";
		boolean		wasWhere = false;

		//	Loop through components and find the primary keys...
		for ( int i = 0; i < cList.length; i++ )
		{
			//	Is it a text field?
			if ( cList[ i ] instanceof AwareTextField )
			{
				//	Instead of casting constantly, use alias...
				AwareTextField	atf = ( AwareTextField )cList[ i ];

				//	Get the values statement...
				s = atf.generateUpdateSQL( false );

				//	Did it change?
				if ( !s.equals( "" ) )
				{
					//	Store the data...
					valueList.addElement( s );
				}

				//	Format up the PK...
				if ( atf.isPrimaryKey() )
				{
					//	If there was a set, add an AND...
					if ( wasWhere )
						whereClause += "AND ";

					//	Build up the WHERE clause...
					whereClause += atf.getSQL( true );

					//	Set the AND flag...
					wasWhere = true;
				}
			}
		}

		//	Bail if nothing to do...
		if ( valueList.size() < 1 )
			return( "" );

		//	Construct a statement...
		s = "UPDATE " + tableName + " SET ";

		//	Add column names...
		for ( int i = 0; i < valueList.size(); i++ )
		{
			s += valueList.elementAt( i );

			if ( i != valueList.size() - 1 )
				s += ", ";
		}

		//	Return the SQL...
		return( s + whereClause );
	}

//****************************************************************************
//* generateInsertSQL                                                        *
//****************************************************************************

	/**
	* Generates an SQL INSERT statement for the AwareTextFields on this
	* panel that have changed.
	* @param tableName The name of the table to update
	*/

	public String
	generateInsertSQL( String tableName )
	{
		Component	cList[] = getComponents();
		Vector		columnList = new Vector( 5 );
		Vector		valueList = new Vector( 5 );
		String		s;

		//	Loop through components and find the primary keys...
		for ( int i = 0; i < cList.length; i++ )
		{
			//	Is it a text field?
			if ( cList[ i ] instanceof AwareTextField )
			{
				//	Instead of casting constantly, use alias...
				AwareTextField	atf = ( AwareTextField )cList[ i ];

				//	Get the values statement...
				s = atf.generateInsertSQL( false );

				//	Did it change?
				if ( !s.equals( "" ) )
				{
					//	Store the data...
					valueList.addElement( s );

					//	Get the column name!
					columnList.addElement( atf.getColumnName() );
				}
			}
		}

		//	Bail if nothing to do...
		if ( valueList.size() < 1 )
			return( "" );

		//	Construct a statement...
		s = "INSERT INTO " + tableName + " ( ";

		//	Add column names...
		for ( int i = 0; i < columnList.size(); i++ )
		{
			s += columnList.elementAt( i );

			if ( i != columnList.size() - 1 )
				s += ", ";
			else
				s += " )";
		}

		s += " VALUES ( ";

		//	Add values...
		for ( int i = 0; i < valueList.size(); i++ )
		{
			s += valueList.elementAt( i );

			if ( i != valueList.size() - 1 )
				s += ", ";
			else
				s += " )";
		}

		//	Return the SQL...
		return( s );
	}

//****************************************************************************
//* setChildValue                                                            *
//****************************************************************************

	public void
	setChildValue( String name, String value )
	{
		AwareTextField		atf;

		if ( ( atf = ( AwareTextField )findChild( name ) ) != null )
		{
			atf.setText( value );
			atf.dataChange = true;
		}
	}

	public void
	setChildValue( String name, int value )
	{
		setChildValue( name, Integer.toString( value ) );
	}

//****************************************************************************
//* findChild                                                                *
//****************************************************************************

	protected Component
	findChild( String name )
	{
		Component	cList[] = getComponents();

		//	Loop through components
		for ( int i = 0; i < cList.length; i++ )
		{
			//	Is it a text field?
			if ( cList[ i ] instanceof AwareTextField )
				if ( ( ( AwareTextField )cList[ i ] ).getColumnName().equalsIgnoreCase( name ) )
					return( cList[ i ] );
		}

		return( null );
	}

//****************************************************************************
//* handleEvent                                                              *
//****************************************************************************

	public boolean
	handleEvent( Event event )
	{
		//	Look for focus events that have been passed up...
		if ( event.id == Event.GOT_FOCUS &&
			event.target instanceof Component )
			setFocus( ( Component )event.target );

		//	Handle tabs nicely...
		if ( event.id == Event.KEY_PRESS && event.key == TAB_KEY )
		{
			if ( event.target instanceof Component )
			{
				if ( !event.shiftDown() )
					focusForward( ( Component )event.target );
				else
					focusBackward( ( Component )event.target );
			}

			//	I handled it...
			return( true );
		}

		//	I don't want this...
		return( super.handleEvent( event ) );
	}

}