//****************************************************************************
//*                                                                          *
//*   Module: CalendarPanel.java                                             *
//*                                                                          *
//****************************************************************************
//*                                                                          *
//*   Author: Gerald H. Ablan, Jr.                                           *
//*  Company: NetGeeks                                                       *
//*                                                                          *
//****************************************************************************
//*                                                                          *
//*     Date: April 13, 1996                                                 *
//*                                                                          *
//* Comments: Creates a calendar panel                                       *
//*                                                                          *
//*                                                                          *
//*                                                                          *
//*                                                                          *
//****************************************************************************

//****************************************************************************
//*                             Revision History                             *
//****************************************************************************
//* 04.13.96: GHA - Creation                                                 *
//* xx.xx.xx:                                                                *
//****************************************************************************

//****************************************************************************
//* Package                                                                  *
//****************************************************************************

package				jif.widgets;

//****************************************************************************
//* Imports                                                                  *
//****************************************************************************

import              java.awt.*;
import              java.util.Date;

//****************************************************************************
//* CalendarPanel Class                                                      *
//****************************************************************************

public class
CalendarPanel extends Panel
{
	//	Constants...
    private	static	final	int	BUTTON_COUNT	= 42;
	private	static	final   int	LABEL_COUNT 	= 7;

	//	Variables...
    Button          days[] = new Button[ BUTTON_COUNT ];
    Label			labs[] = new Label[ LABEL_COUNT ];
    Label			labMonth;
    int             theMonth, theYear;
    int             bHeight, bWidth;
    Panel			cPanel, calPanel;
    Button			butNext, butPrev;
    Date			selectedDate;

//****************************************************************************
//* constructor                                                              *
//****************************************************************************

	public
    CalendarPanel()
    {
        //  Create a font for our calendar and make it the default font...
        setFont( new Font( "Helvetica", Font.BOLD, 14 ) );

        //  investigate its size for our button height and width...
        FontMetrics fm = getFontMetrics( getFont() );

        bHeight = fm.getHeight() + 10;
        bWidth = fm.stringWidth( "31" ) + 10;

        //	Create a border layout for this control...
        setLayout( new BorderLayout() );

        //	Make a few panels...
        cPanel = new Panel();
        calPanel = new Panel();

		cPanel.setLayout( new BorderLayout() );
        calPanel.setLayout( new GridLayout( 7, 7 ) );

		add( "North", cPanel );
		add( "Center", calPanel );

		//	Build the control panel...
		labMonth = new Label( "", Label.CENTER );
		butNext = new Button( ">>" );
		butPrev = new Button( "<<" );
		cPanel.add( "West", butPrev );
		cPanel.add( "Center", labMonth );
		cPanel.add( "East", butNext );

		//	Make the labels...
		constructLabels();

		//	Make the buttons...
		constructButtons();
    }

//****************************************************************************
//* constructLabels                                                          *
//****************************************************************************

    void
    constructLabels()
    {
		String	dayLabels[] = { "S", "M", "T", "W", "Th", "F", "S" };

        //  Make the labels
        for ( int i = 0; i < LABEL_COUNT; i++ )
        {
            labs[ i ] = new Label( dayLabels[ i ], Label.LEFT );
            labs[ i ].resize( bWidth, bHeight );

            //  Add them to the layout...
            calPanel.add( labs[ i ] );
        }
    }

//****************************************************************************
//* constructButtons                                                         *
//****************************************************************************

    void
    constructButtons()
    {
        //  Make the buttons...
        for ( int i = 0; i < BUTTON_COUNT; i++ )
        {
            days[ i ] = new Button( "  " );
            days[ i ].disable();
            days[ i ].resize( bWidth, bHeight );

            //  Add them to the layout...
            calPanel.add( days[ i ] );
        }

        //  Get the current date...
        Date today = new Date();

        //  Get the month and year...
        theMonth = today.getMonth() + 1;
        theYear = today.getYear();

        //  Set the labels...
        setButtonLabels();
    }

//****************************************************************************
//* daysPerMonth                                                             *
//****************************************************************************

    int
    daysPerMonth( int aMonth, int aYear )
    {
        int         tempYear = aYear + 1900;
        int         days = 31;

        switch( aMonth )
        {
            case 2:     //  Feb
                if ( ( tempYear % 4 ) == 0 )
                {
                    if ( ( tempYear % 100 ) == 0 && ( tempYear % 400 ) != 0 )
                        days = 28;
                    else
                        days = 29;
                }
                else
                    days = 28;
                break;

            case 4:     //  Apr
            case 6:     //  Jun
            case 9:     //  Sep
            case 11:     //  Nov
                days = 30;
                break;

            default:
            	days = 31;
            	break;
        }

        return( days );
    }

//****************************************************************************
//* setButtonLabels                                                          *
//****************************************************************************

	void
	setButtonLabels()
	{
		int		i;

		//	Create a new date for the first...
		Date then =	new	Date( theYear, theMonth - 1, 1 );

		//	Which day was the first?
		int		dow	= then.getDay();
		int		realDay	= 1;
		int		lastDay	= daysPerMonth(	theMonth, theYear );

		//	Spin through the entire	month...
		for	( i	= 0; i < BUTTON_COUNT; i++ )
		{
			if ( i < dow )
				days[ i	].hide();
			else
			{
				//	Set	the	labels for the buttons that	are	good...
				days[ i	].setLabel(	Integer.toString( realDay ) );

				//	Did	we slip	into the next month?
				if ( realDay <= lastDay )
				{
					//	Show these guys...
					days[ i	].enable();
					days[ i	].show();
				}
				else
				{
					//	Hide this box...
					days[ i	].hide();
				}

				//	Increment the date...
				realDay++;
				then.setDate( realDay );
			}
		}

		//	Show the month and year...
		setMonthLabel();
	}

//****************************************************************************
//* nextMonth                                                                *
//****************************************************************************

    void
    nextMonth()
    {
        if ( ++theMonth > 12 )
        {
            theMonth = 1;
            theYear++;
        }

        setButtonLabels();
    }

//****************************************************************************
//* lastMonth                                                                *
//****************************************************************************

    void
    lastMonth()
    {
        if ( --theMonth < 1 )
        {
            theMonth = 12;
            theYear--;
        }

        setButtonLabels();
	}

//****************************************************************************
//* handleEvent                                                              *
//****************************************************************************

 	public boolean
	handleEvent( Event event )
	{
		//	Next month button?
		if ( event.target == butNext )
		{
			nextMonth();
			return( true );
		}

		//	Previous month button?
		if ( event.target == butPrev )
		{
			lastMonth();
			return( true );
		}

		//	Still a button?
		if ( event.target instanceof Button )
		{
			for ( int i = 0; i < BUTTON_COUNT; i++ )
			{
				if ( event.target == days[ i ] )
				{
					int selectedDay = Integer.parseInt( days[ i ].getLabel() );

					//	Create a date to return...
					selectedDate = new Date( theYear, theMonth - 1, selectedDay );

					//	Call the action dude...
					Event dse = new Event( this, Event.ACTION_EVENT, selectedDate );
					postEvent( dse );
					return( true );
				}
			}
		}

		//	I didn't handle it
		return( super.handleEvent( event ) );
	}

//****************************************************************************
//* setMonthLabels                                                           *
//****************************************************************************

	void
	setMonthLabel()
	{
	    String	months[] =	{	"January",
					    		"February",
					    		"March",
					    		"April",
					    		"May",
					    		"June",
					    		"July",
					    		"August",
					    		"September",
					    		"October",
					    		"November",
					    		"December"
							};

		labMonth.setText( months[ theMonth - 1 ] + " " + Integer.toString( theYear + 1900 ) );
	}

//****************************************************************************
//* getSelectedDate                                                          *
//****************************************************************************

	public Date
	getSelectedDate()
	{
		return( selectedDate );
	}
}
