//****************************************************************************
//*                                                                          *
//*   Module: Jiflet.java                                                    *
//*   Author: Gerald H. Ablan, Jr.                                           *
//*     Date: April 28, 1996                                                 *
//*                                                                          *
//* Comments: Abstract class that represents a Java Intranet Application     *
//*                                                                          *
//****************************************************************************

//****************************************************************************
//* Package                                                                  *
//****************************************************************************

package							jif.jiflet;

//****************************************************************************
//* Imports                                                                  *
//****************************************************************************

//	Java Imports...
import							java.net.*;
import							java.awt.*;
import							java.io.*;
import							java.util.*;

//	JIF Imports...
import							jif.log.*;
import							jif.util.*;
import							jif.awt.*;
import							jif.widgets.*;
import							jif.sql.*;

//****************************************************************************
//* Jiflet                                                                   *
//****************************************************************************

/**
 * The abstract Jiflet class should be the base class for
 * any Jiflets you create. Implements the Application and
 * Runnable interfaces.
 *
 * @see jif.jiflet.Application
 *
 * @version		1.00, 26 Apr 1996
 * @author		Jerry Ablan, munster@mcs.net
 */

public abstract class Jiflet
extends Frame
implements JifApplication
{

//****************************************************************************
//* Members                                                                  *
//****************************************************************************

	public static final int			DEFAULT_SERVER_PORT = 32109;

	Socket							mySocket;
	Thread							in, out;

	private Properties				appProperties = new Properties();
	private Properties				configProperties;

	protected String				appName;
	protected String				appDefaultConfigFile;
	protected boolean				appVerbosity = false;
	protected DiskLog				appLogFile;
	protected ScreenLog				defaultLog;
	protected StatusBar				myStatusBar = null;

	//	A database connector...
	private JifConnector			myConnector = null;

	private int						oldCursor = -1;

//****************************************************************************
//* Constructors                                                             *
//****************************************************************************

	/**
	 * Creates a generic Jiflet
	 */

	public
	Jiflet()
	{
		this( "Generic Jiflet", "Jiflet", null, false );
	}

	/**
	 * Creates a Jiflet with a title
	 *
	 * @param title The window title
	 */

	public
	Jiflet( String title )
	{
		this( title, "", null, false );
	}

	/**
	 * Creates a Jiflet with a title, a name, and arguments
	 *
	 * @param title The window title
	 * @param name The name of the application
	 * @param args The arguments passed in to the program
	 */

	public
	Jiflet( String title, String name, String args[] )
	{
		this( title, name, args, false );
	}

	/**
	 * Creates a Jiflet with a title, a name, arguments, and optionally
	 * verbose.
	 *
	 * @param title The window title
	 * @param name The name of the application
	 * @param args The arguments passed in to the program
	 * @param verbosity On/Off setting indicating verbosity of log entries
	 * @see #setVerboseMode
	 */

	public
	Jiflet( String title, String name, String args[], boolean verbosity )
	{
		//	Call the superclass...
		super( title );

		//	Copy title to name...
		if ( name.equals( "" ) )
			name = title;

		//	Set the color...
		setBackground( Color.lightGray );

		//	Center and show our window!
		center();

		//	Add a status bar...
		enableStatusBar();

		//	Save my application name...
		appName = name;

		//	Open the log file...
		try
		{
			appLogFile = new DiskLog( appName );
		}
		catch ( IOException e )
		{
			//	Write errors to the screen...
			defaultLog.log( JLL_ERROR, "Error opening log file for [ " +
				appName + "]" );

			appLogFile = null;
		}

		//	Construct a config file name...
		appDefaultConfigFile = name + ".cfg";

		//	Turn on verbose mode...
		setVerboseMode( verbosity );

		//	Denote construction...
		log( "Application [" + appName + "] started at " +
			( new Date() ).toString() );

		//	Parse any passed in arguments...
		parseArguments( args );

		//	Parse the configuration file if available...
		parseConfigFile();

		//	Create a database connector...
		setConnector( new JifConnector( this ) );

		//	Call my init!
		init();

		//	We're outta here!
		setStatus( "Ready" );

		//	Show my window...
		show();
	}

//****************************************************************************
//* setVerboseMode                                                           *
//****************************************************************************

	/**
	 * Turns on and off verbose mode
	 *
	 * @param whichWay A boolean describing the verbosity
	 */

	public void
	setVerboseMode( boolean whichWay )
	{
		appVerbosity = whichWay;
	}

//****************************************************************************
//* verboseLog                                                               *
//****************************************************************************

	/**
	 * Writes an entry to the system log only if the application is in verbose mode.
	 *
	 * @param logLevel One of the logging levels from the JIFIApplication
	 * @param logEntry The string to write to the file
	 * @see #setVerboseMode
	 */

	public void
	verboseLog( char logLevel, String logEntry )
	{
		if ( appVerbosity )
			log( logLevel, logEntry );
	}

	/**
	 * Writes an entry to the system log only if the application is in verbose mode.
	 *
	 * @param logEntry The string to write to the file
	 */

	public void
	verboseLog( String logEntry )
	{
		verboseLog( JLL_INFO, logEntry );
	}

//****************************************************************************
//* errorLog                                                                 *
//****************************************************************************

	/**
	* Writes an error level entry to the system log
	*
	* @param logEntry The string to write to the log file
	*/

	public void
	errorLog( String logEntry )
	{
		log( JLL_ERROR, logEntry );
	}

//****************************************************************************
//* log                                                                      *
//****************************************************************************

	/**
	 * Writes an entry to the system log
	 *
	 * @param logLevel One of the logging levels from the JIFIApplication
	 * @param logEntry The string to write to the file
	 */

	public void
	log( char logLevel, String logEntry )
	{
		if ( appLogFile != null )
		{
			try
			{
				appLogFile.log( logLevel, logEntry );
			}
			catch ( IOException e )
			{
				//	Write log entry to screen upon error...
				defaultLog.log( JLL_ERROR,
					"Error writing to log file [ " + logEntry + "]" );
			}
		}
		else
		{
			//	Default to the default log if the disk log is unavailable...
			defaultLog.log( logLevel, logEntry );
		}
	}

	/**
	 * Writes an informational entry to the system log
	 *
	 * @param logEntry The string to write to the file
	 */

	public void
	log( String logEntry )
	{
		log( JLL_INFO, logEntry );
	}

//****************************************************************************
//* parseArguments                                                           *
//****************************************************************************

	/**
	 * Parses the arguments passed into the application from the command line
	 * and puts them into a Properties object.
	 *
	 * @param args The arguments passed to parse
	 */

	protected int
	parseArguments( String args[] )
	{
		int		i = 0;

		if ( args != null )
		{
			for ( i = 0; i < args.length; i++ )
			{
				StringTokenizer st = new StringTokenizer( args[ i ], "=" );

				if ( st.countTokens() == 2 )
					appProperties.put( st.nextToken(), st.nextToken() );
				else
					errorLog( "Invalid argument string [" + args[ i ] + "]" );
			}
		}

		return( i );
	}

//****************************************************************************
//* parseConfigFile                                                          *
//****************************************************************************

	/**
	 * Checks to see if a configuration file name was passed in via arguments.
	 * If there was not one there, it constructs a default configuration file
	 * name from the application's name. The resultant file is then opened and
	 * the configuration parameters are read into a Properties object.
	 */

	protected int
	parseConfigFile()
	{
		String		cfName;

		//	Create our configuration properties object
		configProperties = new Properties( appProperties );

		//	Search for a passed in one, if not there default to the above one...
		cfName = getParameterString( "ConfigFileName", appDefaultConfigFile );

		//	Try and read in the properties...
		try
		{
			FileInputStream fis = new FileInputStream( cfName );
			configProperties.load( fis );
		}
		catch ( FileNotFoundException e )
		{
			log( "Config file [" + cfName + "] not found" );
		}
		catch ( IOException e )
		{
			errorLog( "Error reading config file [" + cfName + "]: " + e.toString() );
		}

		//	Dump all config entries to the log...
		Enumeration e = configProperties.propertyNames();

		while ( e.hasMoreElements() )
		{
			String s = ( String )e.nextElement();
			log( s + " = " + configProperties.getProperty( s ) );
		}

		return( configProperties.size() );
	}

//****************************************************************************
//* handleEvent                                                              *
//****************************************************************************

	/**
	 * The event handler
	 * @param anEvent The event in question
	 */

	public boolean
	handleEvent( Event anEvent )
	{
		if ( anEvent.id == Event.WINDOW_DESTROY )
		{
			log( "Application [" + appName + "] ended at " +
				( new Date() ).toString() );

			//	Go away gracefully...
			hide();
			dispose();

			System.exit( 0 );
		}

		//	I didn't handle it, pass it up...
		return( super.handleEvent( anEvent ) );
	}

//****************************************************************************
//* suicide                                                                  *
//****************************************************************************

	/**
	 * Allows the Jiflet to kill itself
	 *
	 * @param e An Exception that may have caused the fatality
	 * @param logLine What to write in the log about this suicide
	 * @param level The value to return via the System.exit() call
	 */

	public void
	suicide( Exception e, String logLine, int level )
	{
		verboseLog( "Application [" + appName + "] committed suicide at " +
			( new Date() ).toString() );

		if ( e != null )
		{
			if ( !logLine.equals( "" ) )
				log( JLL_FATAL, logLine + ": " + e );
			else
				log( JLL_FATAL, e.toString() );
		}
		else
			log( JLL_FATAL, logLine );

		System.exit( level );
	}

	/**
	 * Allows the Jiflet to kill itself
	 *
	 * @param logLine What to write in the log about this suicide
	 */

	public void
	suicide( String logLine )
	{
		suicide( null, logLine, 1 );
	}

	/**
	 * Allows the Jiflet to kill itself
	 *
	 * @param logLine What to write in the log about this suicide
	 * @param level The value to return via the System.exit() call
	 */

	public void
	suicide( String logLine, int level )
	{
		suicide( null, logLine, level );
	}

	/**
	 * Allows the Jiflet to kill itself
	 *
	 * @param e An Exception that may have caused the fatality
	 */

	public void
	suicide( Exception e )
	{
		suicide( e, "", 1 );
	}

	/**
	 * Allows the Jiflet to kill itself
	 *
	 * @param e An Exception that may have caused the fatality
	 * @param logLine What to write in the log about this suicide
	 */

	public void
	suicide( Exception e, String logLine )
	{
		suicide( e, logLine, 1 );
	}

//****************************************************************************
//* connectToServer                                                          *
//****************************************************************************

//	/**
//	 * Connects to a JIF server using the host and port specified. This
//	 * connection allows the JIF server to monitor running Jiflets.
//	 *
//	 * @param host The host name where the JIF server is running
//	 * @param port The port on which the JIF server is listening
//	 */
//
//	public void
//	connectToServer( String host, int port )
//	{
//		if ( !host.equals( "" ) && host != null )
//		{
//			try
//			{
//				mySocket = new Socket( host, ( port == 0 ) ? DEFAULT_SERVER_PORT : port );
//				in = new Reader( this );
//				out = new Writer( this );
//				in.setPriority( 6 );
//				out.setPriority( 5 );
//				in.start();
//				out.start();
//			}
//			catch ( IOException e )
//			{
//				log( "Error opening socket [" + e + "]" );
//			}
//		}
//	}

//****************************************************************************
//* center                                                                   *
//****************************************************************************

	/**
	 * Call this method to center the jiflet on the screen
	 */

	public void
	center()
	{
		Rectangle r = bounds();
		Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

		move( ( d.width / 2 ) - ( r.width / 2 ),
			( d.height / 2 ) - ( r.height /2 ) );
	}

//****************************************************************************
//* paint                                                                    *
//****************************************************************************

	/**
	 * In order to capture resizing events for the status bar, we over-
	 * ride the paint() method. Whenever the window is resized, it is
	 * repainted.
	 * @param g The graphics object
	 */

	public void
	paint( Graphics g )
	{
		//	Call dad!
		super.paint( g );

		//	If we have a status bar, then resize and repaint it!
//		if ( myStatusBar != null )
//			myStatusBar.setSize( this );
	}

//****************************************************************************
//* enableStatusBar                                                          *
//****************************************************************************

	/**
	 * Creates a status bar that sits on the bottom of the Frame with text
	 * to display.
	 * @param text The text to display in the status bar
	 * @see #clearStatus
	 * @see #setStatus
	 */

	public void
	enableStatusBar( String text )
	{
		enableStatusBar();

		//	Set the string...
		if ( text != null )
			setStatus( text );
	}

	/**
	 * Creates a status bar that sits on the bottom of the Frame
	 * @param text The text to display in the status bar
	 * @see #clearStatus
	 * @see #setStatus
	 */

	public void
	enableStatusBar()
	{
		//	Create a status bar...
		myStatusBar = new StatusBar();

		//	Set the font...
		myStatusBar.setFont( new Font( "Dialog", Font.PLAIN, 12 ) );

		//	Add to layout!
		add( "South", myStatusBar );
	}

//****************************************************************************
//* setStatus                                                                *
//****************************************************************************

	/**
	 * Sets the text of the status bar
	 * @param text The text to display in the status bar
	 * @see #enableStatusBar
	 * @see #clearStatus
	 */

	public void
	setStatus( String text )
	{
		if ( myStatusBar != null )
			myStatusBar.setText( text );
	}

//****************************************************************************
//* clearStatus                                                              *
//****************************************************************************

	/**
	 * Clears the text of the status bar
	 * @see #enableStatusBar
	 * @see #setStatus
	 */

	public void
	clearStatus()
	{
		if ( myStatusBar != null )
			myStatusBar.clear();
	}

//****************************************************************************
//* setConnector                                                             *
//****************************************************************************

	/**
	 * Sets the Jiflet's connector
	 * @param aConnector The connector
	 * @see #getConnector
	 */

	protected void
	setConnector( JifConnector aConnector )
	{
		//	Disconnect existing connector...
		if ( myConnector != null )
			myConnector.disconnect();

		myConnector = aConnector;
	}

//****************************************************************************
//* getConnector                                                             *
//****************************************************************************

	/**
	 * Returns the current JifConnector
	 * @see #setConnector
	 */

	public JifConnector
	getConnector()
	{
		return( ( myConnector == null ) ? null : myConnector );
	}

//****************************************************************************
//* startWait                                                                *
//****************************************************************************

	/**
	* Turns the cursor to the wait cursor. Use for long operations.
	* @see #endWait
	*/

	public void
	startWait()
	{
		oldCursor = getCursorType();
		setCursor( Frame.WAIT_CURSOR );
	}

//****************************************************************************
//* endWait                                                                  *
//****************************************************************************

	/**
	* Turns the cursor back to the old cursor before the startWait() was
	* called.
	* @see #startWait
	*/

	public void
	endWait()
	{
		if ( oldCursor != -1 )
		{
			setCursor( oldCursor );
			oldCursor = -1;
		}
	}

//****************************************************************************
//* getParameterString                                                       *
//****************************************************************************

	/**
	* Returns the parameter linked with key
	* @param key The key value
	*/

	public String
	getParamenterString( String key )
	{
		return( configProperties.getProperty( key ) );
	}

	/**
	* Returns the parameter linked with key, or the default value if the
	* key is not found.
	* @param key The key value
	* @param defaultValue The default value
	*/

	public String
	getParameterString( String key, String defaultValue )
	{
		return( configProperties.getProperty( key, defaultValue ) );
	}

}
