//****************************************************************************
//*                                                                          *
//*   Module: JifConnector.java                                              *
//*   Author: Gerald H. Ablan, Jr.                                           *
//*     Date: April 28, 1996                                                 *
//*                                                                          *
//* Comments: Encapsulates database access to a JDBC data source             *
//*                                                                          *
//****************************************************************************

//****************************************************************************
//* Package                                                                  *
//****************************************************************************

package							jif.sql;

//****************************************************************************
//* Imports                                                                  *
//****************************************************************************

//	JIF Imports...
import							jif.jiflet.*;

//	JDBC Imports...
import							java.sql.*;
import							imaginary.sql.*;

//****************************************************************************
//* JifConnector                                                             *
//****************************************************************************

/**
 * The JifConnector class encapsulates some of the rudimentary JDBC
 * initialization that is always performed. This is a specialized class
 * as it only connects with mSQL data sources.
 *
 * @see jif.jiflet
 *
 * @version		1.00, 26 Apr 1996
 * @author		Jerry Ablan, munster@mcs.net
 */

public class
JifConnector
{

//****************************************************************************
//* Members                                                                  *
//****************************************************************************

	//	JDBC stuff
	protected iMsqlDriver		myDriver;
	protected Jiflet			myJiflet;
	protected Connection 		myConnection;
	protected boolean			isConnected = false;

	public Statement 			myStatement;

//****************************************************************************
//* Constructors                                                             *
//****************************************************************************

	/**
	 * Creates a JifConnector object
	 * @param jiflet The Jiflet that is using this connection
	 */

	public
	JifConnector( Jiflet jiflet )
	{
		//	Keep track of the Jiflet I belong to
		myJiflet = jiflet;
	}

//****************************************************************************
//* connect                                                                  *
//****************************************************************************

	/**
	 * Establishes a session/connection with the database
	 * @param user The user who is connecting
	 * @param passwd The password of the user
	 * @param host The host name of where the database resides
	 * @param port The port on which the server is listening
	 * @param dbInst The database instance to connect to
	 */

	public boolean
	connect( String user, String passwd, String host, int port, String dbInst )
	{
		String		dbUrl;

		//	If we are connected, disconnect first!
		if ( isConnected )
		{
			if ( !disconnect() )
				return( false );
		}
			
		//	Construct a JDBC:mSQL URL...
		dbUrl = "jdbc:msql:" + "//" + host;
		dbUrl += ":" + Integer.toString( port ) + "/" + dbInst;

		//	All kids love log!
		myJiflet.verboseLog( "Database URL [" + dbUrl + "]" );

		//	Try to make a connection...
		try
		{
			myDriver = new imaginary.sql.iMsqlDriver();
			myConnection = DriverManager.getConnection( dbUrl, user, passwd );
			myStatement = myConnection.createStatement();
		}
		catch ( Exception e )
		{
			myJiflet.log( Jiflet.JLL_ERROR, "Error during driver initialization: " + e.toString() );
			return( false );
		}

		//	It worked!
		isConnected = true;
		return( true );
	}

//****************************************************************************
//* disconnect                                                               *
//****************************************************************************

	/**
	 * Disconnects the currently connected session
	 */

	public boolean
	disconnect()
	{
		//	Are we already disconnected?
		if ( !isConnected )
			return( true );
	
		try
		{
			myStatement.close();
			myConnection.close();
		}
		catch ( SQLException e )
		{
			myJiflet.log( Jiflet.JLL_ERROR, "Error during DB disconnect [" +
				e.toString() + "]" );

			return( false );
		}

		isConnected = false;
		return( true );
	}

//****************************************************************************
//* connected                                                                *
//****************************************************************************

	public boolean	
	connected()
	{
		return( isConnected );
	}
	
}