//****************************************************************************
//*    Class: DiskLog.java                                                   *
//* Comments: A standard logging pathway to the disk                         *
//*                                                                          *
//****************************************************************************

//****************************************************************************
//* package                                                                  *
//****************************************************************************

package							jif.log;

//****************************************************************************
//* imports                                                                  *
//****************************************************************************

//	JIF Imports
import							jif.util.*;

//	Java Imports
import							java.io.*;

//****************************************************************************
//* DiskLog                                                                  *
//****************************************************************************

/**
 * A class that allows log entries to be written to a disk file.
 *
 * @see java.io.RandomAccessFile
 *
 * @version		1.00, 26 Apr 1996
 * @author		Jerry Ablan, munster@mcs.net
 */

public class
DiskLog
extends RandomAccessFile
{

//****************************************************************************
//* Members                                                                  *
//****************************************************************************

	private String				logFileName;
	private String				appName;

//****************************************************************************
//* Constructor(s)                                                           *
//****************************************************************************

	/**
	 * Builds a DiskLog object and creates a log file with a default name.
	 *
	 * @param appName The name of the application using this log
	 */

	public
	DiskLog( String appName )
	throws IOException
	{
		this( createLogFileName(), appName );
	}

	/**
	 * Builds a DiskLog object and creates a log file with the specified name.
	 *
	 * @param logName The name of the log file
	 * @param name The name of the application using this log
	 */

	public
	DiskLog( String logName, String name )
	throws IOException
	{
		super( logName, "rw" );
		logFileName = logName;
		appName = name;

		//	Go to the end of the file...
		seek( length() );
	}

//****************************************************************************
//* createLogFileName                                                        *
//****************************************************************************

	/**
	 * Creates a log file name. Default name is "syslog.<YYMMDD>"
	 */
	protected static String
	createLogFileName()
	{
		return( "syslog." + ( new FileDate() ).toFileString() );
	}

//****************************************************************************
//* log                                                                      *
//****************************************************************************

	/**
	 * Writes an entry to the log file
	 *
	 * @param logLevel The severity of the log entry
	 * @param logEntry The log entry
	 */
	public void
	log( char logLevel, String logEntry )
	throws IOException
	{
		String	logLine = appName + "|";

		//	Use the jif.util.FileDate for this new method...
		logLine += ( new FileDate() ).toFileString() + "|";
		logLine += logLevel + "|";
		logLine += logEntry;
		
		//	Let the system define how lines are terminated...
		logLine += System.getProperty( "line.separator" );

		//	Write it out to disk...
		writeBytes( logLine );
	}
}