//****************************************************************************
//*                                                                          *
//*    Class: AwareTextField.java                                            *
//* Comments: Extends Java's default TextField                               *
//*                                                                          *
//****************************************************************************

//****************************************************************************
//* Package                                                                  *
//****************************************************************************

package							jif.awt;

//****************************************************************************
//* Imports                                                                  *
//****************************************************************************

import							java.awt.*;

//****************************************************************************
//* AwareTextField                                                           *
//****************************************************************************

/**
 * An extension of the default TextField
 *
 * @see java.awt.TextField
 *
 * @version		1.00, 1 May 1996
 * @author		Jerry Ablan, munster@mcs.net
 */

public class AwareTextField
extends TextField
implements Aware
{

//****************************************************************************
//* Constants                                                                *
//****************************************************************************

	public final static int		TAB_KEY = 9;

//****************************************************************************
//* Members                                                                  *
//****************************************************************************

	String						columnName = "";
	String						lastText = "";
	boolean						dataChange = false;
	boolean						initialized = false;
	boolean						primaryKey = false;
	boolean						isNumericData = false;

//****************************************************************************
//* Constructors                                                             *
//****************************************************************************

	public
	AwareTextField( String s )
	{
		super( s );
	}

	public
	AwareTextField( String s, String columnName )
	{
		this( s );
		setColumnName( columnName );
	}

	public
	AwareTextField( String s, String columnName, boolean primaryKey )
	{
		this( s );
		setColumnName( columnName );
		setPrimaryKey( primaryKey );
	}

	public
	AwareTextField( int cols )
	{
		super( cols );
	}

	public
	AwareTextField( int cols, String columnName )
	{
		this( cols );
		setColumnName( columnName );
	}

	public
	AwareTextField( int cols, String columnName, boolean primaryKey )
	{
		this( cols );
		setColumnName( columnName );
		setPrimaryKey( primaryKey );
	}

//****************************************************************************
//* setColumnName                                                            *
//****************************************************************************

	/**
	* Sets the database column name for this field
	* @param columnName The column name
	*/

	public void
	setColumnName( String columnName )
	{
		this.columnName = columnName;
	}

//****************************************************************************
//* getColumnName                                                            *
//****************************************************************************

	public String
	getColumnName()
	{
		return( columnName );
	}

//****************************************************************************
//* setNumeric                                                               *
//****************************************************************************

	public void
	setNumeric( boolean onOff )
	{
		isNumericData = onOff;
	}

//****************************************************************************
//* setPrimaryKey                                                            *
//****************************************************************************

	/**
	* Indicates that this field is a primary key field
	* @param primaryKey Yes/No
	*/

	public void
	setPrimaryKey( boolean primaryKey )
	{
		this.primaryKey = primaryKey;
	}

//****************************************************************************
//* isPrimaryKey                                                             *
//****************************************************************************

	public boolean
	isPrimaryKey()
	{
		return( primaryKey );
	}

//****************************************************************************
//* handleEvent                                                              *
//****************************************************************************

	public boolean
	handleEvent( Event event )
	{
		//	I got a keypress...
		if ( event.id == Event.KEY_PRESS )
		{
			//	Initialize the text field...
			if ( !initialized )
			{
				dataChange = false;
				lastText = getText();
				initialized = true;
			}

			//	Filter for certain events...
			if ( filterEvent( event ) )
				return( false );

			//	Check for change...
			if ( !dataChange && getText() != lastText && initialized )
			{
				dataChange = true;
				lastText = getText();

				//	Post an ACTION_EVENT to notify my dad...
				getParent().postEvent( new Event(
					this,
					Event.ACTION_EVENT,
					( Object ) null ) );
			}
		}

		//	Pass it on up...
		return( super.handleEvent( event ) );
	}

//****************************************************************************
//* didDataChange                                                            *
//****************************************************************************

	public boolean
	didDataChange()
	{
		return( dataChange );
	}

//****************************************************************************
//* generateUpdateSQL                                                        *
//****************************************************************************

	/**
	* Generates an update SQL command if the data has changed.
	* @param addSet If true, adds a SET to the beginning of the statement
	*/

	public String
	generateUpdateSQL( boolean addSet )
	{
		String		result = "";

		if ( didDataChange() )
		{
			if ( addSet )
				result += "set ";

			result += getSQL( true );
		}

		return( result );
	}

//****************************************************************************
//* generateInsertSQL                                                        *
//****************************************************************************

	/**
	* Generates an insert SQL command if the data has changed.
	* @param addParen If true, adds an open parentheses to the beginning
	* of the statement
	*/

	public String
	generateInsertSQL( boolean addParen )
	{
		String		result = "";

		if ( didDataChange() )
		{
			if ( addParen )
				result += "( ";

			result += getSQL( false );
		}

		return( result );
	}

//****************************************************************************
//* getSQL                                                                   *
//****************************************************************************

	/**
	* Returns an SQL statement suitable for a WHERE, INSERT or a SET clause
	* @param forWhere Indicates that the return string should include the
	* column name.
	*/

	public String
	getSQL( boolean forWhere )
	{
		String		result = "";

		if ( !isNumericData )
		{
			if ( forWhere )
				result += getColumnName() + " = ";

			result += "'" + getText() + "' ";
		}
		else
		{
			if ( forWhere )
				result += getColumnName() + " = ";

			result += getText() + " ";
		}

		return( result );
	}

//****************************************************************************
//* filterEvent                                                              *
//****************************************************************************

	/**
	* This method checks for certain events. If the event is one we want,
	* we return true which triggers handleEvent() to not handle it.
	* @param event The event to filter
	*/

	public boolean
	filterEvent( Event event )
	{
		switch ( event.id )
		{
			case Event.KEY_PRESS:		//	Tab was pressed...
				if ( event.key == AwarePanel.TAB_KEY &&
					getParent() instanceof AwarePanel )
					return( true );
				break;

			case Event.GOT_FOCUS:		//	Got the focus
			case Event.LOST_FOCUS:		//	Lost the focus
				return( true );
		}

		//	Don't want any others...
		return( false );
	}

}