/***************************************
  $Header: /home/amb/cxref/RCS/type.c 1.14 1996/06/16 16:43:04 amb Exp $

  C Cross Referencing & Documentation tool. Version 1.2.

  Collects the typedef stuff.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1995,96 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

/*+ Control the output of debugging information in this file. +*/
#define DEBUG 0

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "memory.h"
#include "datatype.h"
#include "cxref.h"

/*+ The file that is currently being processed. +*/
extern File CurFile;

/*+ Whether we are parsing a typedef or not. +*/
extern int in_typedef;

/*+ The defined types that we have seen. +*/
static StringList typedefs={0,NULL};

/*+ The current struct / union or enum definition. +*/
static StructUnion cur_su=NULL;

/*+ The current struct / union if seen in a typedef. +*/
static StructUnion cur_type_su=NULL;

/*+ The last typedef seen, used when two types share a typedef statement. +*/
static Typedef last_typedef=NULL;

static StructUnion CopyStructUnion(StructUnion su);

/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a typedef is seen in the current file. The name of the typedef is stored for future reference.

  char* name The name of the defined type.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenTypedefName(char* name)
{
#if DEBUG
 printf("#Type.c# Type defined '%s'\n",name);
#endif

 AddToStringList(&typedefs,name,0);
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when an IDENTIFIER is seen in the current file, it may be a defined type.

  int HaveISeenType Returns 1 if the argument is a type that has been defined.

  char* name The name of the possible defined type.
  ++++++++++++++++++++++++++++++++++++++*/

int HaveISeenType(char* name)
{
 int i;

 for(i=0;i<typedefs.n;i++)
    if(!strcmp(name,typedefs.s[i]))
       return(1);

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when the start of a struct or union or enum definition is seen.

  char* name The name of the struct type.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenStructUnionStart(char* name)
{
#if DEBUG
 printf("#Type.c# Start Struct / Union '%s'\n",name);
#endif

 cur_su=(StructUnion)Malloc(sizeof(struct _StructUnion));

 cur_su->name=MallocString(name);
 if(!in_typedef)
    cur_su->comment=MallocString(GetCurrentComment());
 else
    cur_su->comment=NULL;
 cur_su->n_comp=0;
 cur_su->comps=NULL;
}


/*++++++++++++++++++++++++++++++++++++++
  Function called when a component of a struct / union / enum is seen.

  char* name The name of the struct / union / enum component.

  int depth The depth within the nested struct / union / enum.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenStructUnionComp(char* name,int depth)
{
 StructUnion s,t=cur_su;
 int i;

#if DEBUG
 printf("#Type.c# Struct / Union Component (%d) '%s'\n",depth,name);
#endif

 for(i=1;i<depth;i++,t=s)
    s=t->comps[t->n_comp-1];

 if(t->comps && strchr(name,'{'))
   {
    char* ob=strchr(name,'{');
    char* cb=strchr(name,'}'),*nb;
    while((nb=strchr(cb+1,'}'))) cb=nb;
    ob[1]=0;
    if(strcmp(name,"enum {") && strcmp(name,"union {") && strcmp(name,"struct {"))
      {
       Typedef typdef;
    
       typdef=(Typedef)Malloc(sizeof(struct _Typedef));

       AddToLinkedList(CurFile->typedefs,Typedef,typdef);

       typdef->comment=MallocString(GetCurrentComment());
       t->comps[t->n_comp-1]->comment=MallocString(typdef->comment);

       typdef->name=MallocString(t->comps[t->n_comp-1]->name);
       typdef->type=NULL;
       typdef->sutype=CopyStructUnion(t->comps[t->n_comp-1]);
       typdef->typexref=NULL;
       typdef->next=NULL;
      }
    Free(t->comps[t->n_comp-1]->name);
    t->comps[t->n_comp-1]->name=MallocString(ConcatStrings(3,name,"...",cb));
   }
 else
   {
    if(!t->comps)
       t->comps=(StructUnion*)Malloc(8*sizeof(StructUnion));
    else
       if(t->n_comp%8==0)
          t->comps=(StructUnion*)Realloc(t->comps,(t->n_comp+8)*sizeof(StructUnion));

    s=(StructUnion)Malloc(sizeof(struct _StructUnion));

    t->comps[t->n_comp++]=s;

    s->name=MallocString(name);
    s->comment=MallocString(GetCurrentComment());
    s->n_comp=0;
    s->comps=NULL;
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when the end of a struct or union or enum definition is seen.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenStructUnionEnd(void)
{
#if DEBUG
 printf("#Type.c# End Struct / Union\n");
#endif

 if(in_typedef)
    cur_type_su=cur_su;
 else
   {
    Typedef xref=CurFile->typedefs;
    Typedef typdef=(Typedef)Malloc(sizeof(struct _Typedef));

    while(xref)
      {
       if(xref->type && !strncmp(cur_su->name,xref->type,strlen(cur_su->name)))
          xref->typexref=typdef;
       xref=xref->next;
      }

    AddToLinkedList(CurFile->typedefs,Typedef,typdef);

    typdef->comment=cur_su->comment; cur_su->comment=NULL;
    typdef->name=MallocString(cur_su->name);
    typdef->type=NULL;
    typdef->sutype=cur_su;
    typdef->typexref=NULL;
    typdef->next=NULL;
   }

 cur_su=NULL;
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a typedef is seen in the current file. This is recorded fully for later output.

  char* name The name of the defined type.

  char* type The type that it is defined to be.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenTypedef(char* name,char* type)
{
 Typedef typdef;

 if(!name)
   {last_typedef=NULL;return;}

#if DEBUG
 printf("#Type.c# Typedef '%s' '%s'\n",name,type);
#endif

 typdef=(Typedef)Malloc(sizeof(struct _Typedef));

 AddToLinkedList(CurFile->typedefs,Typedef,typdef);

 typdef->comment=MallocString(GetCurrentComment());
 typdef->name=MallocString(name);
 typdef->type=MallocString(type);
 typdef->next=NULL;

 if(!cur_type_su)
   {
    Typedef xref=CurFile->typedefs;
    typdef->sutype=NULL;
    typdef->typexref=NULL;
    while(xref)
      {
       if(!strncmp(xref->name,typdef->type,strlen(xref->name)))
          typdef->typexref=xref;
       xref=xref->next;
      }
    if(!typdef->typexref)
       typdef->typexref=last_typedef;
   }
 else
   {
    typdef->sutype=cur_type_su;
    cur_type_su=NULL;
    typdef->typexref=NULL;
   }

 if(!typdef->typexref)
    last_typedef=typdef;
}


/*++++++++++++++++++++++++++++++++++++++
  Clears the memory that is used here.

  The list of defined types and the struct and union definitions.
  ++++++++++++++++++++++++++++++++++++++*/

void DeleteSpareTypes(void)
{
 DeleteStringList(&typedefs);
 InitStringList(&typedefs);
}


/*++++++++++++++++++++++++++++++++++++++
  Free the memory associated with a Struct / Union structure.

  StructUnion su The struct / union to delete.

  This needs to call itself recursively.
  ++++++++++++++++++++++++++++++++++++++*/

void DeleteStructUnion(StructUnion su)
{
 if(su)
   {
    int i;
    for(i=0;i<su->n_comp;i++)
       DeleteStructUnion(su->comps[i]);
    Free(su->name);
    if(su->comment) Free(su->comment);
    if(su->comps)   Free(su->comps);
    Free(su);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Make a copy of the specified Struct / Union structure.

  StructUnion CopyStructUnion Returns a malloced copy of the specified struct / union.

  StructUnion su The struct / union to copy.

  This needs to call itself recursively.
  ++++++++++++++++++++++++++++++++++++++*/

static StructUnion CopyStructUnion(StructUnion su)
{
 StructUnion new;
 int i;

 new=(StructUnion)Malloc(sizeof(struct _StructUnion));

 new->name=MallocString(su->name);
 new->comment=MallocString(su->comment);
 new->n_comp=su->n_comp;
 if(su->n_comp)
   {
    new->comps=(StructUnion*)Malloc(su->n_comp*sizeof(StructUnion));
    for(i=0;i<su->n_comp;i++)
       new->comps[i]=CopyStructUnion(su->comps[i]);
   }
 else
    new->comps=NULL;

 return(new);
}
