/***************************************
  $Header: /home/amb/cxref/RCS/cxref.c 1.16 1996/07/02 20:13:28 amb Exp $

  C Cross Referencing & Documentation tool. Version 1.2.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1995,96 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "parse-yy.h"
#include "memory.h"
#include "datatype.h"
#include "cxref.h"

/*+ The default value of the CPP command. +*/
#ifdef CXREF_CPP
#define CPP_COMMAND CXREF_CPP
#else
#define CPP_COMMAND "gcc -E -C -dD"
#endif

static int DocumentTheFile(char* name);

static char* cpp_command;            /*+ The actual cpp command that is built up, adding -D, -U and -I options. +*/
static unsigned int cpp_command_len; /*+ The length of the cpp command that is built up, adding -D, -U and -I options. +*/

/*+ The command line switch that sets the format of the output, +*/
int option_all_comments=0,              /*+ use all comments. +*/
    option_xref=0,                      /*+ do cross referencing. +*/
    option_warn=0,                      /*+ produce warnings. +*/
    option_index=0,                     /*+ produce an index. +*/
    option_raw=0,                       /*+ produce raw output. +*/
    option_latex=0,                     /*+ produce latex output. +*/
    option_html=0;                      /*+ produce html output. +*/

/*+ The command line switch for the output name, +*/
char *option_odir=".",                  /*+ The directory to use. +*/
     *option_name="cxref";              /*+ The base part of the name. +*/

/*++++++++++++++++++++++++++++++++++++++
  The main function that calls the parser.

  int main Returns the status, zero for normal termination, else an error.

  int argc The command line number of arguments.

  char** argv The actual command line arguments
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc,char** argv)
{
 int i;

 cpp_command=(char*)Malloc(cpp_command_len=1+strlen(CPP_COMMAND));

 strcpy(cpp_command,CPP_COMMAND);

 if(argc==1)
   {
    fputs("Usage: cxref filename [ ... filename]                  ; Filenames to document.\n"
          "             [-Idirname]                               ; The usual gcc switches.\n"
          "             [-Ddefine] [-Udefine]                     ; The usual gcc switches.\n"
          "             [-CPP cpp_program]                        ; The cpp program to use (default '" CPP_COMMAND "')\n"
          "             [-Odirname]                               ; Use dirname as the output directory\n"
          "             [-Nbasename]                              ; Use basename.* as the output filenames\n"
          "             [-all-comments]                           ; Use all comments [Dangerous option!]\n"
          "             [-xref[-all][-file][-func][-var][-type]]  ; Do cross referencing.\n"
          "             [-warn[-all][-comment][-xref]]            ; Produce warnings on comments, cross refs or both.\n"
          "             [-index[-all][-file][-func][-var][-type]] ; Produce an appendix of cross references.\n"
          "             [-raw]                                    ; Produce raw output.\n"
          "             [-latex|-latex2e]                         ; Produce LaTeX output.\n"
          "             [-html]                                   ; Produce HTML output.\n"
          ,stderr);
    exit(1);
   }

 for(i=1;i<argc;i++)
   {
    if(!strncmp(argv[i],"-I",2) || !strncmp(argv[i],"-D",2) || !strncmp(argv[i],"-U",2))
      {
       cpp_command=Realloc(cpp_command,cpp_command_len=cpp_command_len+1+strlen(argv[i]));
       strcat(cpp_command," ");
       strcat(cpp_command,argv[i]); argv[i]=NULL;
       continue;
      }

    if(!strcmp(argv[i],"-CPP"))
      {
       char *opts=&cpp_command[strlen(CPP_COMMAND)],*old=cpp_command;

       if(++i==argc)
         {fprintf(stderr,"cxref: The -CPP option requires a following argument.\n");exit(2);}

       cpp_command=Malloc(cpp_command_len=strlen(opts)+strlen(argv[i])+2);
       strcpy(cpp_command,argv[i]);
       if(opts[0])
          strcat(cpp_command,opts);
       argv[i-1]=argv[i]=NULL;
       Free(old);
       continue;
      }

    if(!strncmp(argv[i],"-O",2))
      {option_odir=&argv[i][2]; argv[i]=NULL; continue;}

    if(!strncmp(argv[i],"-N",2))
      {option_name=&argv[i][2]; argv[i]=NULL; continue;}

    if(!strcmp(argv[i],"-all-comments"))
      {option_all_comments=1; argv[i]=NULL;continue;}

    if(!strncmp(argv[i],"-xref",5))
      {
       char* p=&argv[i][5];

       if(!*p)
          option_xref=XREF_ALL;
       else
          while(*p)
            {
             if(!strncmp(p,"-all" ,4)) {option_xref|=XREF_ALL ; p=&p[4]; continue;}
             if(!strncmp(p,"-file",5)) {option_xref|=XREF_FILE; p=&p[5]; continue;}
             if(!strncmp(p,"-func",5)) {option_xref|=XREF_FUNC; p=&p[5]; continue;}
             if(!strncmp(p,"-var" ,4)) {option_xref|=XREF_VAR ; p=&p[4]; continue;}
             if(!strncmp(p,"-type",5)) {option_xref|=XREF_TYPE; p=&p[5]; continue;}
             break;
            }
       argv[i]=NULL;continue;
      }

    if(!strncmp(argv[i],"-warn",5))
      {
       char* p=&argv[i][5];

       if(!*p)
          option_warn=WARN_ALL;
       else
          while(*p)
            {
             if(!strncmp(p,"-all"    ,4)) {option_warn|=WARN_ALL    ; p=&p[4]; continue;}
             if(!strncmp(p,"-comment",8)) {option_warn|=WARN_COMMENT; p=&p[8]; continue;}
             if(!strncmp(p,"-xref"   ,5)) {option_warn|=WARN_XREF   ; p=&p[5]; continue;}
             break;
            }
       argv[i]=NULL;continue;
      }

    if(!strncmp(argv[i],"-index",6))
      {
       char* p=&argv[i][6];

       if(!*p)
          option_index=INDEX_ALL;
       else
          while(*p)
            {
             if(!strncmp(p,"-all" ,4)) {option_index|=INDEX_ALL ; p=&p[4]; continue;}
             if(!strncmp(p,"-file",5)) {option_index|=INDEX_FILE; p=&p[5]; continue;}
             if(!strncmp(p,"-func",5)) {option_index|=INDEX_FUNC; p=&p[5]; continue;}
             if(!strncmp(p,"-var" ,4)) {option_index|=INDEX_VAR ; p=&p[4]; continue;}
             if(!strncmp(p,"-type",5)) {option_index|=INDEX_TYPE; p=&p[5]; continue;}
             break;
            }
       argv[i]=NULL;continue;
      }

    if(!strcmp(argv[i],"-raw"))
      {option_raw=1; argv[i]=NULL;continue;}

    if(!strcmp(argv[i],"-latex"))
      {option_latex=1; argv[i]=NULL;continue;}
    if(!strcmp(argv[i],"-latex2e"))
      {option_latex=2; argv[i]=NULL;continue;}

    if(!strcmp(argv[i],"-html"))
      {option_html=1; argv[i]=NULL;continue;}

    if(argv[i][0]=='-')
      {fprintf(stderr,"cxref: Unknown option '%s'\n",argv[i]);exit(1);}

    if(!strncmp(argv[i],"./",2))
      {argv[i]+=2; continue;}
   }

 for(i=1;i<argc;i++)
    if(argv[i])
      {
       File file=NewFile(argv[i]);

       if(!DocumentTheFile(argv[i]))
         {
          if(option_xref)
             CrossReference(file);

          if(option_raw || option_warn)
             WriteWarnRawFile(file);
          if(option_latex)
             WriteLatexFile(file);
          if(option_html)
             WriteHTMLFile(file);
         }

       DeleteSpareTypes();
       DeleteSpareProtos();
       DeleteFile(file);

       TidyMemory();
      }

 Free(cpp_command);

 if(option_index)
   {
    StringList files,funcs,vars,types;

    InitStringList(&files);
    InitStringList(&funcs);
    InitStringList(&vars);
    InitStringList(&types);

    CreateAppendix(&files,&funcs,&vars,&types);

    if(option_raw||option_warn)
       WriteWarnRawAppendix(&files,&funcs,&vars,&types);
    if(option_latex)
       WriteLatexAppendix(&files,&funcs,&vars,&types);
    if(option_html)
       WriteHTMLAppendix(&files,&funcs,&vars,&types);
   }

 PrintMemoryStatistics();

 return(0);
}


/*++++++++++++++++++++++++++++++++++++++
  Calls CPP for the file to get all of the needed information.

  int DocumentTheFile Returns 1 in case of error, else 0.

  char* name The name of the file to document.

  The CPP is started as a sub-process, (using popen to return a FILE* for lex to use).
  ++++++++++++++++++++++++++++++++++++++*/

static int DocumentTheFile(char* name)
{
 char* command;
 struct stat stat_buf;
 int error;

 if(stat(name,&stat_buf)==-1)
   {fprintf(stderr,"cxref: Cannot access the file '%s'\n",name);return(1);}

 command=(char*)Malloc(cpp_command_len+2+strlen(name));

 sprintf(command,"%s %s",cpp_command,name);

 yyin=popen(command,"r");

 if(!yyin)
   {fprintf(stderr,"cxref: Failed to start the cpp command '%s'\n",command);exit(1);}

 yyrestart(yyin);

#if YYDEBUG
#if YYDEBUG == 3
 yydebug=1;
#else
 yydebug=0;
#endif
#endif

 error=yyparse();

 pclose(yyin);

 Free(command);

 return(error);
}
