/***************************************
  $Header: /home/amb/cxref/query/RCS/input.c 1.4 1996/07/06 19:17:03 amb Exp $

  C Cross Referencing & Documentation tool. Version 1.2
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1995,96 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../memory.h"
#include "../datatype.h"
#include "../cxref.h"
#include "query.h"

/*+ The names of the function cross reference files. +*/
#define XREF_FUNC_FILE   ".function"

/*+ The names of the variable cross reference files. +*/
#define XREF_VAR_FILE    ".variable"

/*+ The names of the include cross reference files. +*/
#define XREF_INC_FILE    ".include"

/*+ The names of the type cross reference files. +*/
#define XREF_TYPE_FILE   ".typedef"

/*+ The command line switch that sets the amount of cross referencing to do. +*/
extern int option_xref;

/*+ The command line switch for the output name, +*/
extern char *option_odir,       /*+ The directory to use. +*/
            *option_name;       /*+ The base part of the name. +*/

extern File *files;             /*+ The files that are queried. +*/
extern int n_files;             /*+ The number of files referenced. +*/

extern Function *functions;     /*+ The functions that are queried. +*/
extern int n_functions;         /*+ The number of functions referenced. +*/

extern Variable *variables;     /*+ The variables that are queried. +*/
extern int n_variables;         /*+ The number of variables referenced. +*/

extern Typedef *typedefs;       /*+ The type definitions that are queried. +*/
extern int n_typedefs;          /*+ The number of typedefs referenced. +*/

/* Local functions */

static void cross_reference_files(void);
static void cross_reference_functions(void);
static void cross_reference_variables(void);

/*++++++++++++++++++++++++++++++++++++++
  Read in all the information from the cross reference files.
  ++++++++++++++++++++++++++++++++++++++*/

void LoadInCrossRefs(void)
{
 FILE *in;
 char *ifile;

 /* Format: filename [[%]include1] [[%]include2] ... : Files include1, include2, ... are included in filename;
    those with a % are local. */

 /* First do the files */
   {
    char include[128],filename[128],ch;

    ifile=ConcatStrings(4,option_odir,"/",option_name,XREF_INC_FILE);

    in =fopen(ifile,"r");

    if(!in)
      {fprintf(stderr,"cxref-query: Failed to open the include cross reference file '%s'\n",ifile);exit(1);}

    while(fscanf(in,"%s%c",filename,&ch)==2)
      {
       if(n_files)
          files=(File*)Realloc(files,(n_files+1)*sizeof(File*));
       else
          files=(File*)Malloc(sizeof(File*));

       files[n_files]=(File)Calloc(1,sizeof(struct _File));
       files[n_files]->name=MallocString(filename);

       while(ch==' ')
         {
          Include inc=(Include)Calloc(1,sizeof(struct _Include));

          fscanf(in,"%s%c",include,&ch);

          if(include[0]=='%')
             {inc->scope=LOCAL;
              inc->name=MallocString(&include[1]);}
          else
             {inc->scope=GLOBAL;
              inc->name=MallocString(include);}

          AddToLinkedList(files[n_files]->includes,Include,inc);
         }
       n_files++;
      }

    cross_reference_files();

    fclose(in);
   }

 /* Format: filename funcname scope [[%][&]funcname1] [[%][&]funcname2] ... : The function funcname in file filename
    calls or references functions funcname1, funcname2 ... ; those with a % are local, with a & are references. */
 /* Format: filename $ 0 [[%]&funcname1] [[%]&funcname2] ... : The file references functions funcname1, funcname2 ... ;
    those with a % are local.  */

 /* Now do the functions */
   {
    char ch,funcname[64],filename[128],called[64];
    int scope;

    ifile=ConcatStrings(4,option_odir,"/",option_name,XREF_FUNC_FILE);

    in =fopen(ifile,"r");

    if(!in)
      {fprintf(stderr,"cxref-query: Failed to open the functional cross reference file '%s'\n",ifile);exit(1);}

    while(fscanf(in,"%s %s %d%c",filename,funcname,&scope,&ch)==4)
      {
       int i;
       StringList *f_refs;

       for(i=0;i<n_files;i++)
          if(!strcmp(files[i]->name,filename))
             break;

       if(funcname[0]=='$')
          f_refs=&files[i]->f_refs;
       else
         {
          if(n_functions)
             functions=(Function*)Realloc(functions,(n_functions+1)*sizeof(Function*));
          else
             functions=(Function*)Malloc(sizeof(Function*));

          functions[n_functions]=(Function)Calloc(1,sizeof(struct _Function));

          AddToLinkedList(files[i]->functions,Function,functions[n_functions]);

          functions[n_functions]->comment=MallocString(filename); /* Use comment field for filename */
          functions[n_functions]->name=MallocString(funcname);
          functions[n_functions]->scope=scope;

          f_refs=&functions[n_functions]->f_refs;
         }

       while(ch==' ')
         {
          char* c;
          fscanf(in,"%s%c",called,&ch);

          c=called;
          if(c[0]=='%') c++;
          if(c[0]=='&')
            {
             if(c==called)
                AddToStringList(f_refs,c+1,1);
             else
                AddToStringList(f_refs,ConcatStrings(3,c+1," : ",filename),1);
            }
          else
            {
             if(c==called)
                AddToStringList(&functions[n_functions]->calls,c,1);
             else
                AddToStringList(&functions[n_functions]->calls,ConcatStrings(3,c," : ",filename),1);
            }
         }
       n_functions++;
      }

    cross_reference_functions();

    fclose(in);
   }

 /* Format: filename varname scope [$] [[%]funcname1] [[%]funcname2] ... : variable varname is used in
    the file filename if $, and functions funcname1, funcname2 ... Those with a % are local.  */

 /* Now do the variables */
   {
    char varname[64],filename[128],funcname[64],ch;
    int scope;

    ifile=ConcatStrings(4,option_odir,"/",option_name,XREF_VAR_FILE);

    in =fopen(ifile,"r");

    if(!in)
      {fprintf(stderr,"cxref-query: Failed to open the variable cross reference file '%s'\n",ifile);exit(1);}

    while(fscanf(in,"%s %s %d%c",filename,varname,&scope,&ch)==4)
      {
       int i;

       if(n_variables)
          variables=(Variable*)Realloc(variables,(n_variables+1)*sizeof(Variable*));
       else
          variables=(Variable*)Malloc(sizeof(Variable*));

       variables[n_variables]=(Variable)Calloc(1,sizeof(struct _Variable));

       for(i=0;i<n_files;i++)
          if(!strcmp(files[i]->name,filename))
             AddToLinkedList(files[i]->variables,Variable,variables[n_variables]);

       variables[n_variables]->comment=MallocString(filename); /* Use comment field for filename */
       variables[n_variables]->name=MallocString(varname);
       variables[n_variables]->scope=scope;

       while(ch==' ')
         {
          fscanf(in,"%s%c",funcname,&ch);

          if(funcname[0]=='$')
             AddToStringList(&variables[n_variables]->used,ConcatStrings(2,"$",filename),1);
          else
             if(funcname[0]=='%')
                AddToStringList(&variables[n_variables]->used,ConcatStrings(3,&funcname[1]," : ",filename),1);
             else
                AddToStringList(&variables[n_variables]->used,funcname,1);
         }
       n_variables++;
      }

    cross_reference_variables();

    fclose(in);
   }

 /* Format: filename typename type... : For a typedef type.     */
 /* Format: filename #        type... : For a non typedef type. */

 /* Now do the types */
   {
    char typename[128],filename[128],typetype[256];

    ifile=ConcatStrings(4,option_odir,"/",option_name,XREF_TYPE_FILE);

    in =fopen(ifile,"r");

    if(!in)
      {fprintf(stderr,"cxref-query: Failed to open the typedef reference file '%s'\n",ifile);exit(1);}

    while(fscanf(in,"%s %s",filename,typename)==2)
      {
       int i;

       fgets(typetype,256,in);
       typetype[strlen(typetype)-1]=0;

       if(n_typedefs)
          typedefs=(Typedef*)Realloc(typedefs,(n_typedefs+1)*sizeof(Typedef*));
       else
          typedefs=(Typedef*)Malloc(sizeof(Typedef*));

       typedefs[n_typedefs]=(Typedef)Calloc(1,sizeof(struct _Typedef));

       for(i=0;i<n_files;i++)
          if(!strcmp(files[i]->name,filename))
             AddToLinkedList(files[i]->typedefs,Typedef,typedefs[n_typedefs]);

       typedefs[n_typedefs]->comment=MallocString(filename); /* Use comment field for filename */

       if(typename[0]!='#')
         {
          typedefs[n_typedefs]->name=MallocString(typename);
          typedefs[n_typedefs]->type=MallocString(&typetype[1]);
         }
       else
         {
          typedefs[n_typedefs]->name=MallocString(&typetype[1]);
          typedefs[n_typedefs]->type=NULL;
         }

       n_typedefs++;
      }

    fclose(in);
   }

}


/*++++++++++++++++++++++++++++++++++++++
  Performs all of the cross referencing between files, includes and included in.
  ++++++++++++++++++++++++++++++++++++++*/

static void cross_reference_files(void)
{
 int i;

 for(i=0;i<n_files;i++)
   {
    int j;
    Include inc=files[i]->includes;

    while(inc)
      {
       for(j=0;j<n_files;j++)
          if(!strcmp(inc->name,files[j]->name))
            {
             inc->includes=files[j]->includes;
             AddToStringList(&files[j]->inc_in,MallocString(files[i]->name),1);
            }
       inc=inc->next;
      }
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Performs all of the cross referencing between global functions and functions that they call.
  ++++++++++++++++++++++++++++++++++++++*/

static void cross_reference_functions(void)
{
 int i1,j1,i2;
 char* colon;

 for(i1=0;i1<n_functions;i1++)
   {
    Function func1=functions[i1];

    for(j1=0;j1<func1->calls.n;j1++)
      {
       if(!strchr(func1->calls.s[j1],':'))
          for(i2=0;i2<n_functions;i2++)
            {
             Function func2=functions[i2];

             if(!strcmp(func1->calls.s[j1],func2->name))
               {
                char* old=func1->calls.s[j1];
                func1->calls.s[j1]=ConcatStrings(3,func1->calls.s[j1]," : ",func2->comment);
                Free(old);
                break;
               }
            }

       if((colon=strchr(func1->calls.s[j1],':')))
         {
          char* func=func1->calls.s[j1];
          char* file=&colon[2];
          colon[-1]=0;

          for(i2=0;i2<n_functions;i2++)
            {
             Function func2=functions[i2];

             if(!strcmp(func,func2->name) && !strcmp(file,func2->comment))
               {
                AddToStringList(&func2->called,ConcatStrings(3,func1->name," : ",func1->comment),1);
                break;
               }
            }
          colon[-1]=' ';
         }
      }
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Performs all of the cross referencing between global variables and functions that use them.
  ++++++++++++++++++++++++++++++++++++++*/

static void cross_reference_variables(void)
{
 int i1,j1,i2;
 char* colon;

 for(i1=0;i1<n_variables;i1++)
   {
    Variable var1=variables[i1];

    for(j1=0;j1<var1->used.n;j1++)
      {
       if(var1->used.s[j1][0]!='$' && !strchr(var1->used.s[j1],':'))
          for(i2=0;i2<n_functions;i2++)
            {
             Function func2=functions[i2];

             if(!strcmp(var1->used.s[j1],func2->name))
               {
                char* old=var1->used.s[j1];
                var1->used.s[j1]=ConcatStrings(3,var1->used.s[j1]," : ",func2->comment);
                Free(old);
                break;
               }
            }

       if((colon=strchr(var1->used.s[j1],':')))
         {
          char* func=var1->used.s[j1];
          char* file=&colon[2];
          colon[-1]=0;

          for(i2=0;i2<n_functions;i2++)
            {
             Function func2=functions[i2];

             if(!strcmp(func,func2->name) && !strcmp(file,func2->comment))
               {
                AddToStringList(&func2->v_refs,ConcatStrings(3,var1->name," : ",var1->comment),1);
                break;
               }
            }
          colon[-1]=' ';
         }
      }
   }
}
