/***************************************
  $Header: /home/amb/cxref/RCS/html.c 1.11 1996/07/06 19:05:47 amb Exp $

  C Cross Referencing & Documentation tool. Version 1.2.

  Writes the HTML output.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1995,96 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "memory.h"
#include "datatype.h"
#include "cxref.h"

/*+ The name of the output tex file that includes each of the others. +*/
#define HTML_FILE        ".html"
#define HTML_FILE_BACKUP ".html~"

/*+ The name of the output tex file that contains the appendix. +*/
#define HTML_APDX        ".apdx"

/*+ The name of the directory for the output. +*/
extern char* option_odir;

/*+ The base name of the file for the output. +*/
extern char* option_name;

/*+ The directories to go back to get to the base output directory. +*/
static char* goback=NULL;

static void WriteHTMLFilePart(File file);
static void WriteHTMLInclude(Include inc);
static void WriteHTMLSubInclude(Include inc,int depth);
static void WriteHTMLDefine(Define def);
static void WriteHTMLTypedef(Typedef type);
static void WriteHTMLStructUnion(StructUnion su,int depth);
static void WriteHTMLVariable(Variable var);
static void WriteHTMLFunction(Function func);

static void WriteHTMLDocument(char* name,int appendix);
static void WriteHTMLPreamble(FILE* f,char* title,int sourcefile);
static void WriteHTMLPostamble(FILE* f,int sourcefile);

static char* html(char* c);

/*+ The output file for the HTML. +*/
static FILE* of;

/*++++++++++++++++++++++++++++++++++++++
  Write an html file for a complete File structure and all components.

  File file The File structure to output.
  ++++++++++++++++++++++++++++++++++++++*/

void WriteHTMLFile(File file)
{
 char* ofile;
 int i;

 /* Write the including file. */

 WriteHTMLDocument(file->name,0);

 /* Open the file */

 ofile=ConcatStrings(4,option_odir,"/",file->name,HTML_FILE);

 of=fopen(ofile,"w");
 if(!of)
   {
    struct stat stat_buf;
    int i,ofl=strlen(ofile);

    for(i=strlen(option_odir)+1;i<ofl;i++)
       if(ofile[i]=='/')
         {
          ofile[i]=0;
          printf("ofile=%s\n",ofile);
          if(stat(ofile,&stat_buf))
             mkdir(ofile,S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
          ofile[i]='/';
         }

    of=fopen(ofile,"w");
   }

 if(!of)
   {fprintf(stderr,"cxref: Failed to open the HTML output file '%s'\n",ofile);exit(1);}

 for(goback="",i=strlen(file->name);i>0;i--)
    if(file->name[i]=='/')
       goback=ConcatStrings(2,goback,"../");

 /* Write out a header. */

 WriteHTMLPreamble(of,ConcatStrings(5,"Cross reference for ",file->name," of ",option_name,"."),0);

 /*+ The file structure is broken into its components and they are each written out. +*/

 WriteHTMLFilePart(file);

 if(file->includes)
   {
    Include inc =file->includes;
    fprintf(of,"<hr>\n<h2>Included Files</h2>\n\n");
    do{
       WriteHTMLInclude(inc);
      }
    while((inc=inc->next));
   }

 if(file->defines)
   {
    Define def =file->defines;
    fprintf(of,"<hr>\n<h2>Preprocessor definitions</h2>\n\n");
    do{
       if(def!=file->defines)
          fprintf(of,"<p>\n");
       WriteHTMLDefine(def);
      }
    while((def=def->next));
   }

 if(file->typedefs)
   {
    Typedef type=file->typedefs;
    do{
       WriteHTMLTypedef(type);
      }
    while((type=type->next));
   }

 if(file->variables)
   {
    int any_to_mention=0;
    Variable var=file->variables;

    do{
       if(var->scope&(GLOBAL|LOCAL|EXTERNAL|EXTERN_F))
          any_to_mention=1;
      }
    while((var=var->next));

    if(any_to_mention)
      {
       int first_ext=1,first_local=1;
       Variable var=file->variables;
       do{
          if(var->scope&GLOBAL)
             WriteHTMLVariable(var);
         }
       while((var=var->next));
       var=file->variables;
       do{
          if(var->scope&(EXTERNAL|EXTERN_F) && !(var->scope&GLOBAL))
            {
             if(first_ext)
               {fprintf(of,"<hr>\n<h2>External Variables</h2>\n\n"); first_ext=0;}
             else
                fprintf(of,"<p>\n");
             WriteHTMLVariable(var);
            }
         }
       while((var=var->next));
       var=file->variables;
       do{
          if(var->scope&LOCAL)
            {
             if(first_local)
               {fprintf(of,"<hr>\n<h2>Local Variables</h2>\n\n"); first_local=0;}
             else
                fprintf(of,"<p>\n");
             WriteHTMLVariable(var);
            }
         }
       while((var=var->next));
      }
   }

 if(file->functions)
   {
    Function func=file->functions;
    do{
       if(func->scope&GLOBAL)
          WriteHTMLFunction(func);
      }
    while((func=func->next));
    func=file->functions;
    do{
       if(func->scope&LOCAL)
          WriteHTMLFunction(func);
      }
    while((func=func->next));
   }

 WriteHTMLPostamble(of,0);

 fclose(of);

/* Clear the memory in html() */

 html(NULL); html(NULL); html(NULL); html(NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Write a File structure out.

  File file The File to output.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLFilePart(File file)
{
 int i;

 fprintf(of,"<a name=\"file\">\n<h1>File %s</h1>\n</a>\n",html(file->name));

 if(file->comment)
   {
    char *rcs1=strstr(file->comment,"$Header"),*rcs2=NULL;
    if(rcs1)
      {
       rcs2=strstr(&rcs1[1],"$");
       if(rcs2)
         {
          rcs2[0]=0;
          fprintf(of,"<b>RCS %s</b>\n<p>\n",html(&rcs1[1]));
          rcs2[0]='$';
         }
      }
    if(rcs2)
       fprintf(of,"%s\n<p>\n",html(&rcs2[2]));
    else
       fprintf(of,"%s\n<p>\n",html(file->comment));
   }

 if(file->inc_in.n)
   {
    int i;

    fprintf(of,"<dl compact>\n<dt>Included in:\n<dd><menu>\n");
    for(i=0;i<file->inc_in.n;i++)
       fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\">%s</a><br>\n",goback,file->inc_in.s[i],html(file->inc_in.s[i]));
    fprintf(of,"</menu></dl>\n");
   }

 if(file->f_refs.n || file->v_refs.n)
    fprintf(of,"<dl compact>\n");

 if(file->f_refs.n)
   {
    int others=0;
    fprintf(of,"<dt>References Functions:\n<dd><menu>\n");
    for(i=0;i<file->f_refs.n;i++)
      {
       char *temp=strchr(file->f_refs.s[i],':'),*filen;
       if(temp)
         {
          temp[-1]=0;
          filen=&temp[2];
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,filen,file->f_refs.s[i],html(file->f_refs.s[i]),html(filen));
          temp[-1]=' ';
         }
       else
          others++;
      }

    if(others)
      {
       fprintf(of,"<li>");
       for(i=0;i<file->f_refs.n;i++)
          if(!strchr(file->f_refs.s[i],':'))
             fprintf(of,--others?" %s(),":" %s()",html(file->f_refs.s[i]));
       fprintf(of,"\n");
      }
    fprintf(of,"</menu>\n");
   }

 if(file->v_refs.n)
   {
    int others=0;
    fprintf(of,"<dt>References Variables:\n<dd><menu>\n");
    for(i=0;i<file->v_refs.n;i++)
      {
       char *temp=strchr(file->v_refs.s[i],':'),*filen;
       if(temp)
         {
          temp[-1]=0;
          filen=&temp[2];
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#var_%s\">%s  :  %s</a>\n",goback,filen,file->v_refs.s[i],html(file->v_refs.s[i]),html(filen));
          temp[-1]=' ';
         }
       else
          others++;
      }

    if(others)
      {
       fprintf(of,"<li>");
       for(i=0;i<file->v_refs.n;i++)
          if(!strchr(file->v_refs.s[i],':'))
             fprintf(of,--others?" %s,":" %s",html(file->v_refs.s[i]));
       fprintf(of,"\n");
      }
    fprintf(of,"</menu>\n");
   }

 if(file->f_refs.n || file->v_refs.n)
    fprintf(of,"</dl>\n");
}


/*++++++++++++++++++++++++++++++++++++++
  Write an Include structure out.

  Include inc The Include structure to output.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLInclude(Include inc)
{
 if(inc->comment)
    fprintf(of,"%s\n<p>\n",html(inc->comment));

 fprintf(of,"<menu>\n");

 if(inc->scope==LOCAL)
    fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\"><tt>#include &quot;%s&quot;</tt></a>\n",goback,inc->name,html(inc->name));
 else
    fprintf(of,"<li><tt>#include &lt;%s&gt;</tt>\n",html(inc->name));

 if(inc->includes)
    WriteHTMLSubInclude(inc->includes,1);

 fprintf(of,"</menu>\n");
}


/*++++++++++++++++++++++++++++++++++++++
  Write an Sub Include structure out. (An include structure that is included from another file.)

  Include inc The Include structure to output.

  int depth The depth of the include hierarchy.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLSubInclude(Include inc,int depth)
{
 fprintf(of,"<menu>\n");

 while(inc)
   {
    if(inc->scope==LOCAL)
       fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\"><tt>#include &quot;%s&quot;</tt></a>\n",goback,inc->name,html(inc->name));
    else
       fprintf(of,"<li><tt>#include &lt;%s&gt;</tt>\n",html(inc->name));

    if(inc->includes)
       WriteHTMLSubInclude(inc->includes,depth+1);

    inc=inc->next;
   }

 fprintf(of,"</menu>\n");
}


/*++++++++++++++++++++++++++++++++++++++
  Write a Define structure out.

  Define def The Define structure to output.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLDefine(Define def)
{
 int i;
 int pargs=0;

 if(def->comment)
    fprintf(of,"%s\n<p>\n",html(def->comment));

 fprintf(of,"<tt>#define %s",html(def->name));

 if(def->value)
    fprintf(of," %s",html(def->value));

 if(def->args.n)
   {
    fprintf(of,"( ");
    for(i=0;i<def->args.n;i++)
       fprintf(of,i?", %s":"%s",html(def->args.s1[i]));
    fprintf(of," )");
   }
 fprintf(of,"</tt><br>\n");

 for(i=0;i<def->args.n;i++)
    if(def->args.s2[i])
       pargs=1;

 if(pargs)
   {
    fprintf(of,"<dl compact>\n");
    for(i=0;i<def->args.n;i++)
       fprintf(of,"<dt><tt>%s</tt>\n<dd>%s\n",html(def->args.s1[i]),def->args.s2[i]?html(def->args.s2[i]):"");
    fprintf(of,"</dl>\n");
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Write a Typedef structure out.

  Typedef type The Typedef structure to output.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLTypedef(Typedef type)
{
 if(!strncmp("enum",type->name,4))
    fprintf(of,"\n<a name=\"type_enum_%s\">\n",&type->name[5]);
 else
    if(!strncmp("union",type->name,5))
       fprintf(of,"\n<a name=\"type_union_%s\">\n",&type->name[6]);
    else
       if(!strncmp("struct",type->name,6))
          fprintf(of,"\n<a name=\"type_struct_%s\">\n",&type->name[7]);
       else
          fprintf(of,"\n<a name=\"type_%s\">\n",type->name);

 if(type->type)
    fprintf(of,"<hr>\n<h2>Typedef %s</h2>\n</a>\n",html(type->name));
 else
    fprintf(of,"<hr>\n<h2>Type %s</h2>\n</a>\n",html(type->name));

 if(type->comment)
    fprintf(of,"%s\n<p>\n",html(type->comment));

 if(type->type)
    fprintf(of,"<tt>typedef %s</tt><br>\n",html(type->type));

 if(type->sutype)
   {
    fprintf(of,"<menu>\n");
    WriteHTMLStructUnion(type->sutype,0);
    fprintf(of,"</menu>\n");
   }
 else
    if(type->typexref)
      {
       fprintf(of,"<dl compact>\n<dt>See:\n<dd><menu>\n");
       if(type->typexref->type)
          fprintf(of,"<li><a href=\"#type_%s\">Typedef %s</a>\n",type->typexref->name,html(type->typexref->name));
       else
          if(!strncmp("enum",type->typexref->name,4))
             fprintf(of,"<li><a href=\"#type_enum_%s\">Type %s</a>\n",&type->typexref->name[5],html(type->typexref->name));
          else
             if(!strncmp("union",type->typexref->name,5))
                fprintf(of,"<li><a href=\"#type_union_%s\">Type %s</a>\n",&type->typexref->name[6],html(type->typexref->name));
             else
                if(!strncmp("struct",type->typexref->name,6))
                   fprintf(of,"<li><a href=\"#type_struct_%s\">Type %s</a>\n",&type->typexref->name[7],html(type->typexref->name));
       fprintf(of,"</menu></dl>\n");
      }
}


/*++++++++++++++++++++++++++++++++++++++
  Write a structure / union structure out.

  StructUnion su The structure / union to write.

  int depth The current depth within the structure.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLStructUnion(StructUnion su, int depth)
{
 int i;
 char* splitsu=NULL;

 splitsu=strstr(su->name,"{...}");
 if(splitsu) splitsu[-1]=0;

 if(depth && su->comment)
    fprintf(of,"<tt>%s      </tt>%s<br>\n",html(su->name),html(su->comment));
 else
    fprintf(of,"<tt>%s</tt><br>\n",html(su->name));

 if(su->comps)
   {
    fprintf(of,"<menu>\n");

    fprintf(of,"<tt>{</tt><br>\n");

    for(i=0;i<su->n_comp;i++)
       WriteHTMLStructUnion(su->comps[i],depth+1);

    fprintf(of,"<tt>}</tt><br>\n");

    fprintf(of,"</menu>\n");

    if(splitsu)
       fprintf(of,"<tt>%s</tt><br>\n",html(&splitsu[6]));
   }

 if(splitsu) splitsu[-1]=' ';
}


/*++++++++++++++++++++++++++++++++++++++
  Write a Variable structure out.

  Variable var The Variable structure to output.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLVariable(Variable var)
{
 int i;

 if(var->scope&GLOBAL)
    fprintf(of,"<hr>\n<a name=\"var_%s\">\n<h2>Global Variable %s</h2>\n</a>\n",var->name,html(var->name));
 else
    fprintf(of,"<a name=\"var_%s\">\n<b>%s</b>\n</a><br>\n",var->name,html(var->name));

 if(var->comment)
    fprintf(of,"%s\n<p>\n",html(var->comment));

 fprintf(of,"<tt>");

 if(var->scope&LOCAL)
    fprintf(of,"static ");
 else
    if(!(var->scope&GLOBAL) && var->scope&(EXTERNAL|EXTERN_F))
       fprintf(of,"extern ");

 fprintf(of,"%s</tt><br>\n",html(var->type));

 if(var->scope&(GLOBAL|LOCAL))
   {
    if(var->visible.n || var->used.n)
       fprintf(of,"<dl compact>\n");

    if(var->visible.n)
      {
       fprintf(of,"<dt>Visible in:\n<dd><menu>\n");
       for(i=0;i<var->visible.n;i++)
          if(var->visible.s[i][0]=='$')
             fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\">%s</a>\n",goback,&var->visible.s[i][1],html(&var->visible.s[i][1]));
          else
            {
             char *temp=strchr(var->visible.s[i],':'),*file;
             temp[-1]=0;
             file=&temp[2];
             fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,file,var->visible.s[i],html(var->visible.s[i]),html(file));
             temp[-1]=' ';
            }
       fprintf(of,"</menu>\n");
      }

    if(var->used.n)
      {
       fprintf(of,"<dt>Used in:\n<dd><menu>\n");
       for(i=0;i<var->used.n;i++)
         {
          if(var->used.s[i][0]=='$')
             fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\">%s</a>\n",goback,&var->used.s[i][1],html(&var->used.s[i][1]));
          else
            {
             char *temp=strchr(var->used.s[i],':'),*file;
             temp[-1]=0;
             file=&temp[2];
             if(var->scope&LOCAL)
                fprintf(of,"<li><a href=\"#func_%s\">%s()</a>\n",var->used.s[i],html(var->used.s[i]));
             else
                fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,file,var->used.s[i],html(var->used.s[i]),html(file));
             temp[-1]=' ';
            }
         }
       fprintf(of,"</menu>\n");
      }

    if(var->visible.n || var->used.n)
       fprintf(of,"</dl>\n");
   }
 else
    if(var->scope&(EXTERNAL|EXTERN_F) && var->defined)
      {
       fprintf(of,"<dl compact>\n<dt>Defined in:\n<dd><menu>\n");
       fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#var_%s\">%s</a>",goback,var->defined,html(var->name),var->defined);
       fprintf(of,"</menu></dl>\n");
      }
}


/*++++++++++++++++++++++++++++++++++++++
  Write a Function structure out.

  Function func The Function structure to output.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLFunction(Function func)
{
 int i,pret,pargs;
 char* comment2=NULL,*type;

 if(func->scope&GLOBAL)
    fprintf(of,"<hr>\n<a name=\"func_%s\">\n<h2>Global Function %s()</h2>\n</a>\n",func->name,html(func->name));
 else
    fprintf(of,"<hr>\n<a name=\"func_%s\">\n<h2>Local Function %s()</h2>\n</a>\n",func->name,html(func->name));

 if(func->comment)
   {
    comment2=strstr(func->comment,"\n\n");
    if(comment2)
       comment2[0]=0;
    fprintf(of,"%s\n<p>\n",html(func->comment));
   }

 fprintf(of,"<tt>");

 if(func->scope&LOCAL)
    fprintf(of,"static ");
 if(func->scope&INLINED)
   fprintf(of,"inline ");

 if((type=strstr(func->type,"()")))
    type[0]=0;
 fprintf(of,"%s ( ",html(func->type));

 for(i=0;i<func->args.n;i++)
    fprintf(of,i?", %s":"%s",html(func->args.s1[i]));

 if(type)
   {fprintf(of," %s</tt><br>\n",html(&type[1]));type[0]='(';}
 else
    fprintf(of," )</tt><br>\n");

 pret =strncmp("void ",func->type,5) && func->cret;
 for(pargs=0,i=0;i<func->args.n;i++)
    pargs = pargs || ( strcmp("void",func->args.s1[i]) && func->args.s2[i] );

 if(pret || pargs)
   {
    fprintf(of,"<dl compact>\n");
    if(pret)
       fprintf(of,"<dt><tt>%s</tt>\n<dd>%s\n",html(func->type),func->cret?html(func->cret):"");
    if(pargs)
       for(i=0;i<func->args.n;i++)
          fprintf(of,"<dt><tt>%s</tt>\n<dd>%s\n",html(func->args.s1[i]),func->args.s2[i]?html(func->args.s2[i]):"");
    fprintf(of,"</dl>\n");
   }

 if(comment2)
   {
    fprintf(of,"%s\n<p>\n",html(&comment2[2]));
    comment2[0]='\n';
   }

 if(func->protofile || func->calls.n || func->called.n || func->used.n || func->f_refs.n || func->v_refs.n)
    fprintf(of,"<dl compact>\n");

 if(func->protofile)
   {
    fprintf(of,"<dt>Prototyped in:\n<dd><menu>\n");
    fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\">%s</a>\n",goback,func->protofile,html(func->protofile));
    fprintf(of,"</menu>\n");
   }

 if(func->calls.n)
   {
    int others=0;
    fprintf(of,"<dt>Calls:\n<dd><menu>\n");
    for(i=0;i<func->calls.n;i++)
      {
       char *temp=strchr(func->calls.s[i],':'),*file;
       if(temp)
         {
          temp[-1]=0;
          file=&temp[2];
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,file,func->calls.s[i],html(func->calls.s[i]),html(file));
          temp[-1]=' ';
         }
       else
          others++;
      }

    if(others)
      {
       fprintf(of,"<li>");
       for(i=0;i<func->calls.n;i++)
          if(!strchr(func->calls.s[i],':'))
             fprintf(of,--others?" %s(),":" %s()",html(func->calls.s[i]));
       fprintf(of,"\n");
      }
    fprintf(of,"</menu>\n");
   }

 if(func->called.n)
   {
    fprintf(of,"<dt>Called by:\n<dd><menu>\n");
    for(i=0;i<func->called.n;i++)
      {
       char *temp=strchr(func->called.s[i],':'),*file;
       temp[-1]=0;
       file=&temp[2];
       fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,file,func->called.s[i],html(func->called.s[i]),html(file));
       temp[-1]=' ';
      }
    fprintf(of,"</menu>\n");
   }

 if(func->used.n)
   {
    fprintf(of,"<dt>Used in:\n<dd><menu>\n");
    for(i=0;i<func->used.n;i++)
      {
       if(func->used.s[i][0]=='$')
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#file\">%s</a>\n",goback,&func->used.s[i][1],html(&func->used.s[i][1]));
       else
         {
          char *temp=strchr(func->used.s[i],':'),*file;
          temp[-1]=0;
          file=&temp[2];
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,file,func->used.s[i],html(func->used.s[i]),html(file));
          temp[-1]=' ';
         }
      }
    fprintf(of,"</menu>\n");
   }

 if(func->f_refs.n)
   {
    int others=0;
    fprintf(of,"<dt>References Functions:\n<dd><menu>\n");
    for(i=0;i<func->f_refs.n;i++)
      {
       char *temp=strchr(func->f_refs.s[i],':'),*file;
       if(temp)
         {
          temp[-1]=0;
          file=&temp[2];
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",goback,file,func->f_refs.s[i],html(func->f_refs.s[i]),html(file));
          temp[-1]=' ';
         }
       else
          others++;
      }

    if(others)
      {
       fprintf(of,"<li>");
       for(i=0;i<func->f_refs.n;i++)
          if(!strchr(func->f_refs.s[i],':'))
             fprintf(of,--others?" %s(),":" %s()",html(func->f_refs.s[i]));
       fprintf(of,"\n");
      }
    fprintf(of,"</menu>\n");
   }

 if(func->v_refs.n)
   {
    int others=0;
    fprintf(of,"<dt>References Variables:\n<dd><menu>\n");
    for(i=0;i<func->v_refs.n;i++)
      {
       char *temp=strchr(func->v_refs.s[i],':'),*file;
       if(temp)
         {
          temp[-1]=0;
          file=&temp[2];
          fprintf(of,"<li><a href=\"%s%s"HTML_FILE"#var_%s\">%s  :  %s</a>\n",goback,file,func->v_refs.s[i],html(func->v_refs.s[i]),html(file));
          temp[-1]=' ';
         }
       else
          others++;
      }

    if(others)
      {
       fprintf(of,"<li>");
       for(i=0;i<func->v_refs.n;i++)
          if(!strchr(func->v_refs.s[i],':'))
             fprintf(of,--others?" %s,":" %s",html(func->v_refs.s[i]));
       fprintf(of,"\n");
      }
    fprintf(of,"</menu>\n");
   }

 if(func->protofile || func->calls.n || func->called.n || func->used.n || func->f_refs.n || func->v_refs.n)
    fprintf(of,"</dl>\n");
}


/*++++++++++++++++++++++++++++++++++++++
  Write out a file that will include the current information.

  char* name The name of the file.

  int appendix set to non-zero if the appendix file is to be added, else a normal source file.  
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLDocument(char* name,int appendix)
{
 FILE *in,*out;
 char line[256];
 int seen=0;
 char *inc_file,*ofile,*ifile;

 if(appendix)
    inc_file="<!-- Appendix -->\n";
 else
    inc_file=ConcatStrings(6,"<a href=\"",name,HTML_FILE,"#file\">",name,"</a><br>\n");
 ifile=ConcatStrings(4,option_odir,"/",option_name,HTML_FILE);
 ofile=ConcatStrings(4,option_odir,"/",option_name,HTML_FILE_BACKUP);

 in =fopen(ifile,"r");
 if(!in)
   {
    in =fopen(ifile,"w");
    if(!in)
      {fprintf(stderr,"cxref: Failed to open the main HTML output file '%s'\n",ifile);exit(1);}

    WriteHTMLPreamble(in,ConcatStrings(3,"Cross reference of ",option_name,"."),1);
    WriteHTMLPostamble(in,1);
    fclose(in);

    in =fopen(ifile,"r");
   }

 out=fopen(ofile,"w");

 if(!out)
   {fprintf(stderr,"cxref: Failed to open the main HTML output file '%s'\n",ofile);exit(1);}

 while(fgets(line,256,in))
   {
    if(!strcmp(inc_file,line))
      {seen=1;break;}
    if(line[0]=='<' && (!strcmp("<!-- Appendix -->\n",line) || !strcmp("<!-- End-Of-Source-Files -->\n",line)))
       if(appendix)
         {
          fputs("<!-- Appendix -->\n",out);
          fputs("\n",out);
          fputs("<hr>\n",out);
          fputs("<h1>Appendix</h1>\n",out);
          fputs("\n",out);
          fprintf(out,"<a href=\"%s"HTML_FILE"#files\">Files</a><br>\n",name);
          fprintf(out,"<a href=\"%s"HTML_FILE"#functions\">Global Functions</a><br>\n",name);
          fprintf(out,"<a href=\"%s"HTML_FILE"#variables\">Global Variables</a><br>\n",name);
          fprintf(out,"<a href=\"%s"HTML_FILE"#types\">Defined Types</a><br>\n",name);
          fputs("\n",out);
         }
       else
         {
          fputs(inc_file,out);
          fputs("\n",out);
         }
    fputs(line,out);
   }

 fclose(in);
 fclose(out);

 if(!seen)
   {
    unlink(ifile);
    rename(ofile,ifile);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Write out a standard pre-amble.

  FILE* f The file to write the pre amble to.

  char* title The title of the file.

  int sourcefile True if the Source-Files line is to be included.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLPreamble(FILE* f,char* title,int sourcefile)
{
 fputs("<!-- This HTML file generated by cxref. -->\n",f);
 fputs("<!-- cxref program (c) Andrew M. Bishop 1995,1996. -->\n",f);
 fputs("\n",f);
 fputs("<HTML>\n",f);
 fputs("\n",f);
 fputs("<HEAD>\n",f);
 fputs("<TITLE>\n",f);
 fputs(title,f);fputs("\n",f);
 fputs("</TITLE>\n",f);
 fputs("</HEAD>\n",f);
 fputs("\n",f);
 fputs("<BODY>\n",f);
 fputs("\n",f);
 if(sourcefile)
   {
    fputs("<h1>Source Files</h1>\n",f);
    fputs("\n",f);
    fputs("<!-- Begin-Of-Source-Files -->\n",f);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Write out a standard post-amble. This includes the end of document marker.

  FILE* f The file to write the post amble to.

  int sourcefile True if the Source-Files line is to be included.
  ++++++++++++++++++++++++++++++++++++++*/

static void WriteHTMLPostamble(FILE* f,int sourcefile)
{
 if(sourcefile)
   {
    fputs("\n",f);
    fputs("<!-- End-Of-Source-Files -->\n",f);
   }
 fputs("\n",f);
 fputs("</BODY>\n",f);
 fputs("</HTML>\n",f);
}


/*++++++++++++++++++++++++++++++++++++++
  Write out the appendix information.

  StringList* files The list of files to write.

  StringList* funcs The list of functions to write.

  StringList* vars The list of variables to write.

  StringList* types The list of types to write.
  ++++++++++++++++++++++++++++++++++++++*/

void WriteHTMLAppendix(StringList* files,StringList* funcs,StringList* vars,StringList* types)
{
 char* ofile;
 int i;

 /* Write the bits to the including file. */

 WriteHTMLDocument(ConcatStrings(2,option_name,HTML_APDX),1);

 /* Open the file */

 ofile=ConcatStrings(5,option_odir,"/",option_name,HTML_APDX,HTML_FILE);

 of=fopen(ofile,"w");

 if(!of)
   {fprintf(stderr,"cxref: Failed to open the HTML appendix file '%s'\n",ofile);exit(1);}

 /* Write the file structure out */

 WriteHTMLPreamble(of,ConcatStrings(3,"Cross reference index of ",option_name,"."),0);

 fprintf(of,"<h1>Cross References</h1>\n\n");

 /* Write out the appendix of files. */

 if(files->n)
   {
    fprintf(of,"<hr>\n<a name=\"files\">\n<h2>Files</h2>\n</a>\n");
    fprintf(of,"<menu>\n");
    for(i=0;i<files->n;i++)
       fprintf(of,"<li><a href=\"%s"HTML_FILE"#file\">%s</a>\n",files->s[i],html(files->s[i]));
    fprintf(of,"</menu>\n");
   }

 /* Write out the appendix of functions. */

 if(funcs->n)
   {
    fprintf(of,"<hr>\n<a name=\"functions\">\n<h2>Global Functions</h2>\n</a>\n");
    fprintf(of,"<menu>\n");
    for(i=0;i<funcs->n;i++)
      {
       char *temp=strchr(funcs->s[i],':'),*file;
       temp[-1]=0;
       file=&temp[2];
       fprintf(of,"<li><a href=\"%s"HTML_FILE"#func_%s\">%s()  :  %s</a>\n",file,funcs->s[i],html(funcs->s[i]),html(file));
       temp[-1]=' ';
      }
    fprintf(of,"</menu>\n");
   }

 /* Write out the appendix of variables. */

 if(vars->n)
   {
    fprintf(of,"<hr>\n<a name=\"variables\">\n<h2>Global Variables</h2>\n</a>\n");
    fprintf(of,"<menu>\n");
    for(i=0;i<vars->n;i++)
      {
       char *temp=strchr(vars->s[i],':'),*file;
       temp[-1]=0;
       file=&temp[2];
       fprintf(of,"<li><a href=\"%s"HTML_FILE"#var_%s\">%s  :  %s</a>\n",file,vars->s[i],html(vars->s[i]),html(file));
       temp[-1]=' ';
      }
    fprintf(of,"</menu>\n");
   }

 /* Write out the appendix of types. */

 if(types->n)
   {
    fprintf(of,"<hr>\n<a name=\"types\">\n<h2>Defined Types</h2>\n</a>\n");
    fprintf(of,"<menu>\n");
    for(i=0;i<types->n;i++)
      {
       char *temp=strchr(types->s[i],':'),*file;
       temp[-1]=0;
       file=&temp[2];
       if(!strncmp("enum",types->s[i],4))
          fprintf(of,"<li><a href=\"%s"HTML_FILE"#type_enum_%s\">%s  :  %s</a>\n",file,&types->s[i][5],html(types->s[i]),html(file));
       else
          if(!strncmp("union",types->s[i],5))
             fprintf(of,"<li><a href=\"%s"HTML_FILE"#type_union_%s\">%s  :  %s</a>\n",file,&types->s[i][6],html(types->s[i]),html(file));
          else
             if(!strncmp("struct",types->s[i],6))
                fprintf(of,"<li><a href=\"%s"HTML_FILE"#type_struct_%s\">%s  :  %s</a>\n",file,&types->s[i][7],html(types->s[i]),html(file));
             else
                fprintf(of,"<li><a href=\"%s"HTML_FILE"#type_%s\">%s  :  %s</a>\n",file,types->s[i],html(types->s[i]),html(file));
       temp[-1]=' ';
      }
    fprintf(of,"</menu>\n");
   }

 WriteHTMLPostamble(of,0);

 fclose(of);

/* Clear the memory in html() */

 html(NULL); html(NULL); html(NULL); html(NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Make the input string safe to output as HTML ( not < > & " ).

  char* html Returns a safe HTML string.

  char* c A non-safe HTML string.

  The function can only be called four times in each fprintf() since it returns one of only four static strings.
  ++++++++++++++++++++++++++++++++++++++*/

static char* html(char* c)
{
 static char safe[4][256],*malloced[4]={NULL,NULL,NULL,NULL};
 static int which=0;
 int i=0,j=0,len=256-5;              /* 5 is the longest possible inserted amount */
 char* ret;

 which=(which+1)%4;
 ret=safe[which];

 if(malloced[which])
   {Free(malloced[which]);malloced[which]=NULL;}

 if(c)
    do{
       for(;j<len && c[i];i++)
          switch(c[i])
            {
            case '<':
             ret[j++]='&';
             ret[j++]='l';
             ret[j++]='t';
             ret[j++]=';';
             break;
            case '>':
             ret[j++]='&';
             ret[j++]='g';
             ret[j++]='t';
             ret[j++]=';';
             break;
            case '"':
             ret[j++]='&';
             ret[j++]='q';
             ret[j++]='u';
             ret[j++]='o';
             ret[j++]='t';
             ret[j++]=';';
             break;
            case '&':
             ret[j++]='&';
             ret[j++]='a';
             ret[j++]='m';
             ret[j++]='p';
             ret[j++]=';';
             break;
            case '\n':
             if(j && ret[j-1]=='\n')
               {
                ret[j-1]='<';
                ret[j++]='b';
                ret[j++]='r';
                ret[j++]='>';
               }
            default:
             ret[j++]=c[i];
            }

       if(c[i])                 /* Not finished */
         {
          unsigned int est_size=(110*strlen(c)*j)/(100*i); /* guess 10% larger than current estimate */

          if(malloced[which])
             malloced[which]=Realloc(malloced[which],est_size);
          else
            {malloced[which]=Malloc(est_size); strncpy(malloced[which],ret,(unsigned)j);}
          ret=malloced[which];
          len=est_size-4;
         }
       else
         {ret[j]=0; ret=NULL;}
      }
    while(ret);
 else
    safe[which][0]=0;

 return(malloced[which]?malloced[which]:safe[which]);
}
