/***************************************
  $Header: /home/amb/cxref/RCS/func.c 1.12 1996/07/04 20:02:27 amb Exp $

  C Cross Referencing & Documentation tool. Version 1.2.

  Handle Function stuff.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1995,96 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/

/*+ Control the debugging information from this file. +*/
#define DEBUG 0

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "memory.h"
#include "datatype.h"
#include "cxref.h"

/*+ The current parsing options. +*/
extern int option_xref;

/*+ The current file that is being processed. +*/
extern File CurFile;

/*+ The name of the current file. +*/
extern char* parse_file;

/*+ The current function, this is initialised by the start of a possible declaration and maintained until all of the
    arguments have been added and confirmation that it is a definition and not a prototype is seen. +*/
static Function cur_func=NULL;

/*+ The list of function prototypes and the files that they are defined in. +*/
static StringList2 prototypes={0,NULL,NULL};


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a function prototype is seen.

  char* name The name of the function.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenFunctionProto(char* name)
{
#if DEBUG
 printf("#Func.c# Function Prototype '%s'\n",name);
#endif

 AddToStringList2(&prototypes,name,parse_file);
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a function declaration is seen. This may or may not be a function defintion, we will need to wait and see.

  char* name The name of the function.

  int scope The scope of the function definition.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenFunctionDeclaration(char* name,int scope)
{
#if DEBUG
 printf("#Func.c# Function declaration for '%s()'\n",name);
#endif

 if(cur_func)
   {
    if(cur_func->comment) Free(cur_func->comment);
    if(cur_func->name)    Free(cur_func->name);
    if(cur_func->type)    Free(cur_func->type);
    DeleteStringList2(&cur_func->args,1);
    if(cur_func->cret)    Free(cur_func->cret);
    DeleteStringList(&cur_func->calls);
    DeleteStringList(&cur_func->v_refs);
    DeleteStringList(&cur_func->f_refs);
   }
 else
    cur_func=(Function)Malloc(sizeof(struct _Function));

 cur_func->comment=MallocString(GetCurrentComment());
 cur_func->name=MallocString(name);
 cur_func->protofile=NULL;
 cur_func->scope=scope;
 cur_func->type=NULL;
 InitStringList2(&cur_func->args);
 cur_func->cret=NULL;
 InitStringList(&cur_func->calls);
 InitStringList(&cur_func->called);
 InitStringList(&cur_func->used);
 InitStringList(&cur_func->v_refs);
 InitStringList(&cur_func->f_refs);
 cur_func->next=NULL;
}


/*++++++++++++++++++++++++++++++++++++++
  Called when a possible function definition is confirmed.

  char* type The type of the function, or NULL at the end of a definition.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenFunctionDefinition(char* type)
{
 Function *func=&CurFile->functions;
 int i;

#if DEBUG
 printf("#Func.c# Function definition %s for '%s()'\n",type?"start":"end",cur_func->name);
#endif

 if(!type)
   {cur_func=NULL;return;}

 cur_func->type=MallocString(type);

 cur_func->cret=SplitComment(&cur_func->comment,type);
 if(!cur_func->cret)
    cur_func->cret=MallocString(GetCurrentComment());

 if(option_xref&XREF_FUNC)
    for(i=0;i<prototypes.n;i++)
       if(!strcmp(cur_func->name,prototypes.s1[i]))
         {cur_func->protofile=MallocString(prototypes.s2[i]); break;}

 for(i=0;i<cur_func->args.n;i++)
    if(strcmp(cur_func->args.s1[i],"void") && strcmp(cur_func->args.s1[i],"...") && !strchr(cur_func->args.s1[i],' '))
      {
       char *old=cur_func->args.s1[i];
       cur_func->args.s1[i]=ConcatStrings(2,"int ",old);cur_func->args.s2[i]=SplitComment(&cur_func->comment,cur_func->args.s1[i]);
       Free(old);
      }

 while(*func)
   {
    if(strcmp(cur_func->name,(*func)->name)<0)
      {
       Function temp=*func;
       *func=cur_func;
       cur_func->next=temp;
       break;
      }
    func=&(*func)->next;
   }

 if(!cur_func->next)
    *func=cur_func;

}

/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a function argument is seen in the current function declaration.

  char* name The name of the argument.

  char* type The type of the argument, or NULL if a traditional style function definition.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenFunctionArg(char* name,char *type)
{
#if DEBUG
 printf("#Func.c# Function arg %s '%s' in %s()\n",name,type?type:"K&R",cur_func->name);
#endif

 if(type)
   {
    int i;

    for(i=0;i<cur_func->args.n;i++)
       if(!strcmp(cur_func->args.s1[i],name))
         {
          Free(cur_func->args.s1[i]);
          cur_func->args.s1[i]=type;cur_func->args.s2[i]=SplitComment(&cur_func->comment,type);
          break;
         }
    if(i==cur_func->args.n)
       AddToStringList2(&cur_func->args,MallocString(type),SplitComment(&cur_func->comment,type));

    if(!cur_func->args.s2[i])
       cur_func->args.s2[i]=MallocString(GetCurrentComment());
   }
 else
    AddToStringList2(&cur_func->args,MallocString(name),NULL);
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a comment is seen, that may be in a function body.

  int SeenFuncIntComment Returns a true value if the comment was accepted as an function internal comment.

  char* comment The comment that has been seen.
  ++++++++++++++++++++++++++++++++++++++*/

int SeenFuncIntComment(char* comment)
{
 if(!cur_func || !cur_func->type)
    return(0);

#if DEBUG
 printf("#Func.c# Function internal comment '%s' in %s()\n",comment,cur_func->name);
#endif

 if(cur_func->comment)
   {
    char* c=cur_func->comment;

    cur_func->comment=MallocString(ConcatStrings(3,c,"\n\n",comment));
    Free(c);
   }
 else
    cur_func->comment=MallocString(comment);

 return(1);
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a function call is seen in the current function.

  char* name The name of the function that is called.
  ++++++++++++++++++++++++++++++++++++++*/

void SeenFunctionCall(char* name)
{
 if(!(option_xref&XREF_FUNC))
    return;

#if DEBUG
 printf("#Func.c# Function call for '%s()' in %s()\n",name,cur_func->name);
#endif

 AddToStringList(&cur_func->calls,name,1);
}


/*++++++++++++++++++++++++++++++++++++++
  Function that is called when a function or variable is referenced in the current function.

  char* name The name of the function or variable that is referenced.

  int in_a_function Whether the reference is from within a function or at the top level of the file.
  ++++++++++++++++++++++++++++++++++++++*/

void CheckFunctionVariableRef(char* name,int in_a_function)
{
 Variable var =CurFile->variables;
 Function func=CurFile->functions;
 StringList *sl=NULL;

 if(!(option_xref&(XREF_VAR|XREF_FUNC)))
    return;

 if(IsAScopeVariable(name))
    return;

#if DEBUG
 printf("#Func.c# Function/Variable reference for '%s' in %s\n",name,in_a_function?cur_func->name:CurFile->name);
#endif

 if(option_xref&XREF_VAR)
    while(var)
      {
       if(!strcmp(var->name,name))
         {
          if(in_a_function)
             sl=&cur_func->v_refs;
          else
             sl=&CurFile->v_refs;
         }
       var=var->next;
      }

 if(!sl && option_xref&XREF_FUNC)
    while(func)
      {
       if(!strcmp(func->name,name))
         {
          if(in_a_function)
             sl=&cur_func->f_refs;
          else
             sl=&CurFile->f_refs;
         }
       func=func->next;
      }

 if(!sl && option_xref&XREF_FUNC)
   {
    int i;
    for(i=0;i<prototypes.n;i++)
       if(!strcmp(name,prototypes.s1[i]))
         {
          if(in_a_function)
             sl=&cur_func->f_refs;
          else
             sl=&CurFile->f_refs;
         }
   }

 /* Now add the function or variable to the Function / File structure. */

 if(sl)
    AddToStringList(sl,name,1);
}


/*++++++++++++++++++++++++++++++++++++++
  Clears the memory that is used here.
  This is the list of function prototypes and the cur_func variable if not already empty.
  ++++++++++++++++++++++++++++++++++++++*/

void DeleteSpareProtos(void)
{
 DeleteStringList2(&prototypes,0);
 InitStringList2(&prototypes);

 if(cur_func)
   {
    Function func=CurFile->functions;
    int delete_cur_func=1;

    while(func)
      {
       if(func==cur_func)
          delete_cur_func=0;

       func=func->next;
      }

    if(delete_cur_func)
      {
       if(cur_func->comment)   Free(cur_func->comment);
       Free(cur_func->name);
       if(cur_func->type)      Free(cur_func->type);
       if(cur_func->protofile) Free(cur_func->protofile);
       DeleteStringList2(&cur_func->args,1);
       if(cur_func->cret)      Free(cur_func->cret);
       DeleteStringList(&cur_func->calls);
       DeleteStringList(&cur_func->called);
       DeleteStringList(&cur_func->used);
       DeleteStringList(&cur_func->v_refs);
       DeleteStringList(&cur_func->f_refs);
       Free(cur_func);
      }

    cur_func=NULL;
   }
}
