//*************************************************************
//  File name: TTPLOT.c
//
//  Description:
//
//      WinMain, Glyph drrawing, and window procedure routines.
//
//      This sample demonstrates a method of displaying TrueType fonts on a
//      plotter.  It does this by selecting a TrueType font into a screen 
//      display context and calling the GetGlyphOutline() function to 
//      retrieve the curve data points of a TrueType character in the 
//      TrueType font rasterizer's native format.  Then, with the data 
//      it retrieved, it perform's a series of MoveTo() and LineTo() calls 
//      to draw the TrueType character.
//
// Functions for drawing the TrueType characters:
//      int MapFX( FIXED fx )
//      int MapFY( FIXED fy )  : These functions convert FIXED values to integers.
//                                                                                
//      void DrawXMark( HDC hdc, POINTFX ptfx, int xText, int yText)
//                This function will draw a line in a glyph with the MoveTo/LineTo
//                combination.
//
//      FIXED FloatToFixed( double d )
//                This functino converts a double to a fixed.
//
//      void DrawGlyphOutlines(HDC hdc,int xWidth,int yHeight)
//                 This function is the core of the converion of the data returned
//                 from GetGlyphOutline to the actual drawing of the lines of a glyph.
//
// Development Team:
//
//      Greg Binkerd
//
//
// Written by Microsoft Product Support Services, Windows Developer Support
// Copyright (c) 1992 Microsoft Corporation. All rights reserved.
//*************************************************************
#include "global.h"

HANDLE  ghInst          = NULL;
HWND    ghWndMain       = NULL;
BOOL    gbAbort;        // Boolean to tell if user has aborted the print job.
HWND    ghDlgAbort;     // Dialog box handle to the modeless printing dialog box.
HFONT   ghScreenFont;   // Handle to the current selected font.

char    szMainMenu[]    = "MainMenu";
char    szMainClass[]   = "TTPLOTClass";  
int     xOld,yOld;

//*************************************************************
//
//  WinMain()
//
//  Purpose:
//
//              Entry point for all Windows applications.
//
//
//  Parameters:
//
//      HANDLE hInstance     - Handle to current instance
//      HANDLE hPrevInstance - Handle to previous instance
//      LPSTR  lpCmdLine     - Command-line arguments
//      int    nCmdShow      - How window is initially displayed
//      
//
//  Return: (int PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              7/28/93    GGB        Created
//
//*************************************************************

int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;

    if (!hPrevInstance && !InitApplication(hInstance))
	    return(FALSE);       

    if (!InitInstance(hInstance, nCmdShow))
	return(FALSE);

    while (GetMessage(&msg, NULL, NULL, NULL))
    {
	TranslateMessage(&msg);      
	DispatchMessage(&msg);       
    }
    return(msg.wParam);      

} //*** WinMain




//*************************************************************
//
//  MainWndProc()
//
//  Purpose:
//
//              Main Window procedure.
//
//
//  Parameters:
//
//      HWND     hWnd   - Handle to main window
//      unsigned msg    - Message passed to application
//      WORD     wParam - Additional message information
//      LONG     lParam - Additional message information
//      
//
//  Return: (long FAR PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              7/28/93    GGB        Created
//
//*************************************************************

long FAR PASCAL MainWndProc (HWND hWnd, unsigned msg, WORD wParam, LONG lParam)
{
    FARPROC lpProc;

    switch (msg) 
    {
	case WM_CREATE:
	    {
	    LOGFONT lf;
        
        // Let's fill out a LOGFONT structure so that we can have an initial font.
	    lf.lfHeight=40;
	    lf.lfWidth=0;
	    lf.lfEscapement=lf.lfOrientation=0;
	    lf.lfWeight=FW_NORMAL;
	    lf.lfItalic=0;
	    lf.lfUnderline=0;
	    lf.lfStrikeOut=0;
	    lf.lfCharSet=ANSI_CHARSET;
	    lf.lfOutPrecision=OUT_TT_ONLY_PRECIS;
	    lf.lfClipPrecision=CLIP_DEFAULT_PRECIS;
	    lf.lfQuality=DEFAULT_QUALITY;
	    lf.lfPitchAndFamily=FF_DONTCARE;
	    lstrcpy(lf.lfFaceName,"Arial");
	    ghScreenFont = CreateFontIndirect(&lf);
	    break;                       
	    }

	case WM_PAINT:
	    {
	    PAINTSTRUCT ps;
	    HDC         hdc;       
	    RECT        rc;

	    hdc = BeginPaint(hWnd, &ps);
	    GetClientRect(hWnd,&rc);
	    // Draw the current font's glyphs on the client area of the main window.
	    DrawGlyphOutlines(hdc,rc.right-rc.left,rc.bottom-rc.top);
	    EndPaint(hWnd, &ps);
	    }
	    break;

	case WM_COMMAND: 
	    switch ( wParam )
	    {
		case IDM_FONTS:
		     // show ChooseFont common dialog box
		     SelectFont(hWnd);
		     break;

		case IDM_PRINT:
		     // show Print common dialog box
		     GetPrinterReady(hWnd);
		     break;

		case IDM_EXIT:
		     PostMessage( hWnd, WM_SYSCOMMAND, SC_CLOSE, 0L );
		     break;

		case IDM_ABOUT:
		    // display About dialog box
		    lpProc = MakeProcInstance(About, ghInst);
		    DialogBox(ghInst, "AboutBox", hWnd, lpProc);    
		    FreeProcInstance(lpProc);
		break;
	    }
	break;

	case WM_DESTROY:
	    // clean up screen font
	    if (ghScreenFont != NULL)
            DeleteObject(ghScreenFont);
	    PostQuitMessage(0);
	break;
    }
    return(DefWindowProc(hWnd, msg, wParam, lParam));

} //*** MainWndProc


//*************************************************************
//
//  About()
//
//  Purpose:
//
//              The About dialog box procedure.
//
//
//  Parameters:
//
//      HWND     hDlg   - Handle to dialog window
//      unsigned msg    - Message passed to dialog window
//      WORD     wParam - Additional message information
//      LONG     lParam - Additional message information
//      
//
//  Return: (BOOL FAR PASCAL)
//
//      TRUE  - About dialog procedure handled the message.
//      FALSE - About dialog procedure did not handle the message.
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/2/92     DAM        Modified by GGB
//
//*************************************************************

BOOL FAR PASCAL About (HWND hDlg, unsigned msg, WORD wParam, LONG lParam)
{
    switch (msg) 
    {
	case WM_INITDIALOG: 
	    return(TRUE);

	case WM_COMMAND:
	    if (wParam == IDOK || wParam == IDCANCEL) 
	    {
		EndDialog(hDlg, TRUE);         
		return(TRUE);
	    }
	break;
    }
    return(FALSE);      // Didn't process a message

} //*** About

//*************************************************************
//
//  SelectFont
//
//  Purpose:
//              
//    Function is responsible for displaying ChooseFont common
//    dialog.  Also, the function creates a printer font and a 
//    screen font and stores them in their respective global
//    structures.
//
//  Parameters:
//      HWND hWnd
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              7/28/93    GGB        Created
//*************************************************************

void SelectFont (HWND hWnd)
{
    CHOOSEFONT cf;
    HDC        hDC;
    LOGFONT    lfTemp;

    if ((hDC = GetDC (hWnd)) != NULL)
    {
	cf.lStructSize      = sizeof(CHOOSEFONT);
	cf.hwndOwner        = hWnd;
	cf.hDC              = hDC;
	cf.lpLogFont        = &lfTemp;
	cf.Flags            = CF_WYSIWYG | CF_SCALABLEONLY | CF_TTONLY | CF_BOTH;
	cf.lCustData        = 0L;
	cf.rgbColors        = RGB(0,0,0);
	cf.lpfnHook         = NULL;
	cf.lpTemplateName   = (LPSTR)NULL;
	cf.hInstance        = ghInst;
	cf.lpszStyle        = (LPSTR)NULL;
	cf.nFontType        = SCREEN_FONTTYPE;
	cf.nSizeMin         = 0;
	cf.nSizeMax         = 0;


	if (ChooseFont(&cf))
	{
	    // get rid of old screen font, if one 
	    if (ghScreenFont != NULL)
	      {
		  DeleteObject(ghScreenFont);
		  ghScreenFont = NULL;
	      }

	    //create the current screen font
	    ghScreenFont = CreateFontIndirect(&lfTemp);
	    // redraw the client window since font has changed...
	    InvalidateRect(hWnd, NULL, TRUE);
	}
	ReleaseDC(hWnd, hDC);
    }
} //*** SelectFont


//*************************************************************
//
//  GetPrinterReady
//
//  Purpose:
//              
//    Function responsible for displaying Print  common
//    dialog box.  Stores a handle to the current DEVMODE and a
//    handle to the current DEVNAMES in a global PRINTINFOSTRUCT.
//
//  Parameters:
//      HWND hWnd
//      
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              7/28/93    GGB        Created
//*************************************************************

BOOL GetPrinterReady (HWND hWnd)
{
    PRINTDLG pdPrint;
    BOOL     bResult;

    pdPrint.lStructSize = sizeof(PRINTDLG);
    pdPrint.hwndOwner   = hWnd;
    pdPrint.hDevMode    = NULL;
    pdPrint.hDevNames   = NULL;
    pdPrint.nCopies     = 1;
    pdPrint.Flags       = PD_ALLPAGES | PD_NOPAGENUMS | PD_COLLATE |
			  PD_NOSELECTION | PD_HIDEPRINTTOFILE | PD_RETURNDC;

    pdPrint.hInstance   = (HANDLE)ghInst;

    if (PrintDlg(&pdPrint))
      {
      // Plot the TT chars. to the plotter.
	  bResult = PrintIt(hWnd, &pdPrint,pdPrint.hDC);
      return bResult;
      }
    else
	  return FALSE;
} //*** GetPrinterReady


//*************************************************************
//
//  PrintIt
//
//  Purpose:
//              
//    Function that handles all the printing.  Manual banding is
//    not implemented.
//
//  Parameters:
//      HWND hWnd   - The main window of the application.
//      lpPD        - Pointer to a PRINTDLG structure used by PrintDlg()
//      hPrnDC      - Handle to the printer/plotter's device context
//      
//      
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     DAM        Modified by GGB
//*************************************************************

BOOL PrintIt (HWND hWnd, LPPRINTDLG lpPD, HDC hPrnDC)
{
    HDC     hdc;
    DOCINFO di;
    FARPROC lpAbortProc, lpAbortDlg;
    int     nxLogPix, nyLogPix, dxScreen, dyScreen, dxPrinter,dyPrinter;

    // set user abort flag 
    gbAbort = FALSE;

    // Initialize the abort procedure. 
    lpAbortProc = MakeProcInstance((FARPROC)PrintAbortProc, ghInst);
    lpAbortDlg  = MakeProcInstance((FARPROC)PrintAbortDlg,  ghInst);

    // disable main application window 
    EnableWindow(hWnd, FALSE);

    // display abort dialog 
    ghDlgAbort = CreateDialog(ghInst, "PrintDLG", hWnd, (FARPROC)lpAbortDlg);

    if (!gbAbort)
      {
      // set abort procedure 
      if (SetAbortProc(hPrnDC, lpAbortProc) < 0)
        {
	     MessageBox(NULL, "The SetAbortProc function failed.", "ERROR",
		    MB_APPLMODAL|MB_ICONSTOP|MB_OK);

	     DeleteDC(hPrnDC);
	     return FALSE;
        }
      }
      else
        {
        // enable main application window 
        EnableWindow(hWnd, TRUE);
        // get rid of abort dialog 
        DestroyWindow(ghDlgAbort);
        // clean up 
        FreeProcInstance(lpAbortProc);
        FreeProcInstance(lpAbortDlg); 
   	    DeleteDC(hPrnDC);
        return FALSE;
        }
        
    // need for StartDoc 
    di.cbSize      = sizeof(DOCINFO);
    di.lpszDocName = (LPSTR)"PRNCDLG";
    di.lpszOutput  = NULL;

    if (gbAbort)
      {
      // enable main application window 
      EnableWindow(hWnd, TRUE);
      // get rid of abort dialog 
      DestroyWindow(ghDlgAbort);
      // clean up 
      FreeProcInstance(lpAbortProc);
      FreeProcInstance(lpAbortDlg); 
  	  DeleteDC(hPrnDC);
      return FALSE;
      }
      
    // issue STARTDOC 
    if (StartDoc(hPrnDC, &di) < 0)
    {
     // enable main application window 
     EnableWindow(hWnd, TRUE);
     // get rid of abort dialog 
     DestroyWindow(ghDlgAbort);
     // clean up 
     FreeProcInstance(lpAbortProc);
     FreeProcInstance(lpAbortDlg); 
	 DeleteDC(hPrnDC);
	 return FALSE;
    }

    StartPage(hPrnDC);

    // get logical pixels per inch 
    nxLogPix  = GetDeviceCaps(hPrnDC, LOGPIXELSX);
    nyLogPix  = GetDeviceCaps(hPrnDC, LOGPIXELSY);

    // get page resolution 
    hdc=GetDC(hWnd);
    dxPrinter = GetDeviceCaps(hPrnDC, HORZRES);
    dyPrinter = GetDeviceCaps(hPrnDC, VERTRES);
    dxScreen = GetDeviceCaps(hdc, HORZRES);
    dyScreen = GetDeviceCaps(hdc, VERTRES);
    ReleaseDC(hWnd,hdc);

    // Place printer in MM_ISOTROPIC mode so we can get WYSIWYG effects...					      
    SetMapMode(hPrnDC,MM_ISOTROPIC);
    SetWindowExt(hPrnDC,dxScreen,dxScreen);
    SetViewportExt(hPrnDC,dxPrinter,dyPrinter);                                         
    DrawGlyphOutlines(hPrnDC,dxScreen,dyScreen);
    EndPage(hPrnDC);

    // only do if user has not aborted 
    if (!gbAbort)
    {
      if (EndDoc(hPrnDC) < 0)    // end the document
      {
             // enable main application window 
             EnableWindow(hWnd, TRUE);
             // get rid of abort dialog 
             DestroyWindow(ghDlgAbort);
             // clean up 
             FreeProcInstance(lpAbortProc);
             FreeProcInstance(lpAbortDlg); 
             DeleteDC(hPrnDC);
             return FALSE;
      }
    }

    // enable main application window 
    EnableWindow(hWnd, TRUE);

    // get rid of abort dialog 
    DestroyWindow(ghDlgAbort);

    // clean up 
    FreeProcInstance(lpAbortProc);
    FreeProcInstance(lpAbortDlg); 

    DeleteDC(hPrnDC);
} //*** PrintIt



//*************************************************************
//
//  MapFX
//
//  Purpose:
//              
//    Function that converts a FIXED value to an int.
//
//  Parameters:
//      FIXED fx  - The value to be converted to an integer.
//      
//
//  Return: (int)
//
//*************************************************************

int MapFX( FIXED fx )
 {
  long   lx;

  lx = *(LONG *)&fx;
  return (int)((double)(lx)/65536.0);
 }

  
//*************************************************************
//
//  MapFY
//
//  Purpose:
//              
//    Function that converts a FIXED value to an int.
//
//  Parameters:
//      FIXED fy  - The value to be converted to an integer.
//      
//
//  Return: (int)
//
//*************************************************************

int MapFY( FIXED fy )
 {
  long   ly;

  ly = *(LONG *)&fy;
  return (int)((double)(ly) / 65536.0);
 }

  

//*************************************************************
//
//  DrawXMark
//
//  Purpose:
//              
//    Function that draws a line of a TrueType glyph.
//
//  Parameters:
//      
//    HDC hdc      -  Printer's device context
//    POINTFX ptfx -  Fixed values for the x and y coordinates for the line
//    int xText    -  The x starting point of the glyph
//    int yText    -  The y starting point of the glyph
//
//  Return: none
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              7/28/93    GGB        Created
//*************************************************************

void DrawXMark( HDC hdc, POINTFX ptfx, int xText, int yText)
 {
  int  x, y;
  static POINT aPoints[4];


  x = MapFX( ptfx.x );
  y = MapFY( ptfx.y );

  if( xOld != 0x7FFF )
   {
   MoveTo( hdc, xText+xOld, yText-yOld );
   LineTo( hdc, xText+x, yText-y );           
   } 

  xOld = x;
  yOld = y;
 }


//*************************************************************
//
//  FloatToFixed
//
//  Purpose:
//              
//    Function that converts a floting point value to a FIXED.
//
//  Parameters:
//      double d  - The value to be converted to a FIXED
//      
//
//  Return: (FIXED)
//
//*************************************************************

FIXED FloatToFixed( double d )
 {
  long l;

  l = (long)(d * 65536L);
  return *(FIXED *)&l;
 }



//*************************************************************
//
//  DrawGlyphOutlines
//
//  Purpose:
//              
//    Function that draws characters 48-122 as a series of lines.
//
//  Parameters:
//      
//    HDC hdc,int xWidth,int yHeight  
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              7/28/93    GGB        Modified by GGB
//*************************************************************

void DrawGlyphOutlines(HDC hdc,int xWidth,int yHeight)
       {
       DWORD dwSize;
       HANDLE hMem;
       LPTTPOLYGONHEADER lpph;
       LPBYTE            lpb;
       int    nItem;
       long   cbOutline, cbTotal;
       int    nChar;            
       int    x;
       int    y;       
       HDC  hdcGlyph;
       HFONT hOldFont;
       MAT2 m2;    
       GLYPHMETRICS gm;                       
       TEXTMETRIC tm;

       
       hdcGlyph = CreateDC("DISPLAY",NULL,NULL,NULL);
       
       if (ghScreenFont != NULL)
	    hOldFont = SelectObject(hdcGlyph, ghScreenFont);
	    
       GetTextMetrics(hdcGlyph,&tm);
	    
       // Set up the identity matrix.
       m2.eM11 = FloatToFixed(1.0);
       m2.eM12 = FloatToFixed(0.0);
       m2.eM21 = FloatToFixed(0.0);
       m2.eM22 = FloatToFixed(1.0);
	    
       nChar=48;  // Start at char. '0'

       for (y=tm.tmHeight+5;y<(yHeight-tm.tmHeight);y+=tm.tmHeight+5)
          for (x=0; ((x<(xWidth-tm.tmAveCharWidth)) && (nChar<123)); x+=tm.tmAveCharWidth*3)
	        {                                 
	        // Find out the buffer size needed to hold the curve data.
  	        dwSize = GetGlyphOutline(hdcGlyph,nChar,GGO_NATIVE,&gm,0L,NULL,&m2);
		    hMem=GlobalAlloc(GHND,dwSize);
		    lpph=(LPTTPOLYGONHEADER)GlobalLock(hMem);                 
		    // Retrieve the curve data.  This call to GetGlyphOutline will retrieve
		    // the native data of a character.
		    GetGlyphOutline(hdcGlyph,nChar,GGO_NATIVE,&gm,dwSize,lpph,&m2);
		       
	        cbTotal = dwSize;

	        while( cbTotal > 0 )
	          {
		      xOld = yOld = 0x7FFF;
		      DrawXMark( hdc, lpph->pfxStart,x,y );

		      nItem = 0;
		      lpb   = (LPBYTE)lpph + sizeof(TTPOLYGONHEADER);

		      // Calculate size of data needed
              cbOutline = (long)lpph->cb - sizeof(TTPOLYGONHEADER);
              
              // Now, go through all of the points specified in the representation of the
              // character and draw connecting lines with the MoveTo() and LineTo() commands.
              // Note that we are not calculating "spline:" curves.  We are just drawing a
              // straight line instead.  Please read the documentation for GetGlyphOutline
              // for details.
		      while( cbOutline > 0 )
		        {
		        int           i, n;
		        LPTTPOLYCURVE lpc;
 
		        nItem++;
		        lpc = (LPTTPOLYCURVE)lpb;
		        for( i = 0; i < (int)lpc->cpfx; i++ )
			       DrawXMark( hdc, lpc->apfx[i],x,y );
		        n = sizeof(TTPOLYCURVE) + sizeof(POINTFX) * (lpc->cpfx - 1);
		        lpb += n;
		        cbOutline -= n;
		        }                   
		
		      DrawXMark( hdc, lpph->pfxStart,x,y);
		      cbTotal -= lpph->cb;
		      lpph     = (LPTTPOLYGONHEADER)lpb;
			  }
 
		    GlobalUnlock(hMem);
		    GlobalFree(hMem);
		    nChar++;
		    }

       // reselect old font back in to dc
       if (ghScreenFont != NULL)
	    SelectObject(hdcGlyph, hOldFont);
       DeleteDC(hdcGlyph);
 }




// -------------------- Abort Routines --------------------


//*************************************************************
//
//  PrintAbortProc
//
//  Purpose:
//              
//    Function is used to give application some "multi-tasking"
//    through the use of a PeekMessage().  Also, it gives the 
//    application some error control during printing.
//
//  Parameters:
//      HDC hDC
//      short code
//      
//
//  Return: (BOOL FAR PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     DAM        Created
//*************************************************************

BOOL FAR PASCAL PrintAbortProc (HDC hDC, short code)
{
   MSG msg;

   while (!gbAbort && PeekMessage(&msg, 0, 0, 0, PM_REMOVE))
      if (!IsDialogMessage(ghDlgAbort, &msg))
	 {
	 TranslateMessage(&msg);
	 DispatchMessage(&msg);
	 }
   return (!gbAbort);
}  //*** PrintAbortProc


//*************************************************************
//
//  PrintAbortDlg
//
//  Purpose:
//              
//    Dialog procedure for print abort dialog.
//
//  Parameters:
//      HWND hWnd
//      unsigned msg
//      WORD wParam
//      LONG lParam
//      
//
//  Return: (int FAR PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     DAM        Created
//*************************************************************

int FAR PASCAL PrintAbortDlg (HWND hWnd, unsigned msg, WORD wParam, LONG lParam)
{
   switch (msg)
      {
      case WM_INITDIALOG:
	 SetFocus(hWnd);
	 EnableMenuItem(GetSystemMenu(hWnd, FALSE), SC_CLOSE, MF_GRAYED);

	 // set printing info to abort dialog 
	 SetDlgItemText(hWnd, IDM_STATIC1, "Now printing TrueType characters...");

	 return TRUE;

      case WM_COMMAND:
	 gbAbort = TRUE;
	 return TRUE;
      }

   return FALSE;
}  // PrintAbortDlg() 

/*** EOF: TTPLOT.C ***/
