/***********************************************************************
*                                                                      *
*   A tap listener is an independent applications that  opens a queue  *
*   to receive information about file operations occuring in file      *
*   transfers (e.g. via ZMODEM).                                       *
*   These informations can be used for add-in applications like        *
*   auto-unzippers or online viewers.                                  *
*                                                                      *
*   ------------------------------------------------------------------ *
*                                                                      *
*   This listener just waits for download opens (write) and dumps      *
*   all bytes to stdout. The main loop is (unlike TAP.C) not driven    *
*   driven by queue events, but it is driven by requests for data      *
*   (very much like reading from a file).                              *
*                                                                      *
*   For a detailed description of the events take a closer look to     *
*   TAP.C                                                              *
*                                                                      *
*   ------------------------------------------------------------------ *
*                                                                      *
*   Manufacturing: ICC -Ss TAP2.C                                      *
*                                                                      *
*   ------------------------------------------------------------------ *
*                                                                      *
*   No copyright by Markus Schmidt, 1993                               *
*                                                                      *
***********************************************************************/

#define INCL_DOSFILEMGR
#define INCL_DOSQUEUES
#define INCL_DOSMEMMGR
#define INCL_NOCOMMON

#include <os2.h>
#include <stdio.h>

#include "tap.h"


// define events for QueueManager function
enum _QUEUE_OPS {
	WAITFOR,
	GETNEXTBYTE
};


// function prototypes
static void ReadQueue(HQUEUE hq, VOID **ppdata, ULONG *plength, int *px);
static int QueueManager(HQUEUE hq, int op, int op2);


/***********************************************************************
*   Open a queue, wait for an open and then request bytes until EOF    *
*                                                                      *
*                                                                      *
***********************************************************************/
int main()
{
	int error;
	int i;
	char szQueName[]= TAP_QUEUE_NAME;
	HQUEUE hq= 0;


	// Find a free name and open a data queue to receive info
	for (i= 0; i<TAP_MAX && hq==0; i++) {
		DosCreateQueue(&hq, QUE_FIFO, szQueName);
		szQueName[strlen(szQueName)-1]++;
	}

	// queue open?
	if (hq) {
		for (;;) {
			printf("\nWait for open ...\n");
			error= QueueManager(hq, WAITFOR, TAP_EVENT_OPENFORWRITE);
			if (!error) {
				int byte;

				do {
					byte= QueueManager(hq, GETNEXTBYTE, 0);
					printf("%02X  ", byte);
				} while (byte!=EOF);
			}
		}
	}

	// Destroy queue
	DosCloseQueue(hq);

	// terminate
	return (0);
}



/***********************************************************************
*   Wait for a specific tap event or return next byte/EOF              *
*                                                                      *
*                                                                      *
***********************************************************************/
int
QueueManager(HQUEUE hq, int op, int op2)
{
	static unsigned char *data= NULL;
	static int datalength= 0, dataindex=0;
	ULONG length;
	int code, rc=0;
		
		
	// wait for something
	if (op==WAITFOR) {
		// discard old data if waiting for something
		datalength= 0;
		dataindex= 0;
		if (data) {
			DosFreeMem(data);
			data=NULL;
		}

		// wait for it
		do {
			ReadQueue(hq, (VOID**)&data, &length, &code);
			if (code!=op2 && data!=NULL) {
				DosFreeMem(data);
				data= NULL;
			}
		} while (code!=op2);
	}
	else 
	// need input
	if (op==GETNEXTBYTE) {
		// if no data there, wait for a significant event
		if (data==NULL || dataindex>=datalength) {
			// maybe a somehow forgotten block
			if (data!=NULL) {
				DosFreeMem(data);
				data= NULL;
				datalength= 0;
				dataindex= 0;
			}

			// wait for data or close
			for (;;) {

				ReadQueue(hq, (VOID**)&data, &length, &code);

				if (code==TAP_EVENT_CLOSE ||
					code==TAP_EVENT_SEEK) {	// take a seek for CLOSE
					break;
				}
				else 
				if (code==TAP_EVENT_WRITEBLOCK) {
					datalength= length;
					dataindex= 0;
					break;
				}

				if (data!=NULL) {
					DosFreeMem(data);
					data= NULL;
				}
			} 
		}

		if (data==NULL) {
			rc= EOF; // must have been CLOSE or seek
		}
		else {
			rc= data[dataindex++];	// must have been WRITE
		}
	}

	return (rc);
}


/***********************************************************************
*   Read a tap event from the queue                                    *
*                                                                      *
*                                                                      *
***********************************************************************/
void
ReadQueue(HQUEUE hq, VOID **ppdata, ULONG *plength, int *px)
{
	REQUESTDATA rqd;
	BYTE prio;

	DosReadQueue(hq, 
			&rqd,		// ulData-field for eventcode
			plength,	// field for data length or offset
			ppdata,		// field for data pointer
			0, 			// get first element from queue
			0,			// wait synchronously for data
			&prio,		// element priority
			0);			// no semaphore

	*px= rqd.ulData;
	return;
}
