/*************************************************************************
**
**    OLE 2 Automation Controller
**
**    autoctrl.cpp
**
**    Procedures to create application's main window.
**    Procedures to create an automation object, set and get properties 
**    and invoke methods.
**
**     Written by Microsoft Product Support Services, Windows Developer Support
**    (c) Copyright Microsoft Corp. 1993 All Rights Reserved
**
*************************************************************************/

#include <windows.h> 
#include <ole2.h>
#include <dispatch.h>    
#include <olenls.h>     
#include "autoctrl.h"          

// Globals
HINSTANCE g_hinst;                          // Instance of application
HWND      g_hwnd;                           // Toplevel window handle

// String resource buffers
char g_szTitle[STR_LEN];                    // Main window caption
char g_szCannotDisplayReturnValue[STR_LEN]; // "Cannot display return value"
char g_szError[STR_LEN];                    // "Error"

/*
 * WinMain
 *
 * Purpose:
 *  Main entry point of application. Should register the app class
 *  if a previous instance has not done so and do any other one-time
 *  initializations.
 *
 */
int PASCAL WinMain (HINSTANCE hinst, HINSTANCE hinstPrev, LPSTR lpCmdLine, int nCmdShow)
{
   MSG msg;

   LoadString(hinst, IDS_PROGNAME, g_szTitle, STR_LEN);
   LoadString(hinst, IDS_CANNOTDISPLAYRETURNVALUE, g_szCannotDisplayReturnValue, STR_LEN);
   LoadString(hinst, IDS_ERROR, g_szError, STR_LEN);
   
   if (!hinstPrev)
      if (!InitApplication(hinst))
         return (FALSE);

   if(!InitOle())
      return FALSE;
      
   if (!InitInstance(hinst, nCmdShow))
      return (FALSE);

   while (GetMessage(&msg, NULL, NULL, NULL))
   {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }
   
   UninitOle();
   
   return (msg.wParam); // Returns the value from PostQuitMessage
}

/*
 * InitApplication
 *
 * Purpose:
 *  Registers window class
 *
 * Parameters:
 *  hinst       hInstance of application
 *
 * Return Value:
 *  TRUE if initialization succeeded, FALSE otherwise.
 */
BOOL InitApplication (HINSTANCE hinst)
{
   WNDCLASS wc;

   wc.style = CS_DBLCLKS;
   wc.lpfnWndProc = MainWndProc;
   wc.cbClsExtra = 0;
   wc.cbWndExtra = 0;
   wc.hInstance = hinst;
   wc.hIcon = LoadIcon(hinst, "ControlIcon");
   wc.hCursor = LoadCursor(NULL, IDC_ARROW);
   wc.hbrBackground = COLOR_WINDOW + 1;
   wc.lpszMenuName = "ControlMenu";
   wc.lpszClassName = "MainWndClass";
     
   return RegisterClass(&wc);
 }

/*
 * InitInstance
 *
 * Purpose:
 *  Creates and shows main window
 *
 * Parameters:
 *  hinst           hInstance of application
 *  nCmdShow        specifies how window is to be shown
 *
 * Return Value:
 *  TRUE if initialization succeeded, FALSE otherwise.
 */
BOOL InitInstance (HINSTANCE hinst, int nCmdShow)
{
  
   g_hinst = hinst;
   // Create Main Window
   g_hwnd = CreateWindow("MainWndClass", g_szTitle,
                       WS_OVERLAPPEDWINDOW,
                       CW_USEDEFAULT, CW_USEDEFAULT,
                       400, 200,
                       NULL, NULL, hinst, NULL);
   if (!g_hwnd)
      return FALSE;
   
   ShowWindow(g_hwnd, nCmdShow);                  
   UpdateWindow(g_hwnd);            
   return TRUE;
}

/*
 * InitOle
 *
 *  Purpose:
 *   Initialize OLE
 *
 * Return Value:
 *    TRUE if OLE initialization succeeded, FALSE otherwise.
 *
 */
BOOL InitOle(void)
{
    if(OleInitialize(NULL) != 0)
      return FALSE;

    return TRUE;
}

/*
 * UnitinitOle
 *
 *  Purpose:
 *   Unitialize OLE
 *
 */
void UninitOle()
{
    OleUninitialize();
}

/*
 * MainWndProc
 *
 * Purpose:
 *  Window procedure for main window
 *
 */
LRESULT __export CALLBACK MainWndProc (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
   static LPDISPATCH pdisp = NULL;    // Holds IDispatch of automation object
   DLGDATA dlgdata;                   // User entered data from dialogs
   int nRet;                          // Indicates if user pressed OK or Cancel in dialogs
   
   switch (msg)
   {  
      case WM_CREATE:
          // Create Static Window to display return value of properties or methods
          //    exposed by the automation object
          CreateWindow("Static", "", WS_CHILD | WS_VISIBLE | SS_LEFT,
                        20, 20, 200, 100, hwnd, IDC_RETURNVALUE, g_hinst, NULL); 
          return 0L;
                
      case WM_COMMAND:
         switch (wParam)
         {
            case IDM_CREATEOBJECT:  
                // Bring up dialog to prompt for ProgID. Create automation pbject using
                //  ProgID.    
                nRet = DialogBoxParam(g_hinst, MAKEINTRESOURCE(IDD_CREATEOBJECT), 
                               hwnd, (DLGPROC)CreateObjectDlgFunc, (LPARAM)(LPDLGDATA)&dlgdata); 
                if (nRet == IDOK)             
                    CreateObject(dlgdata.szName,  // ProgID
                                 &pdisp);
                return 0L;
                               
            case IDM_SETPROPERTY:
                // Prompt user for property to be set and the value of the property. Set
                //  the property.
                nRet = DialogBoxParam(g_hinst, MAKEINTRESOURCE(IDD_SETPROPERTY), 
                               hwnd, (DLGPROC)SetPropertyDlgFunc, (LPARAM)(LPDLGDATA)&dlgdata); 
                if (nRet == IDOK)    
                    SetProperty(dlgdata.szName,  // Property name
                                dlgdata.szValue, // Property value (stored as a string)
                                dlgdata.vt,      // Property Type
                                dlgdata.lcid,    // Locale ID
                                pdisp);
                return 0L;
                
            case IDM_GETPROPERTY:
                // Prompt user for property to get. Get the property.
                nRet = DialogBoxParam(g_hinst, MAKEINTRESOURCE(IDD_GETPROPERTY), 
                               hwnd, (DLGPROC)GetPropertyDlgFunc, (LPARAM)(LPDLGDATA)&dlgdata); 
                if (nRet == IDOK)    
                    GetProperty(dlgdata.szName, // Property name 
                                dlgdata.lcid,   // Locale ID
                                pdisp);
                return 0L;
              
            case IDM_INVOKEMETHOD:
                // Prompt user for method to invoke. Invoke the method.
                nRet = DialogBoxParam(g_hinst, MAKEINTRESOURCE(IDD_INVOKEMETHOD), 
                               hwnd, (DLGPROC)InvokeMethodDlgFunc, (LPARAM)(LPDLGDATA)&dlgdata); 
                if (nRet == IDOK)    
                    InvokeMethod(dlgdata.szName,  // Property name
                                 dlgdata.lcid,    // Locale ID
                                 pdisp);
                return 0L;
                
            case IDM_RELEASEOBJECT:
                // Release the automation object
                pdisp->Release();
                pdisp = NULL;
                DisplayReturnValue(NULL); // Clear return value display
                return 0L;
         }
         break;
         
      case WM_INITMENUPOPUP:
      {
         HMENU hmenu = (HMENU)wParam;
         
         if (LOWORD(lParam) != 0)
            return 0L;
            
         // Enable or gray the appropriate menu items. pdisp indicates if an automation object
         //  is currently being controlled.   
         EnableMenuItem(hmenu, IDM_CREATEOBJECT,  MF_BYCOMMAND | (pdisp?MF_GRAYED:MF_ENABLED));
         EnableMenuItem(hmenu, IDM_SETPROPERTY,   MF_BYCOMMAND | (pdisp?MF_ENABLED:MF_GRAYED)); 
         EnableMenuItem(hmenu, IDM_GETPROPERTY,   MF_BYCOMMAND | (pdisp?MF_ENABLED:MF_GRAYED));
         EnableMenuItem(hmenu, IDM_INVOKEMETHOD,  MF_BYCOMMAND | (pdisp?MF_ENABLED:MF_GRAYED));
         EnableMenuItem(hmenu, IDM_RELEASEOBJECT, MF_BYCOMMAND | (pdisp?MF_ENABLED:MF_GRAYED));
         return 0L;
      }
      
      case WM_DESTROY:
         if (pdisp)
            pdisp->Release();                
         PostQuitMessage(0);
         break;

      default:                         
         return DefWindowProc(hwnd, msg, wParam, lParam);
   }
   
   return NULL;
}

/*
 * CreateObject
 *
 * Purpose:
 *  Creates an instance of the Automation object and obtains it's IDispatch interface
 *
 * Parameters:
 *  lpszProgID         ProgID of Automation object
 *  ppdisp             Returns IDispatch of Automation object
 *
 * Return Value:
 *  HRESULT indicating success or failure 
 */
HRESULT CreateObject(LPSTR lpszProgID, IDispatch FAR* FAR* ppdisp)
{
    CLSID clsid;                   // CLSID of automation object
    HRESULT hr;
    LPUNKNOWN punk = NULL;         // IUnknown of automation object
    LPDISPATCH pdisp = NULL;       // IDispatch of automation object
    
    *ppdisp = NULL;
    
    // Retrieve CLSID from the progID that the user specified
    hr = CLSIDFromProgID(lpszProgID, &clsid);
    if (FAILED(hr))
    {   
          ReportError(hr, "CLSIDFromProgID", __FILE__, __LINE__);
          goto error;
    }
    
    // Create an instance of the automation object and ask for the IDispatch interface
    hr = CoCreateInstance(clsid, NULL, CLSCTX_SERVER, 
                          IID_IUnknown, (void FAR* FAR*)&punk);
    if (FAILED(hr))
    {   
          ReportError(hr, "CoCreateInstance", __FILE__, __LINE__);
          goto error;
    }           
    hr = punk->QueryInterface(IID_IDispatch, (void FAR* FAR*)&pdisp);
    if (FAILED(hr))
    {   
         ReportError(hr, "IUnknown::QueryInterface - IID_IDispatch", __FILE__, __LINE__);
         goto error;
    }
    *ppdisp = pdisp;
    punk->Release();
    return NOERROR;
     
error:
    if (punk) punk->Release();
    if (pdisp) pdisp->Release();
    DisplayReturnValue(NULL); // Clear return value display
    return hr;
}  

/*
 * SetProperty
 *
 * Purpose:
 *  Sets the value of a property exposed by an automation object. Note that setting
 *  the property requires a named parameter with DISPID == DISPID_PROPERTYPUT which
 *  represents the value of the property.
 *
 * Parameters:
 *  lpszPropName       Name of property
 *  lpszValue          Value to set the property to. (This value was read a string and
 *                      will be converted to the type that the user specified)
 *  vt                 Type of the property value.
 *  lcid               Locale ID indicating language used by the user of the 
 *                     automation controller.
 *  pdisp              IDispatch of automation object.
 *
 * Return Value:
 *  HRESULT indicating success or failure  
 */
HRESULT SetProperty(LPSTR lpszPropName, LPSTR lpszValue, VARTYPE vt, LCID lcid, LPDISPATCH pdisp)
{
      VARIANTARG varg;              
      DISPPARAMS disp;
      HRESULT hr;
      EXCEPINFO excepInfo;
      unsigned int uArgErr;
      DISPID dispidPropertyPut = DISPID_PROPERTYPUT;
      DISPID dispidProperty;
      
      // Set a VARIANTARG to the value to be assigned to the property. The value was
      //   read in from the user as string. This will be converted to the type that
      //   the user specified.
      VariantInit(&varg);
      V_VT(&varg) = VT_BSTR;
      V_BSTR(&varg) = SysAllocString(lpszValue);  
      
      // Change the string that represents the property value to the type that 
      //   the user specified
      hr = VariantChangeType(&varg, &varg, 0, vt);
      if (FAILED(hr))  
      {   
          ReportError(hr, "VariantChangeType", __FILE__, __LINE__);
          goto error;
      }
      
      // Specify information about parameters that the property takes. Setting
      // a property requires an implicit named parameter with DIPID == DISPID_PROPERTYPUT.
      // This named parameter represents the value that the property is assigned. 
      disp.rgvarg = &varg;
      disp.rgdispidNamedArgs = &dispidPropertyPut;
      disp.cArgs = 1;
      disp.cNamedArgs = 1;
      
      // Get the dispid of the property  
      hr = pdisp->GetIDsOfNames(IID_NULL, &lpszPropName, 1, lcid, &dispidProperty);
      if(FAILED(hr))
      {   
          ReportError(hr, "IDispatch::GetIDsOfNames", __FILE__, __LINE__);
          goto error;
      }
      // Set the value of the property                              
      hr = pdisp->Invoke(dispidProperty, IID_NULL, lcid, DISPATCH_PROPERTYPUT, &disp, 
                   NULL, &excepInfo, &uArgErr);
      if(FAILED(hr))
      {   
          ReportError(hr, "IDispatch::Invoke", __FILE__, __LINE__);
          goto error;
      }
      DisplayReturnValue(NULL); // No return value
      return NOERROR;  
      
error:
      VariantClear(&varg); 
      DisplayReturnValue(NULL); // Clear return value display
      return hr;              
}

/*
 * GetProperty
 *
 * Purpose:
 *  Gets the value of a property exposed by an automation object.
 *
 * Parameters:
 *  lpszPropName       Name of property
 *  lcid               Locale ID indicating language used by the user of the 
 *                     automation controller.
 *  pdisp              IDispatch of automation object.
 *
 * Return Value:
 *  HRESULT indicating success or failure  
 */
HRESULT GetProperty(LPSTR lpszPropName, LCID lcid, LPDISPATCH pdisp)
{
      DISPPARAMS disp;
      HRESULT hr;
      VARIANTARG vargResult;
      EXCEPINFO excepInfo;
      unsigned int uArgErr;
      DISPID dispidProperty; 
      
      // Initialize the variant that will hold the return value
      VariantInit(&vargResult);      
      
      // Specify information about the parameters that the property takes. This
      // property takes no parameters.  
      disp.rgvarg = NULL;
      disp.rgdispidNamedArgs = NULL;
      disp.cArgs = 0;
      disp.cNamedArgs = 0;
      
      // Get the dispid of the property   
      hr = pdisp->GetIDsOfNames(IID_NULL, &lpszPropName, 1, lcid, &dispidProperty);
      if(FAILED(hr))
      {   
          ReportError(hr, "IDispatch::GetIDsOfNames", __FILE__, __LINE__);
          goto error;
      }
      // Get the value of the property and display it.                              
      hr = pdisp->Invoke(dispidProperty, IID_NULL, lcid, DISPATCH_PROPERTYGET, &disp, 
                   &vargResult, &excepInfo, &uArgErr);
      if(FAILED(hr))
      {   
          ReportError(hr, "IDispatch::Invoke", __FILE__, __LINE__);
          goto error;
      }
      DisplayReturnValue(&vargResult); 
      VariantClear(&vargResult); // Free any resources contained in the variant              
      return NOERROR;      
      
error:
      DisplayReturnValue(NULL); // Clear return value display
      VariantClear(&vargResult); // Free any resources contained in the variant  
      return hr; 
}

/*
 * InvokeMethod
 *
 * Purpose:
 *  Invokes a method exposed by an automation object. It is assumed that the 
 *  method does not take any parameters.
 *
 * Parameters:
 *  lpszMethodName     Name of method
 *  lcid               Locale ID indicating language used by the user of the 
 *                     automation controller.
 *  pdisp              IDispatch of automation object.
 *
 * Return Value:
 *  HRESULT indicating success or failure  
 */
HRESULT InvokeMethod(LPSTR lpszMethodName, LCID lcid, LPDISPATCH pdisp)
{
      DISPPARAMS disp;
      HRESULT hr;
      VARIANTARG vargResult;
      EXCEPINFO excepInfo;
      unsigned int uArgErr;
      DISPID dispidMethod; 
      
      // Initialize the variant that will hold the return value
      VariantInit(&vargResult);      
      
      // Specify information about the parameters that the method takes. This
      // method takes no parameters.   
      disp.rgvarg = NULL;
      disp.rgdispidNamedArgs = NULL;
      disp.cArgs = 0;
      disp.cNamedArgs = 0;
      
      // Get the dispid of the method 
      hr = pdisp->GetIDsOfNames(IID_NULL, &lpszMethodName, 1, lcid, &dispidMethod);
      if(FAILED(hr))
      {   
          ReportError(hr, "IDispatch::GetIDsOfNames", __FILE__, __LINE__);
          goto error;
      }
      // Invoke the method and display return value, if any.                             
      hr = pdisp->Invoke(dispidMethod, IID_NULL, lcid, DISPATCH_METHOD, &disp, 
                   &vargResult, &excepInfo, &uArgErr);
      if(FAILED(hr))
      {   
          ReportError(hr, "IDispatch::Invoke", __FILE__, __LINE__);
          goto error;
      } 
      DisplayReturnValue(&vargResult);
      VariantClear(&vargResult); // Free any resources contained in the variant        
      return NOERROR;      
      
error:
      DisplayReturnValue(NULL); // Clear return value display
      VariantClear(&vargResult); // Free any resources contained in the variant   
      return hr; 
}

/*
 * DisplayReturnValue
 *
 * Purpose:
 *  Displays return value of a method or property in a static child window.
 *  The return type is coerced to a string so it can be displayed by a static
 *  window.
 *
 * Parameters:
 *  lpvarg             Return value of property or function.
 *  
 */
void DisplayReturnValue(LPVARIANTARG lpvarg)
{
   LPSTR lpszReturnVal;
   HRESULT hr;
   
   if (lpvarg == NULL || (lpvarg && lpvarg->vt == VT_EMPTY))  // No return value 
       lpszReturnVal = ""; 
   else 
   {   
       // Try to coerce the return type to a string, otherwise it can't be displayed 
       hr = VariantChangeType(lpvarg, lpvarg, 0, VT_BSTR);
       if (FAILED(hr))
           lpszReturnVal = g_szCannotDisplayReturnValue;
       else lpszReturnVal = (LPSTR)V_BSTR(lpvarg);
   }    
   SetDlgItemText(g_hwnd, IDC_RETURNVALUE, lpszReturnVal);
}  


