**************************************************************************
*                                                                        *
*  HPDUMP.PRG                                                            *
*                                                                        *
*  Dumps source code listings to printer in a single page format, or     *
*  double page format depending on the value specified for cDescript -   *
*  "1", or "2".                                                          *
*                                                                        *
**************************************************************************

#define PAPER     "LETTER"
#define PRINTER   "II"
#define COPIES    1
#define HEADFONT  { "COURIER 10", "I" }
#define BODYFONT  { "LINEPRT 16", "I" }
#define FOOTFONT  { "COURIER 10", "I" }
#define PAGEAREA1 { 1, 1, 10.5, 8.25 }          // Descript 1
#define PAGEAREA2 { { 1, .2, 8.25, 5.4 }, ;     // Descript 2, panel 1
                    { 1, 5.6, 8.25, 10.75 } }   // Descript 2, panel 2
#define PTOP      1
#define PLEFT     2
#define PBOTTOM   3
#define PRIGHT    4

#define NSTR(x) ALLTRIM(STR(x))

FUNCTION hpDump( cFileName, cDescript )

  IF PCOUNT() == 0
    ErrorShow( "Usage: ESC <filename> [<print format (1 = single page, 2 = double page)>]" )
  ENDIF

  // If second parameter is NIL, assume 1st param is filename
  IF cDescript == NIL
    cDescript := "1"
  ENDIF

  // Locate file to print
  IF ! FILE( cFileName )
    ErrorShow( "File " + UPPER( cFileName ) + " not found!" )
  ENDIF

  SETCOLOR('w/b')
  @ 3,0 CLEAR
  SETCOLOR('w+/bg')
  @ 0,0,2,79 BOX 'Ŀ '
  @ 1,2 SAY 'Escape ver 3.0 - Sample Source Printing Program'

  // Display screen message
  ShowBox('Printing ' + cFileName + '...')

  DO CASE
    CASE cDescript == "1"
      PrintPage( cFileName )
    CASE cDescript == "2"
      Print2Page( cFileName, .F. )
    CASE cDescript == "3"
      Print2Page( cFileName, .T. )
  ENDCASE

RETURN NIL

*****************************************************************************
* FUNCTION ERRORSHOW() : Displays error and quits program
*****************************************************************************
STATIC FUNCTION ErrorShow( cMessage )
  TONE( 1000 )
  hpPrintOff()
  ?
  ? cMessage
  ?
  QUIT
RETURN NIL


*****************************************************************************
* PRINTPAGE() : Prints file using single page, portrait format
*****************************************************************************
STATIC FUNCTION PrintPage( cFileName )

  LOCAL nHandle, cLine
  LOCAL nPage := 0, nLine := 1, nRow := 11
  LOCAL lReturn := .T.              // Carriage return?

  IF( nHandle := FOPEN( cFileName ) ) < 0
    ErrorShow( UPPER( cFileName ) + " could not be opened." )
  ENDIF

  hpPrintOn()

  // configure printer
  hpConfig( PAPER, "P", COPIES, PRINTER )

  // Set to physical mode
  hpPhysical( .T. )

  DO WHILE .T.

    // Compose page elements
    IF nRow > PAGEAREA1[PBOTTOM]-.3

      // Eject page?
      IF nPage > 0
        EJECT
      ENDIF
      nPage++

      // Start row...
      nRow := PAGEAREA1[PTOP] + .1

      // Set font
      hpSetFont( HEADFONT[1],  HEADFONT[2] )

      // Draw lines across top and bottom of page
      hpIBox( PAGEAREA1[PTOP]-.25, PAGEAREA1[PLEFT]-.25, ;
        PAGEAREA1[PTOP]-.25, PAGEAREA1[PRIGHT]+.25, 1 )
      hpIBox( PAGEAREA1[PBOTTOM]-.25, PAGEAREA1[PLEFT]-.25, ;
        PAGEAREA1[PBOTTOM]-.25, PAGEAREA1[PRIGHT]+.25, 1 )

      // Print filename box and filename, and date
      hpIGrayBox( PAGEAREA1[PTOP]-.5, PAGEAREA1[PLEFT], ;
        PAGEAREA1[PTOP]-.25, PAGEAREA1[PLEFT] + 1.75, 0, 20 )
      hpIAtSay( PAGEAREA1[PTOP]-.35, PAGEAREA1[PLEFT]+.1, ;
        hpBold( UPPER( cFileName ) ) )
      hpIRJust( DTOC( DATE() ) + ", " + TIME(), PAGEAREA1[PTOP]-.4 )

      // And the page number
      hpICenter( "Page " + LTRIM( STR( nPage ) ), PAGEAREA1[PBOTTOM] )

      // Set the body font
      hpSetFont( BODYFONT[1],  BODYFONT[2] )

    ENDIF

    // update screen status
    hpPrintOff()
    @ 13,27 SAY PADC('Page '+NSTR(nPage)+' line '+NSTR(nLine), 26 )
    hpPrintOn()

    // If a carriage return exists in previous line, use whole line, 
    // else indent the line
    IF lReturn
      cLine := LineIn( nHandle, PAGEAREA1[PRIGHT]-PAGEAREA1[PLEFT]-.2 )
    ELSE
      cLine := LineIn( nHandle, PAGEAREA1[PRIGHT]-PAGEAREA1[PLEFT]-2.2 )
    ENDIF

    // Have we reached end of file?
    IF cLine == NIL
      EXIT
    ENDIF

    // Print the line
    IF lReturn
      hpIAtSay( nRow, PAGEAREA1[PLEFT], cLine )
    ELSE
      hpIAtSay( nRow, PAGEAREA1[PLEFT]+2, cLine )
    ENDIF

    // Check if current line has a carriage return
    lReturn := CHR(13) $ cLine

    IF lReturn
      nLine++
    ENDIF

    // Increment the row
    nRow += .125

  ENDDO

  hpReset()
  hpPrintOff()

  // close file
  FCLOSE( nHandle )

RETURN .T.


*****************************************************************************
* PRINT2PAGE() : Prints file using double page, landscape format
*****************************************************************************
STATIC FUNCTION Print2Page( cFileName, lNumber )

  LOCAL nHandle, cLine
  LOCAL nPage := 0, nRow := 11, nLine := 1
  LOCAL lReturn := .T.
  LOCAL nPanel := 2

  IF( nHandle := FOPEN( cFileName ) ) < 0
    ErrorShow( UPPER( cFileName ) + " could not be opened." )
  ENDIF

  hpPrintOn()

  // configure printer
  hpConfig( PAPER, "L", COPIES, PRINTER )

  // Set to physical mode
  hpPhysical( .T. )

  DO WHILE .T.

    // Compose page elements
    IF nRow > PAGEAREA2[nPanel,PBOTTOM]-.3

      // Eject page only valid after printing on second panel
      IF nPanel == 2
        IF nPage > 0
          EJECT
        ENDIF
        nPage++
      ENDIF

      // Determine which panel to print next
      IF nPanel == 1
        nPanel := 2
      ELSE
        nPanel := 1
      ENDIF

      // Print page elements (only valid during first panel)
      IF nPanel == 1

        // Set font
        hpSetFont( HEADFONT[1],  HEADFONT[2] )

        // Draw lines across top and bottom of page
        hpIBox( PAGEAREA2[1,PTOP]-.25, PAGEAREA2[1,PLEFT]-.25, ;
          PAGEAREA2[1,PTOP]-.25, PAGEAREA2[2,PRIGHT]+.25, 1 )
        hpIBox( PAGEAREA2[1,PBOTTOM]-.25, PAGEAREA2[1,PLEFT]-.25, ;
          PAGEAREA2[1,PBOTTOM]-.25, PAGEAREA2[2,PRIGHT]+.25, 1 )
        hpIBox( PAGEAREA2[1,PTOP]-.25, 5.5, PAGEAREA2[1,PBOTTOM]-.25, 5.5, 1 )

        // Print filename box and filename, and date
        hpIGrayBox( PAGEAREA2[1,PTOP]-.5, PAGEAREA2[1,PLEFT], ;
          PAGEAREA2[1,PTOP]-.25, PAGEAREA2[1,PLEFT] + 1.75, 0, 20 )
        hpIAtSay( PAGEAREA2[1,PTOP]-.35, PAGEAREA2[1,PLEFT]+.1, ;
          hpBold( UPPER( cFileName ) ) )
        hpIRJust( DTOC( DATE() ) + ", " + TIME(), PAGEAREA2[1,PTOP]-.4 )

        // And the page number
        hpICenter( "Page " + LTRIM( STR( nPage ) ), PAGEAREA2[1,PBOTTOM] )

        // Set the body font
        hpSetFont( BODYFONT[1],  BODYFONT[2] )

      ENDIF

      // Reset the row position
      nRow := PAGEAREA2[nPanel, PTOP] + .1

    ENDIF

    // update screen status
    hpPrintOff()
    @ 13,27 SAY PADC('Page '+NSTR(nPage)+' line '+NSTR(nLine), 26 )
    hpPrintOn()

    // If a carriage return exists in previous line, use whole line, 
    // else indent the line
    IF lReturn
      cLine := LineIn( nHandle, ;
        PAGEAREA2[nPanel,PRIGHT]-PAGEAREA2[nPanel,PLEFT]-.2- ;
        IF( lNumber, .25, 0 ) )
    ELSE
      cLine := LineIn( nHandle, ;
        PAGEAREA2[nPanel,PRIGHT]-PAGEAREA2[nPanel,PLEFT]-1.2 )
    ENDIF

    // Have we reached end of file?
    IF cLine == NIL
      EXIT
    ENDIF

    // Print the line
    IF lReturn
      hpIAtSay( nRow, PAGEAREA2[nPanel,PLEFT], IF( lNumber, STR(nLine,5 ) + " ", "" ) + cLine )
    ELSE
      hpIAtSay( nRow, PAGEAREA2[nPanel,PLEFT]+1, cLine )
    ENDIF

    // Check if current line has a carriage return
    lReturn := CHR(13) $ cLine

    IF lReturn
      nLine++
    ENDIF

    // Increment the row
    nRow += .125

  ENDDO

  hpReset()
  hpPrintOff()

  // close file
  FCLOSE( nHandle )

RETURN .T.


*****************************************************************************
* LINEIN() : Returns a line up to a length of nInchLen or a carriage return
*****************************************************************************
STATIC FUNCTION LineIn( nHandle, nInchLen )
  LOCAL cLine, cNext80, nPos
  STATIC cBuffer := ""

  // Read data until carriage return found
  DO WHILE !( CHR(13) $ cBuffer )
    cNext80 := FREADSTR(nHandle, 80)
    IF cNext80 == ""
      EXIT
    ENDIF
    cBuffer += cNext80
  ENDDO

  // if no characters in buffer, indicate EOF
  IF cBuffer == ""
    RETURN (NIL)
  END

  // Remove line feeds
  cBuffer := STRTRAN(cBuffer, CHR(10))

  IF (nPos := AT(CHR(13), cBuffer)) == 0
    nPos := LEN( cBuffer )
  ENDIF

  // Go backwards to find string of proper length
  DO WHILE hpS2I( LEFT( cBuffer, nPos ) ) > nInchLen .AND. nPos > 0
    nPos--
  ENDDO

  // Get line and remove line from buffer
  cLine := LEFT( cBuffer, nPos )
  cBuffer := SUBSTR( cBuffer, nPos+1 )

RETURN (cLine)


