/***********************************************************************
*$Header:   J:/gedcom/gedlib/vcs/gedmem.c_v   1.8   25 Nov 1992 11:14:26   fhdodj  $
*
*$config$="/K! /L/* /R* /Mgedmem.c"
*!global paths!
*   gedcom\library\gedmem.c
*   gedcom\all\gedmem.c
*!end!
*
*   FILE NAME: GEDMEM.C
*
*   DESCRIPTION:
*       This file contains functions for memory management.
*
*   ROUTINES:
*
*
*$Log:   J:/gedcom/gedlib/vcs/gedmem.c_v  $
 * 
 *    Rev 1.8   25 Nov 1992 11:14:26   fhdodj
 * Fixed bug in ged_alloc_pool. It was not word aligning the pieces it was
 * allocating.
 * 
 *    Rev 1.7   05 Aug 1992 19:04:06   fhdodj
 * split out memory mark functions.
 * 
 *    Rev 1.6   26 Mar 1992 10:31:34   fhdodj
 * Made ged_check_mem a macro.
 * 
 *    Rev 1.5   25 Mar 1992 15:52:36   fhdodj
 * Added the function ged_pool_size(). Changed ged_check_mem to be a little
 * faster.
 * 
 *    Rev 1.4   21 Oct 1991 10:10:48   fhdodj
 * Changed char to byte.
 * 
 *    Rev 1.3   28 Jun 1991 14:52:22   fhdkrf
 * Add keywords for PolyDoc
 * 
 *    Rev 1.2   18 Jun 1991 08:11:54   odj
 * Fixed typo
 * 
 *    Rev 1.1   18 Jun 1991 07:54:42   odj
 * Changed chunk chain so that new chunks are added to the front of the list
 * instead of the end.  Added calls to ged_error() where chunk header has
 * been overwritten, or out of memory.
 * 
 *    Rev 1.0   20 Dec 1990 09:06:26   odj
 * Initial revision.
***********************************************************************/
#include "gedcom.h"

#ifdef NON_ANSI
#define max(a, b) ((a) < (b) ? (b) : (a))
#endif

/********************************************************************
*	 FILE: GEDMEM.C
*
*        DESCRIPTION:
*            This is a proposed starting point for a standardized
*            approach to dynamic memory management. Possible issues
*            to be discussed are:
*                        Extended Memory
*                        Record Caching
*                        Machine Independence
*                        Others
*
********************************************************************/

/* This is where the pool currently in use is kept.                   */
static POOL *curPool = NULL;

/********************************************************************
*!name!
*    ged_check_mem()
*!1!
*
*       NAME: ged_check_mem
*
*       DESCRIPTION:
*           Checks the current pool for corrupted chunk pointers.
*           If a memory violation is found, a message is displayed,
*           and the program exits.
*
*       RETURN VALUE:
*           None.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
#define GED_CHECK_MEM()							\
    {			/*!end!*/ 					\
    if (curPool)							\
        {								\
        CHUNK *tmpChunk = curPool->firstChunk;				\
        byte *bufSize;							\
									\
        while (tmpChunk)						\
            {								\
            if (((bufSize = tmpChunk->bufPtr + tmpChunk->size) !=	\
                                             tmpChunk->endPtr) ||	\
                ((bufSize - tmpChunk->free)  != tmpChunk->curPtr))	\
                ged_error(1, NULL);					\
            tmpChunk = tmpChunk->nextChunk;				\
            }								\
	}								\
    else								\
        return(NULL);							\
    }
/**/
/********************************************************************
*!name!
*    ged_alloc_pool()
*!1!
*
*       NAME: ged_alloc_pool
*
*       DESCRIPTION:
*           Allocates the memory requested from the pool
*
*       RETURN VALUE:
*           A pointer to the beginning of the requested memory or
*           NULL if there is not enough memory in pool.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
byte * ged_alloc_pool(size)
    unsigned long size;
    {			/*!end!*/
    CHUNK *chunk;
    byte  *memAddress = NULL;

        if (size % WORD_SIZE)
            size += WORD_SIZE - (size % WORD_SIZE);
	chunk = ged_get_chunk(size);
	if (chunk)
	    {
	    memAddress = chunk->curPtr;

	    /* Allocate the memory requested from the pool             */
	    chunk->curPtr += size;
	    chunk->free -= size;
	    }
	return(memAddress);
    }
/**/
/********************************************************************
*!name!
*    ged_create_pool()
*!1!
*
*       NAME: ged_create_pool
*
*       DESCRIPTION:
*           Creates a memory pool, and allocates memory for it.
*           Inserts defaults for node size, and succesive chunk
*           sizes that are to be allocated.
*
*       RETURN VALUE:
*           The address of the pool, or NULL if unsuccessful.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
POOL * ged_create_pool(size, nodeSize)
    unsigned size;
    unsigned nodeSize;
    {			/*!end!*/
    POOL *newPool;
    
        if (size % WORD_SIZE)
            size += WORD_SIZE - (size % WORD_SIZE);
        if (nodeSize % WORD_SIZE)
            nodeSize += WORD_SIZE - (nodeSize % WORD_SIZE);

        /* allocate  the memory                                    */
        if ((newPool = (POOL *) (malloc(sizeof(POOL)  +
                                        sizeof(CHUNK) +
                                        size))) == NULL)
            {
            ged_error(2, NULL);
            return(NULL);   /* malloc failed                       */
	    }

        /* initialize the pool's slots                             */
        newPool->firstChunk = (CHUNK *) (newPool + 1);
        newPool->firstChunk->curPtr = newPool->firstChunk->bufPtr =
                        (byte *) (newPool->firstChunk + 1);
        newPool->firstChunk->endPtr = (newPool->firstChunk->bufPtr + size);
        newPool->firstChunk->free = newPool->firstChunk->size = size;
        newPool->size = size;
        newPool->nodeSize = nodeSize;
        newPool->ged_alloc_pool = ged_alloc_pool;
        newPool->ged_reset_pool = ged_reset_pool;
        newPool->ged_destroy_pool = ged_destroy_pool;
        newPool->firstChunk->nextChunk = NULL;
        return(newPool);
    }
/**/
/********************************************************************
*!name!
*    ged_destroy_pool()
*!1!
*
*       NAME: ged_destroy_pool
*
*       DESCRIPTION:
*           free the memory used by the pool
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
void ged_destroy_pool()
    {			/*!end!*/
        ged_reset_pool();
        memset(curPool, 0, (unsigned) curPool->firstChunk->size +
                                      sizeof(CHUNK) + sizeof(POOL));
        free(curPool);
    }

/********************************************************************
*!name!
*    ged_get_pool()
*!1!
*
*        NAME: ged_get_pool
*
*        DESCRIPTION: 
*            gets the pool currently being used.
*
*        RETURN VALUE:
*            A pointer to the pool.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
POOL * ged_get_pool()
    {			/*!end!*/
            
        return(curPool);
    }
/**/
/********************************************************************
*!name!
*    ged_reset_pool()
*!1!
*
*       NAME: ged_reset_pool
*
*       DESCRIPTION:
*           reset the pointer to the beginning of the first chunk,
*            and free the memory in the rest of the chunks.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
short ged_reset_pool()
    {			/*!end!*/
    CHUNK *chunk,
          *nextChunk;

        GED_CHECK_MEM();
        chunk = curPool->firstChunk->nextChunk;
        curPool->firstChunk->curPtr = curPool->firstChunk->bufPtr;
        while (chunk)
            {
            nextChunk = chunk->nextChunk;
            memset(chunk, 0, (unsigned) chunk->size + sizeof(CHUNK));
            free(chunk);
            chunk = nextChunk;
            }
        curPool->firstChunk->nextChunk = NULL;
        curPool->firstChunk->free = curPool->firstChunk->size;
        return(1);
    }

/********************************************************************
*!name!
*    ged_set_pool()
*!1!
*
*        NAME: ged_set_pool
*
*        DESCRIPTION:
*            Sets the current pool.
*
*        RETURN VALUE:
*            The input pool.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
POOL * ged_set_pool(pool)
    POOL *pool;
    {			/*!end!*/
        return(curPool = pool);
    }
/**/
/********************************************************************
*!name!
*    ged_pool_size()
*!1!
*
*       NAME: ged_pool_size
*
*       DESCRIPTION:
*           Returns the size of the pool.
*
*       RETURN VALUE:
*           An unsigned long telling how much memory is being used by the 
*           pool, or NULL if there is no pool.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
unsigned long ged_pool_size()
    {			/*!end!*/
    unsigned long size = 0;
    CHUNK *chunk = NULL;

        GED_CHECK_MEM();
        chunk = curPool->firstChunk;

        /* add up how much memory has been allocated. */
        while (chunk)
	    {
	    size += chunk->size;
            chunk = chunk->nextChunk;
	    }
        return(size);
    }
/********************************************************************
*
*       NAME: ged_get_chunk
*
*       DESCRIPTION:
*           Find the chunk in the pool with the most available memory.
*
*       RETURN VALUE:
*           A pointer to the chunk with most memory or NULL if
*           there is no pool, or no chunk with memory available.
*
********************************************************************/
CHUNK * ged_get_chunk(size)
    unsigned long size;
    {
    CHUNK *chunk = NULL;
    CHUNK *nextChunk;
    unsigned long adjustSize = (unsigned)(size % WORD_SIZE);
    unsigned long allocSize;

	GED_CHECK_MEM();
	chunk = curPool->firstChunk;
	if (adjustSize)
	    size += WORD_SIZE - adjustSize;

	/* find a place in the pool with enough memory left        */
	while (chunk && (chunk->free < size))
	    chunk = chunk->nextChunk;

	/* if there was no room, allocate more memory to the pool  */
	if (!chunk)
	    {
	    allocSize = max(size, curPool->size);
	    nextChunk = curPool->firstChunk->nextChunk;
	    if ((chunk = curPool->firstChunk->nextChunk = (CHUNK *)
			 malloc(sizeof(CHUNK) + (unsigned)allocSize)) == NULL)
		{
		ged_error(3, NULL);
		return(NULL);    /* malloc failed                  */
		}
	    chunk->bufPtr = chunk->curPtr = (byte *) (chunk + 1);
	    chunk->endPtr = chunk->bufPtr + allocSize;
	    chunk->nextChunk = nextChunk;
	    chunk->free = chunk->size = allocSize;
	    }
	return(chunk);
    }

