*-------------------------------------------------------------------------------
*-- Program...: NAVIGATE.PRG
*-- Programmer: Ken Mayer (CIS: 71333,1030)
*-- Date......: 02/23/1993
*-- Notes.....: These are interesting functions designed to help out in 
*--             navigation ... see the file: README.TXT for details on the
*--             use of this library file. 
*--             NOTE -- a few functions have been added into this library
*--             that are duplicated elsewhere (other library files). This is
*--             due to a limitation with dBASE IV, 1.5's handling of libraries.
*--             These functions are (and are from):
*--             STRIP2VAL()   from STRINGS.PRG
*--             STRIPVAL()
*--             STRPBRK()
*--             HAV()         from TRIG.PRG
*--             AHAV()
*--             CSCH()
*--             SINH()
*-------------------------------------------------------------------------------

FUNCTION Correct
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Correction of direction - adjusts direction given, in degrees,
*--               by second number of degrees.  Use to convert a compass
*--               direction to magnetic using deviation as the second argument,
*--               or magnetic to true using variation as the second argument.
*--               Returns a direction in degrees.
*--
*--               A westerly second argument may be given either as a negative
*--               number or as a character value containing "W".  If second
*--               argument is character-type but contains a negative value,
*--               effect of presence or absence of "W" is reversed.  That is,
*--               "-20 W" is treated like "20 E" or the number 20.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Correct(<nDirection>,<xCorrection>)
*-- Example.....: ?Correct(50,"-10 E")
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nDirection  = Heading
*--               xCorrection = amount to 'correct' by, may be numeric or
*--                             character, see above under 'Notes'.
*-------------------------------------------------------------------------------

	parameters nDirection, xCorrection
	private nCval
	if type( "xCorrection" ) = "C"
	  nCval = val( xCorrection )
	  if "W" $ upper( xCorrection )
	    nCval = - nCval
	  endif
	else
	  nCval = xCorrection
	endif
	
RETURN mod( 360 + nDirection + nCval, 360 )
*-- EoF: Correct()

FUNCTION UnCorrect
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Uncorrection of direction - adjusts direction given, in 
*--               degrees, by second number of degrees.  The inverse of 
*--               correct(), see above. Use to convert a true direction to 
*--               magnetic using variation as the second argument, or magnetic
*--               to compass using deviation as the second argument.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: UnCorrect(<nDirection>,<xUnCorr>)
*-- Example.....: ?UnCorrect(50,"-10 E")
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nDirection = Heading
*--               xUnCorr    = amount to 'uncorrect' by, may be numeric or
*--                             character, see above under 'Notes'.
*-------------------------------------------------------------------------------

	parameters nDirection, xUncorr
	private nCval
	if type( "xUncorr" ) = "C"
	  nCval = val( xUncorr )
	  if "W" $ upper( xUncorr )
   	 nCval = - nCval
	  endif
	else
	  nCval = xUncorr
	endif
	
RETURN mod( 360 + nDirection - nCval, 360 )
*-- EoF: UnCorrect()

FUNCTION XAngle
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Angle in degrees ( <= 90 ) at which two vectors in
*--               degrees intersect.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: XAngle(<nVector1>,<nVector2>)
*-- Example.....: ?UnCorrect(20,240)
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nVector1 = First angle
*--               nVector2 = Second angle
*-------------------------------------------------------------------------------

	parameters nVector1, nVector2
	private nResult
	nResult = abs( nVector1 - nVector2)
	do case
	  case nResult > 270
            nResult = 360 - Result
	  case nResult > 180
	    nResult = nResult - 180
	  case nResult > 90
	    nResult = 180 - nResult
	endcase
	
RETURN nResult
*-- EoF: XAngle()

FUNCTION LeftWind
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Whether effect of second vector on first is from the
*--               left or the right.  Returns .T. if from the left, else .F.
*--               Expects vectors in degrees.
*--
*--               For convenience in aviation calculations, the second
*--               argument is expected as the direction FROM which
*--               the wind or current is coming, not the direction TO
*--               which it is going.  If the contrary sense
*--               is more convenient, change the "=" sign in the
*--               function to "#".
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: LeftWind(<nCourse>,<nWindFrom>)
*-- Example.....: ?LeftWind(20,240)
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nCourse   = Direction of heading ...
*--               nWindFrom = Direction wind or current is coming from
*-------------------------------------------------------------------------------

	parameters nCourse, nWindfrom
	
RETURN ( nCourse > nWindfrom ) = ( abs( nCourse - nWindfrom ) < 180 )
*-- EoF: LeftWind()

FUNCTION TailWind
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Whether effect of second vector on first is additive
*--               or subtractive ( from behind or from ahead ).
*-- 
*--               For convenience in aviation calculations, the second
*--               argument is expected as the direction FROM which
*--               the wind or current is coming, not the direction TO
*--               which is going.  If the contrary sense
*--               is more convenient, change the "<" sign in the
*--               function to ">".
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: TailWind(<nCourse>,<nWindFrom>)
*-- Example.....: ?TailWind(20,240)
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nCourse   = Direction of heading ...
*--               nWindFrom = Direction wind or current is coming from
*-------------------------------------------------------------------------------

	parameters nCourse, nWindfrom
	
RETURN ( abs( abs( nCourse - nWindfrom ) - 180 ) < 90 )
*-- EoF: TailWind()

FUNCTION Heading
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Heading required to make good a course.
*--               If using this for boating and the direction of set is
*--               more convenient than the direction from which
*--               it is coming, apply mod( 180 + direction, 360 )
*--               to the fourth argument before calling.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: XANGLE()             Function in NAVIGATE.PRG
*--               LEFTWIND()           Function in NAVIGATE.PRG
*-- Called by...: Any
*-- Usage.......: Heading(<nCourse>,<nAirSpeed>,<nWindFrom>,<nForce>)
*-- Example.....: ?Heading(20,5,240,2)
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nCourse   = Direction of heading ...
*--               nAirSpeed = What it says
*--               nWindFrom = Direction wind or current is coming from
*--               nForce    = Windforce
*-------------------------------------------------------------------------------

	parameters nCourse, nAirspeed, nWindfrom, nForce
	private nCrabAngle
	nCrabAngle = rtod( asin( nForce * sin( dtor( xangle( nCourse, nWindfrom))) ;
		  / nAirspeed ) )
	nCrabAngle = iif( leftwind( nCourse, nWindfrom ), -nCrabAngle, nCrabAngle )
	nCrabAngle = mod( 360 + nCourse + nCrabAngle, 360 )
	
RETURN iif( abs( nCrabAngle ) < 360, nCrabAngle, -1 )
*-- EoF: Heading()

FUNCTION Course
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Course made good given heading, speed and wind direction 
*--               and force.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: XANGLE()             Function in NAVIGATE.PRG
*--               LEFTWIND()           Function in NAVIGATE.PRG
*--               TAILWIND()           Function in NAVIGATE.PRG
*-- Called by...: Any
*-- Usage.......: Course(<nHeading>,<nAirSpeed>,<nWindFrom>,<nForce>)
*-- Example.....: ?Course(20,5,240,2)
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nHeading  = Direction of heading ...
*--               nAirSpeed = What it says
*--               nWindFrom = Direction wind or current is coming from
*--               nForce    = Windforce
*-------------------------------------------------------------------------------

	parameters nHeading, nAirspeed, nWindfrom, nForce
	private nTemp, nCrabAngle
	nTemp = dtor( xangle( nHeading, nWindfrom ) )
	nCrabAngle = nAirspeed - nForce * cos( nTemp ) ;
   	* iif( tailwind( nHeading, nWindfrom ), -1, 1 )
	if nCrabAngle < 0
	  nCrabAngle = 0
	else
	  nCrabAngle = abs( rtod( atan( nForce * sin( nTemp ) / nCrabAngle ) ) )
	  nCrabAngle = iif( leftwind( nHeading,nWindfrom ), nCrabAngle, -nCrabAngle)
	endif
	
RETURN mod( 360 + nHeading + nCrabAngle, 360 )
*-- EoF: Course()

FUNCTION GndSpeed
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Speed over the ground given heading, speed
*--               and wind direction and force.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: XANGLE()             Function in NAVIGATE.PRG
*--               TAILWIND()           Function in NAVIGATE.PRG
*-- Called by...: Any
*-- Usage.......: GndSpeed(<nHeading>,<nAirSpeed>,<nWindFrom>,<nForce>)
*-- Example.....: ?GndSpeed(20,5,240,2)
*-- Returns.....: Numeric (direction in degrees)
*-- Parameters..: nHeading  = Direction of heading ...
*--               nAirSpeed = What it says
*--               nWindFrom = Direction wind or current is coming from
*--               nForce    = Windforce
*-------------------------------------------------------------------------------

	parameters nHeading, nAirspeed, nWindfrom, nForce
	private nTemp
	nTemp  = cos( dtor( xangle( nHeading, nWindfrom ) ) ) ;
   	* iif( tailwind( nHeading, nWindfrom ), -1, 1 )
	nTemp = nAirspeed * nAirspeed + nForce * nForce ;
   	- 2 * nAirspeed * nForce * nTemp

RETURN iif(nTemp<=0,nAirspeed+nForce*iif(tailwind(nHeading,nWindfrom ),1,-1),;
				sqrt(nTemp))
*-- EoF: GndSpeed()

FUNCTION Deg2Num
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Degrees to number: String in the form " 40d50'30.2 N" is 
*--               converted to a number of degrees.  If followed by E or S, 
*-                sign will be reversed.
*--
*--               It doesn't matter what characters are used to separate the
*--               degrees, minutes and seconds, but any of the characters N, E,
*--               W and S or their lowercase equivalents following the last 
*--               digit will be understood as specifying a compass direction.
*-- 
*--               If the degrees or minutes are 0, they must nevertheless be
*--               included in the argument.  Seconds may be omitted if 0, as
*--               may the minutes if 0 and seconds are omitted.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1993 -- Original Release
*-- Calls.......: STRIP2VAL()          Function in STRINGS.PRG
*--               STRIPVAL()           Function in STRINGS.PRG
*--               STRPBRK()            Function in STRINGS.PRG
*-- Called by...: Any
*-- Usage.......: Deg2Num(<cDms>)
*-- Example.....: ?Deg2Num("40d50'30.2 N")
*-- Returns.....: Numeric (degrees)
*-- Parameters..: cDms = Degrees Minutes Seconds
*-------------------------------------------------------------------------------

	parameters cDms
	private nResult, cStrleft
	if type( "cDms" ) $ "NF"
	  RETURN CDms
	endif
	cStrleft = strip2val( cDms )
	nResult = val( cStrleft )
	if "" # strip2val( stripval( cStrleft ) )
	  cStrleft = strip2val( stripval( cStrleft ) )
	  nResult = nResult + val( cStrleft ) / 60
	  if "" # strip2val( stripval( cStrleft ) )
	    cStrleft = strip2val( stripval( cStrleft ) )
	    nResult = nResult + val( cStrleft ) / 3600
	  endif
	endif
	cStrleft = upper( ltrim( stripval( cStrleft ) ) )
	if strpbrk( "NW", cStrleft ) > 0 .or. strpbrk( "ES",cStrleft ) = 0
	  RETURN nResult
	else
	  RETURN -nResult
	endif

*-- EoF: Deg2Num()

FUNCTION BearsDist
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Distance to an object at the time of the second
*--               bearing, given two bearings and the distance run
*--               between them.  Value returned will be in same
*--               units as third argument; first two are in degrees.
*--               Returns -1 if already past the object.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: BearsDist(<nBear1>,<nBear2>,<nRun>)
*-- Example.....: ?BearsDist(200,150,5)
*-- Returns.....: Numeric (degrees)
*-- Parameters..: nBear1 = Bearing of First object
*--               nBear2 = Bearing of Second object
*--               nRun   = Distance (or time) run between bearings
*-------------------------------------------------------------------------------

	parameters nBear1, nBear2, nRun
	if nBear2 > 180
	  if nBear1 < nBear2 .or. nBear2 < 270
   	 RETURN -1
	  else
   	 nBear1 = 360 - nBear1
	    nBear2 = 360 - nBear2
	  endif
	else
	  if nBear2 < nBear1 .or. nBear2 > 90
   	 RETURN -1
	  endif
	endif

RETURN sin( dtor( nBear1 ) ) * nRun / sin( dtor( nBear2 - nBear1 ) )
*-- EoF: BearsDist()

FUNCTION BearsPass
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Distance at which the object will be passed abeam:
*--                                  * <-- Object
*--                            .    /|
*--                      .         / |
*--                1-->-->-->--2  >  3  >
*--               Where 1 = Position at time first bearing to object is
*--                         taken,
*--                     2 = position at second bearing,
*--                     3 = position at which the object will be abeam.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: BearsPass(<nBear1>,<nBear2>,<nRun>)
*-- Example.....: ?BearsPass(200,150,5)
*-- Returns.....: Numeric (degrees)
*-- Parameters..: nBear1 = Bearing of First object
*--               nBear2 = Bearing of Second object
*--               nRun   = Distance (or time) run between bearings
*-------------------------------------------------------------------------------

	parameters nBear1, nBear2, nRun
	private nTemp
	if nBear2 > 180
	  if nBear1 < nBear2 .or. nBear2 < 270
	    RETURN -1
	  else
	    nBear1 = 360 - nBear1
	    nBear2 = 360 - nBear2
	  endif
	else
	  if nBear2 < nBear1 .or. nBear2 > 90
	    RETURN -1
	  endif
	endif
	nTemp = sin( dtor( nBear1 ) ) * nRun / sin( dtor( nBear2 - nBear1 ) )
	
RETURN nTemp * sin( dtor( nBear2 ) )
*-- EoF: BearsPass()

FUNCTION BearsRun
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Distance to run until object will be abeam given two bearings.
*--               Same rules and restrictions as bearsdist().
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: BearsRun(<nBear1>,<nBear2>,<nRun>)
*-- Example.....: ?BearsRun(200,150,5)
*-- Returns.....: Numeric (degrees)
*-- Parameters..: nBear1 = Bearing of First object
*--               nBear2 = Bearing of Second object
*--               nRun   = Distance (or time) run between bearings
*-------------------------------------------------------------------------------

	parameters nBear1, nBear2, nRun
	private nTemp
	if nBear2 > 180
	  if nBear1 < nBear2 .or. nBear2 < 270
	    RETURN -1
	  else
	    nBear1 = 360 - nBear1
	    nBear2 = 360 - nBear2
	  endif
	else
	  if nBear2 < nBear1 .or. nBear2 > 90
	    RETURN -1
	  endif
	endif
	nTemp = sin( dtor( nBear1 ) ) * nRun / sin( dtor( nBear2 - nBear1 ) )

RETURN nTemp * cos( dtor( nBear2 ) )
*-- EoF: BearsRun()

FUNCTION GcDist
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Great circle distance between two points given latitude
*--               and longitude of each.  This function obtains the degrees of
*--               arc along the great circle and simply multiplies by 60 to
*--               convert the degrees to nautical miles.  As this ignores the
*--               eccentricity of the earth, the answer may be in error by
*--               approximately half of one percent.  In general, if the
*--               route lies close to the equator the result of this
*--               function will be smaller than the actual number of nautical
*--               miles, but if the route passes close to the poles
*--               the function result will be larger than the correct number.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: DEG2NUM()            Function in NAVIGATE.PRG
*--               HAV()                Function in TRIG.PRG
*--               AHAV()               Function in TRIG.PRG
*-- Called by...: Any
*-- Usage.......: GCDist(<cLat1>,<cLon1>,<cLat2>,<cLon2>)
*-- Example.....: ?GCDist(200,150,105,200)
*-- Returns.....: Numeric (nautical miles)
*-- Parameters..: cLat1 = Latitude 1
*--               cLon1 = Longitude 1
*--               cLat2 = Latitude 2
*--               cLon2 = Longitude 2
*-------------------------------------------------------------------------------

	parameters cLat1, cLon1, cLat2, cLon2
	private nLa1, nLo1, nLa2, nLo2, nDla, nDlo, nTemp
	nLa1 = dtor( deg2num( cLat1 ) )
	nLo1 = dtor( deg2num( cLon1 ) )
	nLa2 = dtor( deg2num( cLat2 ) )
	nLo2 = dtor( deg2num( cLon2 ) )
	nDla = abs( nLa1 - nLa2 )
	nDlo = abs( nLo2 - nLo1 )
	do case
	  case nDlo = 0 .or. nDla = pi()
	    RETURN 60 * rtod( nDla )
	  case nDlo = pi()
	    RETURN 60 * rtod( ( pi() - nDla ) )
	  case nDlo > pi()
	    nDlo = 2 * pi() - nDlo
	endcase
	nTemp = hav( nDla ) + hav( nDlo ) * cos( nLa1 ) * cos( nLa2 )

RETURN 60 * rtod( ahav( nTemp ) )
*-- EoF: GcDist()

FUNCTION GcCourse
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Initial great circle course between two points given latitude
*--               and longitude of each.  Returns -1 if the points are 
*--               antipodes.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: DEG2NUM()            Function in NAVIGATE.PRG
*--               HAV()                Function in TRIG.PRG
*--               AHAV()               Function in TRIG.PRG
*--               CSCH()               Function in TRIG.PRG
*-- Called by...: Any
*-- Usage.......: GCCourse(<cLat1>,<cLon1>,<cLat2>,<cLon2>)
*-- Example.....: ?GCCourse(200,150,105,200)
*-- Returns.....: Numeric (degrees)
*-- Parameters..: cLat1 = Latitude 1
*--               cLon1 = Longitude 1
*--               cLat2 = Latitude 2
*--               cLon2 = Longitude 2
*-------------------------------------------------------------------------------

	parameters nLat1, nLon1, nLat2, nLon2
	private nLa1, nLo1, nLa2, nLo2, nDla, nDlo, nTemp, lRev
	nLa1 = dtor( deg2num( nLat1 ) )
	nLo1 = dtor( deg2num( nLon1 ) )
	nLa2 = dtor( deg2num( nLat2 ) )
	nLo2 = dtor( deg2num( nLon2 ) )
	nDla = abs( nLa1 - nLa2 )
	nDlo = abs( nLo2 - nLo1 )
	lRev = .F.
	do case
	  case nDla = pi() .or. nDlo = pi () .and. nLa1 + nLa2 = 0
	    RETURN -1
	  case nDlo = 0 .or. nDlo = pi() .or. abs( nLa1 ) = pi() .or.;
			 abs( nLa2 ) = pi()
	    RETURN iif( La1 > La2 , 180, 0 )
	  case nDlo > pi()
	    nDlo = 2 * pi() - nDlo
	    lRev = .T.
	endcase
	nTemp = hav( nDla ) + hav( nDlo ) * cos( nLa1 ) * cos( nLa2 )
	nTemp = rtod( asin( sin( nDlo ) * cos( nLa2 ) * csch( ahav( nTemp ) ) ) )
	nTemp = iif( nLa1 > nLa2, 180 - nTemp, nTemp )
	
RETURN iif( ( nLo2 > nLo1 ) = lRev, nTemp, 360 - nTemp )
*-- EoF: GCCourse()

*-------------------------------------------------------------------------------
*-- For convenience the following routines were brought in from other library
*-- files.
*-------------------------------------------------------------------------------

FUNCTION Strip2Val
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Strip characters from the left of a string until reaching
*--               one that might start a number.
*-- Written for.: dBASE IV
*-- Rev. History: 03/01/1992  -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Strip2Val("<cStr>")
*-- Example.....: ? Strip2Val("Test345")
*-- Returns.....: character string
*-- Parameters..: cStr = string to search
*-------------------------------------------------------------------------------

	parameters cStr
   private cNew
   cNew = cStr
   do while "" # cNew
      if left( cNew, 1 ) $ "-.0123456789"
         exit
	   endif
      cNew = substr( cNew, 2 )
	enddo
	
RETURN cNew
*-- EoF: Strip2Val()

FUNCTION StripVal
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Strip characters from the left of the string until
*--               reaching one that is not part of a number.  A hyphen
*--               following numerics, or a second period,
*--               is treated as not part of a number.
*-- Written for.: dBASE IV
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: StripVal("<cStr>")
*-- Example.....: ? StripVal("123.2Test")
*-- Returns.....: Character
*-- Parameters..: cStr = string to test
*-------------------------------------------------------------------------------

	parameters cStr
   private cNew, cChar, lGotminus, lGotdot
   cNew = cStr
   store .f. to lGotminus, lGotdot
   do while "" # cNew
      cChar = left( cNew, 1 )
	   do case
	      case .not. cChar $ "-.0123456789"
            exit
         case cChar = "-"
	         if lGotminus
               exit
            endif
	       case cChar = "."
	         if lGotdot
               exit
	         else
	            lGotdot = .T.
	         endif
	   endcase
      cNew = substr( cNew, 2 )
	   lGotminus = .T.
	enddo
	
RETURN cNew
*-- EoF: StripVal()

FUNCTION StrPBrk
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Search string for first occurrence of any of the
*--               characters in charset.  Returns its position as
*--               with at().  Contrary to ANSI.C definition, returns
*--               0 if none of characters is found.
*-- Written for.: dBASE IV
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: StrPBrk("<cCharSet>","<cBigStr>")
*-- Example.....: ? StrPBrk("Tt","This is a Test string, with Test data")
*-- Returns.....: Numeric value
*-- Parameters..: cCharSet = characters to look for in cBigStr
*--               cBigStr  = string to look in
*-------------------------------------------------------------------------------

	parameters cCharset, cBigstring
	private nPos, nLooklen
	nPos = 0
	nLooklen = len( cBigstring )
	do while nPos < nLooklen
      nPos = nPos + 1
		if at( substr( cBigstring, nPos, 1 ), cCharset ) > 0
         exit
	   endif
	enddo
	
RETURN iif(nPos=nLookLen,0,nPos)
*-- EoF: StrPBrk()

FUNCTION Hav
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Haversine of an angle in radians
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Hav(<nX>)
*-- Example.....: ?Hav(48)
*-- Returns.....: Numeric
*-- Parameters..: nX = Return Hav of X
*-------------------------------------------------------------------------------

	parameters nX
	
RETURN ( 1 - cos( nX ) ) / 2
*-- EoF: Hav()

FUNCTION AHav
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Inverse haversine - angle size in radians for given
*--               haversine
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: AHav(<nX>)
*-- Example.....: ?AHav(48)
*-- Returns.....: Numeric
*-- Parameters..: nX = Return AHav of X
*-------------------------------------------------------------------------------

	parameters nX
	
RETURN acos( 1 - 2 * nX )
*-- EoF: AHav()

FUNCTION SinH
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Hyperbolic sine of an angle X in radians
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1992 -- Original Release
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: SinH(<nX>)
*-- Example.....: ?SinH(48)
*-- Returns.....: Numeric
*-- Parameters..: nX = Return SinH of X
*-------------------------------------------------------------------------------

	parameters nX
	
RETURN ( exp( nX ) - exp( -nX ) ) / 2
*-- EoF: SinH()

FUNCTION CScH
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (CIS: 70160,340)
*-- Date........: 03/01/1992
*-- Notes.......: Hyperbolic cosecant of an angle X in radians
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/01/1993 -- Original Release
*-- Calls.......: SINH()               Function in TRIG.PRG
*-- Called by...: Any
*-- Usage.......: CScH(<nX>)
*-- Example.....: ?CScH(48)
*-- Returns.....: Numeric
*-- Parameters..: nX = Return CScH of X
*-------------------------------------------------------------------------------

	parameters nX
	
RETURN 1 / sinh( nX )
*-- EoF: CScH()

*-------------------------------------------------------------------------------
*-- EoP: NAVIGATE.PRG
*-------------------------------------------------------------------------------
