/*

  Program......:  FCOPY.PRG
  Purpose......:  CopyFile() function to support enhanced COPY FILE command.
                  This file also contains functions to support the use of an
                  OPTION clause.
                  Requires FCOPY.CH header file.
  Author.......:  Loren Scott, SuccessWare 90, Inc. - (909)699-9657
  Last Update..:  11/02/93

  Compile......:  CLIPPER fcopy /w
  Link.........:  BLINKER FI test, fcopy LI clipper

  --------------------------------------------------------------------------

  CopyFile( cOldFile, cNewFile, [[bOption], nStep], [nBufSize] )

    cOldFile = Source File Name.
    cNewFile = Destination File Name.
    bOption  = Optional Code Block to be evaluated for each <nStep> buffers
               that are copied.
    nStep    = Optional Step value, indicating how often, in buffers-full, to
               evaluate the <bOption> code block.  The default value is 1 (for
               every buffer full).
    nBufSize = Optional buffer size to use in reading/writing the files.
               The default buffer size is 8k, as defined in FCOPY.CH.

  Description:  Copies one file to another file.  Supports optional progress
                indicator (OPTION) and read/write buffer size to be used.

  Returns:  Logical TRUE (.T.) if file was successfully copied.  FALSE (.F.)
            if not.

  --------------------------------------------------------------------------

  _CopyPct( [nVal] )

    nVal = Optional numeric value to store the percentage of the file that
           has been copied thus far.

  Description:  This function simply stores and/or returns a numeric value
                indicating the percentage of the file that has been copied
                so far.

  Returns:  See description.

  --------------------------------------------------------------------------

  _CopyBytes( [nVal] )

    nVal = Optional numeric value to store the number of bytes that have
           been copied thus far.

  Description:  This function simply stores and/or returns a numeric value
                indicating the number of bytes that have been copied so far.

  Returns:  See description.

  --------------------------------------------------------------------------

  _CopySize( [nVal] )

    nVal = Optional numeric value to store the size of the file that is
           being copied.

  Description:  This function simply stores and/or returns a numeric value
                indicating the size of the file that is being copied.

  Returns:  See description.

*/

#include "Fileio.ch"
#include "fCopy.ch"

FUNC CopyFile( cOldFile, cNewFile, bOption, nStep, nBufSize )

LOCAL lRetVal := .F., nFileSize := 0, nPct := 0, nPass := 0
LOCAL hOld := 0, hNew := 0, cBuffer := "", nTotal := 0
LOCAl nBlock := 0
LOCAL lDone := .F.

hOld := fOpen( cOldFile )
hNew := fCreate( cNewFile )

IF hOld == -1 .or. hNew == -1
  Return( lRetVal )
ENDIF

// Get file size
nFileSize := fSeek( hOld, 0, FS_END )
fSeek( hOld, 0, FS_SET )

// Set default values
bOption  := iif( ValType( bOption )=="B", bOption, {|| .T. } )
nStep    := iif( ValType( nStep )=="N", nStep, 1 )
nBufSize := iif( ValType( nBufSize )=="N", nBufSize*1024, BUFSIZE*1024 )
cBuffer  := Space( nBufSize )

_CopyPct( 0 )
_CopyBytes( 0 )
_CopySize( nFileSize )
Eval( bOption )

DO WHILE !lDone
  nBlock := fRead( hOld, @cBuffer, nBufSize )
  IF fWrite( hNew, cBuffer, nBlock ) < nBlock
    lDone := .T.
    lRetVal := .F.
  ELSE
    IF nBlock == 0
      lDone := .T.
      lRetVal := .T.
    ENDIF
  ENDIF
  nTotal += nBlock
  nPct := ( nTotal / nFileSize ) * 100

  _CopyPct( nPct )
  _CopyBytes( nTotal )

  IF ++nPass % nStep == 0 .or. lDone
    IF !Eval( bOption )
      lRetVal := .F.
      lDone := .T.
    ENDIF
  ENDIF

  IF nPct == 100
    lDone := .T.
  ENDIF

ENDDO

fClose( hOld )
fClose( hNew )

_CopyPct( 0 )
_CopyBytes( 0 )

Return( lRetVal )


//----------------------------------//
FUNC _CopyPct( nVal )

STATIC nPct := 0

IF ValType( nVal ) == "N"
  nPct := nVal
ENDIF

Return( nPct )


//----------------------------------//
FUNC _CopyBytes( nVal )

STATIC nBytes := 0

IF ValType( nVal ) == "N"
  nBytes := nVal
ENDIF

Return( nBytes )


//----------------------------------//
FUNC _CopySize( nVal )

STATIC nBytes := 0

IF ValType( nVal ) == "N"
  nBytes := nVal
ENDIF

Return( nBytes )


