/*[]------------------------------------------------------------[]*/
/*|                                                              |*/
/*|     Source code for TDVIDEO.DLL (for TDW.EXE)                |*/
/*|                                                              |*/
/*|     Module for ATI VGA Wonder+ 18800 & 28800 chip sets       |*/
/*|     (16 and 256 color modes only)                            |*/
/*|                                                              |*/
/*|     Copyright (c) 1991 by Borland International              |*/
/*|     All Rights Reserved.                                     |*/
/*|                                                              |*/
/*[]------------------------------------------------------------[]*/

/* Written by Jeffrey J. Peters */

#include <windows.h>
#include <dos.h>       // for geninterrupt()
#include "tdvideo.h"
extern int int2fassist;   // int 2F method used in 640x480x256

char card_ident[] = "TDW video driver for ATI 16 & 256 color";

WORD GetVideoMode (void);
void SetVideoMode (BYTE m);
void SaveTextArea (void);
void RestoreTextArea (void);
void SaveTextArea16 (void);
void SaveTextArea256 (void);
void RestoreTextArea16 (void);
void RestoreTextArea256 (void);
void ReadDAC (BYTE far *);
void WriteDAC (BYTE far *);

WORD port = 0x01CE;
int bits = 256;                   // 16, or 256
BYTE palette16[0x15];             // for 16 color palette table
BYTE DACpal[0x11];

#define BIOS     0x10     // for BIOS interrupt calls
#define ATI_2    0xB2     // index for ATI extended register # 2

//--------------------------------------------------------------------------//
int CheckVideoInfo (void)
/*
   This function is only called from VIDEO.C and is used to check for
   video mode/card specifics.  Use this function to check for the proper
   card (if needed) or the video modes that are supported.
*/
{
  WORD m;
  char buf[20];

  if (0)
    return -2;  // incorrect video card  (later...)

  m = GetVideoMode ();     // get current video mode
  m = m & ~0x80;           // strip off high bit.

  vmode = m;               // set global mode value

  if (m < 0x13)
    return 0;      // tell VideoInit that TDW will be handling the swapping

  switch (m)
  {
    case 0x61:    // 640x400x256  (not very common)
    case 0x62:    // 640x480x256
                  bits = 256;
                  savepalette = 1;

                  GetPrivateProfileString ("VideoOptions", "ATI640x480x256", "0", buf,
                                           10, "TDVIDEO.INI");
                  strupr (buf);
                  if ((buf[0] == 'Y' || buf[0] == '1'))
                  {
                    int2fassist = 1;
                    screen = 1;
                    savepalette = 0;
                  }
                  return 1;  // supported 256 color mode

    case 0x63:    // 800x600x256
                  bits = 256;
                  savepalette = 1;
                  maxplanes = 3;
                  MaxBytesPerPage = 0x8000;
                  return 1;  // supported 256 color mode

    case 0x55:    return -1; // 1024x768x16
                  bits = 16;
                  savepalette = 1;
                  int2fassist = 1;  // use 2F method to get around ATI driver bug
                  screen = 1;       // save whole screen

                  return 2;  // supported 16 color mode

    case 0x54:    // 800x600x16
                  MaxBytesPerPage = 0x8000;
                  maxplanes = 3;
                  savepalette = 1;
                  bits = 16;
                  return 2;   // supported 16 color mode

    case 0x12:    // 640x480x16
                  MaxBytesPerPage = 0x9000;
                  maxplanes = 3;
                  savepalette = 1;
                  screen = 1;
                  bits = 16;
                  return 2;   // supported 16 color mode

    default:
                  return -1;  // unsupported mode
  }
  return 0; // never gets here
}
//--------------------------------------------------------------------------//
WORD GetVideoMode (void)
/*
   This function gets the current video mode from BIOS.  NOTE: some
   SuperVGA modes cannot be entered from BIOS and may not update the
   BIOS data area to reflect the proper mode number.
*/
{
   _AH = 0x0F;
   geninterrupt (BIOS);
   return _AL;
}
//--------------------------------------------------------------------------//
void SetVideoMode (BYTE m)
/*
   This function sets a particular video mode via BIOS.  Some SuperVGA
   modes cannot be entered via BIOS and require direct register
   manipulation.  NOTE: you need to set the high bit of the mode to tell
   BIOS not to clear the screen when switching modes.  If you don't do
   this you will have to save the ENTIRE graphics screen (250k - 1000k)!
*/
{
  if ((m == 3) && (tsize != 25))
  {
    _AX = 0x0083;
    geninterrupt (0x10);
    VideoSetSize (1);
    return;
  }

  _AH = 0x00;
  _AL = m;
  if (screen != 1)
    asm or al, 0x80;   // don't clear screen
  geninterrupt (BIOS);
}
//--------------------------------------------------------------------------//
void SaveTextArea (void)
{
  if (bits == 16)
  {
    ReadDAC (DACpal);
    SaveTextArea16 ();
  }

  if (bits == 256)
    SaveTextArea256 ();
}
//--------------------------------------------------------------------------//
void RestoreTextArea (void)
{
  if (bits == 16)
  {
    RestoreTextArea16 ();
    WriteDAC (DACpal);
  }

  if (bits == 256)
    RestoreTextArea256 ();
}
//--------------------------------------------------------------------------//
void SaveTextArea256 (void)
/*
   This function saves the video memory that will get blown away by
   entering text mode.
*/
{
  WORD i, j;
  WORD plane_hold;
  char far *p;
  LPSTR q;

  p = MK_FP ((WORD)&_A000H, 0);

  outportb (port, ATI_2); // select memory page select reg.
  plane_hold = inportb (port+1);  // save old value

  if (disable_video == 1)
    v_off();       // disable video for faster access to memory
  for (j=0;j<maxplanes;j++)
  {

    outport (port, ATI_2 + 256*((j<<5) + (j<<1))); // select read plane
            // bits D3-D1 select the plane as well as
            // bits D7-D5 select the plane


    q = GlobalLock (v_buf.hplane[j]);

    asm {
      push si
      push di
      push es
      push ds
    }
    _CX = MaxBytesPerPage;
    asm {
      les di, q
      lds si, p
      shr cx, 1
      rep movsw
      adc cx, 0
      rep movsb

      pop ds
      pop es
      pop di
      pop si
    }
    GlobalUnlock (v_buf.hplane[j]);
  }
  if (disable_video == 1)
    v_on();   // enable video again.
  outport (port, ATI_2 + 256 * plane_hold); // reset to old value
}
//--------------------------------------------------------------------------//
void SaveTextArea16 (void)
/*
   This function saves the video memory that will get blown away by
   entering text mode.
*/
{
  WORD i, j;
  WORD plane_hold;
  char far *p;
  LPSTR q;

  p = MK_FP ((WORD)&_A000H, 0);

  if (disable_video == 1)
    v_off();             // disable video for faster access to memory
  for (j=0;j<maxplanes;j++)
  {

    outportb (0x3CE, 0x04);
    outportb (0x3CF, j); // select read plane


    q = GlobalLock (v_buf.hplane[j]);

    asm {
      push si
      push di
      push es
      push ds
    }
    _CX = MaxBytesPerPage;
    asm {
      les di, q
      lds si, p
      shr cx, 1
      rep movsw
      adc cx, 0
      rep movsb

      pop ds
      pop es
      pop di
      pop si
    }
    GlobalUnlock (v_buf.hplane[j]);
  }
  if (disable_video == 1)
    v_on();   // enable video again.

}
//--------------------------------------------------------------------------//
void RestoreTextArea256 (void)
/*
   This function restores the video memory that gets blown away by going
   into text mode.
*/
{
  WORD i, j,k;
  WORD plane_hold;
  char far *p;
  LPSTR q;

  p = MK_FP ((WORD)&_A000H, 0);

  outportb (port, ATI_2); // select memory page select reg.
  plane_hold = inportb (port+1);  // save old value

  if (disable_video == 1)
    v_off();           // disable video for faster access to memory
  for (j=0;j<maxplanes;j++)
  {

    outport (port, ATI_2 + 256*((j<<5) + (j<<1))); // select write plane
            // bits D3-D1 select the plane as well as
            // bits D7-D5 select the plane

    q = GlobalLock (v_buf.hplane[j]);
    asm {
      push si
      push di
      push es
      push ds
    }
    _CX = MaxBytesPerPage;
    asm {
      les di, p
      lds si, q
      shr cx, 1
      rep movsw
      adc cx, 0
      rep movsb

      pop ds
      pop es
      pop di
      pop si
    }
    GlobalUnlock (v_buf.hplane[j]);
  }
  if (disable_video == 1)
    v_on();  // enable video again
  outport (port, ATI_2 + 256 * plane_hold); // reset to old value
}
//--------------------------------------------------------------------------//
void RestoreTextArea16 (void)
/*
   This function restores the video memory that gets blown away by going
   into text mode.
*/
{
  WORD i, j,k;
  WORD plane_hold;
  char far *p;
  LPSTR q;

  p = MK_FP ((WORD)&_A000H, 0);

  if (disable_video == 1)
    v_off();           // disable video for faster access to memory

  outportb (0x3CE, 5);
  outportb (0x3CF, (inportb (0x3CF) & (~3) )); // set write mode 0

  for (j=0;j<maxplanes;j++)
  {
    outportb (0x3C4, 0x2);
    outportb (0x3C5, (1<<j)); // select write plane

    q = GlobalLock (v_buf.hplane[j]);
    asm {
      push si
      push di
      push es
      push ds
    }
    _CX = MaxBytesPerPage;
    asm {
      les di, p
      lds si, q
      shr cx, 1
      rep movsw
      adc cx, 0
      rep movsb

      pop ds
      pop es
      pop di
      pop si
    }
    GlobalUnlock (v_buf.hplane[j]);
  }
  if (disable_video == 1)
    v_on();  // enable video again

}
//--------------------------------------------------------------------------//

void ReadDAC (BYTE far *p)
{
  int j;
  while (inportb (0x3DA) & 8 != 8);  // start with first palette at Vertical retrace

  for (j=0;j<=0x0F;j++)
  {
    inportb (0x3DA);
    outportb (0x3C0,j);
    p[j] = inportb (0x3C1);
  }
  outportb (0x3C0, 0x20);                 // enable palettes
}

void WriteDAC (BYTE far *p)
{
  int j;
  while (inportb (0x3DA) & 8 != 8);  // start with first palette at Vertical retrace

  for (j=0;j<=0x0F;j++)
  {
    inportb (0x3DA);                        // reset flip/flop
    outportb (0x3C0, j);                    // select reg
    outportb (0x3C0, p[j]);                 // modify that reg
    outportb (0x3C0, 0x20);                 // enable palettes
  }
}

