/***************************************************************************
 *  SESSION.C								   *
 *	This file contains following NetBIOS Session related functions	   *
 *									   *
 *	  SendSess   () :- Sends data on an established session 	   *
 *	  ReceiveSess() :- Receives data for a given session		   *
 *	  HangSess   () :- Terminate  a given session			   *
 *	  Call()	:- Establish a session with a remote name	   *
 *	  Listen()	:- Put yourself in "listening" state to receive    *
 *			   remote calls.				   *
 *									   *
 *  History:	Alok Sinha  October, 1991    Created			   *
 *									   *
 ***********************************************************************/

// Include files
#include <ncb.h>
#include <common.h>
#include <namesrv.h>
#include <session.h>
#include <memory.h>
#include <string.h>


/*
 * FUNCTION    :: SendSess()
 * PARAMETERS  ::
 *	 [in]  ucLana	  :- LAN Adaptor number
 *	 [in]  ucLsn	  :- Local Session Number
 *	 [in]  pchData	  :- pointer to a data buffer to send
 *	 [in]  usDataLen  :- data buffer length.
 *
 * Comment:
 *		This function will execute synchronous NCB.SEND command.
 *
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */

unsigned char	SendSess(     unsigned char ucLana,
			      unsigned char ucLsn,
			      char far * pchData,
			      unsigned short usDataLen
			)
{
    NCB Ncb;
    unsigned char ucRc;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command. Synchronous NCB.SEND */
    Ncb.ncb_command = NCBSEND;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* set the Local Session Number */
    Ncb.ncb_lsn = ucLsn;

    /* Set  Data buffer pointer and it's length */
    Ncb.ncb_buffer = pchData;
    Ncb.ncb_length = usDataLen;

    /* Make the OS dependent NetBIOS call */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    return Ncb.ncb_retcode;

}


/*
 * FUNCTION    :: ReceiveSess()
 * PARAMETERS  ::
 *	 [in]  ucLana	  :- Lan Adaptor number
 *	 [in]  ucLsn	  :- Local Session Number
 *	 [in]  pchData	  :- pointer to a data buffer to send
 *	 [in/out]pusDataLen  :- This is set to data buffer length when
 *			    called by application. And it is set by
 *			    to data buffer size actually received, if call
 *			    completes successfully.
 * Comment:
 *		This function will execute synchronous NCB.RECEIVE command.
 *
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */

unsigned char	ReceiveSess(	unsigned char ucLana,
				unsigned char ucLsn,
				char far * pchData,
				unsigned short *pusDataLen
			   )
{
    NCB Ncb;
    unsigned char ucRc;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command synchronous NCB.RECEIVE */
    Ncb.ncb_command = NCBRECV;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* set the Local Session Number */
    Ncb.ncb_lsn = ucLsn;

    /* Set  Data buffer pointer and it's length */
    Ncb.ncb_buffer = pchData;
    Ncb.ncb_length = *pusDataLen;

    /* Make the OS dependent NetBIOS call */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    /* return actual size of data buffer if call was successful */
    if (Ncb.ncb_retcode == NO_ERROR)
	*pusDataLen = Ncb.ncb_length;

    return Ncb.ncb_retcode;

}

/*
 * FUNCTION    :: HangSess()
 * PARAMETERS  ::
 *	 [in]  ucLana	  :- Lan Adaptor number
 *	 [in]  ucLsn	  :- Local Session Number
 *
 * Comment:
 *		This function will execute synchronous NCB.HANGUP command.
 *
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */

unsigned char	HangSess(   unsigned char ucLana,
			    unsigned char ucLsn
			)
{
    NCB Ncb;
    unsigned char ucRc;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command synchronous NCB.HANGUP */
    Ncb.ncb_command = NCBHANGUP;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* set the Local Session Number */
    Ncb.ncb_lsn = ucLsn;

    /* Make the OS dependent NetBIOS call */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    return Ncb.ncb_retcode;

}


/*
 * FUNCTION    :: Call()
 * PARAMETERS  ::
 *	 [in]  ucLana	     :- Lan Adaptor number
 *	 [in]  pchRemoteName :- Remote name to connect to.
 *	 [in]  pchLocalName  :- Local  (process) name.
 *	 [in]  ucSto	     :- Send	Time Out in 1/2 seconds units
 *	 [in]  ucRto	     :- Receive Time Out in 1/2 seconds units
 *	 [out] ucLsn	     :- Local Session Number on successful
 *				completion.
 *
 * Comment:
 *		This function will execute synchronous NCB.CALL command.
 *
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */

unsigned char	Call(	 unsigned char ucLana,
			 char far * pchRemoteName,
			 char far * pchLocalName,
			 unsigned char ucSto,
			 unsigned char ucRto,
			 unsigned char far *ucLsn
		    )
{
    NCB Ncb;
    unsigned char ucRc;

    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command synchronous NCB.CALL */
    Ncb.ncb_command = NCBCALL;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* Set Remote name in ncb_callname */
    memcpy(&Ncb.ncb_callname[0], pchRemoteName, NCBNAMSZ);

    /* Set Local name in ncb_name */
    memcpy(&Ncb.ncb_name[0], pchLocalName, NCBNAMSZ);

    /* set the Send Time Out*/
    Ncb.ncb_sto = ucSto;

    /* set the Receive Time Out*/
    Ncb.ncb_sto = ucRto;

    /* Make the OS dependent NetBIOS call */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    /* Return the Local Session Number */
    if (Ncb.ncb_retcode == NO_ERROR)
	*ucLsn = Ncb.ncb_lsn;

    return Ncb.ncb_retcode;

}

/*
 * FUNCTION    :: Listen()
 * PARAMETERS  ::
 *	 [in]  ucLana	     :- Lan Adaptor number
 *	 [in/out] pchRemoteName :- Remote name to connect to. If this
 *				   is set to '*', NetBIOS will return
 *				   name of calling process.
 *
 *	 [in]  pchLocalName  :- Local  (process) name.
 *	 [in]  ucSto	     :- Send	Time Out in 1/2 seconds units
 *	 [in]  ucRto	     :- Receive Time Out in 1/2 seconds units
 *	 [out] ucLsn	     :- Local Session Number on successful
 *				completion.
 *
 * Comment:
 *		This function will execute synchronous NCB.CALL command.
 *
 * RETURN VALUE:
 *	    NO_ERROR if no errors were encountered, else error returned from
 *	    NetBIOS call in Retcode field.
 */
unsigned char	Listen(	 unsigned char ucLana,
			 char far * pchRemoteName,
			 char far * pchLocalName,
			 unsigned char ucSto,
			 unsigned char ucRto,
			 unsigned char far *ucLsn
		     )
{
    NCB Ncb;
    unsigned char ucRc;
    BOOL     fReturnCallerName;


    /* First clear out the buffer */
    ClearNcb ( &Ncb);

    /*	Set the Command synchronous NCB.LISTEN */
    Ncb.ncb_command = NCBLISTEN;

    /* set the LAN Adaptor number */
    Ncb.ncb_lana_num = ucLana;

    /* Set Remote name in ncb_callname */
    if (pchRemoteName[0]='*')
    {
	Ncb.ncb_callname[0]='*';
	fReturnCallerName = TRUE;
    }
    else
    {
	memcpy(&Ncb.ncb_callname[0], pchRemoteName, NCBNAMSZ);
	fReturnCallerName = FALSE;
    }

    /* Set Local name in ncb_name */
    memcpy(Ncb.ncb_name, pchLocalName, NCBNAMSZ);

    /* set the Send Time Out*/
    Ncb.ncb_sto = ucSto;

    /* set the Receive Time Out*/
    Ncb.ncb_sto = ucRto;

    /* Make the OS dependent NetBIOS call */
    if ((ucRc=NetBiosRequest ( &Ncb))!= NO_ERROR)
	return ucRc ;

    /* Return the Local Session Number
     * and return caller's name if pchRemoteName was set '*'
     */
    if (Ncb.ncb_retcode == NO_ERROR)
    {
	*ucLsn = Ncb.ncb_lsn;
	if (fReturnCallerName)
	    memcpy(pchRemoteName, Ncb.ncb_callname, NCBNAMSZ);
    }

    return Ncb.ncb_retcode;

}
