/*
  process.c

  The file is part of the MIT Artificial Intelligence Laboratory's
  automatic fax transmission system.

  (c) Copyright 1990 by David M. Siegel
      All Rights Reserved.

  Sun Nov 4, 1990 at 11:16:34
*/

#include <stdio.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include "defines.h"

extern int debug;
extern int serial_fd;

#define MAX_USER_LEN	256
#define MAX_PHONE_LEN	256
#define MAX_RECIP_LEN	256
#define MAX_RETURN_LEN	256
#define MAX_SENDER_LEN	256

struct qheader {
    char user[MAX_USER_LEN];
    char phone[MAX_PHONE_LEN];
    char recipient[MAX_RECIP_LEN];
    char returnfax[MAX_RETURN_LEN];
    char sender[MAX_SENDER_LEN];
    int coversheet;		/* If TRUE, send a coversheet */
    int mail_notification;	/* If TRUE, email the sender a status msg */
    int count;
    long time;
};

static void
delete_job(base)
char *base;
{
    char buf[BUFSIZ];
    int page;

    sprintf(buf, "%s/%s.qf", SPOOLDIR, base);
    unlink(buf);

    sprintf(buf, "%s/%s.ps", SPOOLDIR, base);
    unlink(buf);

    /* delete coversheet page if any */
    sprintf(buf, "%s/%s.g3.0", SPOOLDIR, base);
    unlink(buf);

    page = 1;
    do {
	sprintf(buf, "%s/%s.g3.%d", SPOOLDIR, base, page++);
    } while (unlink(buf) == 0);
}

static int
read_qf(base, qhp)
char *base;
struct qheader *qhp;
{
    FILE *fp;
    int found_user = FALSE;
    int found_phone = FALSE;
    int found_recipient = FALSE;
    int found_returnfax = FALSE;
    int found_sender = FALSE;

    char buf[BUFSIZ];
    char *ptr;

    qhp->mail_notification = TRUE; /* defaults to send you mail */
    qhp->coversheet = TRUE;	/* defaults to add annoying coversheet */

    sprintf(buf, "%s.qf", base);
    if ((fp = fopen(buf, "r")) == NULL) {
	log("can't open queue file: %s\n", buf);
	return(-1);
    }

    qhp->time = 0;
    qhp->count = 0;

    while (fgets(buf, sizeof(buf), fp) != NULL) {
	if ((ptr = index(buf, ':')) == NULL) {
	    log("missing ':' in queue file: %s\n", buf);
	    fclose(fp);
	    return (-1);
	}
	*ptr = '\0';
	if (strcmp(buf, "USER") == 0) {
	    found_user = TRUE;
	    strcpy(qhp->user, ptr+2);
	    qhp->user[strlen(qhp->user)-1] = '\0';
	    if (debug)
	      log("got user\n");
	} else if (strcmp(buf, "PHONE") == 0) {
	    found_phone = TRUE;
	    strcpy(qhp->phone, ptr+2);
	    qhp->phone[strlen(qhp->phone)-1] = '\0';
	    if (debug)
	      log("got phone\n");
	} else if (strcmp(buf, "RECIPIENT") == 0) {
	    found_recipient = TRUE;
	    strcpy(qhp->recipient, ptr+2);
	    qhp->recipient[strlen(qhp->recipient)-1] = '\0';
	    if (debug)
	      log("got recipient\n");
	} else if (strcmp(buf, "SENDER") == 0) {
	    found_sender = TRUE;
	    strcpy(qhp->sender, ptr+2);
	    qhp->sender[strlen(qhp->sender)-1] = '\0';
	    if (debug)
	      log("got sender\n");
	} else if (strcmp(buf, "RETURNFAX") == 0) {
	    found_returnfax = TRUE;
	    strcpy(qhp->returnfax, ptr+2);
	    qhp->returnfax[strlen(qhp->returnfax)-1] = '\0';
	    if (debug)
	      log("got returnfax\n");
	} else if (strcmp(buf, "COUNT") == 0) {
	    qhp->count = atoi(ptr+2);
	} else if (strcmp(buf, "TIME") == 0) {
	    qhp->time = atoi(ptr+2);
	} else if (strcmp(buf, "NOMAIL") == 0) {
	    qhp->mail_notification = (! atoi(ptr+2));
	} else if (strcmp(buf, "NOCOVER") == 0) {
	    qhp->coversheet = (! atoi(ptr+2));
	} else {
	    log("unknown entry in queue file: %s\n", buf);
	    fclose(fp);
	    return (-1);
	}
    }

    fclose(fp);

    if (!found_sender) sprintf(qhp->sender, "%s", "");
    if (!found_recipient) sprintf(qhp->recipient, "%s", "");
    if (!found_returnfax) sprintf(qhp->returnfax, "%s", DEFAULT_RETURNFAX);


    if (found_user && found_phone) {
	if (debug)
	  log("queue file %s ok: %s %s %ld\n", 
	      base, qhp->user, qhp->phone, qhp->time);
	return (0);
    } else {
	if (debug)
	  log("queue file %s isn't ok\n", base);
	return (-1);
    }

}

#define SEND_OK     0
#define SEND_FAILED 1
#define SEND_RETRY  2
#define SEND_DENIED 3

#define MAXPAGES 256

/*ARGSUSED*/
static int
do_send(base, qhp)
char *base;
struct qheader *qhp;
{
    char account[BUFSIZ];
    char dial_string[BUFSIZ];
    char buf[BUFSIZ];
    char program[BUFSIZ];
    char *faxfiles[MAXPAGES];
    char *user = qhp->user;
    char *phone = qhp->phone;
    char pagestring[64];
    FILE *fp;
    int page, nfiles;
    int valid_acct;

    if (chdir(LIBDIR) < 0) {
	log("can't change to spool directory: %s\n", LIBDIR);
	exit(2);
    }

    /* Non-local calls must be registered in the faxusers file */
/*   if (get_account(user, account) < 0) 
 *       valid_acct = FALSE; 
 *    else valid_acct = TRUE; */
    
    valid_acct = FALSE;


    /* 
       Phone dialing is done as folows:

       FIRST-DIGIT  ACTION

       +            The remaining dial string is passed literally to the modem.
                    
       otherwise    The users account number is looked up in /etc/faxusers
                    and the sequence for charging to that MIT account is 
		    dialed.

		    If no account number is found, then 
		    A '9,' is prefixed to the string, and it is sent to
                    the modem. A non local call will be refused by the
		    phone service.
    */

    if ( phone[0] == '+' )
      sprintf(dial_string, "%s", phone+1);
    else if (valid_acct)
      sprintf(dial_string, "4,%s,%s", account, phone);
    else 
      sprintf(dial_string, "9,%s", phone);

    sprintf(buf, "%s/%s.ps", SPOOLDIR, base);
    if (access(buf, R_OK) == 0) {
      sprintf(program, "%s -dNODISPLAY pstofaxbits.ps", GSPATH);
	if ((fp = popen(program, "w")) == NULL) {
	    log("can't open gs\n");
	    exit(1);
	}
    
	fprintf(fp, "(%s/%s) ppm1run quit\n", SPOOLDIR, base);
	pclose(fp);
	unlink(buf);
      
	if (qhp->coversheet) {
	    /*
	      Count pages for the coversheet
	    */
	    page = 1;
	    for (;;) {
		sprintf(buf, "%s/%s.ppm.%d", SPOOLDIR, base, page);
		if (access(buf, R_OK) != 0) break;
		page++;
	    }
      
	    /* 
	      usage: coversheet recipient sender returnfax pages psfile 
	      basename.
	    */
	    sprintf(pagestring, "%d", page); /* document pages + 1 */
	    sprintf(program, 
		    "%s/coversheet '%s' '%s' '%s' '%s' coverpage.ps %s/%s",
		    LIBDIR, qhp->recipient, qhp->sender, qhp->returnfax, pagestring,
		    SPOOLDIR, base);
	    system(program);
	}
      
	page = qhp->coversheet ? 0 : 1;
	for (;;) {
	    sprintf(buf, "%s/%s.ppm.%d", SPOOLDIR, base, page);
	    if (access(buf, R_OK) == 0) {
		sprintf(program, "ppmtog3 %s/%s %d", SPOOLDIR, base, page);
		system(program);
	    } else
	      break;
	    page++;
	}
    }

    /* Changed to call faxsend(dialstring, nfiles, filenames, serial_fd) */

    /* if no coversheet g3 pages start at page 1 */
    page = qhp->coversheet ? 0 : 1;

    for (nfiles = 0 ; nfiles < MAXPAGES ; nfiles++) {
      sprintf(buf, "%s/%s.g3.%d", SPOOLDIR, base, page++);
      if (access(buf, R_OK) == 0) {
	faxfiles[nfiles] = (char *) malloc(strlen(buf)+1);
	strcpy(faxfiles[nfiles], buf);
      } else
	break;
    }
    
    if (debug)
      log("calling faxsend\n");

    {
      int val;

    if (faxsend(dial_string, nfiles, faxfiles, serial_fd) == 0) 
      val = (SEND_OK);
    else
      val = (SEND_RETRY);

    /* now deallocate the file name strings */
    while (nfiles > 0) {
      --nfiles;
      free(faxfiles[nfiles]);

    }
      return(val);
    }
  }

static int
update_qf(base, qhp)
char *base;
struct qheader *qhp;
{
    char buf[BUFSIZ];
    FILE *fp;

    sprintf(buf, "%s/%s.qf", SPOOLDIR, base);

    if ((fp = fopen(buf, "w+")) == NULL) {
	log("can't open queue file %s for update\n", base);
	return (-1);
    }

    fprintf(fp, "USER: %s\n", qhp->user);
    fprintf(fp, "PHONE: %s\n", qhp->phone);

    fprintf(fp, "SENDER: %s\n", qhp->sender);
    fprintf(fp, "RECIPIENT: %s\n", qhp->recipient);
    fprintf(fp, "RETURNFAX: %s\n", qhp->returnfax);

    fprintf(fp, "NOMAIL: %d\n", (! qhp->mail_notification));
    fprintf(fp, "NOCOVER: %d\n", (! qhp->coversheet));

    fprintf(fp, "COUNT: %d\n", qhp->count+1);
    fprintf(fp, "TIME: %d\n", time(0));

    fclose(fp);

    return (0);
}

static char *notifications[] = {
#define NOTIFICATION_OK 0
    "Your fax to %s has been successfully sent.\n",
#define NOTIFICATION_FAILED 1
    "There were problems transmitting your fax to %s, it cannot be sent.\n",
#define NOTIFICATION_RETRY  2
    "The transmission of your fax to %s failed, I will try again later.\n",
#define NOTIFICATION_TOOMANY 3
    "The transmission of your fax to %s failed, and the maximum number\n\
of retries has been exceeded.  I'm very sorry, but your fax has\n\
been deleted.\n",
#define NOTIFICATION_DENIED 4
    "I'm sorry, but you don't have long distance access.  Your fax to\n\
%s has been deleted.\n"
};

static void
mail_notification(qhp, stat)
struct qheader *qhp;
int stat;
{
    FILE *fp;
    char program[BUFSIZ];

    if (qhp->mail_notification) {
      sprintf(program, "/usr/ucb/Mail -s \"fax notification\" %s", qhp->user);
      
      if ((fp = popen(program, "w")) == NULL) {
	log("can't open mail program!\n");
	return;
      }
      
      fprintf(fp, notifications[stat], qhp->phone);
      fprintf(fp, "\nSincerely yours,\nThe Fax Daemon\n");
      
      pclose(fp);
  }
}

void
send_process(base)
char *base;
{
    struct qheader qh;

    if (read_qf(base, &qh) < 0) {
	delete_job(base);
	return;
    }

    if (qh.time != 0) {
	if (time(0) - qh.time < RETRY_INTERVAL) {
	    if (debug)
	      log("retry time for %s not reached\n", base);
	    return;
	}
    }

    log("sending fax from %s to %s\n", qh.user, qh.phone);

    switch (do_send(base, &qh)) {
      case SEND_OK:
	delete_job(base);
	mail_notification(&qh, NOTIFICATION_OK);
	log("fax from %s to %s has been sent\n", qh.user, qh.phone);
	billing_log(qh.user, qh.phone);
	break;
      case SEND_FAILED:
	delete_job(base);
	mail_notification(&qh, NOTIFICATION_FAILED);
	log("fax from %s to %s failed\n", qh.user, qh.phone);
	break;
      case SEND_RETRY:
	if (qh.count >= MAX_RETRIES) {
	    delete_job(base);
	    mail_notification(&qh, NOTIFICATION_TOOMANY);
	    log("fax from %s to %s exceeded retry count\n", qh.user, qh.phone);
	} else {
	    update_qf(base, &qh);
	    mail_notification(&qh, NOTIFICATION_RETRY);
	    log("fax from %s to %s has been requeued\n", qh.user, qh.phone);
	}
	break;
      case SEND_DENIED:
	delete_job(base);
	mail_notification(&qh, NOTIFICATION_DENIED);
	log("fax from %s to %s access denied\n", qh.user, qh.phone);
	break;
    }
}
