/* Most of the routines in this file are just quick hacks to get things
 * running. The general idea is that term is only used when the local
 * gethostbyname fails to find anything.  Many of these routines are just
 * quick hacks and can be greatly improved.
 */

/* Everything that follows is an attempt to allow term to easier porting
 * of software to term.  The functions term_connect, term_rcmd, and 
 * term_gethostbyname are written such that ported routines will work
 * even with ordinary network connections.  Unfortunately, I haven't
 * figured out how to make term_bind network invisible yet.
 */

#define I_IOCTL
#define I_SYS
#define I_GETOPT
#define I_SOCKET
#define I_STAT
#define I_SIGNAL
#define I_STRING
#define I_ARGS
#define I_INET
#define I_ERRNO
#define I_CTYPE
#include "includes.h"
#include "client.h"

void get_term_localaddr(unsigned long);
void term_do_exit(void);
struct sockaddr *str_to_sockaddr(char *, unsigned long);
char *sockaddr_to_str(struct sockaddr *, int trans);

int term_debug = 0;
static int Sock = -1;

/* This is a way to disable termnet commands that aren't smart enough */
/* to automatically determine if term commands should be used. */
static int remote_connect = -1;

/* This is needed for term_getsockname. */
struct sockinfo_type {
  int sock;
  int client;
  int pid;
  struct sockaddr *udp_addr;
};

static struct sockinfo_type sockinfo[MAX_CLIENTS];

static struct hostent *str_to_hostent(char *addr){
  static unsigned long int laddr=0;
  static struct hostent hs;
  static char host[258];
  static char *caddr[2]={NULL,NULL};
  static char *host_aliases[2]={NULL,NULL};

  host_aliases[0]=(char *)hs.h_name;

  hs.h_name=host;

  sscanf(addr,"%lx %258s",&laddr,hs.h_name);
  if(! laddr) return NULL;

  laddr = htonl(laddr);
  caddr[0]=(char *) &laddr;

/* I'm using 0 as remote host. */

  hs.h_aliases = host_aliases;
  hs.h_addr_list = caddr;
  hs.h_length = sizeof(unsigned long int);
  hs.h_addrtype = AF_INET; 
  return &hs;
}


/* To keep the lookups accurate, I need this.
 */

static void close_fd(int fd){
  int i, s, j;
#if 0
  int pid;
  pid = getpid();
#endif

  if (remote_term_version > 11862) {
    if (Sock < 0)
      Sock = socket_connect_server(-1,term_server);
    s = Sock;
  }else s = -1;


  if(fd>=0)
    for(i=0;i<MAX_CLIENTS;i++)
      if (fd == sockinfo[i].sock) {

    sockinfo[i].sock = -1;

    if (s >= 0 && sockinfo[i].client >= 0) {
      if (sockinfo[i].udp_addr != NULL) {
        for(j=0;j<MAX_CLIENTS;j++)
          if (sockinfo[j].udp_addr != NULL && sockinfo[j].sock >= 0)
            if (!memcmp(sockinfo[i].udp_addr,
              sockinfo[j].udp_addr,sizeof(struct sockaddr))) break;
        if (j == MAX_CLIENTS) { 
          if ((j = send_command(s, C_CLOSE, 1, "%d", sockinfo[i].client)) < 0) 
            fprintf(stderr, "Term: C_CLOSE 1 %d failed: %s\n",sockinfo[i].client,
              command_result);
        }
      }
    }
    if (sockinfo[i].udp_addr != NULL)
      free(sockinfo[i].udp_addr);
    sockinfo[i].udp_addr = NULL;
  }
}

/* This is a client lookup for sockets necessary for term_accept 
 * and term_sendto
 */

static int store_sockinfo(int s, int client, struct sockaddr *udp_addr) {
  static char term_clients[MAX_CLIENTS*4];
  static int count = -1;
  char *termclients;
  int i;

  if (count < 0) {
#if 0	/* This isn't necessary any more */
    atexit(term_do_exit); 
#endif 
	/* Zero everything */
    count=0;
    for(i=0;i<MAX_CLIENTS;i++) {
      sockinfo[i].sock = -1;
      sockinfo[i].udp_addr = NULL;
    }
	/* Restore any client information */
    termclients = getenv("TERMCLIENTS");
    if (termclients) 
      strcpy(term_clients,termclients); 
    else 
      term_clients[0] = 0;
    termclients = term_clients;
    for(i=0;i<MAX_CLIENTS && *termclients;i++) {
      for(;*termclients && !isdigit(*termclients) && *termclients != '-';
        termclients++);
      if (! *termclients) break;
      sockinfo[i].sock = atoi(termclients);
      for(;*termclients == '-';termclients++);
      for(;isdigit(*termclients) && *termclients;termclients++);
    } 
  }
  if (s < 0) return 0;

  for(i=0;i<MAX_CLIENTS;i++)
    if(sockinfo[i].sock<0||s == sockinfo[i].sock) break;
  if(i==MAX_CLIENTS){
    i = ++count % MAX_CLIENTS;
    count = i;
    close_fd(sockinfo[i].sock);
    fprintf(stderr,"Too many sockets open.  Killing one.\n");
  }
  sockinfo[i].sock = s;
  sockinfo[i].client = client;
  sockinfo[i].pid = getpid();
  if (! udp_addr) {
    sockinfo[i].udp_addr = NULL;
  }else {
    sockinfo[i].udp_addr = (struct sockaddr *)malloc(sizeof(struct sockaddr));
    memcpy(sockinfo[i].udp_addr,udp_addr,sizeof(struct sockaddr));
  }
	/* Store sockinfo for the children */
  strcpy(term_clients,"TERMCLIENTS=");
  for(i=0;i<MAX_CLIENTS;i++) 
    sprintf(&term_clients[strlen(term_clients)],"%d ",sockinfo[i].sock);
  term_putenv(term_clients);

  return 0;
}

void term_do_exit(void) {
  int k;
  for(k=0;k<MAX_CLIENTS;k++)
    if (sockinfo[k].sock > 0) term_close(k);
  if (Sock >= 0) close(Sock);
  Sock = -1;
}


/* This is a lookup routine so I can return the remote peer name instead
 * of the local name.
 */

int term_getpeername(int S, struct sockaddr *name, int *namelen){
  int i;

  if(S<0) return -1;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  for(i=0;i<MAX_CLIENTS;i++)
    if(S==sockinfo[i].sock) break;

  if (i < MAX_CLIENTS) {
    if(namelen != NULL)
      *namelen=(*namelen>sizeof(struct sockaddr_in)) ? sizeof(struct sockaddr_in) : *namelen;
    /* Only open the connection once for a client! */
    if (Sock < 0)
      if((Sock=socket_connect_server(-1,term_server)) < 0) {
      fprintf(stderr, "Term: %s\n", command_result);
      return getpeername(S,name,namelen);
    }
    if(send_command(Sock, C_GETPEERNAME, 0, "%d", sockinfo[i].client) < 0) {
      fprintf(stderr, "C_GETPEERNAME: %s\n", command_result);
      return getpeername(S,name,namelen);
    }

    if(name != NULL && namelen != NULL){
      *namelen = ( *namelen < sizeof(struct sockaddr) ) ? *namelen : sizeof(struct sockaddr);
      memcpy(name,str_to_sockaddr(command_result,htonl(term_remoteaddr)),
        *namelen);
      return 0; 
    }
  };
  return getpeername(S,name,namelen);
}    


/* This is a lookup routine so I can return the remote socket name instead
 * of the local name.
 */

int term_getsockname(int S, struct sockaddr *name, int *namelen){
  int i,j;
  struct hostent *hp;
  struct sockaddr_in *name_in;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if ((j=getsockname(S,name,namelen)) < 0) return j;
  if (S < 0) return j;

  name_in = (struct sockaddr_in *) name;
  if (name_in->sin_family == AF_INET 
      && name_in->sin_addr.s_addr == INADDR_ANY ) { 
    char hostname[259];

#if defined(SYSV) && !defined(DYNIXPTX)
    { struct utsname unam;
      uname(&unam);
      strcpy(hostname, unam.nodename); }
#else
    gethostname (hostname, sizeof(hostname));
#endif
    if (isdigit(hostname[0])) {
      name_in->sin_addr.s_addr = inet_addr(hostname);
    }else if ((hp=gethostbyname(hostname))) {
      memcpy(&name_in->sin_addr, hp->h_addr, hp->h_length);
    }
  }

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  for(i=0;i<MAX_CLIENTS;i++)
    if (S == sockinfo[i].sock) break;
  if (i == MAX_CLIENTS) return j;

  /* Only open the connection once for a client! */
  if (Sock < 0)
    if ((Sock=socket_connect_server(-1,term_server)) < 0) {

    fprintf(stderr, "Term: %s\n", command_result);
    return j;
  }
  if (send_command(Sock, C_GETSOCKNAME, 0, "%d", sockinfo[i].client) < 0) {
    fprintf(stderr, "C_GETSOCKNAME: %d %d : %s\n", S, sockinfo[i].client, command_result);
    return j;
  }

  if(name != NULL && namelen != NULL){
    *namelen = (*namelen>sizeof(struct sockaddr)) ? 
      sizeof(struct sockaddr) : *namelen;
    memcpy(name,str_to_sockaddr(command_result,htonl(term_remoteaddr)),
      *namelen);
    remote_connect=1;
    j=0;
  };
  return j;
}    


/* For term gethostbyname() is executed on the remote machine when
 * the connection is established.  So here, I just list the hostname
 * in a table.  I list the index as 0.0.0.index.  Since I doubt that
 * 0.0.0.index is used by any host in the world as an IP # this should
 * allow routines to work with and without term.  (NOTE: This limits
 * use to 255 term hosts listed in each program.  If you access more
 * than that, the original hostnames will be replaced.)
 */

struct hostent *term_gethostbyname(char *host){
  static char hostname[259];
  static struct hostent *hp;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if (host != NULL)
    if ((hp=gethostbyname(host)))
      return hp;

  /* Copy the passed-in name, to make sure it doesn't get munged */
  if (host != NULL){
    if(!strcmp(host,"127.0.0.254"))
      hostname[0] = '\0';
    else
      strncpy(hostname,host,256);
  }else
    hostname[0] = '\0';
  
  /* Only open the connection once for a client! */
  if (Sock < 0)
    if ((Sock = socket_connect_server(-1,term_server)) < 0) {

    fprintf(stderr, "Term: %s\n", command_result);
    return NULL;
  }
  if (send_command(Sock, C_GETHOSTNAME, 0, "%s", hostname) < 0) {
    fprintf(stderr, "C_GETHOSTNAME: %s\n", command_result);
    return NULL;
  }
  return str_to_hostent(command_result);
}

struct hostent *term_gethostbyaddr(char *addr, int len, int type){
  struct hostent *hs;
  static char host[16];

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if((hs=gethostbyaddr(addr, len, type))!=NULL)
    return hs;

  if(type != AF_INET) return NULL;

  {
    unsigned long int *j;
    struct in_addr tmp;
    memset(&tmp,0,sizeof(tmp));
    j = (unsigned long *)addr; 
    tmp.s_addr = *j;
    strcpy(host,inet_ntoa(tmp));
  };

  return term_gethostbyname(host);
}

int term_shutdown(int fd,int how){
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }
  close_fd(fd);
  return shutdown(fd,how);
}

int term_close(int fd){
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }
  close_fd(fd);
  return close(fd);
}   


/* For term connections, listen doesn't need to do anything.
 */

int term_listen(int fd, int backlog){
  int i;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if(fd>=0)
    for(i=0;i<MAX_CLIENTS;i++)
      if(fd==sockinfo[i].sock){
    return 0;
  };
  return listen(fd,backlog);
}


/* OK now lets try redirecting socket binding.  I'm at a lost in decided
 * how to make this work for both term and non-term connections.  Here is
 * my current algorithm to redirect AF_INET ports:
 *
 * - If a non-zero port # the service must be listed as both "tcp"
 *   and "term" in /etc/service.
 * - If it is a zero port #, then the port is only redirected if the same
 *   program has already opened a term connection to a remote host.
 *
 * For UDP this is even simpler.  If we are connected to term, we use term.
 */

int term_bind(int S, struct sockaddr *my_addr, int addrlen) {
  int s, i, iport, k;
  struct sockaddr_in *name_in;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  for(k=0;k<MAX_CLIENTS;k++)
    if (S == sockinfo[k].sock) break;

  if (k != MAX_CLIENTS && sockinfo[k].udp_addr != NULL) {
    struct sockaddr addr;
    int j = -1, len = sizeof(addr), tmperrno = 0;

    name_in = (struct sockaddr_in *) my_addr;

    if (name_in->sin_addr.s_addr != inet_addr("127.0.0.254") 
	&& name_in->sin_addr.s_addr != htonl(term_remoteaddr)) {         
      j = bind(S, my_addr, addrlen);	/* Try the address requested */
      tmperrno = errno;
      errno = tmperrno;
      if (j == -1  && tmperrno != EBADF) return j;
    }

	/* Next check if this socket is also usable by term */

    if (k == MAX_CLIENTS) return j;  /* Since it is not available just return */
 
	/* Now we redirect a remote port, so that the local socket is */
	/* bound both locally and remotely. */

    iport=ntohs(name_in->sin_port);

    if ((s=Sock) < 0) {
      fprintf(stderr,"Term: Not connect to server\n");
      errno = tmperrno;
      return j;
    }

    if ((i=send_command(s, C_UBIND, 0, "%d %d",
          sockinfo[k].client, iport)) < 0) {
      fprintf(stderr,"Term: C_UBIND failed: %s %d %d\n",command_result,
        sockinfo[k].client, iport);
    }else if ((i=getsockname(S,&addr,&len)) < 0) {
      perror("getsockname()");
    }else if ((i=send_command(s, C_UDPSET, 1, "%d :%s",
      sockinfo[k].client, sockaddr_to_str(&addr,0))) < 0) {
      fprintf(stderr, "Term: C_UDPSET failed: %s\n",command_result);
    }

    if (i < 0) {
      errno = EACCES;
      return -1;
    }else return j;
  }else {  
    int iport,client = -1;
    struct servent *service=NULL;

    name_in=(struct sockaddr_in *) my_addr;
    if (S < 0 || name_in->sin_family != AF_INET
        || (!remote_connect && !name_in->sin_port
            && name_in->sin_addr.s_addr != htonl(term_remoteaddr)
            && name_in->sin_addr.s_addr != inet_addr("127.0.0.254") ))
      return bind(S,my_addr,addrlen);

    if ((iport=ntohs(name_in->sin_port))) {
      service=getservbyport(iport,"tcp");
      if (service) 
        if ((service=getservbyname(service->s_name,"term")))
          iport=ntohs(service->s_port);
      if (!service && !remote_connect &&
          name_in->sin_addr.s_addr != inet_addr("127.0.0.254") )
        return bind(S,my_addr,addrlen);
    }

    if ((s = socket_connect_server(S,term_server)) < 0) {
      fprintf(stderr, "Term: %s\n", command_result);
      i = -1;
    }else if ((i=send_command(s, C_STATS, 1, "%d", -6)) < 0) {
      fprintf(stderr, "Term: C_STATS -6 failed: %s\n",command_result);
    }else{
      client=atoi(command_result);
      if ((i=send_command(s,C_BIND,0,"%d",iport)) < 0) { 
        fprintf(stderr,"Term: C_BIND failed: %s\n",command_result);
      }else{
        store_sockinfo(s, client,NULL);
      };
    };

    if(i<0){
      if(s >= 0) close(s);
      return i;
    }
    return 0;
  }
}


/* Finally for term connections, accept simply continues where
 * bind left off.
 */

int term_accept(int pending, struct sockaddr *addr, int *addrlen){
  int j=MAX_CLIENTS,s,i = -1;
  char port[34];

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if(pending>=0)
    for(j=0;j<MAX_CLIENTS;j++)
      if(pending == sockinfo[j].sock) break;

  if (j == MAX_CLIENTS || addr == NULL || addrlen == NULL )
    return accept(pending, addr, addrlen);

  if (read(pending,port,10) < 0) {
    perror("Term: read port");
    return -1;
  }
 
  if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("Term: term_bind socket:");
    i = -1;
  } else if (sockinfo[j].client != atoi(port)) {
    fprintf(stderr,"Term: Accept mis-match %d != %s\n",
      sockinfo[j].client,port);
    i = -1;
  }else if ((i = socket_connect_server(s,term_server)) < 0) {
    fprintf(stderr, "Term: %s\n", command_result);
  }else if ((i=send_command(s, C_STATS, 1, "%d", -6)) < 0) {
    fprintf(stderr, "Term: C_STATS -6 failed: %s\n",command_result);
  }else{
    store_sockinfo(s, atoi(command_result),NULL);
    if((i=send_command(s, C_ACCEPT, 0, "%d", sockinfo[j].client))<0){
      fprintf(stderr,"Term: C_ACCEPT: %d %s\n",sockinfo[j].client,command_result);
    }else{
      *addrlen = (*addrlen>sizeof(struct sockaddr)) ? sizeof(struct sockaddr) : *addrlen;
      memcpy(addr,str_to_sockaddr(command_result,htonl(term_remoteaddr)),
        *addrlen); 
      send_command(s, C_DUMB, 1, 0);
      return s;
    }
  }
  if (s >= 0) {
    term_close(s);
  }
  fprintf(stderr,"Term: unknown error\n");
  return -1; 
}


int term_connect(int S, struct sockaddr *addr, int addrlen){
  int i = -1, k, s, client = -1;
  struct sockaddr_in *addr_in;
  char host[59],*ihost;
  unsigned long j;
 
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  for(k=0;k<MAX_CLIENTS;k++)
    if (S == sockinfo[k].sock) break;

  addr_in=(struct sockaddr_in *) addr;

  if(addr_in->sin_family != AF_INET)
    return connect(S,addr,addrlen);

  j = addr_in->sin_addr.s_addr;
  ihost=(char *) &j;

  if (j == INADDR_ANY || j == inet_addr("127.0.0.254")) {
    ihost = "\0";
  }else {
    if (gethostbyaddr(ihost,sizeof(long unsigned),addr_in->sin_family) != NULL)
      return connect(S,addr,addrlen);
#define UC(a) (unsigned char) ((a) & 0xff)
    sprintf(host,"%u.%u.%u.%u",UC(ihost[0]),UC(ihost[1]),UC(ihost[2]),UC(ihost[3]));
    ihost=host;
  }

  if (k < MAX_CLIENTS && sockinfo[k].udp_addr != NULL) {
    client = sockinfo[k].client;
    if ((s = Sock) < 0) {
      if(term_debug >= 0) fprintf(stderr,"Term: Not connected to term\n");
    }else if ((i=send_command(s, C_UDPSET, 0, "%d :%s",
        client, sockaddr_to_str(addr,0))) < 0) {
      fprintf(stderr, "Term: C_UDPSET failed: %s\n",command_result);
    }
  }else {
    if ((i = socket_connect_server(S,term_server)) < 0) {
      fprintf(stderr, "Term: %s\n", command_result);
    }else if ((i=send_command(S, C_STATS, 1, "%d", -6)) < 0) {
      fprintf(stderr, "Term: C_STATS -6 failed: %s\n",command_result);
    }else {
      client = atoi(command_result);
      if (! *ihost) {
        if ((i=send_command(S,C_PORT,0,"%d",ntohs(addr_in->sin_port)))< 0) 
          fprintf(stderr,"Term: C_PORT '%d' : %s\n",ntohs(addr_in->sin_port),
            command_result);
      }else {
        if ((i=send_command(S,C_PORT,0,"%s:%d",ihost,ntohs(addr_in->sin_port)))< 0) 
          fprintf(stderr,"Term: C_PORT '%s:%d' : %s\n",ihost,ntohs(addr_in->sin_port),
            command_result);
      }
    }
    if (i < 0) {
      term_close(S);
    }else if (i >= 0) {
      send_command(S, C_DUMB, 1, 0);
      store_sockinfo(S, client, NULL);
    };
  }
  return ((i<0) ? connect(S,addr,addrlen) : 0);
}


/* term_gethostname() - get hostname of remote system, not local
 * ytalk sends gethostbyname of the local hostname to talkd,
 * so we give it the remote hostname
 */

int term_gethostname(char *name, size_t len) {
  int len2;
  struct hostent *hs=NULL;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if (! remote_connect) return gethostname(name, len);

  if ( remote_term_version < 20054) {
    if (!(hs=term_gethostbyname(NULL))) {
      errno = EINVAL;
      return -1;
    }

    hs = str_to_hostent(command_result);
    len2 = strlen(hs->h_name)+1;
  }else {
    len2 = strlen(term_remotehost) + 1;
  }

  if (len2 > len) {
    errno = EINVAL;
    return -1;
  }
  
  if (hs) 
    strncpy(name,hs->h_name,len2);
  else
    strncpy(name,term_remotehost,len2);

  return 0;
}


/* term_socket()
 * Basically, we create a new client, C_USOCK both ends, C_UBIND the local end
 * (to listen to the local program's sendto's) and C_GETPEERNAME the local UDP
 * socket so term_sendto sends to the right place
 */

int term_socket( int domain, int type, int protocol) {
  struct sockaddr udp_addr;
  int s,i,client,len;
  struct sockaddr_in *addr_in=NULL;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  if (domain != AF_INET || type != SOCK_DGRAM || !remote_connect)
    return socket(domain,type,protocol);

  if ((s=socket_connect_server(-1,term_server)) < 0) {
    fprintf(stderr,"Term: connection failed: %s\n",command_result);
    return socket(domain,type,protocol);
  }

  if ((i=send_command(s, C_STATS, 1, "%d", -6)) < 0) {
    fprintf(stderr,"Term: C_STATS -6 failed: %s\n",command_result);
    close(s);
    return -1;
  }
  client = atoi(command_result);

  if (Sock < 0) 
    if ((Sock=socket_connect_server(-1,term_server)) < 0) {
    fprintf(stderr, "Term: %s\n", command_result);
    return socket(domain,type,protocol);
  }

  if ((i=send_command(Sock, C_USOCK, 0, "%d %d", client,
          (UDP_T_SENDSTRIPHDR | UDP_T_RECADDHDR))) < 0) {
    fprintf(stderr,"Term: C_USOCK remote %d failed: %s\n",client,command_result);
  }else if ((i=send_command(Sock, C_USOCK, 1, "%d %d", client,
      (UDP_T_SENDIGNOREHDR))) < 0) {
    fprintf(stderr,"Term: C_USOCK local %d failed: %s\n",client,command_result);
  }

  close(s);

  if (i < 0) return -1;

  if (term_localaddr == INADDR_ANY) get_term_localaddr(inet_addr("127.0.0.1"));
  if ((s = socket(domain,type,protocol)) < 0) {
    perror("Term: socket()");
  }else if ((i=send_command(Sock, C_UBIND, 1, "%d %d", client, 0)) < 0) {
    fprintf(stderr,"Term: C_UBIND failed: %s\n",command_result);
  }else if ((i=send_command(Sock, C_GETSOCKNAME, 1, "%d", client)) < 0) {
    fprintf(stderr,"Term: C_GETSOCKNAME failed: %s\n",command_result);
  }else if (!(addr_in=(struct sockaddr_in *)str_to_sockaddr(command_result,
    htonl(term_localaddr)))) {
    fprintf(stderr,"Can't translate socket address: %s\n",command_result);
    i = -1;
  }

  if (i < 0) {
    if (s >= 0) 
      term_close(s);
    return -1;
  }

  len = sizeof(udp_addr);
  memcpy(&udp_addr, addr_in, len);

  store_sockinfo(s,client,&udp_addr);
  return s;
}

int term_sendto(int s, void *msg, int len, unsigned int flags, 
    struct sockaddr *to, int tolen) {

  int i,k;
  static un_char *buff=NULL;
  static int alloced=0;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  for(k=0;k<MAX_CLIENTS;k++)
    if (s == sockinfo[k].sock && sockinfo[k].udp_addr != NULL) break;
  if (k == MAX_CLIENTS){
    if (to && tolen) 
      return sendto(s, msg, len, flags, to, tolen);
    else 
      return send(s, msg, len, flags);
  }

  if (alloced < (len+HEADER_SIZE)) {	/* malloc it */
    if (! buff) 
      buff = (un_char *)malloc(sizeof(char)*(len+HEADER_SIZE+1)); /* +1 just to make sure... */
    else
      buff = (un_char *)realloc(buff,sizeof(char)*(len+HEADER_SIZE+1));
    alloced = len + HEADER_SIZE + 1;
  }

  memset(buff, 0, alloced);

  {
    struct sockaddr_in *to_addr = (struct sockaddr_in *)to;
    unsigned long int lhost;
    unsigned short int lport;
    if (to_addr != NULL && tolen) if (to_addr->sin_family == AF_INET) {
      lhost = ntohl(to_addr->sin_addr.s_addr);
      lport = ntohs(to_addr->sin_port);
      buff[0] = (lhost>>24)&255;
      buff[1] = (lhost>>16)&255;
      buff[2] = (lhost>>8)&255;
      buff[3] = lhost&255;
      buff[4] = (lport>>8)&255;
      buff[5] = lport&255;
    }
    to_addr = (struct sockaddr_in *)sockinfo[k].udp_addr;
  }
  memcpy(buff+HEADER_SIZE,msg,len);

  i = sendto(s, buff, len+HEADER_SIZE, flags, sockinfo[k].udp_addr, 16);
  if (i < 0) perror("sendto");

  return (i - HEADER_SIZE);
}

/* This simulates rcmd().  locuser is ignored!  Also stderr
 * is piped with stdout.  For stderr I send a closed pipe descriptor.  This
 * seems to keep "rsh" happy.  (Otherwise you'll create lots of zombies.)
 * So far I've only defined "shell", "cmd", "login", and "who".
 *
 * Programs that use rcmd() should be SUID root, so I take extra security
 * measures here.
 */

int term_rcmd(char **ahost,unsigned short inport, char *locuser,
    char *remuser, char *cmd, int *fd2p){
  int i = 0,s = -1;
  char *rcommand, *term;
  struct servent *sp;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

/* If the host is listed by gethostbyname(), don't use term */

  if (! geteuid() && *ahost)
    if (strcmp(*ahost,"remotehost") && strcmp(*ahost,"127.0.0.254"))
      if (gethostbyname(*ahost))
        return rcmd(ahost,inport,locuser,remuser,cmd,fd2p);

/* These values will need to be passed to the rd_exec function */

  if (fd2p!=NULL)
    *fd2p = -1;

  sp = getservbyport(inport,"tcp");
  if(!strcmp(sp->s_name,"shell")||!strcmp(sp->s_name,"cmd")){
    use_term_command(PRIVILEGED);
    setuid(getuid());
    rcommand=(char *)cmd;
  }else if(!strcmp(sp->s_name,"login")){
    use_term_command(PRIVILEGED);
    setuid(getuid());
    rcommand=NULL;
  }else if(!strcmp(sp->s_name,"who")){
    rcommand="rwho";
  }else{
    fprintf(stderr,"%s is not understood by term yet.\n",sp->s_name);
    return -1;
  };

  s = socket(AF_INET, SOCK_STREAM, 0);
  if ((s = socket_connect_server(s,term_server)) <0) {
    fprintf(stderr, "Term: %s\n", command_result);
    i = -1;
  }else{
	/* Convert the "127.0.0.254 remotehost" alias */
    if (! *ahost || ! strcmp(*ahost,"remotehost")
         || !strcmp(*ahost,"127.0.0.254") ) {
      struct hostent *hs;
        *ahost = term_remotehost;
      if (remote_term_version < 11714 ) {
        *ahost = "127.0.0.1";
      }else if (send_command(s, C_GETHOSTNAME, 0, "%s","\0") < 0) {
        fprintf(stderr, "C_GETHOSTNAME failed: %s\n", command_result);
        *ahost = "127.0.0.1";
      }else if ((hs = str_to_hostent(command_result))){
        *ahost = (char *) hs->h_name;
      };
	/* If after all that work we still have "remotehost", just change it */
	/* We don't do this to begin with, because not everyone has listed */
	/* "localhost" in their .rhosts file & hosts.equiv file. */
      if ( !strcmp(*ahost,"127.0.0.254") 
          || !strcmp(*ahost,"remotehost") ) 
        *ahost = "127.0.0.1";
    }
    term = getenv("TERM");
    if ( i >= 0 ) {
      if ((i=send_command(s, C_STATS, 1, "%d", -6)) < 0) {
        fprintf(stderr, "Term: C_STATS -6 failed: %s\n",command_result);
      }else {
        store_sockinfo(s, atoi(command_result),NULL);
        if (! rcommand) {
          if (! term) 
            i=send_command(s,C_EXEC,0,"rlogin %s -l %s",
              *ahost,remuser);
          else
            i=send_command(s,C_EXEC,0,"-DTERM=%s%crlogin %s -l %s",
              term,'\377',*ahost,remuser);
        }else {
          if (! term) 
            i=send_command(s,C_EXEC,0,"rsh %s -l %s %s",
              *ahost,remuser,rcommand);
          else
            i=send_command(s,C_EXEC,0,"-DTERM=%s%crsh %s -l %s %s",
              term,'\377',*ahost,remuser,rcommand);
        };
        if (i < 0) {
          fprintf(stderr,"Term: C_EXEC %s\n",command_result);
          term_close(s);
        }else {
          send_command(s, C_DUMB, 1, 0);
        };
      };
    };
  };

  if (i<0) {
    if (s>=0) {
      close(s);
      s = -1;
    }
  }else if (fd2p!=NULL) {
    int stat_pipe[2];
    if((pipe(stat_pipe))>=0){ /* open a pipe for passing the connect status */
        *fd2p=stat_pipe[0];
        close(stat_pipe[1]);
    };
  };
  return s;
}

/* Replacing exec calls is very tricky.  But normally fork() is called before */
/* exec, so this should patch most of the security problems. */

int term_fork(void) {
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }
  return fork();
}

int term_vfork(void) {
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }
  return vfork();
}

/* term_recvfrom()
 *
 * recvfrom() replacement which correctly assigns the from-address
 * to the remote site instead of localhost, and strip header. -danjo
 */

int term_recvfrom( int s, char *buff, int len, int flags,
		  struct sockaddr *from, int *fromlen)
{
  static un_char *mybuff = NULL;
  static int alloced;
  int avail, myfromlen, k;
  struct sockaddr_in myfrom;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }

  for(k=0;k<MAX_CLIENTS;k++)
    if (s == sockinfo[k].sock && sockinfo[k].udp_addr != NULL) break;
  if (k == MAX_CLIENTS) return recvfrom(s,buff,len,flags,from,fromlen);

  avail = len + HEADER_SIZE;

#if 0  /* This doesn't seem to work right! */
#ifdef FIONREAD
  ioctl(s,FIONREAD,&avail);
  if (avail > len + HEADER_SIZE) 
    avail = len + HEADER_SIZE;
#endif
#endif

  if( alloced < avail) { /* not enough room, let's alloc more */
    if (! mybuff) {
	 mybuff = (un_char *)malloc(sizeof(char)*(avail+1)); /* +1 needed? */
	 alloced = avail+1;
    }else {
	 mybuff = (un_char *)realloc(mybuff,sizeof(char)*(avail+1));
	 alloced = avail+1;
    }
  }

  myfromlen = sizeof(myfrom);
  if((avail=recvfrom(s,mybuff,avail,flags,(struct sockaddr *)&myfrom,
      &myfromlen) - HEADER_SIZE) < 0){
    k=errno;
    if ((avail += HEADER_SIZE) >= 0)
      memcpy(buff, mybuff,avail*sizeof(char));
    errno=k;
    return avail;
  }

  if (fromlen != NULL && from != NULL) {
    myfrom.sin_family = AF_INET;
    myfrom.sin_addr.s_addr =
      htonl((mybuff[0]<<24)+(mybuff[1]<<16)+(mybuff[2]<<8)+mybuff[3]);
    myfrom.sin_port = htons((mybuff[4]<<8)+mybuff[5]);

    if (myfrom.sin_addr.s_addr == inet_addr("127.0.0.1") ||
        myfrom.sin_addr.s_addr == inet_addr("127.0.0.254") ||
        myfrom.sin_addr.s_addr == INADDR_ANY ) 
      myfrom.sin_addr.s_addr = htonl(term_remoteaddr);
       
    *fromlen = (*fromlen > myfromlen) ? myfromlen : *fromlen;
    memcpy(from,&myfrom,*fromlen * sizeof(char));
  };
  memcpy(buff,mybuff+HEADER_SIZE,avail*sizeof(char));

  return avail;
}


/* term_recv()
 *
 * just call term_recvfrom ignoring the garbage -danjo
 */

int term_recv(int s, char *buf, int len, int flags)
{
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }
  return term_recvfrom(s,buf,len,flags,NULL,NULL);
}

/* The following 3 functions all do the same thing, duplicate the 
 * sockinfo when the file descriptor gets duplicated.
 */

int term_dup(int oldfd) {
  int k, newfd;

  if ((newfd = dup(oldfd)) < 0) return newfd;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }else {
    for(k=0;k<MAX_CLIENTS;k++)
      if (oldfd == sockinfo[k].sock) break;
    if (k == MAX_CLIENTS) return newfd;

    store_sockinfo(newfd, sockinfo[k].client, sockinfo[k].udp_addr);
  }
  return newfd;
}


int term_dup2(int oldfd, int newfd) {
  int k, fd;

  if ((fd = dup2(oldfd,newfd)) < 0) return fd;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }else {
    for(k=0;k<MAX_CLIENTS;k++)
      if (oldfd == sockinfo[k].sock) break;
    if (k == MAX_CLIENTS) return fd;

    store_sockinfo(newfd, sockinfo[k].client, sockinfo[k].udp_addr);
  }
  return fd;
}


int term_fcntl(int fd, int cmd, long arg) {
  int k, newfd;

  if ((newfd=fcntl(fd, cmd, arg)) < 0) return newfd;

  if (cmd != F_DUPFD) return newfd;

  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }else {
    for(k=0;k<MAX_CLIENTS;k++)
      if (fd == sockinfo[k].sock) break;
    if (k == MAX_CLIENTS) return newfd;

    store_sockinfo(newfd, sockinfo[k].client, sockinfo[k].udp_addr);
  }
  return newfd;
}

int term_send(int s, void *msg, int len, unsigned int flags) {
  if (remote_connect < 0) {
    remote_connect=use_term_command(PUBLIC);
    store_sockinfo(-1,-1,NULL);
  }
  return term_sendto(s, msg, len, flags, NULL, 0);
}

