//*************************************************************
//  File name: ICONWND.H
//
//  Description:
//    Implementation for the CIconWnd class
//
//  History:    Date       Author     Comment
//              3/7/94     FJB        Created
//
// Written by Microsoft Product Support Services, Windows Developer Support
// Copyright (c) 1994 Microsoft Corporation. All rights reserved.
//*************************************************************

#include "stdafx.h"
#include "multicon.h"
#include "iconwnd.h"
#include "modview.h"

//*************************************************************
//  Class:
//      CIconWnd 
//
//  Description:
//      AppWizard generated CMDIChildWnd class that dynamically displays the
//      output from it's view as an icon.
//
//  Derived from:
//      CMDIChildWnd
//
//  Data Members: 
//      CDC       m_dcXOR;        // a memory CDC for icon's XOR bitmap
//      CDC       m_dcAND;        // a memory CDC for icon's AND plane
//      HBRUSH    m_hbrNull;      // NULL brush for drawing bounding rect      
//
//  Member Functions:
//      CIconWnd          : Constructor
//     ~CIconWnd          : Destructor
//  
//    Implementation:
//      PreCreateWindow   : Registers a NULL icon
//
//    Message Handlers:
//      OnPaint           : Scales the view's output down to icon size.
//      OnQueryDragIcon   : Converts current output to an icon
//      OnCreate          : Does some initialization
//      OnDestroy         : Does some cleanup
//
//  Comments
//      This class requires a CView derived class that exposes a public
//      OnDraw override.
//      
//
//  History:    Date       Author     Comment
//              3/7/94     FJB        Created
//
//*************************************************************  


#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CIconWnd

IMPLEMENT_DYNCREATE(CIconWnd, CMDIChildWnd)

CIconWnd::CIconWnd()
{
}

CIconWnd::~CIconWnd()
{
}    

   
//*************************************************************
//
//  Class
//      CIconWnd
//
//  Member Function:
//      PreCreateWindow
//
//  Purpose:
//     Registers a window class with a NULL icon.
//
//  Parameters:
//      cs - a CREATESTRUCT reference
//
//  Return: BOOL
//     TRUE  - creation should contine
//     FALSE - stop creation
//   
//
//  Comments:
//     In order to provide an animated icon for a minimized window, it is 
//     necessary to use a window class that specifies a NULL icon.  This will 
//     allow the window to receive WM_PAINT messages while it is minimized so
//     it can render it's iconic appearance. 
//
//  History:    Date       Author     Comment
//              3/7/94     FJB        Created
//
//*************************************************************


BOOL CIconWnd::PreCreateWindow (CREATESTRUCT &cs)
{   
   if (CMDIChildWnd::PreCreateWindow(cs))
   {
      cs.lpszClass = AfxRegisterWndClass ( 
                           CS_DBLCLKS | CS_HREDRAW |
                           CS_VREDRAW,
                           AfxGetApp()->LoadStandardCursor(
                                             IDC_ARROW),
                                             (HBRUSH) (COLOR_WINDOW + 1),
                                             0);  // NULL icon
      return TRUE;
    }    
    else 
      return NULL;
}


BEGIN_MESSAGE_MAP(CIconWnd, CMDIChildWnd)
   //{{AFX_MSG_MAP(CIconWnd)
   ON_WM_PAINT()
   ON_WM_QUERYDRAGICON()
   ON_WM_CREATE()
   ON_WM_DESTROY()
   //}}AFX_MSG_MAP
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CIconWnd message handlers


//*************************************************************
//
//  Class
//      CIconWnd
//
//  Member Function:
//      OnPaint
//
//  Purpose:
//     Renders icon by scaling the view's output.
//
//  Comments:
//     When a view is minimized, it's parent (CMDIChildWnd) is responsible
//     for rendering it's iconic appearance.  In other words, the
//     CMDIChildWnd's OnPaint handler will be called when the view is
//     minimized.  Instead of duplicating the view's output code, instead
//     call the view's OnDraw function to perform the output.  Note that
//     MM_ANISOTROPIC mode is used to scale the view's output to fit. 
//
//     CView::OnDraw is protected, meaning it can only be called from a CView
//     member function.  To get around this, CModView is derived from CView,
//     and OnDraw is made public. 
//
//  History:    Date       Author     Comment
//              3/7/94     FJB        Created
//
//************************************************************* 


void CIconWnd::OnPaint()
{
   CPaintDC dc(this); // device context for painting
   CRect rc; 
   // get the view
   CModView* pView = (CModView*) GetActiveView();
   ASSERT(pView); 
   // get the view's dimensions
   pView->GetWindowRect(&rc);  
   // set MM_ANISOTROPIC so we can scale the view's output
   dc.SetMapMode(MM_ANISOTROPIC);
   
   int cxIcon = GetSystemMetrics(SM_CXICON);
   int cyIcon = GetSystemMetrics(SM_CYICON);
      
   dc.SetViewportExt( cxIcon, cyIcon); 
   dc.SetWindowExt(rc.Width(),rc.Height());   

   // have the view do it's thing
   pView->OnDraw(&dc);           
   // select in the NULL brush and draw the bounding rect
   CBrush* pbrOld = dc.SelectObject(CBrush::FromHandle(m_hbrNull));
   dc.Rectangle(0,0,rc.Width()-1,rc.Height()-1);
   dc.SelectObject(pbrOld);
   
}
  

//*************************************************************
//
//  Class
//      CIconWnd
//
//  Member Function:
//      OnQueryDragIcon
//
//  Purpose:
//      Converts current window contents into an icon.
//
//  Comments:
//      When the user clicks the animated icon, the frame window will 
//      receive an WM_QUERYDRAGICON message.  The frame window must convert
//      it's output to an icon using the CreateIcon API.  See the GDIRCRS
//      sample for details on converting to and from icons.    
//
//  History:    Date       Author     Comment
//              3/7/94     FJB        Created
//
//*************************************************************


HCURSOR CIconWnd::OnQueryDragIcon()
{   
   // find out how big an icon is
   int cxIcon = GetSystemMetrics(SM_CXICON);
   int cyIcon = GetSystemMetrics(SM_CYICON); 
   
   // get the CDC* for the frame window
   CDC* pdcSrc = GetDC();

   // create and select the XOR bitmap
   CBitmap bmpXOR;
   bmpXOR.CreateCompatibleBitmap(pdcSrc, cxIcon, cyIcon);
   CBitmap* pbmpOld1 = m_dcXOR.SelectObject(&bmpXOR);
   
   // copy the current window contents to the XOR bitmap
   m_dcXOR.BitBlt(0, 0, cxIcon, cyIcon, pdcSrc,
                  0, 0, SRCCOPY); 
   
   // create and select the AND bitmap (1 bit plane)
   CBitmap bmpAND;
   bmpAND.CreateBitmap(cxIcon, cyIcon, 1, 1, NULL);
   CBitmap* pbmpOld2 = m_dcAND.SelectObject(&bmpAND); 
   
   m_dcAND.BitBlt(0,0,cxIcon, cyIcon, &m_dcXOR, 0, 0, SRCCOPY);

   m_dcXOR.SelectObject(pbmpOld1);
   m_dcAND.SelectObject(pbmpOld2);
   
   // get the bitmap data for ::CreateIcon
   BITMAP bm;
   bmpAND.GetObject(sizeof(BITMAP), &bm);
   int cb = bm.bmWidthBytes * bm.bmHeight * bm.bmPlanes;
   LPBYTE lpbAND = new __far BYTE[cb];  
   bmpAND.GetBitmapBits(cb, lpbAND);
   
   bmpXOR.GetObject(sizeof(BITMAP), &bm);   
   cb = bm.bmWidthBytes * bm.bmHeight * bm.bmPlanes;
   LPBYTE lpbXOR = new __far BYTE[cb];
   bmpXOR.GetBitmapBits(cb, lpbXOR);   
   
   // make the icon
   HICON hIcon = ::CreateIcon( AfxGetInstanceHandle(),
                       cxIcon,
                       cyIcon,
                       bm.bmPlanes,
                       bm.bmBitsPixel, 
                       lpbAND,
                       lpbXOR);
   
   // cleanup and return the icon
   delete (BYTE __far *)lpbAND;
   delete (BYTE __far *)lpbXOR;
   ReleaseDC(pdcSrc);

   return hIcon;   
}

int CIconWnd::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
   if (CMDIChildWnd::OnCreate(lpCreateStruct) == -1)
      return -1;
   CWindowDC dc(this);
   m_dcXOR.CreateCompatibleDC(&dc); // for efficiency, create these once
   m_dcAND.CreateCompatibleDC(&dc);
   m_hbrNull = (HBRUSH) ::GetStockObject(HOLLOW_BRUSH);  
   
   return 0;
}

void CIconWnd::OnDestroy()
{
   CMDIChildWnd::OnDestroy();
   
   m_dcXOR.DeleteDC();
   m_dcAND.DeleteDC();
}



