/*
 *
 * Sample door source to demonstrate some of Doorskel's capabilities
 *
 */

#include "doorskel.h"
#include "keys.h"

static void _fastcall calendar (void);
static int  _fastcall dayno (int yyyy, int mm, int x, int y);
static void _fastcall prt3mon (int yr, int a, char *prtmon);
static void _fastcall moontxt (void);
static void _fastcall do_library (void);
static void _fastcall do_maze (void);


void funckey (int fkey) {  /* just to show how to set function pointers */

  DoorPrintf("\r\nThe DoorSysOp pressed ALT-F%d!  How marvelous!\r\n",fkey);
}


int main (int argc, char **argv)
{

  /*
   * demonstrates using DoorSelectOne and DoorTopMenu
   *
   */

  static char    *options[] = {"Calendar", "Moon Phase", "ANSI", "Help",
                               "Pulldown Demo","Library","The Maze",
                               "Quit"};

/* number of commands available */
#define NUMSELECTS 8

  static char    *prompt = "\r\nChoose: ";
  char           *p;
  int             com;

  DoorSetProgramName("Sample Doorskel Door");
  DoorSetCopyRight("Executable donated to PD; contains PD calendar and maze functions from C_ECHO/Snippets");
  for(com = 1;com < 11;com++)
    DoorSetFunctionKey(com,funckey);

  /*
   * Be very careful to put this first, right after any essential
   * initialization of function keys and Doorskel strings, as above
   * Crashes may result otherwise!
   */
  if(DoorInit(argc,argv))
    return 1;  /* be sure you HEED THE RETURN OF DOORINIT!!! */

  DoorCLS ();

  DoorPuts("\r\n\r\n\r\n\r\nUse your cursor keys or numeric keypad to toggle, [Enter] to select...\r\n");

  DoorLoadHelp("Sample.hlp"); /* load help for pulldown demo */

  for (;;) {
    DoorCheckCarrier ();
    DoorGetTime ();
    if (!DoorGetGraphics())
      com = DoorSelectOne (options, NUMSELECTS, prompt, 1);
    else {
      while(DoorWhereY() < 3)
        DoorPuts("\r\n");
      DoorPuts ("\x1b[s\x1b[2;1H\x1b[K\x1b[u");
      com = DoorTopMenu (options, NUMSELECTS);
      DoorPuts("\x1b[0;1;36m");
    }

    DoorDebugPrintf("\r\n\04User selected #%d",com); /* just to show how it works */

    switch (com) {
    case -1:
      break;

    case 0:
      DoorPrintGraphics("\x1b[s\x1b[1;1H\x1b[K\x1b[u");
      calendar ();
      break;

    case 1:
      DoorPrintGraphics("\x1b[s\x1b[1;1H\x1b[K\x1b[u");
      moontxt ();
      break;

    case 2:
      DoorToggleGraphics();
      DoorCLS();
      break;

    case 3:
      p = DoorSearchPath("DOORSKEL.DOC");
      if(p) {
        DoorPuts("\r\nHelp for this sample would be kind of silly; reading DOORSKEL.DOC"
                 "\r\njust as an example of how it might be done...");
        DoorHitReturn();
        DoorReadText(p,1);
      }
      else {
        DoorPuts("\r\nHelp for this sample would be kind of silly; I was going to read"
                 "\r\nDOORSKEL.DOC just as an example of how it might be done but I can't"
                 "\r\nseem to find it...\r\n");
        DoorHitReturn();
      }
      break;

    case 4:                     /* pulldown menu demo */
      {
        char *suboptions[4];

        if(!DoorGetGraphics())
          DoorPuts("\r\nYou don't get pulldowns without ANSI graphics; you'll\r\n"
                   "get a tree-structured menu instead.\r\n"
                   "Note you can get help by pressing '?'\r\n");
        else {
          DoorCLS();
          DoorPuts("\x1b[15;1H\x1b[KYou can get help by pressing '?' on any option or suboption.\r\n");
        }
        suboptions[0] = "A._Sub1-1 B._Sub1-2 C._Sub1-3 D._Sub1-4";
        suboptions[1] = "A._Sub2-1 B._Sub2-2 C._Sub2-3";
        suboptions[2] = "A._Sub3-1 B._Sub3-2 C._Sub3-3 D._Sub3-4 E._Sub3-5";
        suboptions[3] = NULL;
        com = 0;
DemoLoop:
        com = DoorANSIMenu2(36,33," Pulldown Demo ",
                            "1Header 2Header 3Header Exit_Demo",
                            suboptions,com);
        switch(com) {
          case 0:
            DoorPuts("\r\nYou chickened out and selected no options.\r\n");
            goto DemoLoop;
          case 11:
          case 12:
          case 13:
          case 14:
          case 21:
          case 22:
          case 23:
          case 31:
          case 32:
          case 33:
          case 34:
          case 35:
            DoorPrintf("\r\nYou selected main option #%d suboption #%d.\r\n",
                       com / 10,com % 10);
            goto DemoLoop;
          case 40:
            break;
          default:  /* should never get here... */
            DoorPuts("\r\nUnknown option chosen.\r\n");
            DoorInkey(2000L);
            break;
        }
      }
      break;

    case 5:
      do_library();
      break;

    case 6:
      do_maze();
      break;

    case NUMSELECTS - 1:        /* quit */
      return 0;

    default:
      DoorPuts ("\r\n**Not yet implemented.\r\n");
      DosSleep(1000L);
      break;
    }
  }
}


/*********************************************************************/
/*                                                                   */
/*  This Section Written by Paul Edwards.                            */
/*                                                                   */
/*  (Modified very slightly to work as a door by M. Kimes)           */
/*                                                                   */
/*  The following block of notes are Paul's.                         */
/*                                                                   */
/*********************************************************************/
/*********************************************************************/
/*                                                                   */
/*  Calendar - produce a calendar for any given year.                */
/*                                                                   */
/*  This program takes as a command line parameter a single number   */
/*  which should be a 4 digit year.  It then prints out a calendar   */
/*  corresponding to that year.  This program was inspired by a      */
/*  shareware program which performed a similar function but didn't  */
/*  come with source!                                                */
/*                                                                   */
/*  Many many thanks to Paul Schlyter, Stockholm, Sweden for his     */
/*  absolutely fantastically brilliant dow macro.  However, due to   */
/*  limitations of this macro (which are affected by anomolies of    */
/*  the '/' and '%' operators on negative numbers) this program will */
/*  generate invalid calendars for years such as 4 AD.  But I reckon */
/*  anyone who wants to print out the gregorian calendar for 4 AD is */
/*  a total wanker, especially when you consider the fact that the   */
/*  gregorian calendar only came into existence in 1582.  There is   */
/*  no upper limit to the year the calendar can print.  The program  */
/*  also knows all the rules, ie the 4, 100, 400 year rules for a    */
/*  leapyear, so that even if you don't know that 1900 wasn't a      */
/*  leapyear, yet 2000 will be, this program knows!  Oh yeah, I'm    */
/*  not sure exactly where the lower bound on calendars is, but it   */
/*  will certainly work for anything above 1582.  Have fun!  Paul.   */
/*                                                                   */
/*  This program is dedicated to the public domain.                  */
/*                                                                   */
/*  Written 1990/8/28.                                               */
/*                                                                   */
/*********************************************************************/


static void _fastcall calendar (void)
{

  int             yr,
                  wrkyr,
                  dig1,
                  dig2,
                  dig3,
                  dig4;
  char            yrstr[32];

  strcpy (yrstr, DoorGenericInput (5, 0, 0, 0, NUM,"\r\nYear for calendar? ",
          NULL));
  DoorPuts ("\r\n");
  if (!*yrstr)
    return;

  yr = atoi (yrstr);

  wrkyr = yr;
  dig1 = wrkyr / 1000;
  wrkyr %= 1000;
  dig2 = wrkyr / 100;
  wrkyr %= 100;
  dig3 = wrkyr / 10;
  wrkyr %= 10;
  dig4 = wrkyr;
  DoorPrintf ("\r\n                                %d %d %d %d\r\n\r\n",
           dig1, dig2, dig3, dig4);
  prt3mon (yr, 0, "       JANUARY                 FEBRUARY"
           "                   MARCH");
  prt3mon (yr, 1, "        APRIL                     MAY"
           "                     JUNE");
  prt3mon (yr, 2, "        JULY                    AUGUST"
           "                  SEPTEMBER");
  prt3mon (yr, 3, "       OCTOBER                 NOVEMBER"
           "                 DECEMBER");
}


static void _fastcall prt3mon (int yr, int a, char *prtmon)
{

  /* print 3 months */

  static char    *letters = " s  m  t  w  t  f  s     "
  " s  m  t  w  t  f  s     "
  " s  m  t  w  t  f  s\r\n";
  int             b,
                  i,
                  j,
                  x;

  DoorPrintf ("%s\r\n\r\n", prtmon);
  DoorPrintf ("%s", letters);
  for (i = 0; i < 6; i++) {
    for (b = 1; b <= 3; b++) {
      for (j = 0; j < 7; j++) {
        x = dayno (yr, a * 3 + b, i, j);
        if (x)
      DoorPrintf ("%2d ", x);
        else
      DoorPrintf ("   ");
      }
      DoorPrintf ("    ");
    }
    DoorPrintf ("\r\n");
  }
  return;
}



#define isleap(year) ((((year % 4) == 0) && ((year % 100) != 0)) || \
    ((year % 400) == 0))


#define dow(y,m,d)  \
( ( ( 3 * (y) - (7 * ((y) + ((m) + 9) / 12)) / 4 + (23 * (m)) / 9 + (d) + 2 \
    + (((y) - ((m) < 3)) / 100 + 1) * 3 / 4 - 15 ) % 7 ) )



static int      daytab[] = {31, 28, 31, 30, 31, 30,
31, 31, 30, 31, 30, 31};


static int _fastcall dayno (int yyyy, int mm, int x, int y)
{

  int             a,
                  b,
                  c;

  a = dow (yyyy, mm, 1);
  b = x * 7 + y;
  c = daytab[mm - 1];
  if ((mm == 2) && isleap (yyyy))
    c++;
  if (b < a)
    return (0);
  if ((b - a) >= c)
    return (0);
  return ((b - a + 1));
}


/* =-=-=-=-=-From BSD Unix with modification by Lynn Nash-=-=-=-=-=-=
             further hacked by M. Kimes to work in a door
             the following block of notes are Lynn's (or from BSD):
 * Output the phase of the moon for the given year, month, day.
 * The routine calculates the year's epact (the age of the moon on
 * Jan 1.), adds this to the number of days in the year, and
 * calculates the phase of the moon for this date.
 * In the algorithm:
 *  diy    Is the day of the year - 1 (i.e., Jan 1 is day 0).
 *  golden Is the number of the year in the Mentonic cycle, used to
 *         determine the position of the calender moon.
 *  epact  Is the age of the calender moon (in days) at the beginning
 *         of the year.  To calculate epact, two century-based
 *         corrections are applied:
 *           Gregorian:      (3 * cent)/4 - 12
 *                   is the number of years such as 1700, 1800 when
 *                   leap year was not held.
 *           Clavian:        (((8 * cent) + 5) / 25) - 5
 *                   is a correction to the Mentonic cycle of about
 *                   8 days evry 2500 years.  Note that this will
 *                   overflow 16 bits in the year 409600.  Beware.
 * The algorithm is accurate for the Gregorian calender only.
 * The magic numbers used in the phase calculation are as follows:
 *     29.5    The moon's period in days.
 *    177      29.5 scaled by 6
 *     22      (29.5 / 8) scaled by 6 (this gets the phase)
 *     11      ((29.5 / 8) / 2) scaled by 6
 * Theoretically, this should yield a number in the range 0 .. 7.
 * However, two days per year, things don't work out too well.
 * Epact is calculated by the algorithm given in Knuth vol. 1
 * (calculation of Easter).  See also the article on Calenders in
 * the Encyclopaedia Britannica and Knuth's algorithm in CACM April
 * 1962, page 209.
 */

static char    *phasetxt[] = {
  "new",                        /* totally dark                         */
  "waxing crescent",            /* increasing to full & quarter light   */
  "first quarter",              /* increasing to full & half light      */
  "waxing gibbous",             /* increasing to full & > than half     */
  "full",                       /* fully lighted                        */
  "waning gibbous",             /* decreasing from full & > than half   */
  "last quarter",               /* decreasing from full & half light    */
  "waning crescent"             /* decreasing from full & quarter light */
};

static int      day_year[] = {  /* Days in year for each month       */
  -1, -1, 30, 58, 89, 119, 150, 180, 211, 241, 272, 303, 333
};                              /* Note: Jan. 1 will equal zero      */


static void _fastcall moontxt (void)
{

  int             year;         /* 1978 = 1978 */
  int             month;        /* Jan = 1 */
  int             day;          /* 1 = 1 */
  int             phase;        /* Moon phase                  */
  int             cent;         /* Century number (1979 = 20)  */
  int             epact;        /* Age of the moon on Jan. 1   */
  int             diy;          /* Day in the year             */
  int             golden;       /* Moon's golden number        */
  char            yrstr[32],
                  was;
  int             c;


  strcpy (yrstr, DoorGenericInput (8, 0, 0, 0, DATE,
          "\r\nDate for moon phase? ",NULL));
  DoorPuts ("\r\n");
  if (strlen (yrstr) < 8) {

Invalid:

    DoorPuts ("Invalid date.\r\n");
    return;
  }

  was = yrstr[4];
  yrstr[4] = 0;
  year = atoi (yrstr);
  yrstr[4] = was;
  was = yrstr[6];
  yrstr[6] = 0;
  month = atoi (&yrstr[4]);
  yrstr[6] = was;
  day = atoi (&yrstr[6]);

  if (month < 1 || month > 12)
    goto Invalid;
  c = daytab[month - 1];
  if ((month == 2) && isleap (year))
    c++;
  if (day < 1 || day > c)
    goto Invalid;

  diy = day + day_year[month];  /* Day in the year  */
  if (month > 2 && isleap (year))
    diy++;                      /* Leapyear fixup       */
  cent = (year / 100) + 1;      /* Century number       */
  golden = (year % 19) + 1;     /* Golden number        */
  epact = ((11 * golden) + 20   /* Golden number        */
           + (((8 * cent) + 5) / 25) - 5  /* 400 year cycle       */
           - (((3 * cent) / 4) - 12)) % 30; /* Leap year correction */
  if (epact <= 0)
    epact += 30;                /* Age range is 1 .. 30 */
  if ((epact == 25 && golden > 11) || epact == 24)
    epact++;
  /* Calculate the phase, using the magic numbers defined above. Note that
   * (phase and 7) is equivalent to (phase mod 8) and is needed on two days
   * per year (when the algorithm yields 8). */
  phase = (((((diy + epact) * 6) + 11) % 177) / 22) & 7;
  DoorPrintf ("\r\nPhase of the moon: %s\r\n", phasetxt[phase]);
}


static void _fastcall do_library (void) {

  /*
   * Demonstrates how to list files, get input (filenames, masks)
   * from user via DoorGenericInput, use of DoorANSIMenu(), etc.
   *
   */

  static char options[] = "List_files Read_file ANSI Quit";

/* number of commands available */

  int com;

  DoorCLS();
  srand((unsigned)time(NULL));
  DoorPrintGraphics("\x1b[0m");

  for (;;) {
    DoorCheckCarrier ();
    DoorGetTime ();
    com = DoorANSIMenu (36,options);
    DoorPuts("\x1b[0;1;33m\r\n\r\n");

    switch (com) {
    case 0:
      break;

    case 'L':
      {
        static char  mask[13] = "*.*",build[133];
        char        *p;
        int          numfiles = 0,lines = 0;

        sprintf(build,"Mask: [%s]  ",mask);
        p = DoorGenericInput(12,0,1,0,FLEW,build,NULL);
        if(!p || !*p) {
          if(!*mask)
            strcpy(mask,"*.*");
          DoorPuts(mask);
        }
        else
          strcpy(mask,p);
        DoorPuts("\r\n");
        sprintf(build,".\\%s",mask);
        {
          DOORFINDBUF f;
          int         hDir;

          hDir = HDIR_CREATE;
          if(!DoorFindFirst(&hDir,build,&f,FILE_ARCHIVED |
             FILE_READONLY | FILE_NORMAL)) {
            do {
              DoorPrintf("%-12s  %-6lu bytes\r\n",f.achName,f.cbFile);
              numfiles++;
              if(++lines > DoorGetLength() - 2) {
                lines = 0;
                if(*DoorGenericInput(1,0,1,1,YESNO," More? [Y/n]  ",
                   NULL) == 'N')
                  break;
                else
                  DoorPuts("\r                       \r");
              }
            } while(!DoorFindNext(hDir,&f));
            DoorFindClose(hDir);
          }
        }
        if(!numfiles)
          DoorPuts("\r\nSeems like a waste of time here, doesn't it?\r\n");
        else {
          DoorPrintf("\r\nListed %d file%s.\r\n",
                  numfiles,&"s"[numfiles == 1]);
          if(lines > DoorGetLength() - (1 + (DoorGetGraphics() == 0) * 8))
            DoorHitReturn();
        }
      }
      break;

    case 'R':
      {
        static char name[13],build[133];
        char       *p,ansifile = 0;

        p = DoorGenericInput(12,0,1,0,FLE,"Filename:  ",NULL);
        DoorPuts("\r\n");
        if(!p || !*p)
          break;
        strcpy(name,p);
        sprintf(build,".\\%s",name);
        p = strrchr(build,'.');
        if(p && !stricmp(p,".ANS"))
          ansifile = 1;
        DoorReadText(build,(char)((ansifile) ? 0 : 1));
        if(ansifile)
          DoorInkey(240000L);
      }
      break;

    case 'A':
      DoorToggleGraphics();
      DoorCLS();
      break;

    case 'Q':
      return;

    default:
      break;
    }
  }
}


static void printmaze (int z,int p,int q,int r) {

  /* this maze generator isn't perfect (doesn't always create a solvable
     maze); we try to compensate for it in the play section... */

  static int a[1817];

  memset(a,0,1817 * sizeof(int));
  DoorCLS();
  for(p = 80;q + p - 80;p -= 2 * a[p])
    for(z = 9;z--;)
      q = 3 & (r = rand() + r * 57) / 7,q = q ? q - 1 ? q - 2 ?
          1 - p % 79 ? - 1 : 0 : p % 79 - 77 ? 1 : 0 : p < 1659
          ? 79 : 0 : p > 158 ? - 79 : 0 , q ? !a[p + q * 2] ?
          a[p += a[p += q] = q] = q : 0 : 0;
    for(;q++ - 1817;)
      DoorPrintf(q % 79 ? "%c":"%c\r\n"," *"[!a[q - 1]]);
}

#define GOUP    1
#define GODOWN  2
#define GOLEFT  3
#define GORIGHT 4

static void _fastcall do_maze (void) {

  static char  line[22][81];
  register int x,y;
  int          key,blasts,lastdir = 0;
  static       numblasts = 10;
  clock_t      t1;
  static long  ttuse = 120000L,hiscore = 0L;
  long         score = 0L;

  {
    FILE *fp;
    char s[81];

    fp = DoorFsopen("SAMPLE.HIS","rt",SH_DENYWR);
    if(!fp) {
      DosSleep(200L);
      fp = DoorFsopen("SAMPLE.HIS","rt",SH_DENYWR);
    }
    if(fp) {
      if(fgets(s,80,fp)) {
        fclose(fp);
        DoorPrintf("\r\n%s\r\n",s);
        DoorInkey(5000L);
      }
      else
        fclose(fp);
    }
  }

  blasts = numblasts;
  DoorPrintGraphics("\x1b[0;1;36m");
  printmaze(rand(),rand(),rand(),rand());

  if(!DoorGetGraphics()) {
    DoorPuts("All you can do is look without ANSI graphics...");
    DoorInkey(15000L);
    DoorPuts("\r\n");
    return;
  }

  DoorPuts("\x1b[0;1;37m\x1b[24;1H\x1b[KTime:\x1b[0;1;33m\x1b[22;78H"
           "\x1b[0;1;32m$\x1b[0;1;33m\x1b[24;40HBlasts:     (Spacebar)");

  for(x = 0;x < 22;x++)
    DoorReadScreenLine(x,line[x]);

  x = 76;
  while(x > 1 && line[21][x] == '*') {
    DoorPrintf("\x1b[22;%dH ",x + 1);
    line[21][x] = ' ';
    x--;
  }
  x = 20;
  while(x > 1 && line[x][77] == '*') {
    DoorPrintf("\x1b[%d;78H ",x + 1);
    line[x][77] = ' ';
    x--;
  }

  t1 = DoorTimerSet(ttuse);

  y = 1;
  x = 1;

  for(;;) {
    if(x < 1)
      x = 1;
    if(x > 77)
      x = 77;
    if(y < 1)
      y = 1;
    if(y > 21)
      y = 21;
    if(x == 77 && y == 21) {
      score = (220000L - (ttuse - DoorTimeLeft(t1))) -
              ((long)(numblasts - blasts) * 10000L);
      hiscore = max(score,hiscore);
      if(hiscore == score) {

        FILE *fp;

        fp = DoorFsopen("SAMPLE.HIS","wt",SH_DENYWR);
        if(!fp) {
          DosSleep(100L);
          fp = DoorFsopen("SAMPLE.HIS","wt",SH_DENYWR);
        }
        if(fp) {
          fprintf(fp,"High score:  %s @ %ld points",DoorGetUserName(),hiscore);
          fclose(fp);
        }
      }
      DoorPrintf("\x1b[24;1H\x1b[K\x1b[0;1;37m"
                 "Congratulations!  Score: %ld / Hi: %ld",
                 score,hiscore);
      ttuse = max(25000L,ttuse - 5000L);
      numblasts = max(3,numblasts - 1);
      DoorComFlushOut();
      DosSleep(1000L);
      DoorComPurgeIn();
      DoorPurgeInLocal();
      DoorInkey(15000L);
      break;
    }
    DoorPrintf("\x1b[24;7H%ld \x1b[%d;%dHo\b",DoorTimeLeft(t1) / 1000L,y + 1,x + 1);
    if(DoorTimeUp(t1)) {
      DoorPuts("\x1b[24;1H\x1b[K\x1b[0;2;37mSorry, you're out of time... ");
      DoorComFlushOut();
      DosSleep(1000L);
      DoorComPurgeIn();
      DoorPurgeInLocal();
      DoorInkey(15000L);
      break;
    }
    key = DoorInkey(1000L);
    switch(key) {
      case 0:
        break;

      case ' ':
        if(blasts && lastdir) {
          switch(lastdir) {
            case GOUP:
              if(line[y - 1][x] == '*' && y > 2) {
                line[y - 1][x] = ' ';
                DoorPrintf("\x1b[%d;%dH ",y,x + 1);
              }
              else
                DoorRemoteBell();
              break;
            case GODOWN:
              if(line[y + 1][x] == '*' && y < 21) {
                line[y + 1][x] = ' ';
                DoorPrintf("\x1b[%d;%dH ",y + 2,x + 1);
              }
              else
                DoorRemoteBell();
              break;
            case GOLEFT:
              if(line[y][x - 1] == '*' && x > 2) {
                line[y][x - 1] = ' ';
                DoorPrintf("\x1b[%d;%dH ",y + 1,x);
              }
              else
                DoorRemoteBell();
              break;
            case GORIGHT:
              if(line[y][x + 1] == '*' && x < 77) {
                line[y][x + 1] = ' ';
                DoorPrintf("\x1b[%d;%dH ",y + 1,x + 2);
              }
              else
                DoorRemoteBell();
              break;
          }
          lastdir = 0;
          blasts--;
          DoorPrintf("\x1b[24;49H%d ",blasts);
        }
        else
          DoorRemoteBell();
        break;

      case ESC:
      case CTRL_K:
        DoorPrintf("\x1b[21;1H\r\n\r\n");
        ttuse = max(25000L,ttuse - 500L);
        return;

      case UPP:
      case '8':
        if(line[y - 1][x] != '*') {
          DoorPrintf("\x1b[%d;%dH ",y + 1,x + 1);
          y--;
        }
        lastdir = GOUP;
        break;

      case DN:
      case '2':
        if(line[y + 1][x] != '*') {
          DoorPrintf("\x1b[%d;%dH ",y + 1,x + 1);
          y++;
        }
        lastdir = GODOWN;
        break;

      case LEFT:
      case '4':
        if(line[y][x - 1] != '*') {
          DoorPrintf("\x1b[%d;%dH ",y + 1,x + 1);
          x--;
        }
        lastdir = GOLEFT;
        break;

      case RIGHT:
      case '6':
        if(line[y][x + 1] != '*') {
          DoorPrintf("\x1b[%d;%dH ",y + 1,x + 1);
          x++;
        }
        lastdir = GORIGHT;
        break;

      default:
        DoorRemoteBell();
        break;
    }
  }
}
