#include "..\au.hpp"

#include <time.h>

#define SZ_NAME 80

/***********************************************************************/
#if SUPPORT_DWC || SUPPORT_LZH || SUPPORT_HA
static void my_localtime(time_t t, ARC_RECORD *r)
{
	struct tm *tm;

	tm = localtime(&t); 		  // Expensive function, costs 4K
	r->date.year  = tm->tm_year;
	r->date.month = tm->tm_mon;
	r->date.day   = tm->tm_mday;
	r->hour 	  = tm->tm_hour;
	r->min		  = tm->tm_min;
	r->sec		  = tm->tm_sec;

	r->date.month++;
	if (r->date.month == 13)
	{
		r->date.month = 1;
		r->date.year++;
	}
}
#endif
/***********************************************************************/
static void fill_date(HANDLE file, ARC_RECORD *record)
{
	unsigned char d1,d2;

	d1 = my_getc_1(file);
	d2 = my_getc_1(file);
	record->date.year  = ((d2 & 0xFE) >> 1) + 1980;
	record->date.month = (d2 & 0x01)*8 + ((d1 & 0xE0) >> 5);
	record->date.day   = (d1 & 0x1F);
}
/***********************************************************************/
static void fill_time(HANDLE file, ARC_RECORD *record)
{
	unsigned char d1,d2;

	d1 = my_getc_1(file);
	d2 = my_getc_1(file);
	record->hour = (d2 & 0xF8) >> 3;
	record->min = (d2 & 0x07)*8 + ((d1 & 0xE0) >> 5);
	record->sec = (d1 & 0x1F) * 2;
}/****************************************************************/
#if SUPPORT_ARC
static int get_record_arc7(ARC_FILE *arcFile, ARC_RECORD *record,
						   char *name)
{
	int ch;
	char string[255];

	if (arcFile->pos == 0)
	{
		arcFile->pos+=29;
		for (;;)
		{
			lseek(arcFile->file, arcFile->pos, SEEK_SET);
			ch = my_getc_1(arcFile->file);
			arcFile->pos+=ch;
			lseek(arcFile->file, arcFile->pos+1, SEEK_SET);
			ch = my_getc_1(arcFile->file);
			if (ch != 0)
				break;
		}
	}
	for (;;)
	{
	   lseek(arcFile->file, arcFile->pos+1, SEEK_SET);
	   ch = my_getc_1(arcFile->file);
	   switch (ch)
	   {
		  case EOF:
		  case 0:
			 return EOF;
		  case 2:
			 record->method = 1; break;
		  case 8:
			 record->method = 3; break;
		  case 10:
			 record->method = 25; break;
		  case 21:
			 arcFile->pos+=15;
			 lseek(arcFile->file, arcFile->pos, SEEK_SET);
			 ch = my_getc_1(arcFile->file);
			 arcFile->pos += ch+14;
			 continue;
		  default:
			 record->method = 0;
	   }
	   break;
	}
	get_file_string(arcFile->file, name);
	lseek(arcFile->file, arcFile->pos+15, SEEK_SET);
	record->packed_size = get_file_long(arcFile->file)+29;		   // 15
	fill_date(arcFile->file, record);							   // 19
	fill_time(arcFile->file, record);							   // 21
	record->crc = (unsigned int)get_file_int(arcFile->file);	   // 23
	record->unpacked_size = get_file_long(arcFile->file);		   // 25
	arcFile->pos+=record->packed_size;

	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_ARJ
static int get_record_arj(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	int ch;
	long pos2, old_pos;
	int  fn_len;

	if (arcFile->isFirst)
	{
		pos2 = arcFile->pos + 34;
		lseek(arcFile->file, pos2, SEEK_SET);		// archive name starts here
		arcFile->pos = pos2 + 8;
		while (my_getc_1(arcFile->file) > 0)		// file name
			arcFile->pos++;
		while (my_getc_1(arcFile->file) > 0)		// comment
			arcFile->pos++;
		arcFile->isFirst = FALSE;
	}
	old_pos = arcFile->pos;

	lseek(arcFile->file, arcFile->pos+5, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch == 0)
		return EOF;
	else if (ch == 4)
		arcFile->version = 60;

	lseek(arcFile->file, arcFile->pos+8, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch & 0x01)
		record->encrypted = TRUE;

	ch = my_getc_1(arcFile->file);								   // 9
	switch (ch)
	{
		case EOF:
			return EOF;
		case 0:
			record->method = 1; break;
		case 1: case 2: case 3: case 4:
			record->method = 17 + ch;  break;
		default:
			record->method = 0;
	}

	lseek(arcFile->file, arcFile->pos+12, SEEK_SET);
	fill_time(arcFile->file, record);							   // 12
	fill_date(arcFile->file, record);							   // 14
	record->packed_size   = get_file_long(arcFile->file);		   // 16
	record->unpacked_size = get_file_long(arcFile->file);		   // 20
	record->crc = get_file_long(arcFile->file); 				   // 24

	lseek(arcFile->file, arcFile->pos+34, SEEK_SET);

	fn_len= get_file_string(arcFile->file, name);
	while (my_getc_1(arcFile->file) > 0)  // comment
		fn_len++;

	arcFile->pos += record->packed_size + fn_len + 42;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;
	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_DWC
static int get_record_dwc(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	int ch;

	if (arcFile->pos == 0)
	{
		arcFile->pos = -2;
		while (arcFile->pos > -256)
		{
			lseek(arcFile->file, arcFile->pos, SEEK_END);
			if (my_getc_1(arcFile->file) == 'D')
			{
				lseek(arcFile->file, arcFile->pos-4, SEEK_END);
				// number of embedded files
				arcFile->hold_n = get_file_long(arcFile->file);
				arcFile->pos = lseek(arcFile->file, 0, SEEK_CUR) - 24 -
							   arcFile->hold_n*34;
				if (arcFile->pos < 0)
					return -2;
				goto around;
			}
			arcFile->pos--;
		}
		return -2;
	}
around:
	arcFile->hold_n--;
	if (arcFile->hold_n < 0)
		return EOF;
	lseek(arcFile->file, arcFile->pos, SEEK_SET);
	get_file_string(arcFile->file, name);
	lseek(arcFile->file, arcFile->pos+13, SEEK_SET);
	record->unpacked_size = get_file_long(arcFile->file);		   // 13
	my_localtime(get_file_long(arcFile->file), record); 		   // 17
	record->packed_size = get_file_long(arcFile->file); 		   // 21
	lseek(arcFile->file, arcFile->pos+29, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch == 1)
		record->method = 3;
	else if (ch == 2)
		record->method = 1;
	else
		record->method = 0;
	lseek(arcFile->file, arcFile->pos+32, SEEK_SET);
	record->crc = (unsigned int)get_file_int(arcFile->file);
	arcFile->pos+=34;
	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_HA
static int get_record_ha(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int  ch, ch2;
	int  fn_len;
	char path[FLENGTH];
	char file_name[FLENGTH];
	char *ptr;

	if (arcFile->pos == 0)
		arcFile->pos = 4;

	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch == EOF)
		return EOF;
	else if (ch == 32)
		record->method = 1;    /* Stored */
	else if (ch == 33)
		record->method = 31;   /* ASC */
	else if (ch == 34)
		record->method = 32;   /* HSC */
	else
		record->method = 0;    /* Unknown */

	record->packed_size = get_file_long(arcFile->file); 		   // 1
	record->unpacked_size = get_file_long(arcFile->file);		   // 5
	record->crc = get_file_long(arcFile->file); 				   // 9
	my_localtime(get_file_long(arcFile->file), record); 		   // 13
	get_file_string(arcFile->file, path);						   // 17
	get_file_string(arcFile->file, file_name);					   // 17 + path

	arcFile->pos += record->packed_size + strlen(path) + strlen(file_name) + 22;

	strcpy(name, path);
	if (name[0] != '\0')
	{
		/* Not sure what this is all about, but seems to be necessary */
		if (name[strlen(name)-1] == -1)
			name[strlen(name)-1] = '\0';
		append_backslash(name);
	}
	strcat(name, file_name);

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;

	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_HAP
static int get_record_hap(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int  ch, ch2;
	int  fn_len;
	char path[FLENGTH];
	char file_name[FLENGTH];
	char *ptr;

	if (arcFile->pos == 0)
		arcFile->pos = 15;

	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch == EOF)
		return EOF;

	lseek(arcFile->file, arcFile->pos+4, SEEK_SET);
	record->packed_size = get_file_long(arcFile->file); 		   // 4
	lseek(arcFile->file, arcFile->pos+17, SEEK_SET);
	my_localtime(get_file_long(arcFile->file), record); 		   // 17
	lseek(arcFile->file, arcFile->pos+22, SEEK_SET);
	record->unpacked_size = get_file_long(arcFile->file);		   // 22
	read(arcFile->file, name, 12);								   // 26
	name[12] = '\0';
	lseek(arcFile->file, arcFile->pos+39, SEEK_SET);
	ch = my_getc_1(arcFile->file);								   // 39
	if (ch == 21)
		record->method = 1; 	 /* Stored */
	else if (ch == 22)
		record->method = 34;	 /* Learned */
	else
		record->method = 0; 	 /* Unknown */

	record->crc = 0;
	arcFile->pos += record->packed_size + 40;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;

	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_HPK
static int get_record_hpk(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	int ch, ch2, n;

	if (arcFile->pos == 0)
	{
		arcFile->pos = -3;
		while (arcFile->pos > -256)
		{
			lseek(arcFile->file, arcFile->pos, SEEK_END);
			if (my_getc_1(arcFile->file) == 'H')
			{
				lseek(arcFile->file, arcFile->pos-9, SEEK_END);
				// number of embedded files
				arcFile->hold_n = get_file_int(arcFile->file);
				ch = get_file_int(arcFile->file);
				arcFile->pos = lseek(arcFile->file, - 6L - ch, SEEK_CUR);
				if (arcFile->pos < 0)
					return -2;

				// Find the front of the file names, there should be a better way

				n = arcFile->hold_n;
				while (n > 0)
				{
					ch = my_getc_1(arcFile->file);								   // 0
					ch2 = my_getc_1(arcFile->file); 							   // 1
					(void)get_file_long(arcFile->file);

					if (ch2 & 0x10)
						get_file_int(arcFile->file);

					if (ch2 & 0x80)
						get_file_long(arcFile->file);
					else
						get_file_int(arcFile->file);

					if (ch2 & 0x40)
						get_file_long(arcFile->file);
					else
						get_file_int(arcFile->file);

					if (ch2 & 0x01)
						get_file_int(arcFile->file);
					n--;
				}

				arcFile->trailer_pos = tell(arcFile->file);
				goto around;
			}
			arcFile->pos--;
		}
		return -2;
	}
around:
	arcFile->hold_n--;
	if (arcFile->hold_n < 0)
		return EOF;
	lseek(arcFile->file, arcFile->pos, SEEK_SET);
	ch = my_getc_1(arcFile->file);								   // 0
	ch2 = my_getc_1(arcFile->file); 							   // 1
	record->method = 33;
	record->crc = 0;
	my_localtime(get_file_long(arcFile->file), record); 		   // 2

	if (ch2 & 0x10) 					 /* attributes */
		get_file_int(arcFile->file);

	if (ch2 & 0x80)
		record->unpacked_size = get_file_long(arcFile->file);
	else
		record->unpacked_size = (unsigned int)get_file_int(arcFile->file);

	if (ch2 & 0x40)
		record->packed_size = get_file_long(arcFile->file);
	else
		record->packed_size = (unsigned int)get_file_int(arcFile->file);

	if (ch2 & 0x01)
		get_file_int(arcFile->file);   /* Something having to do with comments */

	lseek(arcFile->file, arcFile->trailer_pos, SEEK_SET);
	get_file_string(arcFile->file, name);
	arcFile->trailer_pos = tell(arcFile->file);

	arcFile->pos += 10;
	if (ch2 & 0x10)
		arcFile->pos += 2;
	if (ch2 & 0x40)
		arcFile->pos += 2;
	if (ch2 & 0x80)
		arcFile->pos += 2;
	if (ch2 & 0x01)
	{
		arcFile->pos += 2;
		goto around;
	}
	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_HYP
static int get_record_hyp(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int  ch, ch2;
	int  fn_len;
	char *ptr;

	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos+1, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch == EOF)
		return EOF;
	else if (ch == 'H')
		record->method = 30;   /* Hypered */
	else
		record->method = 1;    /* Stored */

	lseek(arcFile->file, arcFile->pos+4, SEEK_SET);
	record->packed_size = get_file_long(arcFile->file); 		   // 4
	record->unpacked_size = get_file_long(arcFile->file);		   // 8
	fill_time(arcFile->file, record);							   // 12
	fill_date(arcFile->file, record);							   // 14
	lseek(arcFile->file, arcFile->pos+21, SEEK_SET);
	fn_len = my_getc_1(arcFile->file);							   // 21
	if (fn_len >= SZ_NAME || fn_len < 0)
		return -2;
	read(arcFile->file, name, fn_len);
	name[fn_len] = '\0';

	record->crc = 0;

	arcFile->pos += record->packed_size + fn_len + 22;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;

	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_LBR
static int get_record_lbr(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	DATE lbr_date;
	long old_pos;
	char temp[255];
	int ch;

	if (arcFile->pos == 0)
	{
		lseek(arcFile->file, arcFile->pos+14, SEEK_SET);
		arcFile->hold_n = my_getc_1(arcFile->file) + 2;
		arcFile->pos = 32;
	}

	old_pos = arcFile->pos;
	if (arcFile->hold_n == 0)
		return EOF;

	for (;;)
	{
		lseek(arcFile->file, arcFile->pos, SEEK_SET);
		ch = my_getc_1(arcFile->file);
		if (ch != 0)
		{
			arcFile->pos+=32;
			arcFile->hold_n--;
			if (arcFile->hold_n == 0)
				return EOF;
		}
		else
			break;
	}

	lseek(arcFile->file, arcFile->pos+1, SEEK_SET); 	/* Name */
	read(arcFile->file, name, 8);
	name[8] = '\0';
	rtrim(name);
	strcat(name, ".");
	read(arcFile->file, temp, 3);
	temp[3] = '\0';
	strcat(name, temp);

	lseek(arcFile->file, arcFile->pos+14, SEEK_SET);
	ch = get_file_int(arcFile->file);
	lseek(arcFile->file, arcFile->pos+26, SEEK_SET);
	record->packed_size = ch * 128 - my_getc_1(arcFile->file);
	record->unpacked_size = record->packed_size;

	lseek(arcFile->file, arcFile->pos+16, SEEK_SET);
	record->crc = (unsigned int)get_file_int(arcFile->file);
	{
		lbr_date.year  = 1977;
		lbr_date.month = 12;
		lbr_date.day   = 31;
		date_add(&lbr_date, &record->date, get_file_int(arcFile->file)); /* 18*/
	}
	lseek(arcFile->file, arcFile->pos+22, SEEK_SET);
	fill_time(arcFile->file, record);

	arcFile->hold_n--;
	record->method = 1; 	/* Stored */
	arcFile->pos+=32;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;
	return 0;
}
#endif;
/****************************************************************/
#if SUPPORT_LZS || SUPPORT_LZH
static int get_record_lzh(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int ch, ch2;
	unsigned char d1;
	char string[255];
	char *ptr;

	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos+4, SEEK_SET);

 /* Additional version check */
	ch = my_getc_1(arcFile->file);
	if (ch == 'h')
	{
		if (arcFile->version < 2)	   /* At least Lharc 1.xx */
			arcFile->version = 2;
	}

 /* Archive method */
	ch = my_getc_1(arcFile->file);
	switch(ch)
	{
		case EOF:
			return EOF;
		case '0':
			record->method = 1;
			break;
		case '1':
			record->method = 9;
			break;
		case '4':
		case '5':
			if (arcFile->type == LZH)
			{
				if (ch == '5')
					record->method = 10;
				else
					record->method = 22;

				if (arcFile->version < 3)	 /* At least lharc 2.x */
					arcFile->version = 3;
			}
			else
			{
				if (ch == '5')
					record->method = 11;
				else
					record->method = 35;
			}
			break;
		default:
			return EOF;
	}

	lseek(arcFile->file, arcFile->pos+7, SEEK_SET);
	record->packed_size   = get_file_long(arcFile->file);
	record->unpacked_size = get_file_long(arcFile->file);

	lseek(arcFile->file, arcFile->pos+20, SEEK_SET);
	ch = my_getc_1(arcFile->file);

	d1 = 0;
	if (ch < 2) 	/* Header level of 0 or 1 */
	{
		lseek(arcFile->file, arcFile->pos+15, SEEK_SET);
		fill_time(arcFile->file, record);
		fill_date(arcFile->file, record);
		lseek(arcFile->file, arcFile->pos+21, SEEK_SET);
		d1 = my_getc_1(arcFile->file);
		read(arcFile->file, string, d1);
		string[d1] = '\0';
		record->crc = (unsigned int)get_file_int(arcFile->file);
		if (ch == 1)		 /* Header level of 1 */
		{
			arcFile->pos += 25 + d1;

			my_getc_1(arcFile->file);
			ch2 = get_file_int(arcFile->file);

			while (ch2 > 0)
			{
				ch = my_getc_1(arcFile->file);
				if (ch == 2)
				{
					if (ch2-4 >= SZ_NAME || ch2-4 < 0)
						return -2;
					read(arcFile->file, record->path, ch2-4);
					record->path[ch2-4] = '\0';
					my_getc_1(arcFile->file);
				}
				arcFile->pos+=ch2;
				lseek(arcFile->file, arcFile->pos, SEEK_SET);
				record->packed_size -= ch2;
				ch2 = get_file_int(arcFile->file);
			}

			arcFile->pos += 2 + record->packed_size;
		}
		else
			arcFile->pos+= 24 + d1 +record->packed_size;
	}
	else if (ch == 2)
	{
		lseek(arcFile->file, arcFile->pos+15, SEEK_SET);
		my_localtime(get_file_long(arcFile->file), record);
		lseek(arcFile->file, arcFile->pos+21, SEEK_SET);
		record->crc = (unsigned int)get_file_int(arcFile->file);
		lseek(arcFile->file, arcFile->pos+24, SEEK_SET);

		arcFile->pos += 24;
		ch2 = get_file_int(arcFile->file);
		while (ch2 > 0)
		{
			ch = my_getc_1(arcFile->file);
			if (ch == 1)
			{
				read(arcFile->file, string, ch2-3);
				string[ch2-3] = '\0';
			}
			else if (ch == 2)
			{
				if (ch2-4 >= SZ_NAME || ch2-4 < 0)
					return -2;
				read(arcFile->file, record->path, ch2-4);
				record->path[ch2-4] = '\0';
				my_getc_1(arcFile->file);
			}
			arcFile->pos+=ch2;
			lseek(arcFile->file, arcFile->pos, SEEK_SET);
			ch2 = get_file_int(arcFile->file);
		}

		arcFile->pos+= 2 + record->packed_size;

		if (arcFile->version < 3)
			arcFile->version = 3;
	}
	else
		return -2;

	/* Somehow Amiga .LZHs have the ability to stuff some sort of
	   comment into the name, this should take care of that */
	{
	   if ((ptr = strchr(string, ' ')) != NULL)
		   *ptr = '\0';

	   if (strlen(string) > 80)
		   string[79] = '\0';
	   strcpy(name, string);
	}

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
	   return -2;
	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_MD
static int get_record_md(ARC_FILE *arcFile, ARC_RECORD *record,
						 char *name)
{
	long old_pos;
	int  ch, ch2;
	int  fn_len;
	char path[FLENGTH];
	char file_name[FLENGTH];
	char *ptr;

	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos+24, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	if (ch == EOF)
		return EOF;
	else if (ch == 0)
		record->method = 1;    /* Stored */
	else if (ch == 1)
		record->method = 36;   /* LZW13 */
	else
		record->method = 0;    /* Unknown */

	record->unpacked_size = get_file_long(arcFile->file);		   // 25
	record->packed_size = get_file_long(arcFile->file); 		   // 29
	lseek(arcFile->file, arcFile->pos+35, SEEK_SET);
	fill_time(arcFile->file, record);							   // 35
	fill_date(arcFile->file, record);							   // 37
	record->crc = (unsigned int)get_file_int(arcFile->file);	   // 39
	ch = my_getc_1(arcFile->file);								   // 41
	read(arcFile->file, name, ch);								   // 42
	name[ch] = '\0';

	arcFile->pos += record->packed_size + 122;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;

	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_PAK || SUPPORT_ARC
static int get_record_pak(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int ch, ch2;
	long next;
	char string[255];

	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos+1, SEEK_SET);

 /* Archive method */
	ch = my_getc_1(arcFile->file);
	switch(ch)
	{
		case 0:
			return EOF;
		case 1:
		case 2:
			record->method = 1; break;
		case 3:
			record->method = 7; break;
		case 4:
			record->method = 6; break;
		case 5: case 6:
			record->method = 2;  break;
		case 7: case 8:
			record->method = 3; break;
		case 9:
			if (arcFile->version == 47)
				arcFile->version = 49;
			record->method = 4;
			break;
		case 10:
			record->method = 5; break;
		case 11:
			if (arcFile->version >= 47 || arcFile->version <= 49)
				arcFile->version = 50;
			record->method = 8;
			break;
		default:
			return -2;
	}

 /* name */
	get_file_string(arcFile->file, name);

	lseek(arcFile->file, arcFile->pos+15, SEEK_SET);
	record->packed_size = get_file_long(arcFile->file);

	fill_date(arcFile->file, record);									// 19
	fill_time(arcFile->file, record);									// 21
	record->crc = (unsigned int)get_file_int(arcFile->file);			// 23
	record->unpacked_size = get_file_long(arcFile->file);				// 25

	arcFile->pos += 29 + record->packed_size;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
	   return -2;

	/* Go after the trailer info */
	while (arcFile->trailer_pos > 0)
	{
		lseek(arcFile->file, arcFile->trailer_pos, SEEK_SET);
		ch = my_getc_1(arcFile->file);
		if (ch == 0 || ch == EOF)
			arcFile->trailer_pos = -1;
		else
		{
			ch2 = get_file_int(arcFile->file);
			next = get_file_long(arcFile->file);
			if (ch2 > arcFile->rec_number)
			   break;
			if (ch2 == arcFile->rec_number && ch == 2)
			{
				if (next >= SZ_NAME || next < 0)
					return -2;
				read(arcFile->file, record->path, next);
				record->path[next] = '\0';
				arcFile->trailer_pos += next + 8;
				break;
			}
			arcFile->trailer_pos += next + 1;
			lseek(arcFile->file, arcFile->trailer_pos, SEEK_SET);
		}
	}
	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_SQZ
static int get_record_sqz(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int  ch, ch2;
	int  fn_len;
	char *ptr;

	if (arcFile->pos == 0)
	{
		arcFile->pos = 8;
		lseek(arcFile->file, arcFile->pos, SEEK_SET);
		ch = my_getc_1(arcFile->file);
		if (ch == 1)	   /* header, skip it */
		{
			ch2 = get_file_int(arcFile->file);
			arcFile->pos += ch2 + 3;
		}
		else
			arcFile->pos = 8;
	}
	old_pos = arcFile->pos;
	lseek(arcFile->file, arcFile->pos, SEEK_SET);
	fn_len = my_getc_1(arcFile->file) - 18;
	if (fn_len >= SZ_NAME || fn_len < 0)
		return -2;
	my_getc_1(arcFile->file);	   /* Not sure what this byte is yet */
	ch = my_getc_1(arcFile->file);
	switch (ch)
	{
		case EOF:
			return EOF;
		case 0:
			record->method = 1;
			break;
		case 1: case 2: case 3: case 4:
			record->method = 17 + ch;
			break;
		default:
			record->method = 0;
			break;
	}
	record->packed_size = get_file_long(arcFile->file); 		   // 3
	record->unpacked_size = get_file_long(arcFile->file);		   // 7
	fill_time(arcFile->file, record);							   // 11
	fill_date(arcFile->file, record);							   // 13
	my_getc_1(arcFile->file);	  /* probably the attribute */
	record->crc = get_file_long(arcFile->file); 				   // 16

	read(arcFile->file, name, fn_len);
	name[fn_len] = '\0';

	arcFile->pos += record->packed_size + fn_len + 20;

	/* Remove double backslashes from the name, I don't know why sqz does this */
	while ((ptr = strchr(name, '/')) != NULL)
		*ptr = '\\';
	while ((ptr = strstr(name, "\\\\")) != NULL)
		memmove(ptr, ptr+1, strlen(ptr));

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;

	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_ZIP
static int get_record_zip(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	int ch, ch2;

	old_pos = arcFile->pos;

	lseek(arcFile->file, arcFile->pos+2, SEEK_SET);
	if (my_getc_1(arcFile->file) == 1)
	{
		lseek(arcFile->file, arcFile->pos+4, SEEK_SET);
		if (my_getc_1(arcFile->file) == 11)
			if (arcFile->version < 7 )
				arcFile->version = 7;
		return EOF;
	}

	lseek(arcFile->file, arcFile->pos+6, SEEK_SET);
	if ((ch2 = my_getc_1(arcFile->file)) == EOF)
		return EOF;

	lseek(arcFile->file, arcFile->pos+8, SEEK_SET);
	if ((ch = my_getc_1(arcFile->file)) == EOF)
		return EOF;

	/* Archive method */
	switch(ch)
	{
		case 0: 							  /* Stored */
			record->method = 1; break;
		case 1: 							  /* Shrunk */
			record->method = 12; break;
		case 2: case 3: case 4: case 5: 	  /* Reduced */
			record->method = 11+ch; break;
		case 6: 							  /* Implode */
			record->method = 17;
			/* at least 1.0x since there is imploding done */
			if (arcFile->version < 6)
				arcFile->version = 6;
			break;
		case 8:
			if (arcFile->version < 53)
				arcFile->version = 53;
			switch (ch2 & 0x06)
			{
				case 0:
					record->method = 26;  break;   /* Deflated */
				case 2:
					record->method = 27;  break;   /* Deflated -EX */
				case 4:
					record->method = 28;  break;   /* Deflated -EF */
				case 6:
					record->method = 29;  break;   /* Deflated -ES */
			}
			if (ch2 & 0x01)
				record->encrypted = TRUE;
			break;
		default:
			record->method = 0;    /* Unknown */
	}

	my_getc_1(arcFile->file);										 // 9
	fill_time(arcFile->file, record);								 // 10
	fill_date(arcFile->file, record);								 // 12
	record->crc = get_file_long(arcFile->file); 					 // 14
	record->packed_size   = get_file_long(arcFile->file);			 // 18
	record->unpacked_size = get_file_long(arcFile->file);			 // 22
	ch = get_file_int(arcFile->file);								 // 26
	if (ch >= SZ_NAME || ch < 0)
		return -2;
	ch2 = get_file_int(arcFile->file);								 // 28
	read(arcFile->file, name, ch);									 // 30
	name[ch] = '\0';

	arcFile->pos += 30+record->packed_size;
	arcFile->pos += ch+ch2;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;
	return 0;
}
#endif
/****************************************************************/
#if SUPPORT_ZOO
static int get_record_zoo(ARC_FILE *arcFile, ARC_RECORD *record,
						  char *name)
{
	long old_pos;
	long next_pos;
	int ch, ch2;
	char string[255];

top:
	if (arcFile->pos == 0)
	{
		lseek(arcFile->file, arcFile->pos+24, SEEK_SET);
		ch = my_getc_1(arcFile->file);
		if (ch == 34)
			arcFile->version = 62;	   /* 1.x */
		else
			arcFile->version = 10;	   /* 2.x */

		arcFile->pos = ch+1;
	}
around:
	old_pos = arcFile->pos;

	lseek(arcFile->file, arcFile->pos+4, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	switch (ch)
	{
		case 0:
		   record->method = 1; break;		  // stored
		case 1:
		   record->method = 23; break;		  // Normal
		case 2:
		   record->method = 24; 			  // High
		   if (arcFile->version != 59)
			   arcFile->version = 51;		  // therefore ZOO 2.10
		   break;
		case EOF:
			return EOF;
		default:
			record->method = 0; break;		  // Unknown
	}
	next_pos = get_file_long(arcFile->file)+1;

	lseek(arcFile->file, arcFile->pos+13, SEEK_SET);

	fill_date(arcFile->file, record);								 // 13
	fill_time(arcFile->file, record);								 // 15
	record->crc = (unsigned int)get_file_int(arcFile->file);		 // 17
	record->unpacked_size = get_file_long(arcFile->file);			 // 19
	record->packed_size = get_file_long(arcFile->file); 			 // 23
	if (record->packed_size == 0)
	{
		arcFile->pos+=27;
		while ((ch = my_getc_1(arcFile->file))!=0xA7)
		{
			if (ch == EOF)
				return EOF;
			arcFile->pos++;
		}
		goto around;
	}
	lseek(arcFile->file, arcFile->pos+29, SEEK_SET);
	if (my_getc_1(arcFile->file) == 1)
	{
		/* Deleted file */
		arcFile->pos = next_pos;
		goto top;
	}

	lseek(arcFile->file, arcFile->pos+37, SEEK_SET);
	get_file_string(arcFile->file, string);
	lseek(arcFile->file, arcFile->pos+55, SEEK_SET);
	ch = my_getc_1(arcFile->file);
	ch2= my_getc_1(arcFile->file);

	if (ch != 64)	   // Not sure what this is all about yet
	{
		if (ch != 0)
		{
			if (ch >= SZ_NAME || ch < 0)
				return -2;
			read(arcFile->file, string, ch);
			string[ch] = '\0';
		}
		if (ch2 == 0)
			strcpy(name, string);
		else
		{
			get_file_string(arcFile->file, name);
			strcat(name, "\\");
			strcat(name, string);
		}
	}
	else
		strcpy(name, string);

	arcFile->pos = next_pos;

	if (arcFile->pos <= old_pos || arcFile->pos > old_pos+100000000)
		return -2;

	return 0;
}
#endif
/****************************************************************/
int get_record(AU *au, ARC_FILE *arcFile, ARC_RECORD *record)
{
	int  retCode;
	char name[SZ_NAME];

	record->method = 0; 			 /* unknown until otherwise */
	record->name[0] = '\0';
	record->path[0] = '\0';
	record->encrypted = FALSE;
	arcFile->rec_number++;
	switch (arcFile->type)
	{
#if SUPPORT_ARC
	   case ARC7:
		  retCode = get_record_arc7(arcFile, record, name);
		  break;
#endif
#if SUPPORT_ARJ
	   case ARJ:
		  retCode = get_record_arj(arcFile, record, name);
		  break;
#endif
#if SUPPORT_DWC
	   case DWC:
		  retCode = get_record_dwc(arcFile, record, name);
		  break;
#endif
#if SUPPORT_MD
	   case MD:
		  retCode = get_record_md(arcFile, record, name);
		  break;
#endif
#if SUPPORT_HA
	   case HA:
		  retCode = get_record_ha(arcFile, record, name);
		  break;
#endif
#if SUPPORT_HAP
	   case HAP:
		  retCode = get_record_hap(arcFile, record, name);
		  break;
#endif
#if SUPPORT_HPK
	   case HPK:
		  retCode = get_record_hpk(arcFile, record, name);
		  break;
#endif
#if SUPPORT_HYP
	   case HYP:
		  retCode = get_record_hyp(arcFile, record, name);
		  break;
#endif
#if SUPPORT_LBR
	   case LBR:
		  retCode = get_record_lbr(arcFile, record, name);
		  break;
#endif
#if SUPPORT_LZS || SUPPORT_LZH
	   case LZS:
	   case LZH:
		  retCode = get_record_lzh(arcFile, record, name);
		  break;
#endif
#if SUPPORT_PAK || SUPPORT_ARC
	   case PAK:
	   case ARC:
		  retCode = get_record_pak(arcFile, record, name);
		  break;
#endif
#if SUPPORT_SQZ
	   case SQZ:
		  retCode = get_record_sqz(arcFile, record, name);
		  break;
#endif
#if SUPPORT_ZIP
	   case ZIP:
		  retCode = get_record_zip(arcFile, record, name);
		  break;
#endif
#if SUPPORT_ZOO
	   case ZOO:
		  retCode = get_record_zoo(arcFile, record, name);
		  break;
#endif
	   default: 				/* not supported but known */
		  return -3;
	}

	if (retCode == 0)
	{
		if (record->path[0] == '\0')
		{
			fix_path(name);
			split_file(name, record->path, record->name);
		}
		else
			strcpy(record->name, name);

		if (record->packed_size < 0 || record->unpacked_size < 0)
			return -2;
	}
	Unused(au);
	return retCode;
}

