program txt2coma;
{ program to convert a text file with one record per line
  into a comma delimited text file.  user may specify fields
  by indicating field positions for each field below an actual
  record from the file.  may also specify various delimter characters.
  allows skipping of beginning records by displaying each record and
  asking if you want to skip it until you say no.  then converts the
  rest of the file.  this is useful for skipping headings, "rulers",
  etc. which appear at the beginning of a file, e.g. a text file
  containing a report output from a program. }

uses dos, crt;
const
 bell = #7;
var
  ch      : char;
  fd      : string[255];
  fldno   : integer;
  numflds : word;
  infile  : text;
  infname : string[255];
  outfname: string[255];
  outfile : text;
  inpline  : string[255];
  ruler   : string[255];
  delim   : string[255];
  fldstrt : array [ 1..36 ] of integer;
  fldlen  : array [ 1..36 ] of integer;
  oqdelim : array [ 1..36 ] of string[1];
  cqdelim : array [ 1..36 ] of string[1];
  fldsep  : string[255];
  recsep  : string[255];
  lineno  : longint;

  procedure instructions;
  begin { instructions }
    writeln('TXT2COMA - a program to convert ASCII text files to');
    writeln('           comma delimited files, a format used for');
    writeln('           input to many programs.');
    writeln('You will be asked for an input and output file name.');
    writeln('The input file will be opened and each line will be ');
    writeln('displayed followed by Skip (Y,N)? until you answer N.');
    writeln('This allows skipping the first few lines in the file');
    writeln('which are often headings.  When you answer N, the line');
    writeln('will be redisplayed with instructions allowing you to');
    writeln('indicate the positions of each "field" to be delimited,');
    writeln('the opening and closing delimiter for each field, the');
    writeln('field separator delimiter, and the record separator');
    writeln('delimiter.  TXT2COMMA then displays each records as');
    writeln('it does the conversion.  NOTE: when specifying field');
    writeln('markers, fields are output in 0..9,A..Z order so you');
    writeln('can change the order of the fields in the output file.');
    writeln;
  end; { instructions }

  procedure getparms;
  var
    skipover : boolean;
    ch       : char;
    ior      : integer;
    i        : integer;
    fldno    : integer;
  begin { getparms }

    repeat
      write('Input file name ([RET] for instructions or to quit): ');
      readln(infname);
      if infname = ''
        then begin
          instructions;
          halt;
        end;
      assign(infile,infname);
      {$I-}
      reset(infile);
      {$I+}
      ior := ioresult;
      if ior <> 0
        then writeln('I/O Error opening file. [',ior,']');
    until ior = 0;

    repeat
      write('Output file name: ');
      readln(outfname);
      assign(outfile,outfname);
      {$I-}
      rewrite(outfile);
      {$I+}
      ior := ioresult;
      if ior <> 0
        then writeln('I/O Error opening output file. [',ior,']');
    until ior = 0;


    { read/show each line of infile and ask if skip }
    skipover := true;
    lineno := 0;
    while (not eof(infile)) and (skipover) do begin
      readln(infile,inpline);
      inc(lineno);
      writeln(lineno:2,':',inpline);
      write('Skip (Y,N)? ');
      ch := readkey;
      if ch in ['Y','y']
        then writeln(ch)
        else if ch in ['N','n']
          then begin skipover := false; writeln(ch); end
          else write(bell);
    end; { while }


    if skipover
      then begin
        writeln('All records skipped.  Ending.');
        close(infile);
        halt;
      end;

    { initialize field start positions }
    for fldno := 1 to 36 do fldstrt[fldno] := -1;

    { show 1st data line and ask for field marks 0..9,A..Z }
    clrscr;
    writeln('Indicate positions of fields below with contiguous string of field');
    writeln('  markers [0..9,A..Z] to indicate field numbers.  Example:');
    writeln('xx:Johnson   Arnold  C.   077-39-7768   (401) 233-7809  27.5');
    writeln('FL:111111111 2222222 33   AAAAAAAAAAA   BBBBBBBBBBBBBB  4444');
    writeln;
    writeln(lineno:2,':',inpline);
    write  ('FL:');
    readln (ruler);

    { convert ruler to field number data }
    numflds := 0;
    for i := 1 to length (ruler) do begin
      { convert ruler to upper case }
      ruler[i] := upcase(ruler[i]);
      { determine if position i is in a field }
      fldno := pos(copy(ruler,i,1),'0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ');
      if fldno > 0
        then begin { determine whether this position is start, contin, or err }
          if (fldstrt[fldno] < 0) { first occurrence of field }
            then begin
              fldstrt[fldno] := i;
              fldlen[fldno] := 1;
              inc(numflds);
            end
            else if (fldstrt[fldno]+fldlen[fldno] = i) { fld continues }
              then inc(fldlen[fldno])
              else ruler[i] := ' ';
                { fld number was found non-consecutive, ignore }
        end
        else ruler[i] := ' '; { make it blank if non-field }
    end; { for i }

    { ask for delims for each field }
    if ruler = ''
      then begin
        writeln('No Fields.  Ending.');
        close(infile);
      end;
    clrscr;
    writeln('Enter the Open/Close Quote delimiter below the beginning and end');
    writeln('  of each field.  Use Blank for no delimiter. Example:');
    writeln;
    writeln('xx:Johnson   Arnold  C.   077-39-7768   (401) 233-7809  27.5');
    writeln('FL:111111111 2222222 33   AAAAAAAAAAA   BBBBBBBBBBBBBB  4444');
    writeln('QU:"       " "     " ""   "         "   "            "');
    writeln;
    writeln(lineno:2,':',inpline);
    writeln('FL:',ruler);
    write  ('QU:');
    readln (delim);


    for fldno := 1 to 36 do
      if fldstrt[fldno] > 0
        then begin
          oqdelim[fldno] := copy(delim,fldstrt[fldno],1);
          if oqdelim[fldno] = ' '
            then oqdelim[fldno] := '';
          cqdelim[fldno] := copy(delim,fldstrt[fldno]+fldlen[fldno]-1,1);
          if cqdelim[fldno] = ' '
            then cqdelim[fldno] := '';
        end
        else begin
          oqdelim[fldno] := '';
          cqdelim[fldno] := '';
        end;

    { get field separator }
    writeln;
    write('Enter field separator string (usually ,): ');
    readln(fldsep);
    writeln;
    writeln('Enter record separator string (usually nothing): ');
    readln(recsep);
  end; { getparms }


begin { txt2coma }

  getparms;
  clrscr;

  { read each line and output as fields }
  lineno := 1; { reset lineno to indicate data lines }
  repeat
    clrscr;
    writeln;
    if keypressed
      then begin
        clrscr;
        writeln('Exit program (Y,N)? ');
        ch := readkey;
        if ch in ['Y','y']
          then begin
            close(infile);
            close(outfile);
            halt;
          end
          else clrscr;
      end;
    writeln('Line Number: ',lineno);
    writeln(inpline);
    fd := '';
    for fldno := 1 to 36 do begin
      if fldstrt[fldno] > 0
        then begin
          write(outfile,fd,oqdelim[fldno],
                         copy(inpline,fldstrt[fldno],fldlen[fldno]),
                         cqdelim[fldno]);
          write(fd,oqdelim[fldno],
                copy(inpline,fldstrt[fldno],fldlen[fldno]),
                cqdelim[fldno]);
          fd := fldsep; { set field separator after 1st fld is written }
        end;
    end; { for fldno }
    writeln(outfile);
    writeln;
    readln(infile,inpline);
    inc(lineno);
  until eof(infile);

  close(infile);
  close(outfile);
  writeln(pred(lineno), ' Records converted.');

end. { txt2coma }