/*
 *   Copyright 1992, 1993, 1994 John Melton (G0ORX/N6LYT)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
	downloaded.c

	List downloaded pacsat files.

	John Melton
	G0ORX, N6LYT

	4 Charlwoods Close
	Copthorne
	West Sussex
	RH10 3QZ
	England

	INTERNET:	g0orx@amsat.org
			n6lyt@amsat.org
			john@images.demon.co.uk
			J.D.Melton@slh0613.icl.wins.co.uk

	History:

	0.1	Initial version. 
	0.2	Added Extract.
	0.3	Added view selection
	0.4	Fixed selected entry after update.
*/

#define VERSION_STRING "(version 0.4 by g0orx/n6lyt)"

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/font.h>
#include <xview/sel_attrs.h>
#include <xview/textsw.h>
#include <xview/svrimage.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <malloc.h>
#include <dirent.h>
#include <signal.h>
#include "ftl0.h"
#include "header.h"

char myCall[16];

Frame frame;
Panel panel;
Panel_item panel_list_item;
Panel_item panel_view;
int nList = 0;
int view = 0;


unsigned long selectedFileId;

extern void exit( );
void panel_select( Panel_item item, char * string, caddr_t client_data,
			Panel_list_op op, Event *event );
void LoadFile( char * fileName );
void button_update( void );
void button_extract( void );
void button_view( void );
void button_reply( void );
void button_quit( void );

void LoadDirectory( char * dirName );
void view_notify_proc( Panel_item item, int value, Event *event );

void signal_child( )
{
	int pid, pstatus;

	pid = wait( &pstatus );
}

main( int argc, char ** argv )
{
	char title[80];
	Xv_Font font;

	xv_init( XV_INIT_ARGS, argc, argv, NULL );

	if( getenv( "MYCALL" ) == (char * )0 )
		strcpy( myCall, "NOCALL" );
	else
		strcpy( myCall, getenv( "MYCALL" ) );

	sprintf( title, "downloaded %s", VERSION_STRING );
	frame = (Frame)xv_create( (Frame)NULL, FRAME,
				FRAME_LABEL, title,
				NULL );

	font = (Xv_Font)xv_find( frame, FONT,
				FONT_FAMILY, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
				FONT_SIZE, 12,
				NULL );

	panel = (Panel)xv_create( frame, PANEL,
				PANEL_LAYOUT, PANEL_HORIZONTAL,
				XV_FONT, font,
				NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Quit",
		PANEL_NOTIFY_PROC, button_quit,
		XV_FONT, font,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Update",
		PANEL_NOTIFY_PROC, button_update,
		XV_FONT, font,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Extract",
		PANEL_NOTIFY_PROC, button_extract,
		XV_FONT, font,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "View File",
		PANEL_NOTIFY_PROC, button_view,
		XV_FONT, font,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Reply",
		PANEL_NOTIFY_PROC, button_reply,
		XV_FONT, font,
		NULL );

	panel_view = (Panel_item)xv_create( panel, PANEL_CHOICE_STACK,
		PANEL_LAYOUT, PANEL_HORIZONTAL,
		PANEL_NOTIFY_PROC, view_notify_proc,
		PANEL_LABEL_STRING, "View",
		PANEL_CHOICE_STRINGS, "All",
					"My Mail",
					"Broadcast",
					NULL,
		PANEL_VALUE, 0,
		NULL );

	xv_set( panel, PANEL_LAYOUT, PANEL_VERTICAL, NULL );
	font = (Xv_Font)xv_find( panel, FONT,
				FONT_FAMILY, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
				NULL );

	(void)xv_create( panel, PANEL_MESSAGE,
		XV_WIDTH, 600,
		XV_FONT, font,
		PANEL_LABEL_STRING,
		"        Id Date/Time From     To         Size Title",
		NULL );

	panel_list_item = (Panel_item)xv_create( panel, PANEL_LIST,
		PANEL_LIST_DISPLAY_ROWS, 10,
		PANEL_LIST_WIDTH, 600,
		PANEL_READ_ONLY, 1,
		XV_FONT, font,
		PANEL_NOTIFY_PROC, panel_select,
		NULL );

	window_fit( panel );
	window_fit( frame );

	view = 0;
	button_update( );


	xv_main_loop( frame );
}

void view_notify_proc( Panel_item item, int value, Event *event )
{
	view = value;
	button_update( );
}

void panel_select( Panel_item item, char * string, caddr_t client_data,
			Panel_list_op op, Event *event )
{

	switch( op )
	{
	case PANEL_LIST_OP_SELECT:
		selectedFileId = (unsigned long)client_data;
		break;
	case PANEL_LIST_OP_DESELECT:
		selectedFileId = 0;
		break;
	}
}

void button_update ( )
{
	DIR		*pDir;
	struct dirent	*pDirent;
	int i;

	xv_set( frame, FRAME_BUSY, 1, NULL );
	xv_set( panel_list_item, XV_SHOW, 0, NULL );
	xv_set( panel_list_item, PANEL_CHOOSE_ONE, FALSE, NULL );
	xv_set( panel_list_item, PANEL_CHOOSE_NONE, TRUE, NULL );

	/* flush out the current list */
	for( ; nList>0; nList-- )
		xv_set( panel_list_item, PANEL_LIST_DELETE, 0, NULL );

	/* walk through the directory of files */
	pDir = opendir( "." );
	while( (pDirent = readdir( pDir )) != NULL )
	{
		/* see if it is a .dl file */
		i = 0;
		while( pDirent->d_name[i] && (pDirent->d_name[i] != '.') )
			i++;
		if( strcmp( &pDirent->d_name[i], ".dl" ) == 0 )
			LoadFile( pDirent->d_name );
	}
	closedir( pDir );

	if( nList )
	{
		xv_set( panel_list_item, PANEL_LIST_SORT, PANEL_REVERSE, NULL );
		xv_set( panel_list_item, PANEL_LIST_SELECT, 0, TRUE, NULL );
		xv_set( panel_list_item, PANEL_CHOOSE_ONE, TRUE, NULL );
		selectedFileId = xv_get( panel_list_item, PANEL_LIST_CLIENT_DATA, 0, NULL );
	}
	xv_set( panel_list_item, XV_SHOW, 1, NULL );
	xv_set( frame, FRAME_BUSY, 0, NULL );
}

void LoadFile( char * fileName )
{
	int hFile;
	int nBytes;
	char * pBuffer;
	static struct tm * GMT;
	char szTemp[128];
	int headerSize;
	HEADER *pHeader;
	int j;
	char *p;
	int display;

	/* open the file */
	hFile = open( fileName, O_RDONLY );
	if( hFile == -1 )
	{
		perror( fileName );
		return;
	}

	/* get the header */
	pBuffer = malloc( 1024 );

	/* read in the 1K */
	nBytes = read( hFile, pBuffer, 1024 );

	/* close the file */
	close( hFile );

	/* extracting the header */
	pHeader = ExtractHeader( pBuffer, nBytes, &headerSize );

	if( pHeader == NULL )
	{
		printf( "invalid header entry\n" );
	}
	else
	{
		/* truncate the source and destination */
		j = 0;
		while( isalnum( pHeader->source[j] ) )
			j++;
      		pHeader->source[j] = '\0';
		j = 0;
		while( isalnum( pHeader->destination[j] ) )
			j++;
      		pHeader->destination[j] = '\0';

		GMT = gmtime( &pHeader->uploadTime );

		if( strlen( pHeader->title ) == 0 )
			sprintf( szTemp, "%8x %02d%02d/%02d%02d %-8s %-8s %6d %-40s",
				pHeader->fileId,
				GMT->tm_mon+1, GMT->tm_mday, GMT->tm_hour, GMT->tm_min,
				pHeader->source,
				pHeader->destination,
				pHeader->fileSize,
				pHeader->fileName );
		else
			sprintf( szTemp, "%8x %02d%02d/%02d%02d %-8s %-8s %6d %-40s",
				pHeader->fileId,
				GMT->tm_mon+1, GMT->tm_mday, GMT->tm_hour, GMT->tm_min,
				pHeader->source,
				pHeader->destination,
				pHeader->fileSize,
				pHeader->title );

#ifdef DEBUG
printf( "entry: %s\n", szTemp );
#endif

		p = pHeader->source;
		while( *p )
		{
			if( islower( *p ) )
				*p = toupper( *p );
			p++;
		}

		p = pHeader->destination;
		while( *p )
		{
			if( islower( *p ) )
				*p = toupper( *p );
			p++;
		}

		switch( view )
		{
			case 0:
				display = 1;
				break;
			case 1:
				if( (strcmp( myCall, pHeader->source ) == 0)
					|| (strcmp( myCall,  pHeader->destination ) == 0) )
					display = 1;
				else
					display = 0;
				break;
			case 2:
				if( strcmp( "ALL",  pHeader->destination) == 0 )
					display = 1;
				else
					display = 0;
				break;
		}
		if( display )
		{
			xv_set( panel_list_item, PANEL_LIST_INSERT, nList, NULL );
			xv_set( panel_list_item, PANEL_LIST_STRING, nList, szTemp, NULL );
			xv_set( panel_list_item, PANEL_LIST_CLIENT_DATA, nList, pHeader->fileId, NULL );
			nList++;
		}
		free( pHeader );
	}
	free( pBuffer );
}

void button_extract( )
{
	char userFileName[128];
	char fileName[128];
	int hFile;
	int fileLength;
	char * pBuffer;
	int headerSize;
	HEADER *pHeader;
	char tmp[128];
	int i;


	sprintf( fileName, "%x.dl", selectedFileId );
	hFile = open( fileName, O_RDONLY );
	if( hFile == -1 )
	{
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "Open failed.",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
		return;
	}
	fileLength = lseek( hFile, 0, 2 );
	pBuffer = malloc( fileLength );
	lseek( hFile, 0, 0 );
	if( read( hFile, pBuffer, fileLength ) != fileLength )
	{
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "Read failed.",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
		free( pBuffer );
		return;
	}
	close( hFile );

	if( pHeader = ExtractHeader( pBuffer, fileLength, &headerSize )  )
	{
		mkdir( "msgs" );
		i = strlen( pHeader->userFileName ) - 1;
		while( i>0 )
		{
			if( pHeader->userFileName[i] == '\\' )
			{
				i++;
				break;
			}
			i--;
		}
		if( i < 0 )
			if( strcmp( pHeader->fileExt, "   " ) )
				sprintf( fileName, "msgs/%s.%s", pHeader->fileName, pHeader->fileExt );
			else
				sprintf( fileName, "msgs/%s", pHeader->fileName );
		else
			sprintf( fileName, "msgs/%s", &pHeader->userFileName[i] );
		hFile = creat( fileName, 0660 );
		write( hFile, pBuffer+pHeader->bodyOffset, pHeader->fileSize-pHeader->bodyOffset );
		close( hFile );
		sprintf( tmp, "Message extracted to file %s", fileName );
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, tmp,
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
		free( pHeader );
	}
	else
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "Bad header.",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );

	free( pBuffer );
	
}

void ViewText( char * fileName )
{
	int pid;
	
	signal( SIGCHLD, signal_child );

	if( (pid=fork()) == 0 )
	{
		/* the child process */
		execlp( "viewtext", "viewtext", fileName, NULL );
	}
	else if( pid>0 )
	{
		/* success */
	}
	else
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "cannot fork viewer",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
}

void ViewZIPText( char * fileName )
{
}

void ViewLZHText( char * fileName )
{
}

void ViewImage( char * fileName )
{
	int pid;
	
	signal( SIGCHLD, signal_child );

	if( (pid=fork()) == 0 )
	{
		/* the child process */
		execlp( "xloadimage", "xloadimage", fileName, NULL );
	}
	else if( pid>0 )
	{
		/* success */
	}
	else
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "cannot fork viewer",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
}

void button_view( )
{
	char fileName[128];
	int hFile;
	int nBytes;
	char * pBuffer;
	int headerSize;
	HEADER *pHeader;
	char szCommand[128];
	int pid;

	/* open the file */
	sprintf( fileName, "%x.dl", selectedFileId );
	hFile = open( fileName, O_RDONLY );
	if( hFile == -1 )
	{
		perror( fileName );
		return;
	}

	pBuffer = malloc( 1024 );

	/* read in the header */
	nBytes = read( hFile, pBuffer, 1024 );

	/* close the file */
	close( hFile );

	/* extracting the header */
	pHeader = ExtractHeader( pBuffer, nBytes, &headerSize );

	free( pBuffer );

	strcpy( szCommand, "" );
	switch( pHeader->fileType )
	{
	case 0:
	case 1:
	case 8:
	case 9:
		switch( pHeader->compression )
		{
		case 0:
			ViewText( fileName );
			break;
		case 1:
		case 2:
			(void)notice_prompt( panel, NULL,
				NOTICE_MESSAGE_STRINGS, "Sorry message is compressed.",
				NULL,
				NOTICE_BUTTON, "OK", 100,
				NULL );
			break;
		default:
			(void)notice_prompt( panel, NULL,
				NOTICE_MESSAGE_STRINGS, "Sorry do not know how to uncompress.",
				NULL,
				NOTICE_BUTTON, "OK", 100,
				NULL );
			break;
		}
		break;
	case 221:
		{
			ViewImage( fileName );
		}
		break;
	default:
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "Do not know how to view message",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
		break;
	}

	free( pHeader );

}

void button_reply( )
{
	char fileName[128];
	int hFile;
	int nBytes;
	char * pBuffer;
	int headerSize;
	HEADER *pHeader;
	char source[32];
	char title[128];
	int pid;

	/* open the file */
	sprintf( fileName, "%x.dl", selectedFileId );
	hFile = open( fileName, O_RDONLY );
	if( hFile == -1 )
	{
		perror( fileName );
		return;
	}

	pBuffer = malloc( 1024 );

	/* read in the header */
	nBytes = read( hFile, pBuffer, 1024 );

	/* close the file */
	close( hFile );

	/* extracting the header */
	pHeader = ExtractHeader( pBuffer, nBytes, &headerSize );

	free( pBuffer );

	sprintf( source, "%s", pHeader->source );
	sprintf( title, "Re: %s", pHeader->title );

	free( pHeader );
	
	signal( SIGCHLD, signal_child );

	if( (pid=fork()) == 0 )
	{
		/* the child process */
		execlp( "message", "message", source, title, NULL );
	}
	else if( pid>0 )
	{
		/* success */
	}
	else
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "cannot fork message",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
}

void button_quit( void )
{
	xv_destroy_safe( frame );
}
