/******************************************************************************
* Module    :   Split --- Calculate the number of parts needed, and do
*               even split.
*
* Author    :   John W. M. Stevens
******************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <time.h>
#include    <errno.h>

#include    "post.h"

/*-----------------------------------------------------------------------------
| Routine   :   GenSubLn() --- Generate the subject line.
|
| Inputs    :   PartNo      - Part number.
|               Parts       - Total number of parts.
|               BinData     - Configuration data.
| Outputs   :   SubBfr      - Subject line contents.
-----------------------------------------------------------------------------*/

static
void    GenSubLn(int        PartNo,
                 long       Parts,
                 BIN_DATA   *BinData,
                 char       *SubBfr)
{
    if ( *BinData->PictType )
        sprintf(SubBfr,
                "%s %s(Part %d/%ld) [%s]",
                BinData->FlName,
                ( BinData->RepostFlag )
                    ?   "REPOST "
                    :   "",
                PartNo,
                Parts,
                BinData->PictType
               );
    else
        sprintf(SubBfr,
                "%s %s(Part %d/%ld)",
                BinData->FlName,
                ( BinData->RepostFlag )
                    ?   "REPOST "
                    :   "",
                PartNo,
                Parts
               );
}

/*-----------------------------------------------------------------------------
| Routine   :   WrtArtHdr() --- Write the article header.
|
| Inputs    :   PartFlPtr   - Pointer to part file.
|               HdrFl       - Pointer to header file.
|               PartNo      - Part number.
|               Parts       - Total number of parts.
|               BinData     - Configuration data.
| Outputs   :   SubBfr      - Returns the generated subject line info.
-----------------------------------------------------------------------------*/

HDR_TYPE    HdrLines[] = {
    {   'c',    ""  },
    {   'n',    ""  },
    {   't',    ""  },
    {   '\0',   ""  }
};

static
void    WrtArtHdr(FILE      *PartFlPtr,
                  FILE      *HdrFl,
                  int       PartNo,
                  long      Parts,
                  BIN_DATA  *BinData,
                  char      *SubBfr)
{
    register    int         i;
    auto        time_t      time_val;
    auto        time_t      ExpDays;
    struct      tm          *TmPtr;
    auto        char        TmBfr[BFR_SZ];
    auto        char        HdrBfr[BFR_SZ];
    auto        HDR_TYPE    *HdrPtr;
    extern      char        *Version;

    /*  Seek to begining of header file.    */
    if (fseek(HdrFl, 0L, 0) != 0)
    {
        fprintf(stderr,
                "%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                strerror(errno));
        exit( 1 );
    }

    /*  Read lines from Header File.    */
    while (fgets(HdrBfr, BFR_SZ - 1, HdrFl) != NULL)
    {
        /*  Process string. */
        for (i = 0; HdrBfr[i]; i++)
        {
            /*  If this is a macro character then replace the macro,
            *   otherwise, just copy the character out.
            */
            if (HdrBfr[i] == '%')
            {
                /*  Which macro is this?    */
                switch ( HdrBfr[++i] )
                {
                case 'd':
                    time_val = time( NULL );
                    TmPtr = localtime( &time_val );
					strftime(TmBfr,
							 BFR_SZ,
							 "%A, %d-%b-%y %X %Z",
							 TmPtr);
					fprintf(PartFlPtr,
							TmBfr);
                    break;
                case 'e':
                    /*  Get days figure.    */
                    if (HdrBfr[++i] < '0' || HdrBfr[i] > '9')
                    {
                        i--;
                        ExpDays = (time_t) 604800L;
                    }
                    else
                        ExpDays = HdrBfr[i] - '0';

                    /*  Determine Expiration Date.  */
                    time_val = time( NULL );
                    time_val += ExpDays * (time_t) 86400L;

                    /*  Convert to USENET standard. */
                    TmPtr = localtime( &time_val );
					strftime(TmBfr,
							 BFR_SZ,
							 "%A, %d-%b-%y %X %Z",
							 TmPtr);
					fprintf(PartFlPtr,
							TmBfr);
                    break;
                case 's':
                    GenSubLn(PartNo,
                             Parts,
                             BinData,
                             SubBfr);
                    fprintf(PartFlPtr,
                            SubBfr);
                    break;
                default:
                    /*  Seek for appropriate macro tag in list, and
                    *   if data not already there, ask for info from
                    *   user.
                    */
                    for (HdrPtr = HdrLines; HdrPtr->MacroTag; HdrPtr++)
                    {
                        /*  Found it yet?   */
                        if (HdrBfr[i] == HdrPtr->MacroTag)
                        {
                            /*  If there is no data yet, get it from user.  */
                            if (! *HdrPtr->Bfr)
                            {
                                auto    char    c;

                                /*  Print prompt.   */
                                c = HdrBfr[i - 1];
                                HdrBfr[i - 1] = '\0';
                                fputs(HdrBfr, stdout);
                                HdrBfr[i - 1] = c;

                                /*  If no error, print data.    */
                                 if (gets( HdrPtr->Bfr ) == NULL)
                                    HdrPtr->Bfr[0] = '\0';
                            }

                            /*  Print data. */
                            fprintf(PartFlPtr, HdrPtr->Bfr);
                            break;
                        }
                    }
                    break;
                }
            }
            else
                fputc(HdrBfr[i], PartFlPtr);
        }
    }
}

/*-----------------------------------------------------------------------------
| Routine   :   ExecCmd() --- Generate the subject line.
|
| Inputs    :   CmdBfr      - Command template string buffer.
|               PartFlNm    - Part file name.
|               PartNo      - Part number.
|               Parts       - Total number of parts.
|               BinData     - Configuration data.
|               SubBfr      - Subject line contents.
-----------------------------------------------------------------------------*/

void    ExecCmd(char        *CmdBfr,
                char        *PartFlNm,
                char        *SubBfr)
{
    register    int     i;
    register    int     j;
    auto        int     BfrLen;
    auto        char    *Bfr;
    static      char    SysCmd[BFR_SZ * 2];

    /*  Start copying characters.  If we see a '\', this
    *   is the escape character.  If we see '#', replace
    *   this with the part file name.
    */
    for (i = j = 0;
         i < (2 * BFR_SZ) - 1;
         i++)
    {
        /*  Check for macro.    */
        if (CmdBfr[i] == '%')
        {
            /*  Which macro?    */
            switch ( CmdBfr[++i] )
            {
            case 's':
                Bfr = SubBfr;
                break;
            case 'p':
                Bfr = PartFlNm;
                break;
            default:
                SysCmd[j++] = CmdBfr[i];
                break;
            }

            /*  Check for buffer overflow.  */
            BfrLen = strlen( Bfr );
            if (j + BfrLen < (2 * BFR_SZ) - 1)
            {
                /*  Copy in subject string. */
                strcpy(SysCmd + j, Bfr);
                j += BfrLen;
            }
            else
            {
                fprintf(stderr,
                        "%s %d : Error - Command buffer string overflow\n",
                        __FILE__,
                        __LINE__);
                exit( 1 );
            }
        }
        else
            SysCmd[j++] = CmdBfr[i];
    }

    /*  Execute system command. */
    system( SysCmd );
}

/*-----------------------------------------------------------------------------
| Routine   :   Split() --- Determine number of parts, and create a split,
|               uuencoded file.
|
| Inputs    :   FlName      - Name of binary file to prepare to post.
|               HdrFl       - Header information file.
|               DescFl      - Description file.
|               BinData     - Data for construction of proper subject line.
|               CmdBfr      - Command string buffer.
-----------------------------------------------------------------------------*/

void    Split(FILE      *HdrFl,
              FILE      *DescFl,
              BIN_DATA  *BinData,
              char      *CmdBfr)
{
    register    int         i;
    auto        CRC_TYPE    Crc;
    auto        char        SubBfr[BFR_SZ];
    auto        char        Exten[5];
    auto        char        OutFlNm[MAX_PATH];
    auto        char        OutFlBase[MAX_PATH];
    auto        char        Bfr[BFR_SZ];
    auto        FILE        *BinFlPtr;
    auto        FILE        *PartFlPtr;
    auto        long        FlSize;
    auto        long        Lines;
    auto        long        Parts;
    auto        BIN_TYPES   BinFlType;

    extern      int         MsDosFileNms;
    extern      char        *Version;

    /*  Open file as a binary read-only file.   */
    if ((BinFlPtr = fopen(BinData->FlName, BIN_READ)) == NULL)
    {
        fprintf(stderr,
                "%s %d : Error - Could not open file '%s' for reading.\n",
                __FILE__,
                __LINE__,
                BinData->FlName);
        exit( 1 );
    }

    /*  If we need a CRC, then calculate one.   */
    if ( BinData->CalcCRC )
        Crc = CalcCRC( BinFlPtr );

    /*  Determine binary file type. */
    BinFlType = IdBinFile(BinFlPtr, Exten);

    /*  Position file pointer to end of file.   */
    if (fseek(BinFlPtr, 0L, 2) != 0)
    {
        fprintf(stderr,
                "%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                strerror(errno));
        exit( 1 );
    }
    FlSize = ftell( BinFlPtr );

    /*  Position file pointer to begining of file.  */
    if (fseek(BinFlPtr, 0L, 0) != 0)
    {
        fprintf(stderr,
                "%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                strerror(errno));
        exit( 1 );
    }

    /*  Calculate the number of parts.  */
    Lines = FlSize / BYTES_PER_LINE;
    if (FlSize % BYTES_PER_LINE)
        Lines++;
    Parts = Lines / LINES_PER_PART;
    if (Lines % LINES_PER_PART)
        Parts++;
    if (Lines % Parts)
        Lines = (Lines / Parts) + 1;
    else
        Lines /= Parts;

    /*  Get base file name. */
    BaseName(BinData->FlName, OutFlBase);

    /*  Check for MS-DOS and USENET compatibility.  */
    if ( MsDosFileNms )
        ModifyFlNm(OutFlBase, BinFlType, Exten, BinData->FlName);
    else
        strcpy(BinData->FlName, OutFlBase);

    /*  Strip off extension for part base name. */
    StripExt(BinData->FlName, OutFlBase);

    /*  Now do split.   */
    for (i = 0; i <= Parts; i++)
    {
        /*  Create the part file name.  */
        sprintf(OutFlNm, "%s.%d", OutFlBase, i);

        /*  Open the file to write part to. */
        if ((PartFlPtr = fopen(OutFlNm, TXT_WRITE)) == NULL)
        {
            fprintf(stderr,
                    "%s %d : Error - Could not open file '%s' for writing.\n",
                    __FILE__,
                    __LINE__,
                    OutFlNm);
            exit( 1 );
        }

        /*  Write article header to file.   */
        if ( HdrFl )
            WrtArtHdr(PartFlPtr, HdrFl, i, Parts, BinData, SubBfr);
        else
            GenSubLn(i, Parts, BinData, SubBfr);

        /*  Write POST signature.   */
        fprintf(PartFlPtr,
                "\n%s %s\n\n",
                Version,
                SubBfr);
        if ( BinData->CalcCRC )
            fprintf(PartFlPtr,
                    "POST Crc: 0x%08lx Size: %ld\n\n",
                    Crc,
                    FlSize);

        /*  Get and write the description in part 0.    */
        if (i == 0)
        {
            /*  Print prompt to get description information.    */
            if (DescFl == stdin)
            {
                printf("\nEnter description.  To exit, enter an "
                       "end of file\ncharacter as the first character "
                       "on a blank line.\n");
            }

            /*  Get description information from the description file
            *   pointer.
            */
            while (fgets(Bfr, BFR_SZ, DescFl) != NULL)
                fprintf(PartFlPtr, "  %s", Bfr);
            fprintf(PartFlPtr, "\n");
        }
        else
        {
            /*  Begin uuencode section. */
            fprintf(PartFlPtr,"BEGIN -- Cut Here -- cut here\n");
            if (i == 1)
                fprintf(PartFlPtr, "begin 600 %s\n", BinData->FlName);

            /*  Encode a part.  */
            Encode(BinFlPtr, PartFlPtr, Lines);

            /*  Write end.  */
            if (i == Parts)
                fprintf(PartFlPtr, "`\nend\n");
            fprintf(PartFlPtr,"END   -- Cut Here -- cut here\n");
        }

        /*  Close the part file.    */
        fprintf(PartFlPtr, "\n");
        fclose( PartFlPtr );

        /*  If there is a command to execute, put part number string
        *   into appropriate place in the command, then execute.
        */
        if ( *CmdBfr )
        {
             ExecCmd(CmdBfr,
                    OutFlNm,
                    SubBfr);
        }
    }

    /*  Close the binary file.  */
    fclose( BinFlPtr );
}
