/****************************************************************************\
*                                                                            *
*  FADE.C                                                                    *
*                                                                            *
*  This program demonstrates how to perform a smooth palette fade with       *
*  Fastgraph.  This example assumes a 256-color video mode with 6-bit DAC    *
*  values (i.e., between 0 and 63).  These values are defined at the top of  *
*  this file, so you can change them easily.                                 *
*                                                                            *
*  The fadein() and fadeout() routines in this program were originally       *
*  written by John Wagner, author of the IMPROCES image processing program.  *
*                                                                            *
*  To compile this program and link it with Fastgraph version 4.0:           *
*                                                                            *
*     Borland C++:                                                           *
*        BCC -ms FADE.C FGS.LIB                                              *
*                                                                            *
*     Microsoft C/C++ and Visual C++:                                        *
*        CL /AS FADE.C /link FGS                                             *
*                                                                            *
*     Microsoft QuickC:                                                      *
*        QCL /AS FADE.C /link FGS                                            *
*                                                                            *
*     Microsoft Visual C++ 32-bit Edition with Phar Lap TNT extender:        *
*        CL FADE.C /link /stub:\TNT\BIN\GOTNT.EXE FG32VC.LIB                 *
*                                                                            *
*     Power C:                                                               *
*        PC /ms FADE                                                         *
*        PCL FADE ;FGS                                                       *
*                                                                            *
*     Turbo C and Turbo C++:                                                 *
*        TCC -ms FADE.C FGS.LIB                                              *
*                                                                            *
*     Watcom C/C++ (16 bits):                                                *
*        WCL /ms FADE.C FGS.LIB                                              *
*                                                                            *
*     Watcom C/C++ (32 bits) with Rational Systems DOS/4GW extender:         *
*        WCL386 /l=dos4g FADE.C FG32.LIB FG32DPMI.LIB                        *
*                                                                            *
*  This program also can be linked with Fastgraph/Light 4.0 if you replace   *
*  the FGS library references with FGLS.                                     *
*                                                                            *
*  Fastgraph (tm) and Fastgraph/Light (tm) are graphics libraries published  *
*  by Ted Gruber Software.  For more info, please call, write, or FAX.       *
*                                                                            *
*  Ted Gruber Software                           orders/info (702) 735-1980  *
*  PO Box 13408                                          FAX (702) 735-4603  *
*  Las Vegas, NV  89112                                  BBS (702) 796-7134  *
*                                                                            *
\****************************************************************************/

#include <fastgraf.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* function prototypes */

void main(void);
void fadein(char *,int);
void fadeout(int);

/* these values can be changed for different video modes */

#define NDACS 256
#define NCOLORS 64
#define VIDEO_MODE 19

/* this typedef struct is a clean way to do DACs */

typedef struct
{
   unsigned char r, g, b;
}
RGB;

/* these global arrays hold two complete sets of DAC values */

RGB dacs1[NDACS], dacs2[NDACS];

/* main program */

void main()
{
   int delay;
   int old_mode;

   /* in case we're compiling for protected mode */

   fg_initpm();

   /* make sure the requested graphics mode is available */

   if (fg_testmode(VIDEO_MODE,1) == 0)
   {
      printf("This program requires a %d-color graphics mode.\n",NDACS);
      exit(1);
   }

   /* calculate the base delay between DAC updates */

   delay = fg_measure() / 128;

   /* initialize Fastgraph for the requested video mode */

   old_mode = fg_getmode();
   fg_setmode(VIDEO_MODE);

   /* for each PCX file, fade it in and then back out */

   fadein("TOMMY.PCX",delay);
   fg_waitfor(36);
   fadeout(delay);
   fg_waitfor(18);

   fadein("BALLOONS.PCX",delay*2);
   fg_waitfor(36);
   fadeout(delay*2);
   fg_waitfor(18);

   fadein("MOUSE.PCX",delay*4);
   fg_waitfor(36);
   fadeout(delay*4);

   /* restore the original video mode and screen attributes */

   fg_setmode(old_mode);
   fg_reset();
}

/****************************************************************************\
*                                                                            *
*  fadein                                                                    *
*                                                                            *
*  Display an image by gradually increasing each DAC's RGB components to     *
*  their original values.                                                    *
*                                                                            *
\****************************************************************************/

void fadein(PCXfile,delay)
char *PCXfile;
int delay;
{
   register int i, j;
   int target;

   /* get the target DAC values from the PCX file */

   fg_pcxpal(PCXfile,(unsigned char *)dacs1);

   /* zero all of the DACs */

   memset(dacs2,NDACS*3,0);
   fg_setdacs(0,NDACS,(unsigned char *)dacs2);

   /* display the blacked-out PCX image */

   fg_showpcx(PCXfile,1);

   /* cycle through the DACs, gradually increasing them to their old values */

   for (j = 0; j < NCOLORS; j++)
   {
      /* increment each RGB component if it is below its old value */

      target = NCOLORS - j;

      for (i = 0; i < NDACS; i++)
      {
         if (dacs1[i].r > target && dacs2[i].r < dacs1[i].r) dacs2[i].r++;
         if (dacs1[i].g > target && dacs2[i].g < dacs1[i].g) dacs2[i].g++;
         if (dacs1[i].b > target && dacs2[i].b < dacs1[i].b) dacs2[i].b++;
      }

      /* update the DACs each time through the loop */

      fg_stall(delay);
      fg_setdacs(0,NDACS,(unsigned char *)dacs2);
   }
}

/****************************************************************************\
*                                                                            *
*  fadeout                                                                   *
*                                                                            *
*  Erase an image by gradually fading each DAC's RGB components to black.    *
*                                                                            *
\****************************************************************************/

void fadeout(delay)
int delay;
{
   register int i, j;

   /* load the dacs1 and dacs2 arrays with the current DAC values */

   fg_getdacs(0,NDACS,(unsigned char *)dacs1);
   memcpy(dacs2,dacs1,NDACS*3);

   /* cycle through the DACs, gradually reducing them to 0 (black) */

   for (j = 0; j < NCOLORS; j++)
   {
      /* decrement each RGB component if it is above 0 */

      for (i = 0; i < NDACS; i++)
      {
         if (dacs2[i].r) dacs2[i].r--;
         if (dacs2[i].g) dacs2[i].g--;
         if (dacs2[i].b) dacs2[i].b--;
      }

      /* update the DACs each time through the loop */

      fg_stall(delay);
      fg_setdacs(0,NDACS,(unsigned char *)dacs2);
   }
}
