#include "opendoor.h"      /* Header file for OpenDoors functions */
#include "interbbs.h"      /* Header file for Inter-BBS Toolkit functions */

#include <string.h>        /* Other required C header files */
#include <io.h>
#include <stdio.h>
#include <sys\stat.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>


/* Hard-coded configurable constants */
#define HIGH_SCORES           15       /* Number of high scores in list */
#define TICK_DELAY            1        /* Delay between each game tick */
#define INITIAL_COURSE_WIDTH  30       /* Initial width of ski course */
#define MINIMUM_COURSE_WIDTH  4        /* Minimum width of course */
#define DECREASE_WIDTH_AFTER  100      /* # of ticks before course narrows */
#define CHANGE_DIRECTION      10       /* % of ticks course changes direction */
#define MAX_NAME_SIZE         35       /* Maximum characters in player name */
#define WAIT_FOR_FILE         10       /* Time to wait for access to file */
#define SCORE_FILENAME   "SKIGAME.DAT" /* Name of high score file */


/* High-score file format */
typedef struct
   {
   char szPlayerName[MAX_NAME_SIZE + 1];
   long lnHighScore;
   char szSystemName[SYSTEM_NAME_CHARS + 1];
   char szPlayerLocation[LOCATION_CHARS + 1];
   } tHighScoreRecord;

typedef struct
   {
   tHighScoreRecord aRecord[HIGH_SCORES];
   } tHighScoreFile;


/* Prototypes for functions defined and used in this file */
void DoMaintenance(void);
void DoDoor(void);
int OpenAndReadHighScores(tHighScoreFile *pFileContents);
void CloseHighScores(int hHighScoreFile);
void WriteHighScores(int hfHighScoreFile, tHighScoreFile *pFileContents);
int OpenExclusiveFile(char *pszFileName, int nAccess, time_t SecondsToWait);
tBool FileExists(char *pszFileName);
void ShowHighScores(void);
void PlayGame(void);
void DisplayInstructions(void);
void SpaceRight(int nColumns);
tBool AddHighScore(tHighScoreFile *pHighScores, tHighScoreRecord *pScoreRecord);
long StartTimer(void);
void WaitForTimerToElapse(long lnStartValue, long lnDuration);


/* main() - Program's execution begins here */
int main(int argc, char *argv[])
   {
   /* If the MAINT command-line parameter was specified */
   if(argc >= 2 && stricmp(argv[1], "maint") == 0)
      {
      /* Operate in maintenance mode */
      DoMaintenance();
      }
   else
      {
      /* Otherwise, operate in door mode */
      DoDoor();
      }

   /* Program's execution ends here */
   return(1);
   }


/* DoMaintenance() - Call from main() to operate program in maintenance mode */
void DoMaintenance(void)
   {
   static tIBInfo InterBBSInfo;
   static tIBResult Result;
   static tHighScoreFile HighScoreFile;
   static tHighScoreFile NewScores;
   int iHighScore;
   int hFile;

   printf("Performing maintenance...\n");

   /* Open high score file */
   if((hFile = OpenAndReadHighScores(&HighScoreFile)) == -1)
      {
      printf("Unable to access high scores file\n");
      return;
      }

   /* Read Inter-BBS information from configuration file */
   if(IBReadConfig(&InterBBSInfo, "EXAMPLE.CFG") != eSuccess)
      {
      printf("Unable to access configuration file\n");
      return;
      }

   /* Loop for each new inbound message */
   while((Result = IBGet(&InterBBSInfo, &NewScores, sizeof(tHighScoreFile)))
         == eSuccess)
      {
      printf("Processing Inter-BBS message...\n");

      /* Attempt to add each new score to our file */
      for(iHighScore = 0; iHighScore < HIGH_SCORES; ++iHighScore)
         {
         AddHighScore(&HighScoreFile, &NewScores.aRecord[iHighScore]);
         }
      }

   /* Check that IBGet() failed because there are no more new inbound msgs */
   if(Result != eNoMoreMessages)
      {
      printf("Unable to access inbound messages\n");
      return;
      }

   /* Save and close file */
   WriteHighScores(hFile, &HighScoreFile);
   CloseHighScores(hFile);

   /* Send updated high score list to all other systems */
   if(IBSendAll(&InterBBSInfo, &HighScoreFile, sizeof(tHighScoreFile))
    != eSuccess)
      {
      printf("Unable to send Inter-BBS message\n");
      }
   }


/* DoDoor() - Call from main() to operate program in door mode */
void DoDoor(void)
   {
   char cMenuChoice;

   /* Initialize OpenDoors */
   od_init();

   /* Loop until the user chooses to exit the door */
   do
      {
      /* Display menu */
      od_clr_scr();
      od_printf("`bright white`SKIGAME DOOR\n\r");
      od_printf("------------\n\n\r");
      od_printf("`dark green`[`bright green`P`dark green`] Play Game\n\r");
      od_printf("[`bright green`V`dark green`] View High Scores\n\r");
      od_printf("[`bright green`R`dark green`] Return To BBS\n\r\n\r");
      od_printf("`bright white`Please Press [P], [V] or [R]\n\r");

      /* Get user's choice for list of case-insensitive possibilities */
      cMenuChoice = od_get_answer("PVR");

      /* Perform appropriate action based on user's choice */
      switch(cMenuChoice)
         {
         case 'P':
            /* If user chose to play the game */
            PlayGame();
            break;

         case 'V':
            /* If user chose to view high scores */
            ShowHighScores();
            break;

         case 'R':
            /* If user chose to return to BBS */
            od_printf("\n\rBye from SKIGAME!\n\r");
            break;
         }
      } while(cMenuChoice != 'R');

   /* Exit door at errorlevel 10, and do not hang up */
   od_exit(10, FALSE);
   }


/* OpenAndReadHighScores() - Opens high score file and reads contents. If */
/*                           file does not exist, it is created. File is  */
/*                           locked to serialize access by other nodes on */
/*                           this system.                                 */
int OpenAndReadHighScores(tHighScoreFile *pFileContents)
   {
   int hFile;
   int iHighScore;

   /* If high score file does not exist */
   if(!FileExists(SCORE_FILENAME))
      {
      /* Open and create it */
      hFile = OpenExclusiveFile(SCORE_FILENAME, O_BINARY|O_RDWR|O_CREAT,
                                WAIT_FOR_FILE);

      /* If open was successful */
      if(hFile != -1)
         {
         /* Initialize new high score list */
         for(iHighScore = 0; iHighScore < HIGH_SCORES; ++iHighScore)
            {
            pFileContents->aRecord[iHighScore].lnHighScore = 0L;
            }

         /* Write high score list to the file */
         WriteHighScores(hFile, pFileContents);
         }
      }

   /* If high score file does exit */
   else
      {
      /* Open the existing file */
      hFile = OpenExclusiveFile(SCORE_FILENAME, O_BINARY|O_RDWR,
                                WAIT_FOR_FILE);

      /* Read the contents of the file */
      if(read(hFile, pFileContents, sizeof(tHighScoreFile)) !=
         sizeof(tHighScoreFile))
         {
         close(hFile);
         hFile = -1;
         }
      }

   /* Return handle to high score file, if avilable */
   return(hFile);
   }


/* FileExists() - Returns true if file exists, otherwise returns false */
tBool FileExists(char *pszFileName)
   {
   return(access(pszFileName, 0) == 0);
   }


/* OpenExclusiveFile() - Opens a file for exclusive access, waiting if the */
/*                       file is not currently available.                  */
int OpenExclusiveFile(char *pszFileName, int nAccess, time_t SecondsToWait)
   {
   int hFile;
   time_t StartTime = time(NULL);

   for(;;)
      {
      /* Attempt to open file */
      hFile = open(pszFileName, nAccess | O_DENYALL | O_EXCL, S_IREAD|S_IWRITE);

      /* If file was opened successfuly, then exit */
      if(hFile != -1) break;

      /* If open failed, but not due to access failure, then exit */
      if(errno != EACCES) break;

      /* If maximum time has elapsed, then exit */
      if(StartTime + SecondsToWait < time(NULL)) break;

      /* Give the OpenDoors kernel a chance to execute before trying again */
      od_kernal();
      }

   /* Return handle to file, if opened */
   return(hFile);
   }


/* CloseHighScores() - Closes the high score file, allowing other nodes on */
/*                     system to access it.                                */
void CloseHighScores(int hHighScoreFile)
   {
   if(hHighScoreFile != -1 )
      {
      close(hHighScoreFile);
      }
   }


/* WriteHighScores() - Writes the information from pFileContents to the */
/*                     high score file.                                 */
void WriteHighScores(int hHighScoreFile, tHighScoreFile *pFileContents)
   {
   if(hHighScoreFile != -1)
      {
      lseek(hHighScoreFile, 0L, SEEK_SET);
      write(hHighScoreFile, pFileContents, sizeof(tHighScoreFile));
      }
   }


/* ShowHighScores() - Called From DoDoor() to display list of high scores */
void ShowHighScores(void)
   {
   int hFile;
   tHighScoreFile HighScores;
   int iHighScore;

   /* Clear the screen */
   od_clr_scr();

   /* Attempt to read high scores from file */
   hFile = OpenAndReadHighScores(&HighScores);
   CloseHighScores(hFile);

   if(hFile == -1)
      {
      /* If unable to open high score file, display an error message */
      od_printf("`bright red`Unable to access high score file!\n\r");
      }
   else
      {
      /* Display header line */
      od_printf("`bright green`Player                    Score     "
                "Player Location   BBS Name`dark green`\n\r");

      /* Display high scores */
      for(iHighScore = 0; iHighScore < HIGH_SCORES; ++iHighScore)
         {
         /* Exit loop when we have reached the end of the high scores */
         if(HighScores.aRecord[iHighScore].lnHighScore == 0L) break;

         /* Display next high score */
         od_printf("%-24.24s  %8.8ld  %-16.16s  %-24.24s\n\r",
                   HighScores.aRecord[iHighScore].szPlayerName,
                   HighScores.aRecord[iHighScore].lnHighScore,
                   HighScores.aRecord[iHighScore].szPlayerLocation,
                   HighScores.aRecord[iHighScore].szSystemName);
         }
      }

   /* Wait for user to press a key */
   od_printf("`bright white`\n\rPress [ENTER]/[RETURN] to continue: ");
   od_get_answer("\n\r");
   }


/* PlayGame() - Called from DoDoor() when user chooses to play a game. */
void PlayGame(void)
   {
   int nLeftEdge = 1;
   int nRightEdge = nLeftEdge + 1 + INITIAL_COURSE_WIDTH;
   int nPlayerPos = nLeftEdge + 1 + (INITIAL_COURSE_WIDTH / 2);
   long lnScore = 0;
   int nDistanceSinceShrink = 0;
   tBool bMovingRight = TRUE;
   char cKeyPress;
   tHighScoreRecord ScoreRecord;
   int hFile;
   tHighScoreFile HighScores;
   long lnTimerStart;

   /* Display Instructions to User */
   DisplayInstructions();

   /* Clear the Screen */
   od_clr_scr();

   /* Set current display colour to white */
   od_set_attrib(L_WHITE);

   /* Re-seed random number generator */
   randomize();

   /* Loop until game is over */
   for(;;)
      {
      /* Start timer */
      lnTimerStart = StartTimer();

      /* Display current line */
      SpaceRight(nLeftEdge - 1);
      od_putch(bMovingRight ? '\\' : '/');
      SpaceRight(nPlayerPos - nLeftEdge - 1);
      od_set_attrib(L_YELLOW);
      od_putch('o');
      od_set_attrib(L_WHITE);
      SpaceRight(nRightEdge - nPlayerPos - 1);
      od_putch(bMovingRight ? '\\' : '/');

      /* Loop for each key pressed by user */
      while((cKeyPress = od_get_key(FALSE)) != '\0')
         {
         if(cKeyPress == 'q' || cKeyPress == 'Q')
            {
            /* Move left */
            --nPlayerPos;
            }
         else if(cKeyPress == 'w' || cKeyPress == 'W')
            {
            /* Move right */
            ++nPlayerPos;
            }
         }

      /* Check whether course should turn */
      if((rand() % 100) < CHANGE_DIRECTION)
         {
         bMovingRight = !bMovingRight;
         }
      else
         {
         /* If no change in direction, then position moves */
         /* Adjust course position appropriately */
         if(bMovingRight)
            {
            ++nLeftEdge;
            ++nRightEdge;
            }
         else
            {
            --nLeftEdge;
            --nRightEdge;
            }
         }

      /* Check whether course size should shink */
      if(++nDistanceSinceShrink >= DECREASE_WIDTH_AFTER)
         {
         /* Reset distance */
         nDistanceSinceShrink = 0;

         /* Randomly choose a side to shrink */
         if((rand() % 100) < 50)
            {
            ++nLeftEdge;
            }
         else
            {
            --nRightEdge;
            }
         }

      /* Change course direction if it collides with edge of screen */
      if(nLeftEdge < 1)
         {
         bMovingRight = TRUE;
         ++nLeftEdge;
         ++nRightEdge;
         }
      else if(nRightEdge > 79)
         {
         bMovingRight = FALSE;
         --nLeftEdge;
         --nRightEdge;
         }

      /* Check that player is still within the course */
      if(nPlayerPos <= nLeftEdge || nPlayerPos >= nRightEdge)
         {
         /* Player has left course - game over! */
         od_clr_scr();
         od_printf("`flashing bright red`       !!! Game Over !!!\n\r\n\r");
         od_printf("`dark green`You have veered off the course!\n\r\n\r");
         od_printf("Your Score is: %ld\n\r", lnScore);

         /* Create a score record */
         ScoreRecord.lnHighScore = lnScore;
         strncpy(ScoreRecord.szPlayerName, od_control.user_name, MAX_NAME_SIZE);
         ScoreRecord.szPlayerName[MAX_NAME_SIZE] = '\0';
         strncpy(ScoreRecord.szPlayerLocation, od_control.user_location,
                 LOCATION_CHARS);
         ScoreRecord.szPlayerLocation[LOCATION_CHARS] = '\0';
         strncpy(ScoreRecord.szSystemName, od_control.system_name,
                 SYSTEM_NAME_CHARS);
         ScoreRecord.szSystemName[SYSTEM_NAME_CHARS] = '\0';

         /* Attempt to read high scores from file */
         hFile = OpenAndReadHighScores(&HighScores);

         if(hFile == -1)
            {
            /* If unable to open high score file, display an error message */
            od_printf("`bright red`Unable to access high score file!\n\r");
            }
         else
            {
            /* Check whether user made it to high score list */
            if(AddHighScore(&HighScores, &ScoreRecord))
               {
               od_printf("Congratulations! You have set a new record!\n\r");
               /* If so, write the new high score list */
               WriteHighScores(hFile, &HighScores);
               }

            /* Close and unlock file */
            CloseHighScores(hFile);
            }

         /* Wait for user to press enter */
         od_printf("`bright white`\n\rPress [ENTER]/[RETURN] to return to menu: ");
         od_get_answer("\n\r");

         return;
         }

      /* Wait for tick time to elapse */
      WaitForTimerToElapse(lnTimerStart, TICK_DELAY);

      /* Increase score */
      ++lnScore;

      /* Move to next line */
      od_printf("\r\n");
      }
   }


/* DisplayInstructions() - Displays brief instructions before game begins */
void DisplayInstructions(void)
   {
   /* Clear the screen */
   od_clr_scr();

   /* Display instructions */
   od_printf("`dark green`Get Ready To Ski!\n\r\n\r");
   od_printf("Press the [`bright green`Q`dark green`] key to ski left\n\r");
   od_printf("Press the [`bright green`W`dark green`] key to ski right\n\r\n\r");
   od_printf("All that you have to do is ski within the slalam course.\n\r");
   od_printf("It may sound easy - but beware - it gets harder as you go!\n\r\n\r");

   /* Wait for user to press enter */
   od_printf("`bright white`Press [ENTER]/[RETURN] to begin game: ");
   od_get_answer("\n\r");
   }


/* SpaceRight() - Moves right the specified number of columns. In ANSI mode, */
/*                uses the move cursor right control sequence. Otherwise,    */
/*                uses od_repeat(), which is optimized for ASCII and AVATAR  */
/*                modes.                                                     */
void SpaceRight(int nColumns)
   {
   /* If we don't have a positive column count, then return immediately */
   if(nColumns <= 0) return;

   /* If operating in ANSI mode */
   if(od_control.user_ansi)
      {
      /* Move cursor right using ESC[nC control sequence */
      od_emulate(27);
      od_emulate('[');
      od_emulate('0' + nColumns / 10);
      od_emulate('0' + nColumns % 10);
      od_emulate('C');
      }

   /* If not operating in ANSI mode */
   else
      {
      od_repeat(' ', nColumns);
      }
   }


/* AddHighScore() - Adds a new score to the high score list, if it is high   */
/*                  enough. Returns TRUE if score is added, FALSE otherwise. */
tBool AddHighScore(tHighScoreFile *pHighScores, tHighScoreRecord *pScoreRecord)
   {
   int iHighScore;
   int iExistingScore;

   /* Loop through each existing high score */
   for(iHighScore = 0; iHighScore < HIGH_SCORES; ++iHighScore)
      {
      /* If new score is greater than or equal to this one, then its */
      /* position has been found.                                    */
      if(pHighScores->aRecord[iHighScore].lnHighScore <=
       pScoreRecord->lnHighScore)
         {
         /* Move remaining scores down one in list */
         for(iExistingScore = HIGH_SCORES - 1; iExistingScore >= iHighScore + 1;
          --iExistingScore)
            {
            pHighScores->aRecord[iExistingScore] =
               pHighScores->aRecord[iExistingScore - 1];
            }

         /* Add new score to list */
         pHighScores->aRecord[iHighScore] = *pScoreRecord;

         /* Return with success */
         return(TRUE);
         }
      }

   /* Score did not make it to list */
   return(FALSE);
   }


long StartTimer(void)
   {
   return(*(long far *)0x46cL);
   }


void WaitForTimerToElapse(long lnStartValue, long lnDuration)
   {
   while(*(long far *)0x46cL >= lnStartValue &&
         lnStartValue + lnDuration >= *(long far *)0x46cL)
      {
      od_kernal();
      }
   }
